/*****************************************************************************
 * File:	utils.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	March 6, 1993
 *
 * Modified:	March 20, 1993, by Rhett "Jonzy" Jones.
 *		Cleaned up some of the routines.
 *
 *		March 28, 1993, by Rhett "Jonzy" Jones.
 *		Added support for varargs.h, which is needed on a sun
 *		and included modifying DoSystemCall().
 *
 *		March 28, 1993, by Rhett "Jonzy" Jones.
 *		Fixed a problem using vsprintf() compiled on a sun.
 *
 *		March 30, 1993, by Rhett "Jonzy" Jones.
 *		Rewrote DoSystemCall() and added Mysprint().
 *
 *		April 4, 1993, by Rhett "Jonzy" Jones.
 *		Added the routine NumberOfLines().
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		April 30, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE which is a rewrite
 *		of StrToLower() to make this routine ISO-8859-1 clean.
 *		Thank you Maes@elec.ucl.ac.BE for the code.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones with code from
 *		Maes@elec.ucl.ac.BE.  Added to the code strncoll()
 *		to support the use of strncmp() in an ISO 8859-1
 *		character set environment.
 *		Optimized some of the code.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones, with code from
 *		Maes@elec.ucl.ac.BE.  Added the code StrCmp() and
 *		StrnCmp() to support both ASCII and ISO-XXXX-Y
 *		character sets.  Changed all calls to strcmp() to
 *		StrCmp().  Removed use of LOCALE, and no longer use
 *		strncoll().
 *
 *		May 9, 1994, by Rhett "Jonzy" Jones.
 *		Fixed a bug that slipped by, and was reported along
 *		with the fix by Maes@elec.ucl.ac.BE.  Pascal your a
 *		life saver.  Thanks for your assistance.
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code for lint.
 *
 *		May 23, 1994, by Rhett "Jonzy" Jones.
 *		Added support for writting to 'rptPtr', which points to
 *		the logFile file, if need be.
 *
 *		Jun 3, 1994, by Rhett "Jonzy" Jones.
 *		Added the include sprint.h which is built via "make config"
 *		to assist in local configuration.
 *
 *		Jun 7, 1994, by Rhett "Jonzy" Jones.
 *		Modified Mysprint() to make it somewhat cleaner.
 *
 * Description:	Contains various utlity routines.  See "Routines" for more
 *		information.
 *
 * Routines:	char	*StrToLower(char *str);
 *		long	GetLong(FILE *fp);
 *		int	GetInt(FILE *fp);
 *		char	*GetStr(FILE *fp,char *str,int limit);
 *		int	Str2Int(char *s);
 *		char	*Reverse(char *s, char *r);
 *		int	StrCmp(char *s1,char *s2);
 *		int	StrnCmp(char *s1,char *s2,int n);
 *		short	StrRcmp(char *s1, char *s2);
 *		char	*Mysprint(va_dcl va_alist | char *format,...);
 *		int	DoSystemCall(char *command);
 *		char	*MyStrTok(char *str,char delimeter);
 *		char	*OnlyDigits(char *s);
 *		char	*RemoveCRLF(char *line);
 *		int	strncoll(char *s1,char *s2,int n);	* Not used. *
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#include "sprint.h"

#include <stdio.h>
#include <ctype.h>

extern FILE	*rptPtr;	/* Defined in "jughead.c". */

/*****************************************************************************
 * NumberOfLines returns the number of lines in the file "fileName".
 ****************************************************************************/
long NumberOfLines(fileName)
	char		*fileName;	/* Name of the file to count lines. */
{	FILE		*fp;		/* Pointer the the file 'fileName'. */
	long		numLines = 0;	/* The number of lines to return. */
	register int	c;

	if (fp = fopen(fileName,"r"))
		{
		while ((c = fgetc(fp)) != EOF)
			if (c == '\n')
				numLines++;
		(void)fclose(fp);
		}
	else
		(void)fprintf(rptPtr,"error: NumberOfLines could not open %s\n",fileName);
	return(numLines);

}	/* NumberOfLines */

/*****************************************************************************
 * StrToLower returns the string 'str' passed in, which gets converted to
 * lower case.  This routine supports the ISO Latin character set 1.
 * Thank you Maes@elec.ucl.ac.BE for the code modifications.
 ****************************************************************************/
char *StrToLower(str)
	char		*str;	/* The string we are converted to lower case. */
{	register char	*s;	/* Pointer to the position we are converting. */
	unsigned char	c;	/* To support ISO latin set 1 characters. */

	for (s = str; *s; s++)
		if (isascii(*s))
			*s = tolower(*s);
		else	switch (c = (unsigned char)*s)
			{
			case 208:	/* ETH */
			case 215:	/* MULT */
				break;	/* Nothing to do with these. */
			default:
				if (c > 191 && c < 222)	/* Uppercase */
					*s = c + 32;
			}

	return(str);

}	/* StrToLower */

/*****************************************************************************
 * GetLong returns a long acquired from the stream 'fp'.  This routine
 * supports a '+' or '-' as input as the first character to set the sign.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
long GetLong(fp)
	FILE	*fp;		/* The stream we are getting input from. */
{	long	i;		/* The value we return. */
	int	c,		/* A character from the stream. */
		sign = 1;	/* Initial sign of the result. */

	while (isspace(c = fgetc(fp)));		/* Skip any whitespace. */
	if (c == '+' || c == '-')		/* Check the sign. */
		{
		sign = (c == '+') ? 1 : -1;
		c = fgetc(fp);
		}
	for (i = 0; isdigit(c); c = fgetc(fp))	/* Get the number. */
		i = i * 10 + c - '0';
#if(0)
	if (c != EOF)				/* Put the character back. */
		ungetc(c,fp);
#endif
	return(i * sign);

}	/* GetLong */

/*****************************************************************************
 * GetInt returns an int acquired from the stream 'fp'.  This routine
* supports a '+' or '-' as input as the first character to set the sign.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
int GetInt(fp)
	FILE	*fp;		/* The stream we are getting input from. */
{	long	i;		/* The value we return. */
	int	c,		/* A character from the stream. */
		sign = 1;	/* Initial sign of the result. */

	while (isspace(c = fgetc(fp)));		/* Skip any whitespace. */
	if (c == '+' || c == '-')		/* Check the sign. */
		{
		sign = (c == '+') ? 1 : -1;
		c = fgetc(fp);
		}
	for (i = 0; isdigit(c); c = fgetc(fp))	/* Get the number. */
		i = i * 10 + c - '0';
#if(0)
	if (c != EOF)				/* Put the character back. */
		ungetc(c,fp);
#endif
	return(i * sign);


}	/* GetInt */

/*****************************************************************************
 * GetStr returns a string acquired from the stream 'fp'.
 * This routine was written because of the problems with fscanf().
 ****************************************************************************/
char *GetStr(fp,str,limit)
	FILE		*fp;	/* The stream we are getting input from. */
	char		*str;	/* Where to place the characters. */
	short		limit;	/* Max chars to get. */
{	register int	i;	/* A loop counter. */
	int		c;	/* A character from the stream. */


	while (isspace(c = fgetc(fp)));		/* Skip any whitespace. */
	for (i = 0; i < limit && !isspace(c); c = fgetc(fp))
		str[i++] = (char)c;
	str[i] = '\0';
#if(0)
	if (c != EOF)				/* Put the character back. */
		ungetc(c,fp);
#endif
	return(str);

}	/* GetStr */

/*****************************************************************************
 * Str2Int returns the integer equivilent of str.
 ****************************************************************************/
int Str2Int(s)
	char	*s;	/* The string we are converting to an integer. */
{	int	i;	/* The value returned by this routine. */

	for (i = 0; *s && isdigit(*s); s++)
		i = i * 10 + *s - '0';
	return(i);

}	/* Str2Int */

/*****************************************************************************
 * Reverse copies 's' into 'r' in reverse order and returns 'r'.
 ****************************************************************************/
char *Reverse(s,r)
	char		*s,	/* The string in forward order. */
			*r;	/* The result in reverse order. */
{	register int	i;	/* A loop counter. */
	int		l;	/* The length of 's'. */

	for (l = strlen(s), i = 0; i < l; i++)
		r[i] = s[l - i - 1];
	r[l] = '\0';
	return(r);

}	/* Reverse */

/*****************************************************************************
 * StrCmp compares the strings s1 and s2, returns 0 if they are equal,
 * a value less then 0 if s1 is less than s2, a value greater then 0
 * if s1 is greater then s2.  This routine was written to support both
 * the ASCII and ISO-XXXX-Y character sets.
 ****************************************************************************/
int StrCmp(s1,s2)
	char	*s1,	/* The first string. */
		*s2;	/* The second string. */
{
	for ( ;(unsigned char)*s1 == (unsigned char)*s2; s1++, s2++)
		if (!*s1)
			return(0);
	return((unsigned char)*s1 - (unsigned char)*s2);

}	/* StrCmp */

/*****************************************************************************
 * StrnCmp compares at most n characters while comparing the strints s1
 * and s2, returns 0 if they are equal, a value less then 0 if s1 is less
 * than s2, a value greater than 0 if s1 is greater than s2.  This routine
 * was written to support both the ASCII and ISO-XXXX-Y character sets.
 * This routine returns -1 if n is less than 0, for safety only.
 ****************************************************************************/
int StrnCmp(s1,s2,n)
	char	*s1,	/* The first string. */
		*s2;	/* The second string. */
	int	n;	/* The number of characters to compare. */
{
	if (n < 0)	/* Better safe than sorry. */
		return(-1);

	for ( ; n; n--, s1++, s2++)
		if ((unsigned char)*s1 != (unsigned char)*s2)
			return((unsigned char)*s1 - (unsigned char)*s2);
	return(0);

}	/* StrnCmp */

/*****************************************************************************
 * StrRcmp returns true if 's1' in reverse is contained at the begining of
 * 's2' in reverse.  Otherwise it returns false.
 * Example: say 's1' = ".some.site", and 's2' = "this.some.site", if you reverse
 * both of these strings you get
 * 's1' = "etis.emos."
 * 's2' = "etis.emos.siht"  which evalutaes to true.  If 's1' is not contained
 * in 's2' we return false.
 * This routine was written to support the use of domains as hosts contained
 * in 'hosts2search'.
 ****************************************************************************/
short StrRcmp(s1,s2)
	char	*s1,	/* The string which must be in s2 (in reverse order). */
		*s2;	/* The string which must contain s1. */
{	char	r1[255],/* The string s1 in reverse. */
		r2[255];/* The string s2 in reverse. */

	for (s1 = Reverse(s1,r1), s2 = Reverse(s2,r2); *s1 && *s2; s1++, s2++)
		if (*s1 != *s2)
			return(0);
	if (*s1 && !*s2)
		return(0);
	return(1);
			
}	/* StrRcmp */

/*VARARGS*/

/*****************************************************************************
 * Mysprint takes a variable number of arguments similar to fprintf() or
 * or vsprintf() inparticular, and puts the formated information into a string
 * and returns the string.  This routine make use of the defines USE_VARARGS_H
 * and VSPRINTF_RETURNS_STR.  If USE_VARARGS_H is defined we use <varargs.h>
 * otherwise <stdarg.h> gets included and used.  If VSPRINTF_RETURNS_STR is
 * define it is assumed the call to vsprintf() returns a char* otherwise it
 * must return an int.
 ****************************************************************************/
#ifdef USE_VARARGS_H
 char *Mysprint(va_alist)
	va_dcl
{
	char		*format;	/* Format of the arguments. */
#else /* We are using <stdarg.h>. */
 char *Mysprint(char *format,...)
{
#endif /* Using <stdarg.h>. */
	static char	str[1024];	/* The string we send to the system. */
#ifdef VSPRINTF_RETURNS_STR
	char		*result;	/* The result of calling vsprintf. */
#else /* vsprintf() returns an int. */
	int		result;		/* The result of calling vsprintf. */
#endif /* vsprintf() returns an int. */
	va_list		ap;		/* The arguments beyond 'format'. */

#ifdef USE_VARARGS_H
	va_start(ap);
	format = va_arg(ap,char *);
#else  /* We are using <stdarg.h>. */
	va_start(ap,format);
#endif  /* Using <stdarg.h>. */

	result = vsprintf(str,format,ap);
	va_end(ap);

#ifdef VSPRINTF_RETURNS_STR
	if (StrCmp((char *)result,str))
		{
		(void)fprintf(rptPtr,"error: Mysprint had vsprintf fail returning a (char *).\n");
		return((char *)0);
		}
#else /* vsprintf() returns an int. */
	if (result < 0)
		{
		(void)fprintf(rptPtr,"error: Mysprint had vsprintf fail returning an (int).\n");
		return((char *)0);
		}
#endif /* vsprintf() returns an int. */

	return(str);

}	/* Mysprint */

/*ARGSUSED*/

/*****************************************************************************
 * DoSystemCall does a system on the result of 'format'.
 ****************************************************************************/
 int DoSystemCall(command)
	char	*command;		/* The command to send to system(). */
{	int	result,			/* The result of calling vsprintf. */
		error = 0;		/* Did things turn out ok? */

	if (command)
		if (result = system(command))
			error = fprintf(rptPtr,"error: DoSystemCall [%s] failed with %d\n",command,result);
	return(error);

}	/* DoSystemCall */

/*****************************************************************************
 * MyStrTok returns breaks 'str' into tokens as per the token seperator
 * 'delimeter'.  This routine basicly does the samething as strtok().
 ****************************************************************************/
char *MyStrTok(str,delimeter)
	char		*str,		/* The string to tokenize. */
			delimeter;	/* The delimeter character. */
{	static char	*theStr,	/* Keep a copy the string to tokenize. */
			*s,		/* A temporary pointer. */
			*token;		/* The token to return. */

	if (str)	/* A new string to tokenize. */
		theStr = str;

	token = theStr;	/* This is the token. */

	/* Break up theStr into a token. */
	if (*theStr != delimeter)
		for (s = theStr; *s && *s != delimeter; s++)
			;

	/* Nill out the token and do some cleaning up. */
	*s++ = '\0';
	theStr = s;
	return(token);

}	/* MyStrTok */

/*****************************************************************************
 * OnlyDigits returns a null terminated string containing only the digit
 * characters contained in 's'.  What we do is convert the first non-digit
 * character in 's' to the null character and return the string.
 ****************************************************************************/
char *OnlyDigits(s)
	char	*s;		/* The input string. */
{	char	*digits;	/* The string with digits only. */

	for (digits = s; *s; s++)
		if (!isdigit(*s))
			{
			*s = '\0';
			return(digits);
			}
			
	return(digits);

}	/* OnlyDigits */

/*****************************************************************************
 * RemoveCRLF removes the carriage return or line feed by changing it to
 * the null character.
 ****************************************************************************/
char *RemoveCRLF(line)
	char		*line;		/* The line we are dealing with. */
{	register char	*s;		/* Pointer into 'line'. */

	for (s = line; *s && *s != '\r' && *s != '\n'; s++)
		;	/* Go to the CR or LF position. */
	*s = '\0';
	return(s);

}	/* RemoveCRLF */

#if(0)  /* As of May 6, 1994 this is no longer needed. */
/*****************************************************************************
 * strncoll compares 'n' characters of the strings s1 and s2.  This routine
 * returns a number less than 0 if s1 is less than s2, returns a number
 * greater than zero is s1 is greater than s2, and 0 if s1 equals s2.
 * The strings s1 and s2 are interpreted as appropriate to the LC_COLLATE
 * category of the current locale.  This routine is only compiled if
 * LOCALE is defined.  Thank you Maes@elec.ucl.ac.BE for the code.
 ****************************************************************************/
int strncoll(s1,s2,n)
	char	*s1,		/* The first string we compare. */
		*s2;		/* The other string for comparison. */
	int	n;		/* The number of characters to compare. */
{	char	*t1,		/* A copy of s1. */
		*t2;		/* A copy of s2. */
	int	result;		/* The result of the comparison. */

	t1 = (char *) malloc((unsigned)n+1);
	t2 = (char *) malloc((unsigned)n+1);

	strncpy(t1, s1, n);
	*(t1+n) = '\0';
	strncpy(t2, s2, n);
	*(t2+n) = '\0';

	result = StrnCmp(t1, t2);

	free((char *)t1); 
	free((char *)t2);

	return (result);

} 	/* strncoll */
#endif
