/*****************************************************************************
 * File:	tree.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	February 27, 1993
 *
 * Modified:	March 6, 1993, by Rhett "Jonzy" Jones
 * 		to support positions, by adding InList() and BuildList().
 *		Also altered CreateBranch(), BuildTree(), and PrintTree().
 *
 *		March 9, 1993, by Rhett "Jonzy" Jones.
 *		Modified BuildList() and added DestroyList(), and
 *		NumberOfListNodes().
 *
 *		March 10, 1993, by Rhett "Jonzy" Jones.
 *		Added BinarySearch(), LessOrEqual(), and NumberOfLeafs().
 *
 *		March 13, 1993, by Rhett "Jonzy" Jones.
 *		Modified PrintTree(), so we can also print the host tree
 *		in a more readable format, and added LongestStr2tab() and
 *		PrintHostTree().
 *
 *		March 16, 1993, by Rhett "Jonzy" Jones.
 *		Changed the type returned by BinarySearch() from a short
 *		to a long.  Oops.
 *
 *		May 5, 1993, by Rhett "Jonzy" Jones.
 *		Modified LessOrEqual() and BinarySearch() to support
 *		partial word searches.
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		May 1, 1994, by Rhett "Jonzy" Jones.
 *		Fixed a bug when looking for "a*", it would not find "a".
 *		Thank you Kuypers@sri.ucl.ac.BE for pointing this out.
 *		Also cleaned up the code by changing asterikPos in
 *		BinarySearch() and numChars in LessOrEqual() from type
 *		size_t to an int.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Optimized some of the code.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Changed all calls to strcmp() to StrCmp() and strncmp()
 *		to StrnCmp() to support both ASCII and ISO-XXXX-Y character
 *		sets.  Removed use of LOCALE.
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code for lint.
 *
 *		May 23 1994, by Rhett "Jonzy" Jones.
 *		Added use of rptPtr in PrintTree() and PrintHostTree()
 *		to support use of the new -l logFile flag.
 *		Modified PrintTree() and PrintHostTree() to print
 *		four leading spaces prior to the data for readability.
 *
 *		June 17, 1994, by Rhett "Jonzy" Jones.
 *		Added #ifdef MAX to undefine it for use on the NeXT.
 *		This was done to prevent the warning message and to
 *		ensure my MAX() definition gets used on all platforms.
 *
 * Description:	Contains list, tree and binary tree routines.
 *
 * Routines:	 	short    InList(TreeType *node,long where);
 *			void	 DestroyList(ListType *list);
 *		static	void     BuildList(TreeType *node,long where);
 *		static	short	 LessOrEqual(char *a,char *b,char * partialWord,
 *						int numChars);
 *			long	 BinarySearch(char *what2Find,Element *data,
 *						long size,
 *					      char **asterik,int *asterikPos);
 *			long	 NumberOfLeafs(TreeType *node);
 *		static	short    WhatDirection(char *s1,char *s2);
 *			TreeType *WhatBranch(TreeType *node,char *word);
 *		static	TreeType *CreateBranch(char *word,long where);
 *			void     BuildTree(TreeType **root,char *word,
 *						long where);
 *			long	 NumberOfListNodes(ListType *positions);
 *		static	int	 LongestStr2tab(Tree *node);
 *		static	void	 PrintHostTree(Tree *node,int maxHostLen);
 *			void     PrintTree(TreeType *node,int hostTre);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#include <stdio.h>
#ifdef NEXT
#	include <libc.h>
#else
#	include <stdlib.h>
#endif
#include <string.h>
#include "tree.h"

#define GOLEFT		-1
#define GORITE		1
#define THISONE		0
#ifdef MAX
#	undef MAX
#endif
#define MAX(a,b)	((a) > (b)) ? (a) : (b)

static short	limb = 0;		/* What limb do we insert on? */
       TreeType	*root = (TreeType *)NIL;/* The root of the tree. */
static TreeType	*lastNode = (TreeType *)NIL;	/* The last tree node we looked at. */
static ListType	*lastList = (ListType *)NIL;	/* The last list node we looked at. */

extern long	lineNumber;		/* Defined in "dirTree.c". */

extern int	debug;			/* Defined in "jughead.c". */
extern FILE	*rptPtr;		/* Defined in "jughead.c". */

/*****************************************************************************
 * InList return true if 'where' is already in the list of the tree nodes.
 * Otherwise if returns false.
 ****************************************************************************/
short InList(node,where)
	TreeType	*node;		/* The tree node we looking at. */
	long		where;		/* Index we are looking for. */
{	ListType	*positions;	/* List with the indexes. */

	for (positions = node->positions; positions; positions = positions->next)
		{
		lastList = positions;
		if (positions->where == where)
			return(1);
		}
	return(0);

}	/* InList */

/*****************************************************************************
 * DestroyList destroys the list 'list' by freeing up the memory occupied by
 * the entire list.
 ****************************************************************************/
void DestroyList(list)
	ListType	*list;	/* The list to destroy. */
{	ListType	*node;	/* The node to free up. */

	while (list)
		{
		node = list;
		list = list->next;
		free((char *)node);
		}

}	/* DestroyList */

/*****************************************************************************
 * BuildList returns the head of a list which gets built.  Each node contains
 * 'where' and gets appended to the end of the list.
 ****************************************************************************/
ListType *BuildList(node,where)
	ListType	*node;		/* The tree node we looking at. */
	long		where;		/* Index we are looking for. */
{	ListType	*positions;	/* List with the positions. */

	if (positions = (ListType *)malloc(sizeof(ListType)))
		{
		positions->where = where;
		positions->next = (ListType *)NIL;
		if (!node)
			return(lastList = positions);
		else
			{
			lastList->next = positions;
			lastList = positions;
			return(node);
			}
		}
	else
		(void)fprintf(rptPtr,"error: BuildList could not get memory for the index.\n");

	return(node);		/* We better never get here. */

}	/* BuildList */

/*****************************************************************************
 * LessOrEqual returns true if string 'a' is less than or equal to string 'b'.
 * Either StrnCmp() or StrCmp() is used for the string comparison depending
 * on the value of 'partialWord'.
 ****************************************************************************/
static short LessOrEqual(a,b,partialWord,numChars)
	char	*a,		/* The first string we a comparing. */
		*b,		/* The other string we a comparing. */
		*partialWord;	/* Is this a partial word search? */
	int	numChars;	/* Number of characters to look at. */
{
	if (partialWord)
		return(StrnCmp(a,b,numChars) <= 0);
	else
		return(StrCmp(a,b) <= 0);

}	/* LessOrEqual */

/*****************************************************************************
 * BinarySearch returns the position in 'data' where 'what2Find' exists or
 * should exist.  The search is done when 'leftBounds' is greater than  or
 * equal to 'rightBounds', or if we are doing a partial word search which
 * 'asterik' tells us, we return as soon as StrnCmp() finds a match.
 ****************************************************************************/
long BinarySearch(what2Find,data,size,asterik,asterikPos)
	char		*what2Find;	/* The string we are looking for. */
	Element		*data;		/* An array of strings in sorted order. */
	long		size;		/* The number of elements in 'data'. */
	char		**asterik;	/* Do we have a partial word search? */
	int		*asterikPos;	/* The position of the asterik. */
{	register long	leftBounds,	/* The left side of the search. */
			rightBounds,	/* The right side of the search. */
			midPoint;	/* The mid-point of the search. */

	/* See if this is a partial word search. */
	if (*asterik = strchr(what2Find,'*'))
		*asterikPos = (int)(*asterik - what2Find);

	/* Start the bounds as the entire array. */
	leftBounds = 0;
	rightBounds = size - 1;

	/* Keep finding the center of the bounds until the bounds converge. */
	while (leftBounds < rightBounds)
		{
		midPoint = (leftBounds + rightBounds) / 2;
		if (*asterik)
			if (!StrnCmp(what2Find,data[midPoint].word,*asterikPos))
				return(midPoint);
		if (LessOrEqual(what2Find,data[midPoint].word,*asterik,*asterikPos))
			rightBounds = midPoint;
		else
			leftBounds = midPoint + 1;
		}

	if ((*asterik && !StrnCmp(what2Find,data[leftBounds].word,*asterikPos)) ||
	                 !StrCmp(what2Find,data[leftBounds].word))
		return(leftBounds);

	return(-1);

}	/* BinarySearch */

/*****************************************************************************
 * NumberOfLeafs returns the nuber of the leaves contained in the tree 'node'.
 ****************************************************************************/
long NumberOfLeafs(node)
	TreeType	*node;		/* The node to have summed. */
{
	if (!node)
		return(0);
	else
		return(1 + NumberOfLeafs(node->left) + NumberOfLeafs(node->right));

}	/* NumberOfLeafs */

/*****************************************************************************
 * WhatDirection returns GOLEFT if 's1' is less than 's2', returns GORITE if
 * if 's1' is greater than 's2', or returns THISONE if 's1' is the same as
 * 's2'.
 ****************************************************************************/
static short WhatDirection(s1,s2)
	char	*s1,			/* String contained in the leaf. */
		*s2;			/* String we are looking for. */
{	short	direction;		/* What direction do we look? */

	if ((direction = StrCmp(s1,s2)) < 0)
		return(GOLEFT);
	else if (direction > 0)
		return(GORITE);
	else
		return(THISONE);

}	/* WhatDirection */

/*****************************************************************************
 * WhatBranch returns the limb of the tree containing 'word'.  This routine
 * also assigns to 'limb' the direction we are looking in the tree and assigns
 * to 'lastNode' the last node encountered.  These assignments are done to
 * speed the time required when building the tree.
 ****************************************************************************/
TreeType *WhatBranch(node,word)
	TreeType	*node;		/* The node we are looking at. */
	char		*word;		/* The string we are looking for. */
{
	if (!node)		/* Insert at last position. */
		return(node);
	else switch (WhatDirection(word,node->word))
		{
		case GOLEFT:
			limb = GOLEFT;
			lastNode = node;
			return(WhatBranch(node->left,word));
		case GORITE:
			limb = GORITE;
			lastNode = node;
			return(WhatBranch(node->right,word));
		case THISONE:
			return(node);
		default:
			(void)fprintf(rptPtr,"warning: WhatBranch encountered bad default.\n");
			return((TreeType *)NIL);	/* I hope not. */
		}

}	/* WhatBranch */

/*****************************************************************************
 * CreateBranch returns a node in the tree containing 'word'.
 ****************************************************************************/
static TreeType *CreateBranch(word,where)
	char		*word;		/* The string to put in this node. */
	long		where;		/* Index of the line the word is from. */
{	TreeType	*node;		/* The node we are creating. */
	ListType	*positions;	/* List with the positions. */

	if (debug)
		(void)fprintf(rptPtr,"CreateBranch working on %s %ld\n",word,where);

	if (node = (TreeType *)malloc(sizeof(TreeType)))
		if (node->word = (char *)malloc((unsigned)strlen(word) + 1))
			{
			node->left = node->right = (TreeType *)NIL;
			(void)strcpy(node->word,word);
			if (where < 0)
				node->positions = (ListType *)NIL;
			else if (positions = (ListType *)malloc(sizeof(ListType)))
				{
				positions->where = where;
				positions->next = (ListType *)NIL;
				node->positions = positions;
				}
			else
				{
				free((char *)node->word);
				free((char *)node);
				node = (TreeType *)NIL;
				(void)fprintf(rptPtr,"error: CreateBranch could not get memory for the index.\n");
				}
			}
		else
			{
			free((char *)node);
			node = (TreeType *)NIL;
			(void)fprintf(rptPtr,"error: CreateBranch could not get memory for the word [%s].\n",word);
			}
	else
		(void)fprintf(rptPtr,"error: CreateBranch could not get memory for the node.\n");

	return(node);

}	/* CreateBranch */

/*****************************************************************************
 * BuildTree adds to the tree pointed to by 'root'.  No node will be added
 * to the tree if 'word' already exists in the tree.
 ****************************************************************************/
void BuildTree(root,word,where)
	TreeType	**root;		/* Root of the tree to build. */
	char		*word;		/* The string to put in the tree. */
	long		where;		/* Index of the line the word is from. */
{	TreeType	*node;		/* The node we are adding. */

	if (!*root)
		{
		if (debug)
			(void)fprintf(rptPtr,"BuildTree creating the root\n");
		*root = CreateBranch(word,where);
		}
	else if (!(node = WhatBranch(*root,word)))
		{
		node = CreateBranch(word,where);
		switch (limb)
			{
			case GOLEFT:
				lastNode->left = node;
				break;
			case GORITE:
				lastNode->right = node;
				break;
			default:
				(void)fprintf(rptPtr,"warning: BuildTree encountered bad default.\n");
			}
		}
	else if (where >= 0)			/* Tree has a list in it. */
		if (!InList(node,where))	/* word is already in the tree. */
			node->positions = BuildList(node->positions,where);
	/* else both word and where are already in the tree. */

}	/* BuildTree */

/*****************************************************************************
 * NumberOfListNodes returns the number of nodes contained in the list
 * 'positions'.
 ****************************************************************************/
long NumberOfListNodes(positions)
	ListType	*positions;	/* List with the indexes. */
{	register long	num;		/* The number of nodes found. */

	for (num = 0; positions; positions = positions->next)
		num++;
	return(num);

}	/* NumberOfListNodes */

/*****************************************************************************
 * LongestStr2tab returns the greatest number of characters prior to the tab
 * character within the tree pointed to by 'node'.  This routine is used to
 * give a formated output of the host and port contained at a given node.
 ****************************************************************************/
static int LongestStr2tab(node)
	TreeType	*node;		/* The tree we are looking at. */
{	static int	maxLen = 0;	/* Maximum length encountered. */
	int		leftLen,	/* The max len found on the left side. */
			riteLen;	/* The max len found on the right side. */

	if (!node)
		return(maxLen);
	else
		{
		leftLen = LongestStr2tab(node->left);
		riteLen = LongestStr2tab(node->right);
		maxLen  = MAX(maxLen,leftLen);
		maxLen  = MAX(maxLen,riteLen);
		maxLen  = MAX(maxLen,strchr(node->word,'\t') - node->word);
		return(maxLen);
		}

}	/* LongestStr2tab */

/*****************************************************************************
 * PrintHostTree is the routine that prints the contents of the tree pointed
 * to by 'node' including the port.  This routine prints the contents of the
 * node such that the hostname will get printed followed by the port, where
 * the port for all lines will be lined up.
 ****************************************************************************/
static void PrintHostTree(node,maxHostLen)
	TreeType	*node;		/* The node to have printed. */
	int		maxHostLen;	/* Max chars to a tab. */
{	char		*tab,		/* Position of the tab. */
			*hStr,		/* The host string. */
			*pStr;		/* The port string. */

	if (!node)
		return;
	else
		{
		PrintHostTree(node->left,maxHostLen);

		/* Break the string up into the host and port parts. */
		hStr = node->word;
		tab = strchr(hStr,'\t');
		*tab = '\0';
		pStr = tab + 1;

		if (maxHostLen)
			(void)fprintf(rptPtr,"    %*.*s\t%8s\n",-maxHostLen,maxHostLen,hStr,pStr),++lineNumber;
		else
			(void)fprintf(rptPtr,"    %s\n",hStr),++lineNumber;

		/* Restore the information in case we want to use it again. */
		*tab = '\t';

		PrintHostTree(node->right,maxHostLen);
		}

}	/* PrintHostTree */

/*****************************************************************************
 * PrintTree is the routine that prints the contents of the tree pointed by
 * 'node' when not printing the port information.
****************************************************************************/
void PrintTree(node,hostTree)
	TreeType	*node;		/* The node to have printed. */
	int		hostTree;	/* Are we printing the host tree? */
{	ListType	*positions;	/* List with the positions. */

	if (hostTree)
		PrintHostTree(node,(hostTree == 2) ? LongestStr2tab(node) : 0);
	else if (!node)
		return;
	else
		{
		PrintTree(node->left,hostTree);
		(void)fprintf(rptPtr,"    %d\t%s",strlen(node->word) + 1,node->word);
		(void)fprintf(rptPtr,"\t%ld",NumberOfListNodes(node->positions));
		for (positions = node->positions; positions; positions = positions->next)
			(void)fprintf(rptPtr,"\t%ld", positions->where);
		(void)fprintf(rptPtr,"\n"),++lineNumber;
		PrintTree(node->right,hostTree);
		}

}	/* PrintTree */

