/*****************************************************************************
 * File:	getargs.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	May 20, 1993
 *
 * Modifed:	May 21, 1993, by Rhett "Jonzy" Jones.
 *		Had to fix the -m flag.  The menus would not be displayed
 *		with or without this flag set.
 *
 *		May 22, 1993, by Rhett "Jonzy" Jones.
 *		Altered GetArguments() to work on a list of hosts and ports
 *		or selection strings.
 *
 *		May 23, 1993, by Rhett "Jonzy" Jones.
 *		Forgot to assign the correct value to 'initialHost', for use
 *		with the -t flag.
 *
 *		November 14, 1993, by Rhett "Jonzy" Jones.
 *		Modified the code to support the 'm' flag when building the
 *		the hash tables.  This way no output will go to the screen
 *		and can thus help speed up a lengthy process.
 *		Support for this was requested by: doylej@liberty.uc.wlu.edu
 *
 *		Added use of "#ifdef NEXT_3.0" to support NeXT running 3.0.
 *		This patch was provided by hoymand@joe.uwex.edu
 *
 *		February 25, 1994, by Rhett "Jonzy" Jones.
 *		Added the use of MAXPROCS and numProcs to support the maximum
 *		number of forked jughead processes.
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Moved many definitions from the Makefile into
 *		jughead.conf.h and now include jughead.conf.h.
 *		Optimized some of the code.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Changed the use of #ifdef NEXT_3.0 to #ifdef NEXT.
 *		Removed use of LOCALE.
 *		Changed the call to strcmp() to StrCmp().
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Added use of the variable 'logFile', to generate a
 *		logFile to assist in logging the results of building
 *		the datafile via a script for use with cron.
 *		Removed support for 'onlyDoHosts' and 'onlyDoHostsT'
 *		which didn't function anymore.
 *		Cleaned up the code for lint.
 *
 *		June 11, 1994, by Rhett "Jonzy" Jones.
 *		Added #ifdef NEED_MALLOC_H to give support for BSD.
 *
 *		June 17, 1994, by Rhett "Jonzy" Jones.
 *		Added the #ifndef NEXT include <unistd.h> to support
 *		NeXT system version 2.1.
 *
 *		June 18, 1994, by Rhett "Jonzy" Jones.
 *		Fixed the most embarassing bug.  Changed "if (logFile)"
 *		to "if (*logFile)" in GetArguments().  I have no idea how
 *		I missed this, nor how long this potential disaster has
 *		even existed.
 *
 *		July 24, 1994, by Rhett "Jonzy" Jones.
 *		Renamed 'searchPort' to 'port2use' and 'maxProcs' to
 *		'maxprocs' for clarification.
 *		Removed inclusion of jughead.conf.h.
 *
 *		September 20, 1994, by Rhett "Jonzy" Jones.
 *		Gave support for the -V flag which is used in conjunction
 *		with the -b flag to build a database file for veronica
 *		which does not contain any "Disallow" gopher paths.
 *
 * Description:	Gets the various arguments from the command line for use
 *		with jughead.
 *
 * Routines:	static void	ShowArguments(void);
 *		static short	UsageError(char *progName);
 *		static short	PrintVersion(char *progName);
 *		static short	GetParamsFromStdin(void);
 *		static int	GetArg(char ***argv,char **argument);
 *		       int	GetArguments(int argc,char *argv[],
 *					     char **fileName,char **logFile);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#ifdef NEED_MALLOC_H
#	include <malloc.h>
#endif
#include <stdio.h>
#include <string.h>
#ifndef NEXT
#	include <unistd.h>
#endif

#include "jughead.h"

#define VERONICAXTENTION	"veronica"

extern void	exit();		/* For lint. */

extern List	*CreateNode();	/* Defined in "jughead.c". */
extern char	*logFile;	/* Defined in "search.c". */

/*****************************************************************************
 * ShowArguments simply displays the status of the arguments passed in, and
 * is solely for debugging.
 ****************************************************************************/
static void ShowArguments()
{	register int	i;	/* A loop counter. */
	List		*t;	/* Pointer into the list of nogos. */

	(void)fprintf(stderr,"flags are [%c%c%c%c%c%c%c%c%c%c%c%c%c]",
				printDTree	 ? 'a' : '\0',
				printDTreeDirs	 ? 'A' : '\0',
				buildIndex	 ? 'B' : '\0',
				debug		 ? 'd' : '\0',
				listHosts	 ? 'h' : '\0',
				listHostsNPorts	 ? 'H' : '\0',
				info4dirsOnly	 ? 'i' : '\0',
				info4allItems	 ? 'I' : '\0',
				maxprocs	 ? 'M' : '\0',
				menuFlag	 ? 'm' : '\0',
				printLineNumbers ? 'n' : '\0',
				doSearch	 ? 'S' : '\0',
				time2process	 ? 't' : '\0');
	if (buildDataFile)
		{
		(void)fprintf(stderr," -%cb [%s]",veronica ? 'V' : '\0',fileName);
		if (StrCmp(portStr,DEFAULTPORT))
			(void)fprintf(stderr," -p [%s]",portStr);
		}
	else if (doSearch)
		if (port2use != 3000)	/* ??? 3000 -> PORT2USE ??? */
			(void)fprintf(stderr," -p [%d]",port2use);

	if (selStr[0])
		(void)fprintf(stderr," -s [%s]",selStr);

	if (nogos)
		for (t = nogos; t; t = t->next)
			if (t->info.sStr[0])
				(void)fprintf(stderr," -X [%s]",t->info.sStr);
			else
				(void)fprintf(stderr," -x [%s] [%s]",t->info.hStr,t->info.pStr);
	if (logFile)
		(void)fprintf(stderr," -l [%s]",logFile);
	if (maxprocs)
		(void)fprintf(stderr," -M [%d]",maxprocs);
	if (buildIndex || doSearch)
		(void)fprintf(stderr," dataFile=[%s]",fileName);
	if (numSearchHosts)
		{
		(void)fprintf(stderr," hosts2search=(");
		for (i = 0; i < numSearchHosts; i++)
			(void)fprintf(stderr," [%s]",searchHosts[i]);
		(void)fprintf(stderr," )");
		}
	if (userName)
		(void)fprintf(stderr," -u [%s]",userName);
	if (!buildIndex && !doSearch && hostStr)
		(void)fprintf(stderr," host=[%s]",hostStr);
	(void)fprintf(stderr,"\n");

}	/* ShowArguments */

/*****************************************************************************
 * UsageError prints the usage of the program and returns false so we can
 * terminate.
 ****************************************************************************/
static short UsageError(progName)
	char	*progName;	/* Name of the program. */
{	int	numSpaces;	/* Number of spaces to print or the second
				 * part of the main usage line. */

	numSpaces = strlen("usage: ") + strlen(progName) + 1;
	(void)fprintf(stderr,"usage: %s [-?aAdhHiImntv] [-[V]b dataFile] [-l log] [-s selStr] [-p port#]\n",progName);
	(void)fprintf(stderr,"%*c[[-x hostNot2search port#]...] [[-X selStrNot2Search]...]\n",numSpaces,' ');
	(void)fprintf(stderr,"%*c[- | hosts2search...] host\n",numSpaces,' ');
	(void)fprintf(stderr,"usage: %s -B[?dmtv] [-l log] dataFile\n",progName);
	(void)fprintf(stderr,"usage: %s -S[?dtv] [-l log] [-M maxProcs] [-p port#] [-u username] dataFile\n",progName);
	return(0);

}	/* UsageError */

/*****************************************************************************
 * PrintVersion prints the version of the program and returns false so we
 * can terminate.
 ****************************************************************************/
static short PrintVersion(progName)
	char	*progName;		/* The name of this program. */
{
	(void)fprintf(stderr,"This version of %s is %s\n",progName,VERSION);
	return(0);

}	/* PrintVersion */

/*****************************************************************************
 * GetParamsFromStdin gets words from stdin until end-of-file is
 * encounterd.  This routine also takes care of backspaces.
 ****************************************************************************/
static short GetParamsFromStdin()
{	int		c;		/* A character from stdin. */
	register int	i;		/* A loop counter. */
	short		done = 0;	/* Are we done yet? */
	char		*s;		/* The string we are building. */

#ifdef NEXT
	if (isatty(0))
#else
	if (isatty(STDIN_FILENO))
#endif
		{
		(void)fprintf(stderr,"Please enter the hosts you would like to traverse\n");
		(void)fprintf(stderr,"separated by whitespace.  When you are finished\n");
		(void)fprintf(stderr,"enter ^D [control-d].\n  --> ");
		}

	for (i = 0; (c = fgetc(stdin)) && !done && numSearchHosts < MAXHOSTS; i++)
		if (isspace(buffer[i] = (char)c) || c == EOF)
			{
			if (c == EOF)
				done = 1;
			buffer[i] = '\0';

			if (i)
				if (s = (char *)malloc((unsigned)i * sizeof(char)))
					searchHosts[numSearchHosts++] = StrToLower(strcpy(s,buffer));
				else
					{
					(void)fprintf(stderr,"error: could not get memory for [%s]\n",buffer);
					exit(-2);
					}
			i = -1;
			}
		else if (c == '\b')	/* Take care of backspace. */
			{
			i -= 2;
			if (i < -1)
				i = -1;
			}

	if (numSearchHosts >= MAXHOSTS)
		(void)fprintf(stderr,"warning: only the first %d hosts2search are being used.\n",MAXHOSTS);

	return(1);

}	/* GetParamsFromStdin */

/*****************************************************************************
 * GetArg returns true if the current argument is exhausted, as in empty.
 * Otherwise this routine returns false.  If returning true we assign the
 * next argument in the list to 'argument', and adjust the list such that
 * it is pointing to the second to last character in 'argument'.
 ****************************************************************************/
static int GetArg(argv,argument)
	char	***argv,		/* The list of arguments. */
		**argument;		/* The argument to assign to. */
{
	if (!(*++**argv))		/* We're at the end of the list. */
		{
		*argument = *++*argv;	/* Get the next argument. */
		while (***argv && *(**argv + 1))
			++**argv;
		return(1);
		}
	else
		return(0);

}	/* GetArg */

/*****************************************************************************
 * GetArguments returns true if we were able to initialize variables from
 * argc and argv.  Otherwise this routine calls UsageError() which exits, if
 * something was wrong with the way the arguments were passed in.
 ****************************************************************************/
int GetArguments(argc,argv,fileName,logFile)
	int	argc;
	char	*argv[];
	char	**fileName,	/* Name of the file to open or create. */
		**logFile;	/* Name of the file to log to if searching. */
{	char	*programName,	/* The name of the program. */
		*s,		/* A temporary string. */
		*exceptionHost,
		*exceptionPort,
		*thePort;	/* The port to use. */
	int	i,		/* Loop counter. */
		vFlag = 0;	/* Are we doing a veronica file? */

	thePort = (char *)NIL;

	for (programName = *argv, argc--, argv++; *argv; argv++, argc--)
		if (**argv == '-')
			if (!*++*argv)
				(void)GetParamsFromStdin();
			else for ( ; **argv; ++*argv)
				switch(**argv)
					{
					case '?':
						return(UsageError(programName));
					case 'A':
						printDTree = printDTreeDirs = 1;
						break;
					case 'a':
						printDTree = 1;
						break;
					case 'B':
						buildIndex = 1;
						break;
					case 'b':
						buildDataFile = 1;
						if (argc > 1 && GetArg(&argv,fileName))
							argc--;
						else
							return(UsageError(programName));
						break;
					case 'd':
						debug = 1;
						break;
					case 'H':
						listHosts = listHostsNPorts = 1;
						break;
					case 'h':
						listHosts = 1;
						break;
					case 'I':
						info4allItems = 1;
						break;
					case 'i':
						info4dirsOnly = 1;
						break;
					case 'l':
						if (argc > 1 && GetArg(&argv,logFile))
							argc--;
						else
							return(UsageError(programName));
						break;
					case 'M':
						if (argc > 2 && GetArg(&argv,&s))
							{
							argc --;
							maxprocs = 0;
							for ( ; *s; s++)
								if (!isdigit(*s))
									return(UsageError(programName));
								else
									maxprocs = maxprocs * 10 + *s - '0';
							if (!maxprocs)
								{
								(void)fprintf(stderr,"error: maxprocs must be greater than 0\n");
								return(UsageError(programName));
								}
							}
					case 'm':
						menuFlag = !menuFlag;
						break;
					case 'n':
						printLineNumbers = 1;
						break;
					case 'p':
						if (argc > 1 && GetArg(&argv,&thePort))
							{
							argc--;
							for (s = thePort; *s; s++)	/* Validate the port. */
								if (!isdigit(*s))
									return(UsageError(programName));
							}
						else
							return(UsageError(programName));
						break;
					case 'S':
						doSearch = 1;
						break;
					case 's':
						if (argc > 1 && GetArg(&argv,&selStr))
							argc--;
						else
							return(UsageError(programName));
						break;
					case 't':
						time2process = 1;
						break;
					case 'V':
						vFlag = 1;
						break;
					case 'v':
						return(PrintVersion(programName));
					case 'X':
						if (argc > 1 && GetArg(&argv,&s))
							{
							argc--;
							if (!nogos)
								nogos = CreateNode(s,EMPTYSTRING,EMPTYSTRING);
							else
								{
								List *t = CreateNode(s,EMPTYSTRING,EMPTYSTRING);
								t->next = nogos;
								nogos = t;
								}
							}
						else
							return(UsageError(programName));
						break;
					case 'x':
						if (argc > 2 && GetArg(&argv,&s))
							{
							argc -= 2;
							exceptionHost = StrToLower(s);
							if (GetArg(&argv,&exceptionPort))
								for (s = exceptionPort; *s; s++)	/* Validate the port. */
									{
									if (!isdigit(*s))
										return(UsageError(programName));
									}
							else
								return(UsageError(programName));
							if (!nogos)
								nogos = CreateNode(EMPTYSTRING,exceptionHost,exceptionPort);
							else
								{
								List *t = CreateNode(EMPTYSTRING,exceptionHost,exceptionPort);
								t->next = nogos;
								nogos = t;
								}
							}
						else
							return(UsageError(programName));
						break;
					case 'u':
						if (argc > 1 && GetArg(&argv,&userName))
							argc--;
						else
							return(UsageError(programName));
						break;
					default:
						return(UsageError(programName));
					}
		else if (argc == 1)
			if (buildIndex || doSearch)
				*fileName = *argv;
			else
				initialHost = hostStr = *argv;
		else if (numSearchHosts < MAXHOSTS)
			searchHosts[numSearchHosts++] = StrToLower(*argv);
		else
			(void)fprintf(stderr,"warning: Too many hosts to search.  Ignoring %s\n",*argv);

	if (buildDataFile)		/* Tweek the menu flag. */
		menuFlag = !menuFlag;

	if (thePort)				/* User wants to use a different port. */
		if (!doSearch && !buildIndex)
			portStr = thePort;
		else if (doSearch)
			port2use = atoi(thePort);
		else if (buildIndex)
			return(UsageError(programName));

	/* If no hosts to search just search the host. */
	if (!numSearchHosts && !buildIndex && !doSearch)
		searchHosts[numSearchHosts++] = hostStr;

	if (!*fileName && !hostStr)		/* Make sure we got the last argument. */
		return(UsageError(programName));

	/* Make sure the proper flags have been used correctly:				*
	 * jughead [-?aAdhHiImnoOtv] [-[V]b dataFile] [-l log] [-s selStr] [-p port#]	*
	 *         [-x hostNot2search port#] [-X selStrNot2Search] 			*
	 *         [- | hosts2search...] host						*
	 * jughead -B[?dtv] [-l log] dataFile						*
	 * jughead -S[?dtv] [-l log] [-M maxProcs] [-p port#] [-u username] dataFile	*/
	if ((buildIndex && doSearch) || (!doSearch && userName) || (buildIndex && thePort))
		return(UsageError(programName));
	if ((buildIndex || doSearch) && (nogos ||
					 selStr[0] ||
					 printDTree ||
					 printDTreeDirs ||
					 listHosts ||
					 listHostsNPorts ||
					 info4allItems ||
					 info4dirsOnly ||
/*					 !menuFlag ||			*/
					 printLineNumbers ||
					 numSearchHosts))
		return(UsageError(programName));

	if (vFlag || doSearch)
		if (vFlag && !buildDataFile)
			return(UsageError(programName));
		else if (veronica = (char *)malloc(strlen(*fileName) + strlen("veronica") + 2))
			{
			(void)strcpy(veronica,*fileName);
			(void)strcat(veronica,".");
			(void)strcat(veronica,VERONICAXTENTION);
			}
		else
			{
			(void)fprintf(stderr,"error: could not get memory [%s.veronica] name\n",fileName);
			return(0);
			}

	if (debug)
		{
		ShowArguments();
		if (*logFile)	/* Turn off writting to the logFile file. */
			{
			(void)fprintf(stderr,"-l %s turned off for debugging\n",*logFile);
			*logFile = (char *)NIL;
			}
		}

	/* Now make sure that none of the searchHosts exceed 255 characters. */
	for (i = 0; i < numSearchHosts; i++)
		if (strlen(searchHosts[i]) > 255)
			{
			(void)fprintf(stderr,"error: [%s] has too many characters\n",searchHosts[i]);
			return(0);
			}

	return(1);

}	/* GetArguments */
