package PGopherLib::Misc;

use strict;
use warnings;
no warnings 'exec';
use Socket;
use Fcntl qw/F_GETFL F_SETFL O_NONBLOCK/;
use POSIX qw/:errno_h/;
use PGopherLib::Globals qw/$PAGER $BOOKMARKS @pageropt @history @menu $rows
	$cols $sigint $sigpipe $utf8/;
use PGopherLib::Init qw/quit/;
use PGopherLib::Term qw/newtty oldtty cls clreol curpos appkeys/;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(msg prompt errmsg mysystem myconnect isutf8 escape
	unescape splithpts center page getbookmark help menu);

my %icons = (
    '0' => '<TXT>',
    '1' => '<DIR>',
    '2' => '<CSO>',
    '3' => '<ERR>',
    '4' => '<HQX>',
    '5' => '<ARC>',
    '6' => '<UUE>',
    '7' => '<NDX>',
    '8' => '<TEL>',
    '9' => '<BIN>',
    'T' => '<IBM>',
    'g' => '<GIF>',
    'I' => '<IMG>',
    'h' => '<HTM>',
    'P' => '<PDF>',
    's' => '<SND>',
    ';' => '<MOV>',
    'i' => '     '
);

my %escapes;
foreach (0..255) {
    $escapes{chr($_)} = sprintf("%%%02X", $_);
}


sub msg {
    curpos($rows, 1); clreol; print shift;
}

sub prompt {
    my $prompt = shift;
    msg $prompt;
    $/ = "\015"; defined($_ = <STDIN>) or quit; chomp; $/ = "\012";
    msg $prompt;
    return $_;
}

sub errmsg {
    prompt "Error: " . shift() . " ";
}

sub mysystem {
    oldtty;
    my $r = system(@_);
    newtty;
    return $r;
}

sub myconnect {
    my ($host, $port) = @_;
    my $iaddr = inet_aton($host) or do
	    {errmsg "no host `$host'"; return undef};
    my $paddr = sockaddr_in($port, $iaddr);
    socket(SOCK, PF_INET, SOCK_STREAM, 6) or do
	    {errmsg "socket: $!"; return undef};
    CONNECT: {
	my $flags = fcntl(SOCK, F_GETFL, 0) or do
		{errmsg "fcntl: $!"; last CONNECT};
	fcntl(SOCK, F_SETFL, $flags | O_NONBLOCK) or do
		{errmsg "fcntl: $!"; last CONNECT};
	$sigint = 0;
	my $start = time;
	while (! connect(SOCK, $paddr)) {
	    if ($! != EINPROGRESS and $! != EALREADY) {
		$! == EISCONN and last;
		# FreeBSD returns EINVAL instead of ECONNREFUSED
		$! == EINVAL and $! = ECONNREFUSED;
		errmsg "connect: $!";
		last CONNECT;
	    }
	    $sigint and last CONNECT;
	    if (time - $start >= 10) {
		errmsg "connect: timeout";
		last CONNECT;
	    }
	    select(undef, undef, undef, 0.1);
	}
	fcntl(SOCK, F_SETFL, $flags) or do
		{errmsg "fcntl: $!"; last CONNECT};
	return \*SOCK;
    }
    close(SOCK);
    return undef;
}

sub isutf8 {
    return shift =~ /^([\x00-\x7F] |		# ASCII
	    [\xC2-\xDF][\x80-\xBF] |		# non-overlong 2-byte
	    \xE0[\xA0-\xBF][\x80-\xBF] |	# excluding overlongs
	    [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2} |	# straight 3-byte
	    \xED[\x80-\x9F][\x80-\xBF] |	# excluding surrogates
	    \xF0[\x90-\xBF][\x80-\xBF]{2} |	# planes 1-3
	    [\xF1-\xF3][\x80-\xBF]{3} |		# planes 4-15
	    \xF4[\x80-\x8F][\x80-\xBF]{2}	# plane 16
    )*$/x;
}

sub escape {
    my $text = shift;
    $text =~ s/([^A-Za-z0-9\-_.!~*'()\/])/$escapes{$1}/g;
    return $text;
}

sub unescape {
    my $text = shift;
    $text =~ s/%([0-9A-Fa-f]{2})/chr(hex($1))/eg;
    return $text; 
}

sub splithpts {
    my ($host, $port, $type, $sel) = (shift =~
	    m#^([^:/]+)(:(\d+))?(/(.)(.*))?$#)[0,2,4,5] or return ();
    defined $type or $type = "1";
    defined($port) or $port = ($type eq "2") ? 105 :
	    ($type =~ /[8T]/) ? 23 : 70;
    defined($sel) or $sel = "";
    return ($host, $port, $type, $sel);
}

sub center {
    my $text = shift;
    return " " x ($cols/2 - length($text)/2) . $text;
}

sub page {
    if (scalar(@_) > $rows - 2) {
	oldtty; appkeys 1;
	open(PIPE, '|-', $PAGER, @pageropt) or do {
	    newtty; appkeys 0;
	    errmsg "open: $!";
	    return 0;
	};
	$sigint = $sigpipe = 0;
	foreach (@_) {
	    last if $sigint || $sigpipe;
	    print PIPE;
	}
	close(PIPE);
	newtty; appkeys 0;
	print "\n";
    } else {
	print foreach (@_);
    }
    return 1;
}

sub getbookmark {
    -f $BOOKMARKS or return;
    open(FILE, $BOOKMARKS) or do {errmsg "open: $!"; return ()};
    my @bookmarks = ();
    while (<FILE>) {
	chomp;
	/^\s*$/ and next;
	my ($desc, $hpts) = split(/\t/, $_, 2);
	my ($host, $port, $type, $sel) = splithpts($hpts) or do {
	    close(FILE);
	    errmsg "invalid string `$hpts' in bookmark file";
	    return ();
	};
	push @bookmarks, [$desc, $host, $port, $type, $sel];
    }
    close(FILE);
    cls;
    my $n = 1;
    my @text = ();
    my $hdr = substr("BOOKMARKS", 0, $cols);
    push @text, "\n", center($hdr) . "\n", "\n";
    foreach (@bookmarks) {
	my ($type, $desc) = @$_[3, 0];
	push @text, sprintf("%4s %s %." . ($cols - 11) . "s\n", $n,
		(exists $icons{$type} ?  $icons{$type} : "<???>"), $desc);
	$n++;
    }
    page(@text) or return;
    $_ = prompt("Bookmark: ") or return ();
    !/^\d+$/ and do {errmsg "invalid entry"; return ()};
    $_ < 1 || $_ > scalar(@bookmarks) and do
	    {errmsg "invalid entry"; return ()};
    return @{$bookmarks[$_ - 1]};
}

sub help {
    cls;
    my $hdr = substr("COMMAND HELP", 0, $cols);
    print "\n", center($hdr) . "\n", "\n";
    print <<EOF;
	(# = menu item number)

	#	View/Download menu item
	d#	Download menu item
	?[#]	Show information about menu or menu item
	.	Redisplay current menu
	,	Go back one menu
	b	Bookmarks
	B[#]	Add menu or menu item to bookmarks
	h	Command help (this text)
	!	Shell
	q	Quit

	Any other character sequence is taken as a server to connect to       
	in the following format:

	    host[:port][/type[selector]]

	Press Control-C to abort a transfer in progress.

EOF
    prompt("Press RETURN to continue ");
}

sub menu {
    cls;
    scalar(@history) or return;
    my $n = 1;
    my @text = ();
    my ($type, $desc) = @{$history[$#history]};
    $type eq "7" and $desc = "Search Results";
    my $hdr = substr("MENU" . ($desc ? " [$desc]" : ""), 0, $cols);
    push @text, "\n", center($hdr) . "\n", "\n";
    foreach (@menu) {
	my ($type, $desc) = @$_;
	push @text, sprintf("%4s %s %." . ($cols - 11) . "s\n",
		($type eq "i" ? "" : $n),
		(exists $icons{$type} ? $icons{$type} : "<???>"), $desc);
	$n++;
    }
    page(@text);
}

1;
