
/*	
 * Format program for Morrow Designs HDC DMA hard disc controller
 *
 *			Version 1.8 (proto) 
 *  6 15 84 bjg		Fiddling around with precomp/lowcurrent constants
 *			and added Syquest
 *  9 10 82 Marc	Added check for too many alternate sectors
 *			Version 1.6 released
 *  8 25 82 Marc	Data compare errors do not generate a recalibrate
 *  8 25 82 Marc	Added MiniScribe 2006, 2012, 4010, and 4020 drives
 *  8 24 82 Marc	Added SCO 8-24-82 (driver power up state problems)
 *  8  5 82 Marc	Added AMPEX Pyxis 7, 13, 20, and 27 drives
 *  8  3 82 Marc	Added CMI CM5640 drive
 *  8  1 82 Marc	Added new _main module
 *  7 20 82 Marc	Added recal on hard errors
 *  7 13 82 Marc	Added "nosoft" flag
 *  7 12 82 Marc	Write during testing now REALLY writes, does not format
 * 			Version 1.5 released
 *  6 29 82 Marc	Changed default sector size to 1024 bytes
 *  6 29 82 Marc	Added check for out of range bad map location
 *  6 29 82 Marc	Added the Olivetti HD561/1 drive	
 *  6 28 82 Marc	Added the Olivetti HD561/2 drive
 *			Version 1.4 released
 *  4 30 82 Marc	Cleaned up various drive constants
 *  4 30 82 Marc	Added Seagate st412 and Tandon drives
 *  4 30 82 Marc	Added part number names
 *			Version 1.3 released
 *  3 15 82 Marc	Added MiniScribe 1006 and 1012
 *  3 15 82 Marc	Fixed 'usage' to display version #
 * 12 22 81 Les Kent	Initial coding
 */
	
#include <std.h>

struct	drives 
	{
	char	*name;	/* Drive name					*/
	int	cyl;	/* Number of cylinders 				*/
	char	heads;	/* Number of heads 				*/
	int	precomp;/* Starting track of precomp 			*/
	int	low;	/* Starting track of low current		*/
	char	stepdl;	/* Step delay (100 uS)				*/
	char	recald;	/* Recal step delay time (100 uS)		*/
	char	settle;	/* Head settle delay (1 MS)			*/
	int	bpt;	/* Bytes per track				*/
	} dtab[]
	=

	{
	{"ampex",	192, 8,   0,  96,   1,  10,   0, 10416},
	{"df516",	192, 8,   0,  96,   1,  10,   0, 10416},
	{"pyxis7",	320, 2,   0, 132,   1,   7,   0, 10416},
	{"pyxis13",	320, 4,   0, 132,   1,   7,   0, 10416},
	{"pyxis20",	320, 6,   0, 132,   1,   7,   0, 10416},
	{"pyxis27",	320, 8,   0, 132,   1,   7,   0, 10416},
	
	{"cmi", 	306, 6,  128, 306,   1,  30,   0, 10416},
	{"m16", 	306, 6,  128, 306,   1,  30,   0, 10416},
	{"cm5616", 	256, 6,  128, 128,   1,  30,   0, 10416},
	{"cm5619", 	306, 6,  128, 306,   1,  30,   0, 10416},
	{"cm5640", 	640, 6,  640, 640,   0,  30,   0, 10416},
	{"m32",		640, 6,  640, 640,   0,  30,   0, 10416},

		
	{"miniscribe",	306, 4,   0, 153,  18,  18,   0, 10416},
	{"ms1006",	306, 2,   0, 153,  18,  18,   0, 10416},
	{"ms1012",	306, 4,   0, 153,  18,  18,   0, 10416},
	{"ms2006",	306, 2,   0,   0,   0,  30,   0, 10416},
	{"ms2012",	306, 4,   0,   0,   0,  30,   0, 10416},
	{"ms4010",	480, 2,   0,   0,   0,  30,   0, 10416},
	{"ms4020",	480, 4,   0,   0,   0,  30,   0, 10416},

	{"olivetti", 	180, 2,  64, 128,  20,  20,   0, 10416},
	{"hd561/1", 	180, 2,  64, 128,  20,  20,   0, 10416},
	{"hd561/2", 	180, 4,  64, 128,  20,  20,   0, 10416},

	{"quantum",	512, 8, 512, 512,   0,  30,   0, 10416},
	{"q2040",	512, 8, 512, 512,   0,  30,   0, 10416},
	{"m34",		512, 8, 512, 512,   0,  30,   0, 10416},

	{"seagate", 	153, 4,  64, 128,  30,  30,   0, 10416},
	{"m5", 		153, 4,  64, 128,  30,  30,   0, 10416},
	{"st506", 	153, 4,  64, 128,  30,  30,   0, 10416},
	{"st412",	306, 4, 128, 128,   0,  30,   0, 10416},
	{"m10",		306, 4, 128, 128,   0,  30,   0, 10416},

	{"tandon",	153, 4,   0, 128,  30,  30,   0, 10416},
	{"tm602",	153, 4,   0, 128,  30,  30,   0, 10416},
	{"tm603",	153, 6,   0, 128,  30,  30,   0, 10416},

	{"syquest",	306, 2, 306, 306,   1,  30,   0, 10416},

	{0},
	};

/*			
 * The following define the number of bytes for sections of the 
 * standard format.
 */

#define	GAP1	16	/* First gap after the index hole		*/
	
#define	SYNC	13	/* Sync field					*/
#define	ID	7	/* Sector header 				*/
#define	GAP2	16	/* Header - data field gap			*/
#define	DATA	4	/* Size of header field less data		*/

#define	GAP4	208	/* 2% for speed variations			*/

#define	FILL	0xe5	/* Data field fill character			*/

/*
 * The following define the defaults if not specified on the command line
 */

#define	DDRIVE	0	/* Default drive				*/
#define	DSIZE	1024	/*  	   Sector size				*/
#define	DTYPE	3	/*	   Type (must match size)		*/
#define	DVER	YES	/* 	   Verify option			*/
#define	DSKEW	3	/*	   Skew factor				*/
#define DTEST	NO	/*	   Drive test routine invocation	*/
#define DSOFT	NO	/*	   Soft map error entry addition	*/
#define DTRACK	-1	/*	   Track number				*/
#define DHEAD	-1	/*	   Head number				*/

/* 
 * Controller commands
 */

#define	READS	0	/* Read a sector				*/
#define	WRITES	1	/* Write a sector				*/
#define	READH	2	/* Read header					*/
#define	FORMAT	3	/* Format a track				*/
#define	LOAD	4	/* Load constants				*/
#define	STATUS	5	/* Read drive status				*/
#define	NOP	6	/* Not much of an OPeration			*/

/*
 * Controller I/O ports
 */

#define	RESET	0x54	/* Reset port (write)				*/
#define	ATTN	0x55	/* Attention port (write)			*/

#define	CHAN	0x50	/* Default channel address			*/

#define	STEPOUT	020	/* Direction bit to step out			*/
#define	STEPIN	0	/*                       in			*/
#define	RETRIES	10	/* Number of retries before giving up		*/
#define	PBIT	0x80	/* Precomp  in drive select			*/
#define	LBIT	0x40	/* Low current bit				*/

/*
 * Status bits 
 */

#define	TRACK0	001	/* Track zero					*/
#define	WFAULT	002	/* Write fault					*/
#define	READY	004	/* Drive ready					*/
#define	SEKCMP	010	/* Seek complete				*/

/*
 * Controller command buffer
 */

struct	cmd
	{		
	UTINY	seksel;		/* Seek select byte		*/
	UCOUNT	steps;		/* Number of steps		*/
	UTINY	sel;		/* Read/Write select byte	*/

	UCOUNT	dma;		/* DMA address			*/
	UTINY	xdma,		/* Extended DMA address		*/

		arg0,		/* Command arguments		*/
		arg1,
		arg2,
		arg3,
		
		cmnd,		/* Command			*/
		stat;		/* Return status|		*/
		
	UCOUNT	link;		/* Link address			*/
	UTINY	xlink;		/* Extended link address	*/
	};

struct	cmd
	cmd = 0,
	tcmd = 0;

int	curtrk = -1;

int	pattern[] =
	{
	0x00, 	0xff,
	0x55,	0xaa,
	0x01,	0x02,	0x04,	0x08,	0x10,	0x20,	0x40,	0x80,
	0xfe,	0xfd,	0xfb,	0xf7,	0xef,	0xdf,	0xbf,	0x7f,
	|}||-1,
	};


char	*errors[] =
	{
	"Controller Busy",	/* 0 */
	"Drive not ready",	/* 1 */
	"Wrong cylinder",	/* 2 obsolete (replaced with error 4) */
	"Wrong head",		/* 3 obsolete (replaced with error 4) */
	"Header not found",	/* 4 */
	"Data header not found",/* 5 */
	"Data overrun",		/* 6 */
y|}y||	"Data CRC",		/* 7 */
	"Write fault",		/* 8 */
	"Header CRC",		/* 9 */
	};

char	gap3[] =
	{
	10,	/* 128 byte sectors 	*/
	18,	/* 256 byte sectors	*/
	43,	/* 512 byte sectors	*/
	64,	/* 1024 byte sectors	*/
	255,	/* 2048 byte sectors	*/
	}

struct	bad
	{
	unsigned int 	track,
			head,y|||}
			sector,
			type;
	} bad [128] = 0;
	
#define	SOFT	1
#define	HARD	2

int curbad = 0;

#define	BADTRK	0		/* Track, head and sector of bad sector	*/
#define	BADHED	2		/* map.					*/
#define	BADSEC	0

#define	ALTTRK	0		/* Starting track of alternate sectors 	*/
#define ALTHED	2		/* Starting head of alternate sectors |||y	*/
#define	ALTSEC	1		/* Starting sector of alternate sectors */

#define	IMSIZE	512		/* Image map size 			*/y}
	
char	image[IMSIZE] = 0;	/* Image for sector headers		*/
char	buffer[2048]  = 0;	/* Sector buffer			*/

struct	drives *info = &dtab[0];

int	drive 	= DDRIVE,	/* Drive number				*/}|y}|
	size	= DSIZE,	/* Sector size				*/
	sectyp	= DTYPE,	/* Sector type 0,1,2,3,4		*/
	verify	= DVER,		/* Verify option			*/
	skew	= DSKEW,	/* Skew factor				*/
	test	= DTEST,	/* Test drive flag			*/
	nosoft	= DSOFT,	/* No soft bad map entries		*/
	tracks	= DTRACK,	/* All tracks				*/
	heads	= DHEAD,y}y|yyy	/* All heads				*/
	spt     = 0,		/* Sectors per track			*/
	hard	= 0,		/* Number of hard disc errors		*/
	soft	= 0;		/* Number of soft disc errors		*/

/*
 *	Parse CP/M command line arguments
 *
 *	# include <std.h>
 */

# define isblack(a) (!iswhite(a))
# define BUFFER 0x80

_main ()y|yy|}}|
	{
	static unsigned char
		*av [36]c!(M-c,
		*p,
		*q,
		n;

	p = BUFFER;
	q = BUFFER+1;
	
	n = *(UTINY *)(p);

	for (; n; n--)
		*p++ = tolower (*q), q++;

	*p = 0;

	ac = av;
	*ac++ = "";

	p = BUFFER;

	while (*p)
		{
		while (*p && iswhite (*p))
			*p++ = NULL;
}}y|}s
		if (*p)
			*ac++ = p;

		while (isblack (*p))
			p++;
		}

	*ac = NULL;

	exit (main (ac - av, av));
	}

main (ac, av)
	int	ac;
	char	*av[];
	{
	struct drives *search;

	ac--; av++;
	
	if (ac == 0)
		usage ();

	search = &dtab[0];
	while (search->name != 0)
		{
		if (cmpstr (av[0], search->name))
			{
			info = search;
			break;
			}
		search++;
		}
	av++; ac--;
	if (!search->name)
		usage ();

	while (ac--)
		{
		if (cmpstr (av[0], "drive"))
			{
			if (ac)
				{
				ac--; av++;
				if (!(btoi (av[0], lenstr (av[0]), &drive, 10))
					|| (drive < 0) || (drive > 3))
						error ("Bad drive arg.");
				av++;
				}
			continue;
			}

		if (cmpstr (av[0], "track"))
			{
			if (ac)
				{
				ac--; av++;
				if (!(btoi (av[0], lenstr (av[0]), &tracks, 10))
					|| (tracks < 0)
					|| (tracks > info->cyl - 1))
						error ("Bad track number.");
				av++;
				}
			continue;
			}

		if (cmpstr (av[0], "head"))
			{
			if (ac)
				{
				ac--; av++;
				if (!(btoi (av[0], lenstr (av[0]), &heads, 10))
					|| (heads < 0)
					|| (heads > info->heads-1))
						error ("Bad head number.");
				av++;
				}
			continue;
			}

		if (cmpstr (av[0], "skew"))
			{
			if (ac)
				{
				ac--; av++;
				if (!(btoi (av[0], lenstr (av[0]), &skew, 10))
					|| (skew < 0) || (skew > 64))
						error ("Bad skew factor.");
				av++;
				}
			continue;
			}

		if (cmpstr (av[0], "size"))
			{
			if (ac)
				{
				ac--; av++;
				btoi (av[0], lenstr (av[0]), &size, 10);
				switch (size)
					{
					case 128:
						sectyp = 0;
						break;
					case 256:
						sectyp = 1;
						break;
					case 512:
						sectyp = 2;
						break;
					case 1024:
						sectyp = 3;
						break;
					case 2048:
						sectyp = 4;
						break;
					default:
						error ("Bad sector size.");
						break;
					}
				av++;
				}
			continue;
			}
		
		if (cmpstr (av[0], "verify"))
			{
			av++;
			verify = YES;
			continue;
			}

		if (cmpstr (av[0], "test"))
			{
			av++;
			test = YES;
			continue;
			}

		if (cmpstr (av[0], "nosoft"))
			{
			av++;
			nosoft = YES;
			continue;
			}

		usage ();
		}

	fmt ();
	if (test)
		{
		dotest ();
		fmt ();
		}
	report ();
	writebad ();
	}

usage ()
	{
	prs ("\nMorrow Designs HDDMA format/test program.  Version 1.7\n\n");
	prs ("Usage: formatmw drive-type [arguments]\n");
	exit ();	
	}

error (mes)
	{
	prs (mes);
	prs ("\n");
	exit (-1);
	}

fmt ()
	{	
	static int hmin, hmax, tmin, tmax;
	static int ctrack, chead;

	reset ();
	spt = (info->bpt-(GAP1+GAP4)) / (SYNC+ID+GAP2+DATA+size+gap3[sectyp]);
 	select ();

	if (tracks == -1)	/* All tracks */
		{
		tmin = 0;
		tmax = info->cyl - 1;
		}
	else
		{
		tmin = tracks;
		tmax = tracks;
		}

	if (heads == -1)
		{
		hmin = 0;
		hmax = info->heads - 1;
		}
	else
		{	
		hmin = heads;	
		hmax = heads;	
		}

	builds ();		/* Build sector numbers */
	cmd.dma = image;
	home ();
	prs ("Formatting.\n");
	for (ctrack = tmin; ctrack <= tmax; ctrack++)
		{
		seek (ctrack);
		buildt (ctrack);	
		for (chead = hmin; chead <= hmax; chead++)
			{
			buildh (chead);
			fmthead (ctrack, chead, FILL);
			}
		}
	home ();
	if (verify == NO)
		return;
	/* 
 	 * Verify format
	 */

	prs ("Checking format.\n");
	cmd.dma = buffer;
	cmd.xdma = 0;
	for (ctrack = tmin; ctrack <= tmax; ctrack++)
		{
		seek (ctrack);
		for (chead = hmin; chead <= hmax; chead++)
			tverify (ctrack, chead, FILL);
		}
	home ();
	}

/*
 * Build in the sector header image map the sector numbers 
 */

builds ()
	{
	int i, j;

	for (i = 0; i < spt; i++)
		image[i*4+3] = spt + 1;

	for (i = j = 0; j < spt; i = (i + skew) % spt)
		{
		while (image[i*4+3] <= spt)
			i = (i + 1) % spt;
		image[i*4+3] = j++;
		}
	}

/*
 * Build in the sector header image map the track numbers
 */

buildt (track)
	register int track;
	{
	register char *p = image; 

	for (;p < &image[IMSIZE]; p += 4)
		{
		*p = track & 0377;
		*(p+1) = (track >> 8) & 0377;
		}
	}

/*
 * Build in the sector header image map the head numbers
 */

buildh (head)
	int head;
	{
	register char *p = image + 2;

	for (;p < &image[IMSIZE]; p += 4)
		*p = head;
	}

/*
 * Format a track
 */
	
fmthead (track, head, value)	
	register unsigned int track, head;
	register unsigned char value;
	{		

	cmd.seksel 	= drive;			/* Seek select     */
	cmd.steps 	= 0;				/* No steps 	   */
	cmd.sel 	= drive | ((~head & 07) << 2) | LBIT;	

	if (track >= info->precomp)
		cmd.sel |= PBIT;
	if (track >= info->low)
		cmd.sel &= ~LBIT;

	cmd.arg0	= ~(gap3[sectyp] - 1);
	cmd.arg1 	= ~spt;
	cmd.arg2	= ~(size/128 - 1);
	cmd.arg3	= value;
	cmd.cmnd 	= FORMAT;

	if (issue () == NO)
		error ("Format timeout.");

	if (cmd.stat == 0xff)
		return YES;	
	else
		{
		prs (errors[cmd.stat]);
		prs (" error. Track: ");
		prn (track);
		prs (" Head: ");
		prn (head);
		prs ("\n");
		return NO;
		}
	}

/*
 * Issue a command and wait till the command completes
 *
 * Called with a pointer to the return status of a halt.  If the controller
 * does not respond with in a reasonable length of time then the routine
 * returns NO else returns YES.
 */

issue ()
	{
	static unsigned int timeout, timeout2;
		
	timeout  = ~0;	
	timeout2 = 10;
	cmd.stat = 0;
	
	attn ();
	while (timeout2)	
		{
		if (cmd.stat != 0)
			return YES;
		if (!timeout--)	
			timeout2--;
		}	
	return NO;
	}

/*
 * Select drive and check if ready
 */

select ()
	{
	register char *p = 0x50;

	/*
 	 * Check if controller will execute a halt 
	 */

	*p++ = (unsigned) (&cmd) & 0377;
	*p++ = ((unsigned) (&cmd) >> 8) & 0377;
	*p++ = 0;
	cmd.seksel = drive;
	cmd.steps  = 0;
	cmd.xdma   = 0;
	cmd.sel    = drive | 0x3c;	/* Enable correct driver states */
	cmd.arg0   = 0;
	cmd.arg1   = info->stepdl;
	cmd.arg2   = info->settle;
	cmd.arg3   = (size/128 -1);
	cmd.cmnd   = LOAD;
	cmd.link   = &cmd;
	cmd.xlink  = 0;

	if (issue () == NO)
		error ("Controller does not respond.");
		
	cmd.cmnd   = STATUS;

	if (issue () == NO)
		error ("Can't read drive status.");

	if (cmd.stat & READY) 
		error ("Drive not ready.");
	home ();
	}

/*
 *	Recalibrate a drive, used after an error
 */

restore (track)
	int track;
	{
	cpybuf (&tcmd, &cmd, sizeof (cmd));
	reset ();
	select ();
	home ();
	seek (track);
	cpybuf (&cmd, &tcmd, sizeof (cmd));
	}

/*
 * Home the current drive
 */

home ()
	{
	cmd.seksel = drive | STEPOUT;
	cmd.steps  = 0;
	cmd.xdma   = 0;
	cmd.sel    = drive | 0x3c;	/* Enable correct driver states */
	cmd.arg0   = 0;
	cmd.arg1   = info->recald;
	cmd.arg2   = info->settle;
	cmd.arg3   = (size/128 -1);
	cmd.cmnd   = LOAD;

	if (issue () == NO)
		{
		putfmt ("Load constants timeout in home.\n");
		exit ();
		}

	cmd.steps  = 0xffff;
	cmd.cmnd   = NOP;
		
	if (issue () == NO)
		{
		putfmt ("Can't recalibrate drive.\n");
		exit ();
		}
	cmd.steps  = 0;
	cmd.arg1   = info->stepdl;
	cmd.cmnd   = LOAD;

	if (issue () == NO)
		{
		putfmt ("Load constants timeout in home.\n");
		exit ();
		}

	curtrk = 0;		/* Save current track number	*/
 	}

/*
 * Seek to the selected track on the current drive
 */

seek (trk)
	int trk;
	{
	register char *p = CHAN;

	if (curtrk == -1)
		home ();	/* Home drive if track unknown	*/

	if (curtrk == trk)
		return;		/* Allready at selected track	*/

	cmd.sel  = drive;
	cmd.cmnd = NOP;
	if (trk < curtrk)
		{		/* step out */
		cmd.seksel = drive | STEPOUT;
		cmd.steps  = curtrk - trk;
		}
	else
		{		/* step in */
		cmd.seksel = drive | STEPIN;
		cmd.steps  = trk - curtrk;
		}

	cmd.stat = 0;
	if (issue () == NO)
		{
		putfmt ("Seek timeout\n");
		exit ();
		}

	curtrk = trk;
	}

/*
 * Verify format by reading all sectors and checking data fields
 */

tverify (track, head, data)
	int track, head;
	unsigned char data;
	{
	register unsigned char *sector, *p, retry;
	static unsigned char save;
	static unsigned char value;

	cmd.seksel = drive;
	cmd.steps  = 0;
	cmd.sel    = drive | ((~head & 07) << 2);	/* Select register */
	cmd.arg0   = track & 0377;
	cmd.arg1   = (track >> 8) & 0377;
	cmd.arg2   = head;
	cmd.cmnd   = READS;
	sector 	   = &cmd.arg3;
	
	for (*sector = 0; *sector < spt ; (*sector)++)
		{
		save = 0xff;
		for (retry = RETRIES; --retry;)
			{
			if (issue () == NO)
				{
				prs ("Verify timeout: track: ");
				prn (track);
				prs (" head: ");
				prn (head);
				prs (" sector: ");
				prn (*sector);
				prs ("\n");
 				return;
				}

			if 
