/*
 * four-in-a-row game - terminal/textmode functions
 * Copyright (c) 2017-2019,2022,2023 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* SPDX-License-Identifier: AGPL-3.0-or-later */

/*
For Unix a xterm compatible terminal emulation is assumed,
with either the "DEC Special Graphics" charset available, or UTF-8.
It has X11 mouse support.

It should be backward compatible with a plain vt100 (emulation),
without mouse and without colors.
<http://vt100.net>

The macro "VT100" is only used for the charset, not the terminal type.

For DOS codepage 437 is assumed, 850 is also okay.
*/

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include "row4.h"

#define COPYRIGHT "Copyright (c) 2023 Andreas K. Foerster"
#define HOMEPAGE "https:/" "/akfoerster.de/p/row4/"

#define ESC  "\033"		/* Escape */
#define CSI  ESC "["		/* Control Sequence Introducer */

#define MENULINE 1
#define TITLELINE 2

static void windowsize P_ ((int, int));
static void value P_ ((unsigned int));
static void clear_line P_ ((int));
static void centered P_ ((int, const char *));
static void finish P_ ((void));
static void show_score1 P_ ((int, int));
static void show_score P_ ((void));
static void numbers P_ ((void));
static void show_license P_ ((void));
static void clear_board P_ ((void));
static int mouse_board P_ ((int, int));
static int mouse_players P_ ((int, int));
static void player_menu P_ ((void));


static const char playerchip[4] = { ' ', 'X', 'O', '#' };

static unsigned int score[2];
static void (*border) P_ ((const char *));
static void (*refresh) P_ ((void));
static int screenwidth, halfscreenwidth;
static int boardtop, boardleft, promptline, winline, scoreline;


extern void
messages (l)
     enum languages l;
{
  if (l == unknown_language)
    l = system_language ();

  if (l == unknown_language)
    l = english;

  msg.language = l;
  msg.copyright = COPYRIGHT;

  switch (l)
    {
    default:
    case english:
      msg.language_name = "English";
      msg.title = "Four in a row";
      msg.optionerror = "Unknown option";
      msg.key = "Press any key...";
      msg.win = "? wins!";
      msg.player_1 = "1 player";
      msg.player_2 = "2 players";
      msg.license = "License";
      msg.quit = "Quit";
      msg.new_start = "New start";
      msg.license_text = COPYRIGHT "\r\n\n" HOMEPAGE "\r\n\n\
This program is free software: you can redistribute it and/or modify\r\n\
it under the terms of the GNU Affero General Public License as\r\n\
published by the Free Software Foundation, either version 3 of the\r\n\
License, or (at your option) any later version.\r\n\
\r\n\
This program is distributed in the hope that it will be useful,\r\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\r\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\r\n\
GNU Affero General Public License for more details.\r\n\
\r\n\
You should have received a copy of the GNU Affero General Public License\r\n\
along with this program.  If not, see <http://www.gnu.org/licenses/>.";
      break;

    case esperanto:
      msg.language_name = "Esperanto";
      msg.title = "Kvar en vico";
      msg.optionerror = "Nevalida argumento";
      msg.key = "Premu klavon por daurigi...";
      msg.win = "? venkis!";
      msg.player_1 = "1 Ludanto";
      msg.player_2 = "2 Ludantoj";
      msg.license = "Permesilo";
      msg.quit = "Chesu";
      msg.new_start = "Nova komenco";
      msg.license_text = COPYRIGHT "\r\n\n" HOMEPAGE "\r\n\n\
Chi tiu verko estas libera programo; vi povas ghin pludistribui kaj/au\n\
modifi je la kondichoj de la GNUa Afferoa Ghenerala Publika Permesilo,\n\
eldonita de Free Software Foundation, lau la versio 3 de tiu Permesilo\n\
au, se vi preferas, ajna posta versio.\n\
\n\
Ni distribuas chi tiun programon esperante ke ghi estos utila, tamen\n\
SEN IA AJN GARANTIO, i.a. sen la implica garantio pri SURMERKATIGEBLO\n\
au TAUGECO POR IU KONKRETA CELO. Pliajn detalojn vidu en la GNUa\n\
Afferoa Ghenerala Publika Permesilo.\n\
\n\
Ekzemplero de la GNUa Afferoa Ghenerala Publika Permesilo devas esti\n\
liverita al vi kun chi tiu programo;  se vi ghin ne ricevis, vidu\n\
<http://www.gnu.org/licenses/>.";
    break;

    case deutsch:
      msg.language_name = "Deutsch";
      msg.title = "Vier in einer Reihe";
      msg.optionerror = "Unbekannte Option";
      msg.key = "Bitte beliebige Taste druecken...";
      msg.win = "? gewinnt!";
      msg.player_1 = "1 Spieler";
      msg.player_2 = "2 Spieler";
      msg.license = "Lizenz";
      msg.quit = "Ende";
      msg.new_start = "Neuanfang";
      msg.license_text = COPYRIGHT "\r\n\n" HOMEPAGE "\r\n\n\
Dieses Programm ist freie Software. Sie koennen es unter den Bedingungen\r\n\
der GNU Affero General Public License, wie von der Free Software\r\n\
Foundation veroeffentlicht, weitergeben und/oder modifizieren, entweder\r\n\
gemaess Version 3 der Lizenz oder (nach Ihrer Option) jeder spaeteren\r\n\
Version.\r\n\
\r\n\
Die Veroeffentlichung dieses Programms erfolgt in der Hoffnung, dass es\r\n\
Ihnen von Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, sogar ohne\r\n\
die implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT FUER EINEN\r\n\
BESTIMMTEN ZWECK. Details finden Sie in der GNU Affero General Public\r\n\
License.\r\n\
\r\n\
Sie sollten ein Exemplar der GNU Affero General Public License zusammen\r\n\
mit diesem Programm erhalten haben.\r\n\
Falls nicht, siehe <http://www.gnu.org/licenses/>.";
      break;
    }
}


#if defined(__DOS__) || defined(__MSDOS__) || defined(_MSDOS)
#define DOS
/*
FreeDOS is the last one actively developed. And it's free software.
<http://www.freedos.org>
*/
#undef __unix__			/* DJGPP thinks it is a Unix */
#endif

#if defined(_WIN32) || defined(__WIN32__) || defined(__WINDOWS__)
#define WINDOWS
#undef DOS
#undef __unix__
#endif

#if !defined(ASCII) && !defined(CP437) && !defined(VT100)
#if defined(DOS) || defined(WINDOWS)
#define CP437
#endif
#if defined(__unix__)
#define VT100
#endif
#endif

/* ------------------------------------------------------------------ */

#if defined(DOS) || defined(WINDOWS)
#define C_BLUE 1
#define C_GREEN 2
#define C_RED 4
#define C_INTENSITY 8		/* just foreground */
#define C_LIGHTGRAY (C_BLUE | C_GREEN | C_RED)
#define C_BACKGROUND(c) (((c) & 7) << 4)

#define ATTR_SYSTEM      (C_LIGHTGRAY)	/* don't change */
#define ATTR_NORMAL      (C_LIGHTGRAY)
#define ATTR_EMPHASIZED  (C_LIGHTGRAY | C_INTENSITY)
#define ATTR_BOARD       (C_LIGHTGRAY | C_BACKGROUND(C_BLUE))
#define ATTR_P1          (C_LIGHTGRAY | C_BACKGROUND(C_GREEN))
#define ATTR_P2          (C_LIGHTGRAY | C_BACKGROUND(C_RED))
#endif

#ifdef DOS
/* For DOS the screensize is fixed. */

static int mouse;

#ifdef __DJGPP__
#undef __STRICT_ANSI__
#include <conio.h>
#include <dpmi.h>
#include <dos.h>		/* delay */

static void initialize_textmode P_ ((void));
static void djgpp_show_mouse P_ ((int));
static int djgpp_mouse_status P_ ((int *, int *));


static void
initialize_textmode ()
{
  __dpmi_regs r;

  /* initialize and check for mouse */
  r.x.ax = 0;
  __dpmi_int (0x33, &r);

  mouse = (r.x.ax != 0);
}


static void
djgpp_show_mouse (s)
     int s;
{
  __dpmi_regs r;

  r.x.ax = s ? 1 : 2;
  __dpmi_int (0x33, &r);
}


static int
djgpp_mouse_status (x, y)
     int *x, *y;
{
  __dpmi_regs r;

  memset (&r, 0, sizeof (r));
  r.x.ax = 3;
  __dpmi_int (0x33, &r);

  *x = (int) r.x.cx;
  *y = (int) r.x.dx;

  return (int) r.x.bx;
}


#define clear_screen() clrscr()
#define clear_rest() clreol()
#define position(x,y) gotoxy((x), (y))
#define print(s) cputs(s)
#define character(c) putch(c)
#define key_pressed()  _conio_kbhit()
#define get_key()  getch()
#define idle()  __dpmi_yield()
#define show_mouse(on)  djgpp_show_mouse(on)
#define mouse_status(x,y)  djgpp_mouse_status(x,y)
#define systemattributes()  textattr(ATTR_SYSTEM)
#define emphasized() textattr(ATTR_EMPHASIZED)
#define normal() textattr(ATTR_NORMAL)
#define boardattributes() textattr(ATTR_BOARD)
#define playerattributes(p)  \
  textattr((p==1)?ATTR_P1:(p==2)?ATTR_P2:ATTR_NORMAL)
#endif


#ifdef __BCC__
#include <conio.h>
#include "bcc-syst.h"

static void initialize_textmode P_ ((void));

static void
initialize_textmode ()
{
  mouse = (bcc_check_mouse () != 0);
}

#define delay(l)  bcc_usleep ((l)*1000UL)
#define clear_screen() bcc_clear_screen()
#define clear_rest() bcc_clear_rest()
#define position(x,y) bcc_position((x), (y))
#define print(s) bcc_print(s)
#define character(c)  bcc_character(c)
#define key_pressed()  kbhit()
#define get_key()  getch()
#define idle()  bcc_idle()
#define show_mouse(on)  bcc_show_mouse(on)
#define mouse_status(x,y)  bcc_mouse_status(x,y)
#define systemattributes()  bcc_set_attribute (ATTR_SYSTEM)
#define normal()  bcc_set_attribute (ATTR_NORMAL)
#define emphasized()  bcc_set_attribute (ATTR_EMPHASIZED)
#define boardattributes()  bcc_set_attribute (ATTR_BOARD)
#define playerattributes(p)  \
  bcc_set_attribute ((p==1)?ATTR_P1:(p==2)?ATTR_P2:ATTR_NORMAL)
#endif /* __BCC__ */

/* still DOS */

#define home()  position (1, 1)

static int choice P_ ((int (*)P_ ((int, int))));

static int
choice (mouse_handler)
     int (*mouse_handler) P_ ((int, int));
{
  int e;
  int x, y;

  if (mouse)
    {
      /* wait for release of mouse button */
      while (mouse_status (&x, &y) != 0)
	idle ();

      show_mouse (TRUE);
    }

  e = -1;
  while (e < 0)
    {
      if (key_pressed ())
	e = (int) get_key ();
      else if (mouse && mouse_status (&x, &y))
	{
	  if (mouse_handler)
	    e = mouse_handler ((x / 8) + 1, (y / 8) + 1);
	  else
	    e = '\r';		/* any key */
	}
      else
	idle ();
    }

  if (mouse)
    show_mouse (FALSE);

  return e;
}

#endif /* DOS */
/* ------------------------------------------------------------------ */

#ifdef WINDOWS
#include <windows.h>

/* I don't like Windows! */

#define delay(l)  Sleep(l)

static void *woe_input, *woe_output;
static unsigned long woe_input_mode;
static unsigned int woe_codepage;

static void woe_reset_textmode P_ ((void));
static void initialize_textmode P_ ((void));
static void woe_clear_screen P_ ((void));
static void woe_clear_rest P_ ((void));
static void woe_position P_ ((int, int));
static int woe_choice P_ ((int (*)P_ ((int, int))));
static void woe_character P_ ((int));
static void woe_print P_ ((const void *));


static void
woe_reset_textmode ()
{
  SetConsoleMode (woe_input, woe_input_mode);
  SetConsoleOutputCP (woe_codepage);
}

static void
initialize_textmode ()
{
  CONSOLE_SCREEN_BUFFER_INFO info;

  /* why easy when it can be made complicated? ;-\ */
  woe_input = GetStdHandle (STD_INPUT_HANDLE);
  woe_output = GetStdHandle (STD_OUTPUT_HANDLE);

  if (woe_input == INVALID_HANDLE_VALUE || woe_output == INVALID_HANDLE_VALUE)
    exit (EXIT_FAILURE);

  woe_codepage = GetConsoleOutputCP ();
  GetConsoleMode (woe_input, &woe_input_mode);

  SetConsoleOutputCP (437);	/* for UTF-8 use 65001 */
  SetConsoleTitle (msg.title);
  SetConsoleMode (woe_input, ENABLE_MOUSE_INPUT | ENABLE_WINDOW_INPUT);

  if (GetConsoleScreenBufferInfo (woe_output, &info))
    windowsize ((int) info.dwSize.X, (int) info.dwSize.Y);

  atexit (woe_reset_textmode);
}


static void
woe_clear_screen ()
{
  unsigned long length, written;
  COORD coord;
  CONSOLE_SCREEN_BUFFER_INFO info;

  /* this can change dynamically */
  if (!GetConsoleScreenBufferInfo (woe_output, &info))
    return;

  coord.X = coord.Y = 0;
  length = info.dwSize.X * info.dwSize.Y;

  FillConsoleOutputAttribute (woe_output, info.wAttributes, length, coord,
			      &written);
  FillConsoleOutputCharacter (woe_output, (TCHAR) ' ', length,
			      coord, &written);
}


static void
woe_clear_rest ()
{
  unsigned long length, written;
  COORD coord;
  CONSOLE_SCREEN_BUFFER_INFO info;

  if (!GetConsoleScreenBufferInfo (woe_output, &info))
    return;

  coord.X = info.dwCursorPosition.X;
  coord.Y = info.dwCursorPosition.Y;
  length = info.dwSize.X - info.dwCursorPosition.X;

  FillConsoleOutputAttribute (woe_output, info.wAttributes, length, coord,
			      &written);
  FillConsoleOutputCharacter (woe_output, (TCHAR) ' ', length,
			      coord, &written);
}


static void
woe_position (x, y)
     int x, y;
{
  COORD coord;

  coord.X = x - 1;
  coord.Y = y - 1;

  SetConsoleCursorPosition (woe_output, coord);
}


static int
woe_choice (mouse_handler)
     int (*mouse_handler) P_ ((int, int));
{
  int e;

  e = -1;
  while (e < 0)
    {
      unsigned long gotten;
      INPUT_RECORD b;

      ReadConsoleInput (woe_input, &b, 1, &gotten);

      switch (b.EventType)
	{
	  /* do windows users actually know what the keyboard is for? */
	case KEY_EVENT:
	  if (b.Event.KeyEvent.bKeyDown)
	    e = (int) b.Event.KeyEvent.uChar.UnicodeChar;
	  break;

	case MOUSE_EVENT:
	  /* only button presses */
	  if (b.Event.MouseEvent.dwEventFlags == 0
	      && b.Event.MouseEvent.dwButtonState != 0)
	    {
	      if (mouse_handler)
		{
		  register int x, y;

		  x = b.Event.MouseEvent.dwMousePosition.X;
		  y = b.Event.MouseEvent.dwMousePosition.Y;
		  e = mouse_handler (x + 1, y + 1);
		}
	      else
		e = '\r';	/* any key */
	    }
	  break;

	case WINDOW_BUFFER_SIZE_EVENT:
	  windowsize ((int) b.Event.WindowBufferSizeEvent.dwSize.X,
		      (int) b.Event.WindowBufferSizeEvent.dwSize.Y);
	  break;
	}
    }

  return e;
}


static void
woe_character (c)
     int c;
{
  char ch;
  unsigned long written;

  ch = (char) c;
  WriteConsole (woe_output, &ch, sizeof (ch), &written, NULL);
}


static void
woe_print (s)
     const void *s;
{
  unsigned long written;

  WriteConsole (woe_output, s, strlen (s), &written, NULL);
}

#define choice(x) woe_choice(x)
#define print(s)  woe_print(s)
#define character(c)  woe_character(c)
#define clear_screen()  woe_clear_screen()
#define position(x,y)  woe_position((x),(y))
#define home()  woe_position (1, 1)
#define clear_rest()  woe_clear_rest()
#define systemattributes()  SetConsoleTextAttribute (woe_output, ATTR_SYSTEM)
#define normal()  SetConsoleTextAttribute (woe_output, ATTR_NORMAL)
#define emphasized()  SetConsoleTextAttribute (woe_output, ATTR_EMPHASIZED)
#define boardattributes()  SetConsoleTextAttribute (woe_output, ATTR_BOARD)
#define playerattributes(p)  \
  SetConsoleTextAttribute (woe_output, \
                            (p==1)?ATTR_P1:(p==2)?ATTR_P2:ATTR_NORMAL)
#endif /* WINDOWS */
/* ------------------------------------------------------------------ */

#ifdef __unix__
#include <time.h>
#include <signal.h>
#include <termios.h>
#include <sys/ioctl.h>

static struct termios terminal_settings;

static void playerattributes P_ ((int));
static void reset_textmode P_ ((void));
static void get_windowsize P_ ((void));
static void resize P_ ((int));
static void initialize_textmode P_ ((void));
static void character P_ ((int));
static int get_key P_ ((void));
static void position P_ ((int, int));
static int choice P_ ((int (*)P_ ((int, int))));
static void delay P_ ((unsigned long));

#define print(s)  write (STDOUT_FILENO, (s), strlen (s))
#define control(s)  write (STDOUT_FILENO, "" s "", sizeof (s)-1)

#define home()              control (CSI "H")
#define clear_screen()      control (CSI "2J")
#define clear_rest()        control (CSI "K")
#define emphasized()        control (CSI "0;1;4m")	/* bold, underlined */
#define normal()            control (CSI "m")
#define boardattributes()   control (CSI "0;37;44m")
#define systemattributes()  control (CSI "m")

static void
playerattributes (player)
     int player;
{
  switch (player)
    {
    case 1:
      control (CSI "0;42m");
      break;

    case 2:
      control (CSI "0;41m");
      break;

    default:			/* hole */
      control (CSI "m");
      break;
    }
}


static void
reset_textmode ()
{
  /* disable X11 mouse support, autowrap on */
  control (CSI "?1000l" CSI "?7h");

  tcsetattr (STDIN_FILENO, TCSANOW, &terminal_settings);
}

static void
get_windowsize ()
{
#ifdef TIOCGWINSZ
  struct winsize size;

  if (!ioctl (STDOUT_FILENO, TIOCGWINSZ, &size))
    windowsize ((int) size.ws_col, (int) size.ws_row);
#endif
}

static void
resize (sig)
     int sig;
{
  (void) sig;
  get_windowsize ();
}

static void
initialize_textmode ()
{
  struct termios n;
  struct sigaction sa;

  if (!isatty (STDIN_FILENO))
    exit (EXIT_FAILURE);

  /* store previous settings */
  tcgetattr (STDIN_FILENO, &terminal_settings);
  atexit (reset_textmode);

  /*
     Set terminal in noncanonical mode, so there is no need to press Enter.
     Also disable ctrl-c and other signals.
   */
  tcgetattr (STDIN_FILENO, &n);
  n.c_lflag &= ~(ICANON | ECHO | ISIG | IEXTEN);
  n.c_cc[VMIN] = 1;
  n.c_cc[VTIME] = 0;

  tcsetattr (STDIN_FILENO, TCSAFLUSH, &n);

#ifdef VT100
  /* set G0 to default */
  /* load vt100 special graphics character set into G1 */
  control (ESC "(B" ESC ")0");
  /*  if the terminal is locked to UTF-8 these sequences do nothing */
#endif

  /* enable X11 mouse support, autowrap off */
  control (CSI "?1000h" CSI "?7l");

#ifdef TIOCGWINSZ
  get_windowsize ();

  /* resize on signal SIGWINCH */
  sa.sa_handler = resize;
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART;
  sigaction (SIGWINCH, &sa, NULL);
#endif
}


static void
delay (l)
     unsigned long l;
{
  struct timespec t;

  t.tv_sec = l / 1000L;
  t.tv_nsec = (l % 1000L) * 1000000L;

  nanosleep (&t, NULL);
}


static void
character (c)
     int c;
{
  char ch = (char) c;
  write (STDOUT_FILENO, &ch, sizeof (ch));
}

static int
get_key ()
{
  unsigned char ch;

  if (read (STDIN_FILENO, &ch, sizeof (ch)) <= 0)
    exit (EXIT_FAILURE);	/* hung up */

  return (int) ch;
}

/* starts with 1, 1 in upper left */
static void
position (x, y)
     int x, y;
{
  control (CSI);
  value (y);
  character (';');
  value (x);
  character ('H');
}

static int
choice (mouse_handler)
     int (*mouse_handler) P_ ((int, int));
{
  int e;

  e = -1;
  while (e < 0)
    {
      e = get_key ();

      /* Escape sequence? */
      if (e == 033)
	{
	  e = -1;

	  /* mouse sequence? */
	  if (get_key () == '[' && get_key () == 'M')
	    {
	      int b, x, y;

	      b = get_key () - 0x20;
	      x = get_key () - 0x20;
	      y = get_key () - 0x20;

	      /* any button, but not on button release */
	      if ((b & 3) != 3)
		e = mouse_handler ? mouse_handler (x, y) : '\r';
	    }
	}
    }

  return e;
}


#endif /* __unix__ */

/* some very old compilers do not have \a */
#define bell()  character('\007')

/* ------------------------------------------------------------------ */

/*
 * End of system specific code
 */

extern int
main (argc, argv)
     int argc;
     char *argv[];
{
  return run (argc, argv);
}

static void
windowsize (width, height)
     int width, height;
{
  screenwidth = width;
  halfscreenwidth = screenwidth / 2;
  boardleft = halfscreenwidth - 15;

  if (height >= 24)
    {
      int topoffset;

      topoffset = (height / 2) - (24 / 2);
      boardtop = 5 + topoffset;
      promptline = 20 + topoffset;
      winline = 22 + topoffset;
      scoreline = 24 + topoffset;
    }
  else				/* small height */
    {
      boardtop = 3;
      promptline = 17;
      winline = 18;
      scoreline = height;

      /* note: I currently have a device with 53x20 chars */
    }

  if (refresh)
    refresh ();
}


/* print unsigned value */
static void
value (v)
     unsigned int v;
{
  char *p, buffer[40];
  register unsigned int i;

  p = &buffer[sizeof (buffer) - 1];
  *p = '\0';

  i = v;
  do
    {
      --p;
      *p = digit (i % 10);
      i /= 10;
    }
  while (i);

  print (p);
}


static void
clear_line (y)
     int y;
{
  position (1, y);
  clear_rest ();
}


static void
centered (y, s)
     int y;
     const char *s;
{
  position ((int) (halfscreenwidth - (strlen (s) / 2)), y);
  print (s);
}


#ifndef CP437
/* for DOS and Windows only one codepage is relevant */

static void ascii_border P_ ((register const char *));
static void utf8_border P_ ((register const char *));
static void vt100_border P_ ((const char *));


static void
ascii_border (s)
     register const char *s;
{
  for (; *s; ++s)
    {
      int c;

      if ('x' == *s)
	c = '|';
      else if ('o' <= *s && *s <= 's')
	c = '-';
      else if ('j' <= *s && *s <= 'w')
	c = '+';
      else
	c = *s;

      character (c);
    }
}


static void
utf8_border (s)
     register const char *s;
{
  static const char vt100_u8[32][4] = {
    " ", "\342\231\246", "\342\226\222", "\342\220\211", "\342\220\214",
    "\342\220\215", "\342\220\212", "\302\260", "\302\261", "\342\220\244",
    "\342\220\213", "\342\224\230", "\342\224\220", "\342\224\214",
    "\342\224\224", "\342\224\274", "\342\216\272", "\342\216\273",
    "\342\224\200", "\342\216\274", "\342\216\275", "\342\224\234",
    "\342\224\244", "\342\224\264", "\342\224\254", "\342\224\202",
    "\342\211\244", "\342\211\245", "\317\200", "\342\211\240",
    "\302\243", "\302\267"
  };
  char b[133 * 3];
  size_t l;

  for (l = 0; *s && l < (sizeof (b) - 4); ++s)
    {
      if ('_' <= *s && *s <= '~')
	{
	  size_t ul;
	  const char *u;

	  u = vt100_u8[*s - '_'];
	  ul = strlen (u);
	  memcpy (b + l, u, ul);
	  l += ul;
	}
      else
	b[l++] = *s;
    }

  b[l] = '\0';
  print (b);
}
#endif /* not CP437 */

#if defined(VT100)
static void
vt100_border (s)
     const char *s;
{
  character (0x0E);		/* SO (shift out) */
  print (s);
  character (0x0F);		/* SI (shift in) */
}

#define native_border  vt100_border
#endif

#ifdef CP437
static void cp437_border P_ ((register const char *));
static int vt100_cp437 P_ ((int));


/* VT100 graphics to CP437 translation, incomplete */
static int
vt100_cp437 (c)
     int c;
{
  switch (c)
    {
      /* missing: graphical representations of control codes "bcdehi" */

    case '_':
      return 0x20;		/* blank */

    case '`':
      return 0x04;		/* diamond */

    case 'a':
      return 0xB1;		/* checkerboard (error indicator) */

    case 'f':
      return 0xF8;		/* degree symbol */

    case 'g':
      return 0xF1;		/* plus/minus */

    case 'j':
      return 0xD9;		/* lower-right corner */

    case 'k':
      return 0xBF;		/* upper-right corner */

    case 'l':
      return 0xDA;		/* upper-left corner */

    case 'm':
      return 0xC0;		/* lower-left corner */

    case 'n':
      return 0xC5;		/* crossing lines */

      /* horizontal lines of various heights */
    case 'o':			/* highest */
    case 'p':
    case 'q':			/* 'q' center for box drawing */
    case 'r':
    case 's':			/* lowest */
      return 0xC4;

    case 't':
      return 0xC3;		/* left "T" */

    case 'u':
      return 0xB4;		/* right "T" */

    case 'v':
      return 0xC1;		/* bottom "T" */

    case 'w':
      return 0xC2;		/* top "T" */

    case 'x':
      return 0xB3;		/* vertical bar */

    case 'y':
      return 0xF3;		/* less than or equal */

    case 'z':
      return 0xF2;		/* greater than or equal */

    case '{':
      return 0xE3;		/* Pi */

      /* missing: '|': not equal to */

    case '}':
      return 0x9C;		/* UK pound */

    case '~':
      return 0xFA;		/* centered dot */
    }

  return c;
}


static void
cp437_border (s)
     register const char *s;
{
  char b[133];
  register size_t l;

  for (l = 0; *s && l < (sizeof (b) - 1); ++s, ++l)
    b[l] = vt100_cp437 (*s);

  b[l] = '\0';
  print (b);
}

#define native_border cp437_border
#endif /* CP437 */

#ifndef native_border
#define native_border  ascii_border
#endif


static void
finish ()
{
  systemattributes ();
  clear_screen ();
  home ();
}


extern void
initialize (mode)
     int mode;
{
#ifdef CP437
  (void) mode;
  border = cp437_border;
#else
  if (MOD_UTF8 == mode)
    border = utf8_border;
  else if (MOD_ASCII == mode)
    border = ascii_border;
  else
    border = native_border;
#endif

  refresh = NULL;
  windowsize (80, 24);

  initialize_textmode ();
  normal ();
  atexit (finish);
}


/* put chip into specified position */
static void
put_chip (x, y, player)
     int x, y, player;
{
  playerattributes (player);
  position (boardleft + (4 * x) + 2, boardtop + (2 * (5 - y)) + 1);
  character (playerchip[get_player (player)]);
  normal ();
}


extern void
chip (x, y, player)
     int x, y, player;
{
  int i;

  if (outside (x, y))
    return;

  /* animation */
  for (i = 5; i > y; --i)
    {
      put_chip (x, i, player);
      home ();
      delay (100);
      put_chip (x, i, NONE);
    }

  put_chip (x, y, player);
}


static void
clear_board ()
{
  int i;

  for (i = 0; i <= 5; ++i)
    {
      int x;
      home ();
      delay (100);

      for (x = 0; x <= 6; ++x)
	{
	  int y;

	  memmove (&board[x][0], &board[x][1], 5 * sizeof (board[0][0]));
	  board[x][5] = NONE;	/* top */

	  for (y = 5; y >= 0; --y)
	    put_chip (x, y, board[x][y]);
	}
    }
}


static void
show_score1 (player, x)
     int player;
     int x;
{
  position (x, scoreline);
  character (playerchip[player]);
  print (": ");
  value (score[player - 1]);
}


static void
show_score ()
{
  clear_line (scoreline);
  show_score1 (1, halfscreenwidth - 10);
  show_score1 (2, halfscreenwidth + 8);
}


static void
numbers ()
{
  register int i;

  for (i = '1'; i <= '7'; ++i)
    {
      character (i);
      print ("   ");
    }
}


extern void
draw_board ()
{
  int x, y;

  clear_screen ();
  emphasized ();
  centered (TITLELINE, msg.title);
  normal ();

  if (boardtop >= 4)
    {
      position (boardleft + 2, boardtop - 1);
      numbers ();
    }

  boardattributes ();
  for (y = 0; y < 2 * 6; y += 2)
    {
      position (boardleft, boardtop + y);
      border (y ? "tqqqnqqqnqqqnqqqnqqqnqqqnqqqu"
	      : "lqqqwqqqwqqqwqqqwqqqwqqqwqqqk");
      position (boardleft, boardtop + y + 1);
      border ("x   x   x   x   x   x   x   x");
    }

  position (boardleft, boardtop + y);
  border ("mqqqvqqqvqqqvqqqvqqqvqqqvqqqj");

  normal ();
  position (boardleft + 2, boardtop + y + 1);
  numbers ();

  for (x = 0; x <= 6; ++x)
    for (y = 0; y <= 5; ++y)
      put_chip (x, y, board[x][y]);
  /* the function put_chip extracts the players already */

  position (1, MENULINE);
  print ("N: ");
  print (msg.new_start);
  position (screenwidth - 7, MENULINE);
  print ("Q: ");
  print (msg.quit);

  show_score ();
  refresh = draw_board;
}


static void
show_license ()
{
  refresh = NULL;
  clear_screen ();
  position (1, 1);

  emphasized ();
  print (msg.title);
  normal ();
  print ("\r\n");
  print (msg.license_text);
  print ("\r\n\n");
  print (msg.key);

  choice (NULL);
}


/*
 * mouse input for board view
 * returns an ASCII character or -1
 */
static int
mouse_board (x, y)
     int x, y;
{
  int e;

  e = -1;
  if (MENULINE == y)
    {
      if (x <= 12)
	e = 'N';
      else if (x >= (screenwidth - 12))
	e = 'E';
    }
  else if ((boardtop - 1) <= y && y <= (boardtop + (2 * 6) + 1))
    {
      int slot;

      slot = x - boardleft;
      if ((slot % 4) != 0)	/* not on the border */
	{
	  slot /= 4;
	  if (0 <= slot && slot <= 6)
	    e = '1' + slot;
	}
    }

  return e;
}


/* returns slot number, or -1 */
extern int
ask_slot (player)
     int player;
{
  int r, ch;

  ch = playerchip[get_player (player)];

  r = -1;
  while (r < 0)
    {
      char c;

      clear_line (promptline);
      position ((int) (halfscreenwidth - 2), promptline);
      character (ch);
      print (": ");

      c = choice (mouse_board);
      switch (c)
	{
	case 'n':
	case 'N':
	case ' ':
	  clear_board ();
	  return -1;

	case 'e':
	case 'E':
	case 'q':
	case 'Q':
	case 3:		/* ctrl-c */
	  exit (EXIT_SUCCESS);

	case 'r':
	case 'R':
	  draw_board ();
	  break;

	case 'l':
	case 'L':
	  show_license ();
	  draw_board ();
	  break;

	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	  r = c - '1';
	  break;
	}
    }

  return r;
}

static int
mouse_players (x, y)
     int x, y;
{
  (void) x;

  switch (y)
    {
    case 4:
      return '1';

    case 6:
      return '2';

    case 8:
      return 'L';

    case 10:
      return 'T';

    case 12:
      return 'E';
    }

  return -1;
}


static void
player_menu ()
{
  char line[81];

  clear_screen ();
  emphasized ();
  centered (1, msg.title);
  normal ();
  centered (2, msg.copyright);
  centered (4, msg.player_1);
  centered (6, msg.player_2);
  memcpy (line, "L: ", 3);
  strcpy (line + 3, msg.license);
  centered (8, line);
  memcpy (line, "T: ", 3);
  strcpy (line + 3, msg.language_name);
  centered (10, line);
  memcpy (line, "Q: ", 3);
  strcpy (line + 3, msg.quit);
  centered (12, line);
  position (1, 1);
}

extern int
ask_players ()
{
  int r;

  r = -1;
  while (r < 0)
    {
      char c;

      player_menu ();
      refresh = player_menu;

      c = choice (mouse_players);

      switch (c)
	{
	case '1':
	  r = 1;
	  break;

	case '2':
	  r = 2;
	  break;

	case 'l':
	case 'L':
	  show_license ();
	  break;

	case 't':
	case 'T':
	  next_language ();
	  break;

	case 'e':
	case 'E':
	case 'q':
	case 'Q':
	case 3:		/* ctrl-c */
	  exit (EXIT_SUCCESS);
	}
    }

  refresh = NULL;
  return r;
}


extern void
win (player)
     int player;
{
  char t[81];

  if (1 > player || player > 2)
    return;

  /* if maximum score is reached, drop both scores first, to be fair */
  if (score[0] == UINT_MAX || score[1] == UINT_MAX)
    score[0] = score[1] = 0U;

  score[player - 1] += 1;
  show_score ();

  strcpy (t, msg.win);
  t[0] = playerchip[player];

  clear_line (winline);
  emphasized ();
  centered (winline, t);
  normal ();
  bell ();
}
