/*
 * Copyright © 2022,2023 Andreas K. Förster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* SPDX-License-Identifier: GPL-3.0-or-later */

#include "SDL.h"
#include "row4.h"
#include "akftext.h"
#include <stdlib.h>
#include <limits.h>

#undef  Text
#define Text(t)  Text_u8 (t)
#define COPYRIGHT u8"Copyright © 2023 Andreas K. Förster"
#define HOMEPAGE u8"https://akfoerster.de/p/row4/"

#define WIDTH 800
#define HEIGHT 600

/*                        Red,  Green, Blue */
#define BACKGROUND_COLOR  0x00, 0x00, 0x00
#define BOARD_COLOR       0x8B, 0x45, 0x13
#define TRIGGER_COLOR     0x69, 0x23, 0x01
#define LINES_COLOR       0x00, 0x00, 0x00
#define PLAYER1_COLOR     0xCC, 0x00, 0x00
#define PLAYER2_COLOR     0xCC, 0xCC, 0x00
#define TEXT_COLOR        0xDE, 0xB8, 0x87
#define TEXT_COLOR2       0x8B, 0x45, 0x13

extern const unsigned char B_Logo[];
extern const unsigned int B_Logo_len;
extern const unsigned char B_Disc[];
extern const unsigned int B_Disc_len;

static SDL_Window *window;
static SDL_Renderer *renderer;
static SDL_Texture *Logo, *Disc;
static SDL_Rect boardrect, LogoRect;
static int cellsize, padding, character_width, character_height;

static SDL_INLINE int inside_rectangle (int, int, const SDL_Rect *);
static void build_board (void);
static void clear_board (void);
static void animation_delay (void);
static bool info (int, int, const char *);
static void textbox (int, int);
static void resize (int, int);
static void quit (void);
static void put_chip (int, int, int);
static const char *playername (int);
static void license (void);
static void initialize_graphic (void);
static SDL_Surface SDL_INLINE *load_graphic (const unsigned char *, size_t);


/*
 * main must be here like this,
 * because SDL may replace it
 * with a macro on some systems
 */
int
main (int argc, char *argv[])
{
  return run (argc, argv);
}


extern void
messages (enum languages l)
{
  if (l == unknown_language)
    l = system_language ();

  if (l == unknown_language)
    l = english;

  msg.language = l;
  msg.copyright = COPYRIGHT;

  /* *INDENT-OFF* */
  switch (l)
    {
    default:
    case english:
      msg.language_name = u8"English";
      msg.title = u8"Four in a row";
      msg.optionerror = u8"Unknown option";
      msg.key = u8"Press any key…";
      msg.win = u8"wins!";
      msg.player_1 = u8"1 player";
      msg.player_2 = u8"2 players";
      msg.license = u8"License";
      msg.quit = u8"Quit";
      msg.new_start = u8"New start";
      msg.license_text = COPYRIGHT u8"\r\n\n" HOMEPAGE u8"\r\n\n\
This program is free software: you can redistribute it and/or modify\n\
it under the terms of the GNU Affero General Public License as published by\n\
the Free Software Foundation, either version 3 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but \016without any warranty\017; without even the implied warranty of\n\
\016merchantability\017 or \016fitness for a particular purpose\017.  See the\n\
GNU Affero General Public License for more details.\n\
\n\
You should have received a copy of the GNU Affero General Public License\n\
along with this program.  If not, see «http://www.gnu.org/licenses/».";
      break;

    case esperanto:
      msg.language_name = u8"Esperanto";
      msg.title = u8"Kvar en vico";
      msg.optionerror = u8"Nevalida argumento";
      msg.key = u8"Premu klavon por daŭrigi…";
      msg.win = u8"venkis!";
      msg.player_1 = u8"1 Ludanto";
      msg.player_2 = u8"2 Ludantoj";
      msg.license = u8"Permesilo";
      msg.quit = u8"Ĉesu";
      msg.new_start = u8"Nova komenco";
      msg.license_text = COPYRIGHT u8"\r\n\n" HOMEPAGE u8"\r\n\n\
Ĉi tiu verko estas libera programo; vi povas ĝin pludistribui kaj/aŭ\n\
modifi je la kondiĉoj de la GNUa Afferoa Ĝenerala Publika Permesilo,\n\
eldonita de «Free Software Foundation», laŭ la versio 3 de tiu Permesilo\n\
aŭ, se vi preferas, ajna posta versio.\n\
\n\
Ni distribuas ĉi tiun programon esperante ke ĝi estos utila, tamen\n\
\016sen ia ajn garantio\017, i.a. sen la implica garantio pri \016surmerkatigeblo\017\n\
aŭ \016taŭgeco por iu konkreta celo\017. Pliajn detalojn vidu en la GNUa\n\
Afferoa Ĝenerala Publika Permesilo.\n\
\n\
Ekzemplero de la GNUa Afferoa Ĝenerala Publika Permesilo devas esti\n\
liverita al vi kun ĉi tiu programo;  se vi ĝin ne ricevis, vidu\n\
«http://www.gnu.org/licenses/».";
      break;

    case deutsch:
      msg.language_name = u8"Deutsch";
      msg.title = u8"Vier in einer Reihe";
      msg.optionerror = u8"Unbekannte Option";
      msg.key = u8"Beliebige Taste drücken…";
      msg.win = u8"gewinnt!";
      msg.player_1 = u8"1 Spieler";
      msg.player_2 = u8"2 Spieler";
      msg.license = u8"Lizenz";
      msg.quit = u8"Ende";
      msg.new_start = u8"Neuanfang";
      msg.license_text = COPYRIGHT u8"\r\n\n" HOMEPAGE u8"\r\n\n\
Dieses Programm ist freie Software. Sie können es unter den Bedingungen\n\
der GNU Affero General Public License, wie von der Free Software\n\
Foundation veröffentlicht, weitergeben und/oder modifizieren, entweder\n\
gemäß Version 3 der Lizenz oder (nach Ihrer Option) jeder späteren\n\
Version.\n\
\n\
Die Veröffentlichung dieses Programms erfolgt in der Hoffnung, dass es\n\
Ihnen von Nutzen sein wird, aber \016ohne irgendeine Garantie\017, sogar ohne\n\
die implizite Garantie der \016Marktreife\017 oder der \016Verwendbarkeit für einen\n\
bestimmten Zweck\017. Details finden Sie in der GNU Affero General Public\n\
License.\n\
\n\
Sie sollten ein Exemplar der GNU Affero General Public License zusammen\n\
mit diesem Programm erhalten haben.\n\
Falls nicht, siehe «http://www.gnu.org/licenses/».";
      break;
    }
  /* *INDENT-ON* */
}


/* are the coordinates inside the rectangle? */
static SDL_INLINE int
inside_rectangle (int x, int y, const SDL_Rect * r)
{
  return ((x >= r->x) && (x < r->x + r->w) &&
	  (y >= r->y) && (y < r->y + r->h));
}


extern int
ask_players (void)
{
  int players;
  bool redraw;
  const SDL_Rect *ta;

  ta = Text_Get_Area ();
  /* ta is a constant pointer, the content may change */

  players = -1;
  redraw = true;

  while (1 > players || players > 2)
    {
      SDL_Event e;

      if (redraw)
	{
	  char text[256];
	  SDL_snprintf (text, sizeof (text),
			"\016%s\017\n\n%s\n%s\n\nF1  %s\nF2  %s\nEsc %s",
			msg.title, msg.player_1, msg.player_2, msg.license,
			msg.language_name, msg.quit);

	  textbox (20, 8);
	  Text (text);
	  SDL_RenderPresent (renderer);
	  redraw = false;
	}

      SDL_WaitEvent (&e);

      switch (e.type)
	{
	case SDL_KEYDOWN:
	  switch (e.key.keysym.sym)
	    {
	    case SDLK_1:
	    case SDLK_KP_1:
	      players = 1;
	      break;

	    case SDLK_2:
	    case SDLK_KP_2:
	      players = 2;
	      break;

	    case SDLK_ESCAPE:
	      exit (EXIT_SUCCESS);
	      break;

	    case SDLK_F1:
	      license ();
	      redraw = true;
	      break;

	    case SDLK_F2:
	      next_language ();
	      redraw = true;
	      break;
	    }
	  break;		/* case SDL_KEYDOWN: */

	case SDL_MOUSEBUTTONUP:
	  if (inside_rectangle (e.button.x, e.button.y, ta))
	    {
	      int line = (e.button.y - ta->y) / character_height;

	      /* first line is 0 */
	      switch (line)
		{
		case 2:	/* 1 player */
		  players = 1;
		  break;

		case 3:	/* 2 players */
		  players = 2;
		  break;

		case 5:	/* license */
		  license ();
		  redraw = true;
		  break;

		case 6:	/* language */
		  next_language ();
		  redraw = true;
		  break;

		case 7:	/* quit */
		  exit (EXIT_SUCCESS);
		  break;
		}
	    }
	  break;		/* case SDL_MOUSEBUTTONUP: */

	case SDL_QUIT:
	  exit (EXIT_SUCCESS);
	  break;

	case SDL_WINDOWEVENT:
	  if (e.window.event == SDL_WINDOWEVENT_EXPOSED)
	    redraw = true;
	  else if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED)
	    resize (e.window.data1, e.window.data2);
	  break;
	}			/* switch */
    }				/* while */

  return players;
}


static void
animation_delay (void)
{
  SDL_Event e;

  while (SDL_PollEvent (&e))
    {
      switch (e.type)
	{
	case SDL_KEYDOWN:
	  if (e.key.keysym.sym == SDLK_ESCAPE)
	    exit (EXIT_SUCCESS);
	  break;

	case SDL_QUIT:
	  exit (EXIT_SUCCESS);
	  break;

	case SDL_WINDOWEVENT:
	  if (e.window.event == SDL_WINDOWEVENT_EXPOSED)
	    draw_board ();
	  else if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED)
	    resize (e.window.data1, e.window.data2);
	  break;
	}
    }

  SDL_Delay (100);
}


extern void
chip (int x, int y, int player)
{
  int i;

  if (outside (x, y))
    return;

  /* animation */
  board[x][y] = NONE;
  for (i = 5; i > y; --i)
    {
      byte old = board[x][i];	/* store threads */

      board[x][i] = player;
      draw_board ();
      board[x][i] = old;
      animation_delay ();
    }

  board[x][y] = player;
  draw_board ();
}


/* animation */
static void
clear_board (void)
{
  for (int y = 0; y <= 5; ++y)
    {
      animation_delay ();

      for (int x = 0; x <= 6; ++x)
	{
	  SDL_memmove (&board[x][0], &board[x][1], 5 * sizeof (board[0][0]));
	  board[x][5] = NONE;	/* top */
	}

      draw_board ();
    }
}


/* returns slot number, or -1 */
extern int
ask_slot (int player)
{
  int slot;

  draw_board ();

  slot = -1;
  while (slot < 0)
    {
      SDL_Event e;

      SDL_WaitEvent (&e);

      switch (e.type)
	{
	case SDL_KEYDOWN:
	  switch (e.key.keysym.sym)
	    {
	    case SDLK_ESCAPE:
	      exit (EXIT_SUCCESS);
	      break;

	    case SDLK_F1:
	      license ();
	      draw_board ();
	      break;

	    case SDLK_F2:
	      next_language ();
	      break;

	    case SDLK_SPACE:
	      clear_board ();
	      return -1;	/* clear board */
	      break;

	    case SDLK_1:
	    case SDLK_KP_1:
	      slot = 0;
	      break;

	    case SDLK_2:
	    case SDLK_KP_2:
	      slot = 1;
	      break;

	    case SDLK_3:
	    case SDLK_KP_3:
	      slot = 2;
	      break;

	    case SDLK_4:
	    case SDLK_KP_4:
	      slot = 3;
	      break;

	    case SDLK_5:
	    case SDLK_KP_5:
	      slot = 4;
	      break;

	    case SDLK_6:
	    case SDLK_KP_6:
	      slot = 5;
	      break;

	    case SDLK_7:
	    case SDLK_KP_7:
	      slot = 6;
	      break;
	    }
	  break;		/* case SDL_KEYDOWN: */

	case SDL_MOUSEBUTTONUP:
	  /* below board? */
	  if (e.button.y > boardrect.y + boardrect.h)
	    {
	      clear_board ();
	      return -1;
	    }
	  else			/* on board? */
	    {
	      int i = e.button.x - boardrect.x;
	      if (0 <= i && i <= boardrect.w)
		slot = i / cellsize;
	    }
	  break;

	case SDL_QUIT:
	  exit (EXIT_SUCCESS);
	  break;

	case SDL_WINDOWEVENT:
	  if (e.window.event == SDL_WINDOWEVENT_EXPOSED)
	    draw_board ();
	  else if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED)
	    resize (e.window.data1, e.window.data2);
	  break;
	}			/* switch */
    }				/* while */

  return slot;
}


/* returns true when the language changed */
static bool
info (int box_width, int box_height, const char *text)
{
  bool redraw;

  redraw = true;
  while (1)
    {
      SDL_Event e;

      if (redraw)
	{
	  textbox (box_width, box_height + 2);
	  Text (text);
	  Text_New_Line ();
	  Text_New_Line ();
	  Text_Color (TEXT_COLOR2);
	  Text (msg.key);
	  SDL_RenderPresent (renderer);
	  redraw = false;
	}

      SDL_WaitEvent (&e);

      switch (e.type)
	{
	case SDL_KEYDOWN:
	  switch (e.key.keysym.sym)
	    {
	    case SDLK_ESCAPE:
	      exit (EXIT_SUCCESS);
	      break;

	    case SDLK_F1:
	      license ();
	      redraw = true;
	      break;

	    case SDLK_F2:
	      next_language ();
	      return true;
	      break;

	    default:
	      return false;
	      break;
	    }
	  break;		/* case SDL_KEYDOWN: */

	case SDL_MOUSEBUTTONUP:
	  return false;
	  break;

	case SDL_QUIT:
	  exit (EXIT_SUCCESS);
	  break;

	case SDL_WINDOWEVENT:
	  if (e.window.event == SDL_WINDOWEVENT_EXPOSED)
	    redraw = true;
	  else if (e.window.event == SDL_WINDOWEVENT_SIZE_CHANGED)
	    resize (e.window.data1, e.window.data2);
	  break;
	}			/* switch */
    }				/* while */

  return false;
}


/* show which player has won */
extern void
win (int player)
{
  static unsigned int score[2];
  char text[256];

  /* if maximum score is reached, drop both scores first, to be fair */
  if (score[0] == UINT_MAX || score[1] == UINT_MAX)
    score[0] = score[1] = 0;

  score[player - 1] += 1;

  do
    {
      /* field widths with UTF-8 strings are problematic */
      SDL_snprintf (text, sizeof (text),
		    "\t%s %s\n\n%10u %s\n%10u %s",
		    playername (player), msg.win,
		    score[0], playername (1), score[1], playername (2));
    }
  while (info (25, 4, text));
}


static void
license (void)
{
  bool r = true;

  while (r)
    switch (msg.language)
      {
      case deutsch:
	r = info (71, 19, msg.license_text);
	break;

      case esperanto:
	r = info (72, 17, msg.license_text);
	break;

      case english:
      default:
	r = info (72, 16, msg.license_text);
	break;
      }
}


static const char *
playername (int player)
{
  const char *name;

  /* *INDENT-OFF* */
  switch (msg.language)
    {
    case deutsch:
      name = (player == 1) ? u8"Rot" : u8"Gelb";
      break;

    case esperanto:
      name = (player == 1) ? u8"Ruĝoj" : u8"Flavoj";
      break;

    case english:
    default:
      name = (player == 1) ? u8"Red" : u8"Yellow";
      break;
    }
  /* *INDENT-ON* */

  return name;
}


/* show chip on board */
static void
put_chip (int x, int y, int player)
{
  SDL_Rect r;

  switch (player)
    {
    case 0:
    default:
      SDL_SetTextureColorMod (Disc, BACKGROUND_COLOR);
      break;

    case 1:
      SDL_SetTextureColorMod (Disc, PLAYER1_COLOR);
      break;

    case 2:
      SDL_SetTextureColorMod (Disc, PLAYER2_COLOR);
      break;
    }

  r.w = r.h = cellsize - padding - padding;
  r.x = boardrect.x + padding + (cellsize * x);
  r.y = boardrect.y + padding + (cellsize * (5 - y));

  SDL_RenderCopy (renderer, Disc, NULL, &r);
}


static void
textbox (int box_width, int box_height)
{
  SDL_Rect r;

  build_board ();

  Text_Size (character_width, character_height);

  /* box */
  r.w = (box_width + 2) * character_width;
  r.h = (box_height + 2) * character_height;
  r.x = boardrect.x + ((boardrect.w / 2) - (r.w / 2));
  r.y = boardrect.y + ((boardrect.h / 2) - (r.h / 2));

  SDL_SetRenderDrawBlendMode (renderer, SDL_BLENDMODE_BLEND);
  SDL_SetRenderDrawColor (renderer, 0, 0, 0, 150);
  SDL_RenderFillRect (renderer, &r);
  SDL_SetRenderDrawBlendMode (renderer, SDL_BLENDMODE_NONE);

  /* border */
  SDL_SetRenderDrawColor (renderer, 0x55, 0x55, 0x55, SDL_ALPHA_OPAQUE);
  SDL_RenderDrawRect (renderer, &r);

  /* content */
  r.x += character_width;
  r.w -= 2 * character_width;
  r.y += character_height;
  r.h -= 2 * character_height;

  Text_Area (&r);
  Text_Color (TEXT_COLOR);
}


extern void
draw_board (void)
{
  build_board ();
  SDL_RenderPresent (renderer);
}


static void
build_board (void)
{
  SDL_Rect r;

  SDL_SetRenderDrawBlendMode (renderer, SDL_BLENDMODE_NONE);
  SDL_SetRenderDrawColor (renderer, BACKGROUND_COLOR, SDL_ALPHA_OPAQUE);
  SDL_RenderClear (renderer);

  if (Logo)
    SDL_RenderCopy (renderer, Logo, NULL, &LogoRect);

  SDL_SetRenderDrawColor (renderer, BOARD_COLOR, SDL_ALPHA_OPAQUE);
  SDL_RenderFillRect (renderer, &boardrect);

  SDL_SetRenderDrawColor (renderer, TRIGGER_COLOR, SDL_ALPHA_OPAQUE);
  r.w = boardrect.w;
  r.h = cellsize / 4;
  r.x = boardrect.x;
  r.y = boardrect.y + boardrect.h;
  SDL_RenderFillRect (renderer, &r);

  SDL_SetRenderDrawColor (renderer, LINES_COLOR, SDL_ALPHA_OPAQUE);

  /* vertical lines */
  for (int x = 1; x <= 6; ++x)
    {
      int xp = boardrect.x + (cellsize * x);
      SDL_RenderDrawLine (renderer, xp, boardrect.y,
			  xp, boardrect.y + boardrect.h);
    }

  /* horizontal lines */
  for (int y = 1; y <= 5; ++y)
    {
      int yp = boardrect.y + (cellsize * y);
      SDL_RenderDrawLine (renderer, boardrect.x, yp,
			  boardrect.x + boardrect.w, yp);
    }

  /* chips / holes */
  for (int x = 0; x <= 6; ++x)
    for (int y = 0; y <= 5; ++y)
      put_chip (x, y, get_player (board[x][y]));

  /* column numbers */
  Text_Color (TEXT_COLOR);
  Text_Size (Text_CharacterWidth, Text_CharacterHeight);
  r.w = boardrect.w;		/* larger than needed */
  r.h = Text_CharacterHeight;
  r.x = boardrect.x + (cellsize / 2) - (Text_CharacterWidth / 2);
  r.y = boardrect.y - Text_CharacterHeight;
  Text_Area (&r);

  for (int x = 1; x <= 7; ++x)
    {
      Text_Position ((x - 1) * cellsize, 0);
      Text_Character (digit (x));
    }
}


static void
resize (int width, int height)
{
  cellsize = SDL_min (width, height) / 7;
  padding = cellsize / 8;

  boardrect.w = 7 * cellsize;
  boardrect.h = 6 * cellsize;

  /* center position */
  boardrect.x = (width / 2) - (boardrect.w / 2);
  boardrect.y = (height / 2) - (boardrect.h / 2);
  LogoRect.x = width - LogoRect.w - LogoRect.y;

  character_width =
    (width >= 1500) ? 2 * Text_CharacterWidth : Text_CharacterWidth;
  character_height =
    (height >= 950) ? 2 * Text_CharacterHeight : Text_CharacterHeight;
}


static SDL_INLINE SDL_Surface *
load_graphic (const unsigned char *g, size_t l)
{
  SDL_Surface *s;

  s = SDL_LoadBMP_RW (SDL_RWFromConstMem (g, l), SDL_TRUE);
  if (!s)
    {
      SDL_Log ("%s", SDL_GetError ());
      exit (EXIT_FAILURE);
    }

  return s;
}


static void
initialize_graphic (void)
{
  SDL_Surface *s;

  /* Logo / Icon */
  s = load_graphic (B_Logo, B_Logo_len);
  SDL_SetColorKey (s, SDL_TRUE, 15);
  SDL_SetWindowIcon (window, s);
  Logo = SDL_CreateTextureFromSurface (renderer, s);
  LogoRect.w = s->w;
  LogoRect.h = s->h;
  LogoRect.y = 10;
  LogoRect.x = WIDTH - LogoRect.w - LogoRect.y;
  SDL_FreeSurface (s);

  /* Disc */
  s = load_graphic (B_Disc, B_Disc_len);
  SDL_SetColorKey (s, SDL_TRUE, 1);
  Disc = SDL_CreateTextureFromSurface (renderer, s);
  SDL_FreeSurface (s);
}


static void
quit (void)
{
  Text_Done ();
  SDL_DestroyTexture (Logo);
  SDL_DestroyTexture (Disc);
  SDL_DestroyRenderer (renderer);
  SDL_DestroyWindow (window);
  SDL_Quit ();
}


extern void
initialize (int mode)
{
  (void) mode;

  SDL_SetHint (SDL_HINT_VIDEO_ALLOW_SCREENSAVER, "1");
  SDL_SetHint (SDL_HINT_RENDER_SCALE_QUALITY, "nearest");

  if (SDL_Init (SDL_INIT_VIDEO) < 0)
    {
      SDL_Log ("%s", SDL_GetError ());
      exit (EXIT_FAILURE);
    }

  atexit (quit);

  window =
    SDL_CreateWindow (msg.title,
		      SDL_WINDOWPOS_CENTERED,
		      SDL_WINDOWPOS_CENTERED,
		      WIDTH, HEIGHT, SDL_WINDOW_RESIZABLE);

  if (!window)
    {
      SDL_Log ("%s", SDL_GetError ());
      exit (EXIT_FAILURE);
    }

  renderer = SDL_CreateRenderer (window, -1, 0);

  if (!renderer)
    {
      SDL_Log ("%s", SDL_GetError ());
      exit (EXIT_FAILURE);
    }

  initialize_graphic ();
  Text_Init (renderer);

  SDL_SetRenderDrawColor (renderer, BACKGROUND_COLOR, SDL_ALPHA_OPAQUE);
  SDL_RenderClear (renderer);
  SDL_RenderPresent (renderer);

  resize (WIDTH, HEIGHT);

  SDL_EventState (SDL_KEYUP, SDL_IGNORE);
  SDL_EventState (SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
  SDL_EventState (SDL_MOUSEMOTION, SDL_IGNORE);
  SDL_EventState (SDL_MOUSEWHEEL, SDL_IGNORE);
}
