/*
 * four-in-a-row game - logic
 * Copyright (c) 2017,2018 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* SPDX-License-Identifier: AGPL-3.0-or-later */

#include <string.h>
#include "row4.h"

byte board[7][6], filled[7];
short int chips;

static short int column;

static int won P_ ((int, int, int, int *));
static void analyze P_ ((int, byte *));
static void mark_threat P_ ((int, int, int));
static void seek_threat P_ ((int, int, int));


extern void
reset_board ()
{
  memset (board, NONE, sizeof (board));
  memset (filled, 0, sizeof (filled));

  chips = 42;
  column = -1;
}


static int
won (player, x, y, num)
     int player, x, y, *num;
{
  if (get_player (board[x][y]) == player)
    *num += 1;
  else
    *num = 0;

  return (*num == 4);
}


extern int
wincheck ()
{
  int x, y, player, num;
  register int i, j;

  x = column;
  if (x < 0)
    return FALSE;

  y = filled[x] - 1;
  player = get_player (board[x][y]);

  /* vertical */
  for (i = num = 0; i <= 5; ++i)
    if (won (player, x, i, &num))
      return TRUE;

  /* horizontal */
  for (i = num = 0; i <= 6; ++i)
    if (won (player, i, y, &num))
      return TRUE;

  /* ascending diagonal */
  for (i = num = 0, j = y - x; i <= 6; ++i, ++j)
    if (0 <= j && j <= 5 && won (player, i, j, &num))
      return TRUE;

  /* descending diagonal */
  for (i = num = 0, j = y + x; i <= 6; ++i, --j)
    if (0 <= j && j <= 5 && won (player, i, j, &num))
      return TRUE;

  return FALSE;
}


/* drop chip into slot x */
/* returns y or -1 */
extern int
drop (player, x)
     int player, x;
{
  int y;

  if (0 > x || x > 6 || 1 > player || player > 2)
    return -1;

  /* one above */
  y = filled[x];
  if (y > 5)
    return -1;			/* full */

  /* a filled field is no longer a threat */
  board[x][y] = player;
  column = x;
  filled[x] += 1;
  chips -= 1;

  seek_threat (x, y, player);

  return y;
}


/*
 * computer logic
 */

/* gets line of chips, turns it into line of threats */
static void
analyze (player, line)
     int player;
     byte *line;
{
  int i, threat;
  char s[8], *c;

  /* turn line into string */
  for (i = 0; i <= 6; ++i)
    {
      /* pl ignores threats */
      int pl = get_player (line[i]);
      if (pl == NONE)
	s[i] = '.';
      else if (pl == player)
	s[i] = 'P';
      else
	s[i] = 'C';
    }

  s[7] = '\0';

  memset (line, 0, 7);
  threat = player << 2;

  c = strstr (s, "P.PP.P");
  if (c)
    {
      int p = c - s;
      line[p + 1] = line[p + 4] = threat;
    }

  c = strstr (s, ".PPP.");
  if (c)
    {
      int p = c - s;
      line[p] = line[p + 4] = threat;
    }

  c = strstr (s, ".PPP");
  if (c)
    line[c - s] = threat;

  c = strstr (s, "PPP.");
  if (c)
    line[c - s + 3] = threat;

  c = strstr (s, "P.PP");
  if (c)
    line[c - s + 1] = threat;

  c = strstr (s, "PP.P");
  if (c)
    line[c - s + 2] = threat;
}


static void
mark_threat (x, y, threat)
     int x, y, threat;
{
  if (threat && 0 <= y && y <= 5 && get_player (board[x][y]) == NONE)
    board[x][y] |= threat;
}


static void
seek_threat (x, y, player)
     int x, y, player;
{
  byte line[7];
  register int i, j;

  /* look for horizontal threat */
  for (i = 0; i <= 6; ++i)
    line[i] = board[i][y];
  analyze (player, line);
  for (i = 0; i <= 6; ++i)
    mark_threat (i, y, line[i]);

  /* look for vertical threat */
  for (i = 0; i <= 5; ++i)
    line[i] = board[x][i];
  line[6] = NONE;
  analyze (player, line);
  for (i = 0; i <= 5; ++i)
    mark_threat (x, i, line[i]);

  /* look for ascending diagonal threat */
  for (i = 0, j = y - x; i <= 6; ++i, ++j)
    line[i] = (0 > j || j > 5) ? NONE : board[i][j];
  analyze (player, line);
  for (i = 0, j = y - x; i <= 6; ++i, ++j)
    mark_threat (i, j, line[i]);

  /* look for descending diagonal threat */
  for (i = 0, j = y + x; i <= 6; ++i, --j)
    line[i] = (0 > j || j > 5) ? NONE : board[i][j];
  analyze (player, line);
  for (i = 0, j = y + x; i <= 6; ++i, --j)
    mark_threat (i, j, line[i]);
}


extern int
compute (player)
     int player;
{
  /* prefered order, starting from center */
  static const int order[7] = { 3, 2, 4, 1, 5, 0, 6 };
  int mythreat;
  register int i;

  if (1 > player || player > 2)
    return -1;

  /* own threat, but not both */
  mythreat = player << 2;

  /* no chip set yet - the center is the best start */
  if (chips == 42)
    return 3;

  /* one chip - go next to it */
  if (chips == 41)
    return ((column < 4) ? column + 1 : column - 1);

  /* special case: do I own the bottom row? (false positive doesn't harm) */
  if (chips == 38)
    {
      int n = 0;

      for (i = 0; i < 6; ++i)
	{
	  int p = get_player (board[i][0]);

	  if (p == player)
	    ++n;
	  else if (p)
	    break;		/* other player */

	  if (n == 2)
	    return i + 1;
	}
    }

  /* can I win? */
  for (i = 0; i <= 6; ++i)
    {
      int f = filled[i];	/* above */
      if (f < 6 && (board[i][f] & mythreat) != 0)
	return i;
    }

  /* close threats */
  for (i = 0; i <= 6; ++i)
    {
      int f = filled[i];	/* above */
      if (f < 6 && threatened (board[i][f]))
	return i;
    }

  /* vertically two successive holes? */
  for (i = 0; i <= 6; ++i)
    {
      int y;

      for (y = filled[i] + 1; y < 5; ++y)
	{
	  if (board[i][y] == mythreat && (board[i][y + 1] & mythreat) != 0)
	    return i;

	  /* there may be no thread by other player underneath */
	  if (threatened (board[i][y]) && board[i][y] != mythreat)
	    break;
	}
    }

  /* seek field that is not indirectly threatened - avoid current column */
  for (i = 0; i <= 6; ++i)
    {
      int f, x;

      x = order[i];
      if (x == column)
	continue;

      f = filled[x];
      if (f == 5 || (f < 5 && !threatened (board[x][f + 1])))
	return x;
    }

  /* check the same for current column */
  {
    int f = filled[column];
    if (f == 5 || (f < 5 && !threatened (board[column][f + 1])))
      return column;
  }

  /* all are indirectly threatened => give up an own threat */
  for (i = 0; i <= 6; ++i)
    {
      int x = order[i];
      int f = filled[x];
      if (f < 5 && board[x][f + 1] == mythreat)
	return x;
    }

  /* last resort: search for any free column */
  for (i = 0; i <= 6; ++i)
    {
      int x = order[i];
      if (!full (x))
	return x;
    }

  return -1;
}
