/*
 * Text output for SDL2
 * Copyright (c) 2020-2023 Andreas K. Foerster <akf@akfoerster.de>
 * https://akfoerster.de/p/akftext/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* SPDX-License-Identifier: GPL-3.0-or-later */

// This API may still change

#ifndef AKFTEXT_H
#define AKFTEXT_H

#include "SDL.h"
#include <uchar.h>
#include <stdio.h>
#include <stdbool.h>

typedef void (*Text_Char_Func) (const char *);

extern const int Text_CharacterWidth, Text_CharacterHeight;

// returns 0 on success, -1 on error
int Text_Init (SDL_Renderer *);
void Text_Done (void);

void Text_Color (Uint8 red, Uint8 green, Uint8 blue);
void Text_Alpha (Uint8);	// 255=opaque, 0=invisible
void Text_Size (int width, int height);	// 0 = normal
void Text_Underline (bool);
void Text_Strike (bool);

#define Text_Colour(r,g,b)  Text_Color(r,g,b)

// positions are in pixels, not characters
void Text_Area (const SDL_Rect *);
SDL_Rect *Text_Get_Area (void);
void Text_Home (void);
void Text_Position (int x, int y);
int Text_Get_X (void);
int Text_Get_Y (void);

// automatic linebreak after end of line?
void Text_Wrap (bool);

// Output
// Strings may be NULL

#define Text(s)            Text_u8(s)
#define Text_New_Line()    Text_Character(10)
#define Text_Line_Down()   Text_Character(132)
#define Text_Line_Up()     Text_Character(141)

// Unicode
void Text_Character (char32_t);
void Text_u32 (const char32_t *);
void Text_u16 (const char16_t *);
void Text_u8 (const char *);

// wchar_t
void Text_wchar (const wchar_t *);
void Text_wprintf (const wchar_t *fmt, ...);

// ISO-8859-1 or ASCII
void Text_l1 (const char *);
#define Text_Latin1(s)  Text_l1(s)

// locale (category LC_CTYPE)
void Text_locale (const char *);

// Esperanto X-system (X-sistemo)
void Text_xs (const char *);
/*
Input method for Esperanto.
The letter x is used as modifier after the letters c,g,h,j,s,u.
Otherwise characters are interpreted as ISO-8859-1 (Latin-1).
Example: "Cxi tio ankaux estas GNU/Linukso-masxino."
If you must, use DEL as invisible separator, as in "Linu\177x"
*/

// output numbers left aligned (fast)
// radix (2-36): 10=decimal, 16=hexadecimal, 2=binary
void Text_Number (long int value, int radix);
void Text_Natural (unsigned long int value, int radix);

// formatted output (like printf)
// example: Text_Format (Text_u8, u8"price: %ld\u20AC\n", price);
void Text_Format (Text_Char_Func, const char *fmt, ...)
     SDL_PRINTF_VARARG_FUNC(2);

// read text from wide oriented stream
wchar_t *Text_read (FILE *);

/*
 * for Text_locale, Text_read
 * you must use setlocale() to get more than just ASCII
 */

#endif
