/*
 * lesen - CGI-Programm zum Lesen langer Texte
 * Copyright © 2014-2019 Andreas K. Förster
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

 /*
  * Für CGI/1.1 (RFC-3875)
  * oder als HTTP/1.1-Server (wenn HTTP definiert ist)
  */

#define VERSION "2019-01-24:akf"

#define _ISOC99_SOURCE
#define _ISOC11_SOURCE
#define _POSIX_C_SOURCE 200112L

#include "config.h"
#include "Methoden.h"

#ifdef HTTP
#include "Medientypen.h"
#undef NPH
#define NPH
#endif

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include <time.h>
#include <errno.h>

// POSIX
#include <strings.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>


#if __STDC_VERSION__+0 >= 201112L
#include <stdnoreturn.h>
#elif defined(__GNUC__)
#define noreturn  __attribute__((__noreturn__))
#else
#define noreturn		/* leer */
#endif

// printf zu vermeiden spart bei statischem Linken mit dietlibc
#pragma GCC poison  printf fprintf sprintf snprintf asprintf
#pragma GCC poison  scanf fscanf sscanf
#pragma GCC poison  setlocale

#define SEKUNDEN(x) (x)
#define MINUTEN(x)  ((x)*60L)
#define STUNDEN(x)  ((x)*3600L)
#define TAGE(x)     ((x)*86400L)
#define WOCHEN(x)   ((x)*604800L)
#define MONATE(x)   ((x)*2628000L)	// ((x)*(TAGE(365L)/12))

#ifndef EACCES
#define EACCES 0
#endif

// Code von UTF-8 abhängig
#define ZEICHENKODIERUNG "UTF-8"

#ifndef BUCHHALTBARKEIT
#define BUCHHALTBARKEIT 0
#endif

#ifndef INDEXHALTBARKEIT
#define INDEXHALTBARKEIT 0
#endif

#if (BUCHHALTBARKEIT) < 0L || (INDEXHALTBARKEIT) < 0L
#error "Haltbarkeiten dürfen nicht kleiner als 0 sein."
#endif

#if (BUCHHALTBARKEIT) > 31536000L || (INDEXHALTBARKEIT) > 31536000L
#error "Haltbarkeiten dürfen nicht über einem Jahr sein."
#endif

#ifndef FEHLEREINLEITUNG
#define FEHLEREINLEITUNG ""
#endif

#ifndef FEHLERANHANG
#define FEHLERANHANG ""
#endif

#if !defined(ANDERE) || !defined(ANDEREURL)
#undef ANDERE
#undef ANDEREURL
#endif

#if !defined(ALTERNATIVE) || !defined(ALTERNATIVEURL)
#undef ALTERNATIVE
#undef ALTERNATIVEURL
#endif

#if !defined(HOMEPAGE) || !defined(HOMEPAGEURL)
#undef HOMEPAGE
#undef HOMEPAGEURL
#endif

#if !defined(IMPRESSUM) || !defined(IMPRESSUMURL)
#undef IMPRESSUM
#undef IMPRESSUMURL
#endif

#if !defined(SOFTWARE) || !defined(SOFTWAREURL)
#undef SOFTWARE
#undef SOFTWAREURL
#endif


#define NL  "\r\n"
#define DATEIERWEITERUNG  ".txt"	// im Dateinamen
#define TEXTERWEITERUNG  ".txt"	// in der URL
#define XHTMLERWEITERUNG ".xhtml"
#define ZEILENBUFFER  1024

// Dies bitte auf das Allernötigste beschränken
#ifndef CSS
#define CSS \
  ".Kopf,.Kapitelnavigation{text-align:center}" NL \
  "a.Navigation{text-decoration:none}"
#endif

// Das ist für die Gesamtausgabe eines Buches
#ifndef GESAMTCSS
#define GESAMTCSS \
  "html{color:black;background-color:#F4ECD8}" NL \
  "#Inhalt{max-width:45em;margin:auto;padding:1ex;font-size:x-large;" NL \
  "font-family:\"URW Palladio L\",Palatino,serif;text-align:justify;line-height:1.5}" NL \
  "div.Kopf{text-align:center}" NL \
  "p{margin:0;text-indent:2em}" NL \
  "i{font-style:normal;font-weight:normal;font-family:sans-serif}" NL \
  "a{color:inherit}" NL \
  "a:hover{background-color:#CFC}" NL \
  ".Intro{border:solid thin black;border-radius:1em;padding:1ex}" NL \
  ".Intro p{text-align:left;text-indent:0;margin:1em 0;}" NL \
  "h2+p{text-indent:0}" NL \
  "h2+p:first-letter{font-size:300%;font-weight:bold}" NL \
  "h2+p:first-line{line-height:100%}" NL \
  "hr:before{content:\"☙\"}" NL \
  "hr{display:block;color:inherit;border:none;margin:1em auto;padding:0;" \
  "text-align:center;font-size:200%}" NL \
  NL "@media print {" NL \
  "html,#Inhalt{background-color:white}" NL \
  "#Inhalt{font-size:medium}" NL \
  "a{color:black;text-decoration:none}" NL \
  "div.Kapitel{page-break-before:always}" NL \
  "}"
#endif


#define XHTML_DEKLARATION \
  "<?xml version=\"1.0\" encoding=\"" ZEICHENKODIERUNG "\" ?>" NL \
  "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"" NL \
  "   \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\" >" NL

#ifdef SPRACHE
#define XHTML_ANFANG \
  XHTML_DEKLARATION \
  "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"" SPRACHE "\"" \
  " lang=\"" SPRACHE "\">" NL NL
#else
#define XHTML_ANFANG \
  XHTML_DEKLARATION \
  "<html xmlns=\"http://www.w3.org/1999/xhtml\">" NL NL
#endif

#define ja 1
#define nein 0

// Konstanten für Kapitelangaben
#define UNBEKANNT (-1)
#define GESAMTTEXT (-2)

/*
time_t könnte theoretisch vorzeichenlos sein.
Zeiten vor oder bei Epochenbeginn werden nicht unterstützt.
Das ist auch ein Schutz vor Überläufen.
*/
#define gueltige_Zeit(x)  ((x)>0 && (x)!=(time_t)(-1))

#define fertig()  exit (EXIT_SUCCESS)

enum Fehlerarten
{
  Fehlerfrei, Buchfehler, Kapitelfehler, Zugriffsfehler,
  Verfuegbarkeitsfehler, Methodenfehler, Unterstuetzungsfehler,
  Anfragefehler, Scheisse
};

enum Anfragetyp
{
  Inhaltanfrage, Gesamttextanfrage, Gesamtxhtmlanfrage, Kapitelanfrage,
  Indexanfrage, Dateianfrage
};

enum Ausgabetyp
{ Formatlos, XHTML, TEXT };

// tatsächlich unterstützte Methoden
#define ALLOW  "Allow: GET,HEAD"

// bis zu welcher Methode unterstützt wird
#define maximale_Anfragemethode  HEAD

static const char Wochentagsabk[7][4] =
  { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

static const char Monatsabk[12][4] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
  "Nov", "Dec"
};

// angefordertes Kapitel, -1 für Inhaltsverzeichnis, -2 für Gesamttext
static int Kapitel;
static char Kurzname[256], Buchtitel[256], Autor[256];
static bool xhtml_body, betont, Alternativschrift;
static time_t jetzt, Modifikationszeit;
static enum Anfragemethoden Anfragemethode;
static enum Ausgabetyp Format;

// Zeit im Browser-Cache (If-Modified-Since)
static struct tm Cachezeit;


/*
 * Eingabe-Datei
 */

typedef FILE *Datei;
#define oeffne_Datei(Pfad)  fopen(Pfad, "rm")
#define oeffne_Binaerdatei(Pfad)  fopen(Pfad, "rb")
#define schliesse_Datei(d)  fclose(d)
#define lies_Byte(d)  fgetc(d)
#define Dateianfang(d)  rewind(d)
#define Stelle  fpos_t
#define Stelle_merken(d,s)  fgetpos(d, s)
#define Stelle_wiederherstellen(d,s)  fsetpos(d, s)

#ifndef FILTER_S
#define lies_Zeile(d, Buffer, Groesse)  fgets((Buffer), (Groesse), (d))
#else

// langes s rausfiltern (U+017F)
static char *
lies_Zeile (Datei d, char *Buffer, int Groesse)
{
  char *e, *q;

  e = fgets (Buffer, Groesse, d);

  if (e && (q = strstr (e, "\xC5\xBF")))
    {
      char *z = q;
      while (*q)
	{
	  if (q[0] != '\xC5' || q[1] != '\xBF')
	    *z++ = *q++;
	  else
	    {
	      *z++ = 's';
	      q += 2;
	    }
	}

      *z = '\0';
    }

  return e;
}

#endif


/*
 * Ausgabe
 */

#define Byte(z)  putchar (z)
#define Zeilenende()  fputs (NL, stdout)

static inline void
Code (const char *s)
{
  fputs (s, stdout);
}


// Für reine Textausgabe
#define Klartext(s)  Code(s)


// Gibt Ganzzahl dezimal aus
// printf zu vermeiden spart bei statischem Linken
static void
Zahl (long int Wert)
{
  register unsigned long int z;

  if (Wert >= 0)
    z = (unsigned long int) Wert;
  else
    {
      Byte ('-');
      z = (unsigned long int) (-Wert);
      // zwar undefiniert für kleinste Zahl,
      // aber funktioniert dennoch
    }

  char Puffer[44], *p;

  // der Puffer wird von hinten gefüllt
  p = &Puffer[sizeof (Puffer) - 1];
  *p = '\0';

  do
    {
      --p;
      *p = '0' + (z % 10);
      z /= 10;
    }
  while (z && p != Puffer);

  Code (p);
}


// Gibt zweistellige Zahl aus, evtl. mit führender 0
static void
zweistellig (unsigned int z)
{
  z %= 100;
  Byte ('0' + (z / 10));
  Byte ('0' + (z % 10));
}


// Gibt Hexadezimal-Ziffer aus für Werte 0-15
static inline void
Hexziffer (unsigned char c)
{
  Byte ((c < 10) ? '0' + c : ('A' - 10) + c);
}


// Gibt Byte hexadezimal aus
static inline void
Hexbyte (unsigned char c)
{
  Hexziffer (c >> 4);
  Hexziffer (c & 0x0F);
}


// kopiere s in Ziel, soweit es passt
static void
kopiere (char *Ziel, size_t Groesse, const char *s)
{
  char *p = Ziel;

  if (s)
    {
      while (*s && --Groesse)
	*p++ = *s++;
    }

  *p = '\0';
}


// gibt Text mit angegebener Länge aus, "&" wird maskiert
static void
uri_ausgeben (const char *s, size_t Laenge)
{
  while (Laenge--)
    {
      if (*s != '&')
	Byte (*s);
      else
	Code ("&#38;");
      ++s;
    }
}


static const char *
uri (const char *s)
{
  size_t Laenge;

  // wie lang ist die URI?
  // nur bestimmte Zeichen sind zulässig (RFC-1738)
  Laenge = strspn (s, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
		   "0123456789;/?:@&=+$,-_.!~*#()[]%");

  Code ("<a href=\"");

#ifdef NBN_RESOLVER
  if (!strncasecmp (s, "urn:nbn:", 8))
    Code (NBN_RESOLVER);
#endif

  uri_ausgeben (s, Laenge);
  Code ("\" class=\"URI\" " EXTERNER_LINK ">");
  uri_ausgeben (s, Laenge);
  Code ("</a>");

  return s + Laenge;
}


static inline bool
uri_erkannt (const char *s)
{
  int l = tolower (*s);
  return ('h' == l &&
	  (!strncasecmp (s, "http://", 7)
	   || !strncasecmp (s, "https://", 8)))
    || ('f' == l && (!strncasecmp (s, "ftp://", 6)
		     || !strncasecmp (s, "ftps://", 7)
		     || !strncasecmp (s, "file:", 5)))
    || ('m' == l && !strncasecmp (s, "mailto:", 7))
    || ('s' == l &&
	(!strncasecmp (s, "shttp://", 8)
	 || !strncasecmp (s, "sftp://", 7)
	 || !strncasecmp (s, "ssh://", 6)))
    || ('t' == l &&
	((!strncasecmp (s, "tel:", 4) && !isspace (s[4]))
	 || !strncasecmp (s, "telnet://", 9)))
    || ('u' == l && !strncasecmp (s, "urn:", 4))
    || ('g' == l && !strncasecmp (s, "gopher://", 9));
}


// Gibt maskierte Textzeile aus
// Nur bei HTML-Ausgabe, nicht für reines Textformat!
static void
Text (const char *s)
{
  if (!s)
    return;

  while (*s)
    {
      switch (*s)
	{
	case '<':
	  Code ("&#60;");
	  break;

	case '>':
	  Code ("&#62;");
	  break;

	case '&':
	  Code ("&#38;");
	  break;

	case '{':
	case '\x0E':		// ASCII: Shift Out
	  if (s[0] == '{' && s[1] == '{')
	    Byte (*s++);
	  else if (xhtml_body && !betont)
	    {
	      Code ("<em>");
	      betont = ja;
	    }
	  break;

	case '}':
	case '\x0F':		// ASCII: Shift In
	  if (s[0] == '}' && s[1] == '}')
	    Byte (*s++);
	  else if (xhtml_body && betont)
	    {
	      Code ("</em>");
	      betont = nein;
	    }
	  break;

	case '[':
	  if (s[0] == '[' && s[1] == '[')
	    Byte (*s++);
	  else if (xhtml_body && !Alternativschrift)
	    {
	      Code (ALTERNATIVTEXTANFANG);
	      Alternativschrift = ja;
	    }
	  break;

	case ']':
	  if (s[0] == ']' && s[1] == ']')
	    Byte (*s++);
	  else if (xhtml_body && Alternativschrift)
	    {
	      Code (ALTERNATIVTEXTENDE);
	      Alternativschrift = nein;
	    }
	  break;

	  // geschütztes Leerzeichen
	case '_':
	  if (s[1] != '_')
	    Code ("\u00A0");
	  else
	    {
	      Byte ('_');
	      ++s;
	    }
	  break;

	case '/':
	  if (s[1] == '/')
	    {
	      ++s;
	      if (xhtml_body)
		Code ("<br />");
	    }
	  else
	    Byte ('/');
	  break;

	  // "://" intakt lassen, da es in URLs verwendet wird
	  // nicht alle URLs werden erkannt
	case ':':
	  if (s[1] == '/' && s[2] == '/')
	    {
	      s += 2;
	      Code ("://");
	    }
	  else
	    Byte (':');
	  break;

	case '\n':
	case '\v':
	  Zeilenende ();
	  break;

	  // andere Freizeichen zu einzelnem Leerzeichen zusammenziehen,
	  // das spart Bandbreite
	case ' ':
	case '\t':
	case '\f':
	  Byte (' ');
	  while (isspace (s[1]))
	    ++s;
	  break;

	case '\x7F':		// ASCII: Delete
	  break;

	  // C1-Steuerzeichen herausfiltern (UTF-8)
	case '\xC2':
	  if (s[1] >= '\xA0')	// kein Steuerzeichen
	    {
	      // druckbare 2-Byte-Sequenz (U+00A0 - U+00BF)
	      Byte (0xC2);
	      Byte (s[1]);
	    }

	  ++s;
	  break;

	default:
	  if (xhtml_body && uri_erkannt (s))
	    s = uri (s) - 1;
	  else if ((unsigned char) *s >= ' ')
	    Byte (*s);
	  break;
	}

      ++s;
    }
}


static void
urlkodiert (const char *s)
{
  while (*s)
    {
      if (isalnum (*s) || strchr ("-_.~:()", *s))
	Byte (*s);
      else			// Prozent-Kodierung
	{
	  Byte ('%');
	  Hexbyte (*s);
	}

      ++s;
    }
}


// gibt Wert von hexadezimaler Ziffer aus (ohne zu prüfen)
static inline unsigned char
Hexwert (char c)
{
  return isdigit (c) ? c - '0' : (c | 0x20) - ('a' - 10);
}


// dekodiert URL-Kodierung an Ort und Stelle
static void
url_dekodieren (char *s)
{
  char *q, *z;
  // q: Quelle, z: Ziel

  for (q = z = s; *q; ++q)
    {
      if ('+' == *q)
	*z++ = ' ';
      else if ('%' == *q)
	{
	  *z++ = (char) ((Hexwert (q[1]) << 4) | Hexwert (q[2]));
	  q += 2;
	}
      else
	*z++ = *q;
    }

  *z = '\0';
}


static inline char *
Textanfang (const char *s)
{
  while (*s && isspace (*s))
    ++s;

  return (char *) s;
}


// Zeitangabe für HTTP-Kopf (festes Format)
static void
HTTP_Zeit (const char *Name, const struct tm *z)
{
  if (z && z->tm_mday)
    {
      Code (Name);
      Code (": ");
      Code (Wochentagsabk[z->tm_wday]);
      Code (", ");
      zweistellig (z->tm_mday);
      Byte (' ');
      Code (Monatsabk[z->tm_mon]);
      Byte (' ');
      Zahl (1900 + z->tm_year);
      Byte (' ');
      zweistellig (z->tm_hour);
      Byte (':');
      zweistellig (z->tm_min);
      Byte (':');
      zweistellig (z->tm_sec);
      Code (" GMT" NL);
    }
}


static void
HTTP_Zeitangabe (const char *Name, time_t Zeit)
{
  // falls Zeit irgendwie seltsam ist, lieber gar nichts senden
  if (gueltige_Zeit (Zeit))
    HTTP_Zeit (Name, gmtime (&Zeit));
}


static void
Content_Type (const char *Typ)
{
  Code ("Content-Type: ");
  Code (Typ);
  Zeilenende ();
}


static void
Cache_Kontrolle (long int Haltbarkeit)
{
  if (Haltbarkeit > 0)
    {
      if (gueltige_Zeit (jetzt))
	HTTP_Zeitangabe ("Expires", jetzt + Haltbarkeit);

      // doppelt gemoppelt hält besser
      Code ("Cache-Control: max-age=");
      Zahl (Haltbarkeit);
      Zeilenende ();
    }
}


static void
HTTP_Anfang (const char *Status)
{
  static bool http_angefangen;

  if (!http_angefangen)
    {

#ifdef NPH
      // NPH = non-parsed header (auch wenn HTTP definiert ist)

      Code ("HTTP/1.1 ");
      Code (Status ? Status : "200 OK");
      Zeilenende ();
      HTTP_Zeitangabe ("Date", jetzt);
      Code ("Server: lesen/" VERSION " (Andreas K. Foerster)" NL);
      Code ("Connection: close" NL);

#else // CGI

      // Status nicht unbedingt senden,
      // erlaubt Server evtl. mit 304 zu antworten
      if (Status)
	{
	  Code ("Status: ");
	  Code (Status);
	  Zeilenende ();
	}

#endif

      http_angefangen = ja;
    }
}


static void
HTTP_Ende (void)
{
  // Leerzeile
  Zeilenende ();

  if (Anfragemethode == HEAD)
    fertig ();
}


// Links im HTTP-Kopf (RFC-5988)
// insbesondere für die Textschnittstelle für spezielle Clients
static void
HTTP_Links (int vorheriges, int naechstes)
{
  if (vorheriges < 0 && naechstes < 0)
    return;

  HTTP_Anfang (NULL);

  Code ("Link:");

  if (Kapitel >= 0)
    {
      Code (" <index");
      Code ((Format == TEXT) ? TEXTERWEITERUNG : XHTMLERWEITERUNG);
      Code (">; rel=\"contents\"");
    }

  if (vorheriges >= 0)
    {
      if (Kapitel >= 0)
	Byte (',');

      Code (" <");
      Zahl (vorheriges);
      Code ((Format == TEXT) ? TEXTERWEITERUNG : XHTMLERWEITERUNG);
      Code (">; rel=\"prev\"");
    }

  // nächstes Kapitel wird evtl. auch schon im Vorraus geladen
  if (naechstes >= 0)
    {
      if (Kapitel >= 0 || vorheriges >= 0)
	Byte (',');

      Code (" <");
      Zahl (naechstes);
      Code ((Format == TEXT) ? TEXTERWEITERUNG : XHTMLERWEITERUNG);
      Code (">; rel=\"next\"");
    }

  Zeilenende ();
}


// Kopf für XHTML-Ausgabe (nicht für Fehlermeldungen!)
static void
Kopf (const char *Titel, long int Haltbarkeit)
{
  HTTP_Anfang (NULL);
  Cache_Kontrolle (Haltbarkeit);
  HTTP_Zeitangabe ("Last-Modified", Modifikationszeit);
  Content_Type (TYP ";charset=" ZEICHENKODIERUNG);

#ifdef SPRACHE
  Code ("Content-Language: " SPRACHE NL);
#endif

  if (GESAMTTEXT == Kapitel)
    Code ("Content-Disposition: attachment" NL);

  HTTP_Ende ();

  Code (XHTML_ANFANG);
  Code ("<head>" NL);

  // nur für Rückwärtskompatibilität mit klassischem HTML,
  // wird ignoriert, wenn es als XHTML gelesen wird.
  Code ("<meta http-equiv=\"Content-Type\" "
	"content=\"text/html; charset=" ZEICHENKODIERUNG "\" />" NL);

  Code ("<title>");
  Text (Titel);
  Code ("</title>" NL);

  if (*Autor)
    {
      Code ("<meta name=\"author\" content=\"");
      Text (Autor);
      Code ("\" />" NL);
    }

  // Bitte drin lassen
  Code ("<meta name=\"generator\" content=\"lesen/" VERSION
	" von Andreas K. Foerster - https://akfoerster.de/p/lesen/\" />" NL);

#ifdef METADATEN
  Code (METADATEN NL);
#endif

  Code ("<style type=\"text/css\">" NL);
  Code ((Kapitel == GESAMTTEXT) ? GESAMTCSS : CSS);
  Code (NL "</style>" NL);

#ifdef CSSURL
  if (Kapitel != GESAMTTEXT)
    Code ("<link rel=\"stylesheet\" href=\"" CSSURL "\" />" NL);
#endif

  Code ("</head>" NL NL "<body><div id=\"Inhalt\">" NL NL);

  xhtml_body = ja;
}


static inline noreturn void
Ende (void)
{
  Code (NL "</div></body></html>" NL);
  fertig ();
}


static void
Hauptlinks (void)
{
#ifdef HOMEPAGE
  Code ("<li><a href=\"" HOMEPAGEURL "\" " EXTERNER_LINK ">");
  Text (HOMEPAGE);
  Code ("</a></li>" NL);
#endif

#ifdef IMPRESSUM
  Code ("<li><a href=\"" IMPRESSUMURL "\" " EXTERNER_LINK ">");
  Text (IMPRESSUM);
  Code ("</a></li>");
#endif
}


// Einleitung für reine Textausgabe (nicht für Fehlermeldungen!)
static void
Textkopf (long int Haltbarkeit)
{
  HTTP_Anfang (NULL);
  Cache_Kontrolle (Haltbarkeit);
  HTTP_Zeitangabe ("Last-Modified", Modifikationszeit);
  Content_Type ("text/plain;charset=" ZEICHENKODIERUNG);

#ifdef SPRACHE
  Code ("Content-Language: " SPRACHE NL);
#endif

  if (GESAMTTEXT == Kapitel)
    Code ("Content-Disposition: attachment" NL);

  HTTP_Ende ();
}


static noreturn void
Textfehlerseite (const char *Meldung)
{
  Content_Type ("text/plain;charset=" ZEICHENKODIERUNG);
  HTTP_Ende ();

  Klartext ("* ");

  if (*FEHLEREINLEITUNG)
    Klartext (FEHLEREINLEITUNG);

  Klartext (Meldung);

  if (*FEHLERANHANG)
    Klartext (FEHLERANHANG);

  Zeilenende ();

  fertig ();
}


// HTTP_Anfang() muss shon aufgerufen worden sein
static noreturn void
Fehlerseite (const char *Meldung, const char *Objekt)
{
  // keine Modifikationszeit oder Cache-Kontrolle
  if (Format == TEXT)
    Textfehlerseite (Meldung);

  // XHTML-Ausgabe
  Content_Type (TYP ";charset=" ZEICHENKODIERUNG);

  HTTP_Ende ();

  Code (XHTML_ANFANG);
  Code ("<head>" NL);
  Code (CSS);

#ifdef CSSURL
  Code ("<link rel=\"stylesheet\" href=\"" CSSURL "\" />" NL);
#endif

  Code ("<title>");
  Text (Meldung);
  Code ("</title>" NL);

#ifdef METADATEN_FEHLERSEITE
  Code (METADATEN_FEHLERSEITE NL);
#endif

  Code ("</head>" NL NL "<body class=\"Fehler\">"
	"<div id=\"Inhalt\" class=\"Fehler\">" NL);
  xhtml_body = ja;

  Code ("<h1 class=\"Fehler\">");

  if (*FEHLEREINLEITUNG)
    Text (FEHLEREINLEITUNG);

  Text (Meldung);

  if (*FEHLERANHANG)
    Text (FEHLERANHANG);

  if (Objekt)
    {
      Code (NL "<br />(");
      Text (Objekt);
      Byte (')');
    }

  Code ("</h1>" NL NL);

  Code ("<div id=\"Fuss\">" NL "<ul>" NL);
  Hauptlinks ();
  Code (NL "</ul>" NL "</div>" NL);
  Code (NL "</div></body></html>" NL);

  fertig ();
}


static noreturn void
Fehler (enum Fehlerarten Art)
{
  switch (Art)
    {
    case Buchfehler:
      HTTP_Anfang ("404 Not Found");
      Fehlerseite (FALSCHES_BUCH, Kurzname);
      break;

    case Kapitelfehler:
      HTTP_Anfang ("404 Not Found");
      Fehlerseite (FALSCHES_KAPITEL, NULL);
      break;

    case Zugriffsfehler:
      HTTP_Anfang ("403 Forbidden");
      Fehlerseite (GESPERRT, Kurzname);
      break;

    case Methodenfehler:
      HTTP_Anfang ("405 Method Not Allowed");
      Code (ALLOW NL);
      Fehlerseite (UNUNTERSTUETZT, Methodenname[Anfragemethode]);
      break;

    case Verfuegbarkeitsfehler:
      HTTP_Anfang ("410 Gone");
      Fehlerseite (VERSCHWUNDEN, Kurzname);
      break;

    case Unterstuetzungsfehler:
      HTTP_Anfang ("501 Not Implemented");
      Code (ALLOW NL);
      Fehlerseite (UNUNTERSTUETZT, NULL);
      break;

    case Anfragefehler:
      HTTP_Anfang ("400 Bad Request");
      Fehlerseite (UNUNTERSTUETZT, NULL);
      break;

    default:
      HTTP_Anfang ("500 Internal Server Error");
      Fehlerseite (SERVERFEHLER, NULL);
      break;
    }
}


// analysiert HTTP/1.1 Zeitangabe (RFC-1123, keine älteren Formate)
// tm_yday bleibt ungesetzt
// Bei Fehler ist tm_mday==0
static void
analysiere_Zeitangabe (struct tm *z, const char *Zeit)
{
  // Auf jeden Fall alles löschen, einschließlich undokumentierter Felder
  memset (z, 0, sizeof (*z));

  if (!Zeit)
    return;

  Zeit = Textanfang (Zeit);

  if (strlen (Zeit) < 29 || Zeit[3] != ',' || Zeit[28] != 'T')
    return;

  z->tm_mday = strtol (Zeit + 5, NULL, 10);
  z->tm_year = strtol (Zeit + 12, NULL, 10) - 1900;
  z->tm_hour = strtol (Zeit + 17, NULL, 10);
  z->tm_min = strtol (Zeit + 20, NULL, 10);
  z->tm_sec = strtol (Zeit + 23, NULL, 10);
  z->tm_isdst = 0;		// GMT

  int i;
  const char *p;

  // Monat
  p = Zeit + 8;
  for (i = 0; i < 12; ++i)
    if (!memcmp (p, Monatsabk[i], 3))
      {
	z->tm_mon = i;
	break;
      }

  // Wochentag
  p = Zeit;
  for (i = 0; i < 7; ++i)
    if (!memcmp (p, Wochentagsabk[i], 3))
      {
	z->tm_wday = i;
	break;
      }
}


// Vergleicht Zeitangaben ohne Berücksichtigung der Zeitzone
// Ergebnis ist kleiner 0, gleich 0, oder größer 0
static int
Zeitenvergleich (const struct tm *z1, const struct tm *z2)
{
  int e;

  if (!(e = z1->tm_year - z2->tm_year)
      && !(e = z1->tm_mon - z2->tm_mon)
      && !(e = z1->tm_mday - z2->tm_mday)
      && !(e = z1->tm_hour - z2->tm_hour) && !(e = z1->tm_min - z2->tm_min))
    e = z1->tm_sec - z2->tm_sec;

  return e;
}


// muss der Inhalt überhaupt gesendet werden?
static void
pruefe_Modifikationszeit (long int Haltbarkeit)
{
  if (Cachezeit.tm_mday && gueltige_Zeit (Modifikationszeit)
      && Zeitenvergleich (&Cachezeit, gmtime (&Modifikationszeit)) >= 0)
    {
      HTTP_Anfang ("304 Not Modified");
      if (Haltbarkeit > 0)
	Cache_Kontrolle (Haltbarkeit);
      HTTP_Ende ();
      // hier darf überhaupt kein Inhalt mehr folgen
      fertig ();
    }
}


#ifdef HTTP

static char Host[256];

static inline void
Basisadresse (void)
{
  Code ("http://");
  // Host enthält auch den Port, falls nötig
  Code (Host);
  Byte ('/');
}

#else
// CGI/NPH

static void
Basisadresse (void)
{
  const char *port = getenv ("SERVER_PORT");
  const char *https = getenv ("HTTPS");

  // http oder https?
  // Erkennung nicht wirklich standardisiert
  if ((https && *https) || (port && !strcmp ("443", port)))
    {
      Code ("https://");
      if (port && !strcmp ("443", port))
	port = NULL;
    }
  else
    {
      Code ("http://");
      if (port && !strcmp ("80", port))
	port = NULL;
    }

  const char *host = getenv ("SERVER_NAME");
  if (!host || !*host)
    host = "localhost";

  Code (host);

  if (port && *port)
    {
      Byte (':');
      Code (port);
    }

  const char *script_name = getenv ("SCRIPT_NAME");
  if (script_name)
    Code (script_name);

  Byte ('/');
}

#endif


static noreturn void
Umleitung (const char *Buch)
{
  HTTP_Anfang ("301 Moved Permanently");
  Code ("Location: ");

  Basisadresse ();

  // falls ein Buch gewählt wurde
  if (Buch && *Buch)
    {
      urlkodiert (Buch);

      if (Kapitel != GESAMTTEXT)
	Byte ('/');
    }

  if (Kapitel == UNBEKANNT)
    Code ("index");
  else if (Kapitel >= 0)
    Zahl (Kapitel);

  Code ((Format == TEXT) ? TEXTERWEITERUNG : XHTMLERWEITERUNG);
  Zeilenende ();

  /*
   * Es ist nicht wirklich ein Fehler,
   * die Seite wird aber nur angezeigt,
   * wenn Weiterleitungen deaktiviert sind...
   */
  Fehlerseite (ADRESSFEHLER, NULL);
}


static void
ermittle_Dateimodifikationszeit (Datei d)
{
  struct stat status;
  if (fstat (fileno (d), &status) == 0 && status.st_mtime <= jetzt)
    Modifikationszeit = status.st_mtime;
}


// fängt die Datei mit einem UTF-8 BOM an? (diesem nutzlosen Ärgernis)
static void
pruefe_BOM (Datei d)
{
  if (lies_Byte (d) != '\xEF'
      || lies_Byte (d) != '\xBB' || lies_Byte (d) != '\xBF')
    Dateianfang (d);
}


static Datei
oeffne_Text (void)
{
  size_t Kurznamenlaenge = strlen (Kurzname);
  char Pfad[Kurznamenlaenge + sizeof (DATEIERWEITERUNG)];
  // sizeof() beinhaltet Terminator

  strcpy (Pfad, Kurzname);
  strcpy (Pfad + Kurznamenlaenge, DATEIERWEITERUNG);

  Datei d = oeffne_Datei (Pfad);

  if (!d)
    Fehler (errno == EACCES ? Zugriffsfehler : Buchfehler);

  ermittle_Dateimodifikationszeit (d);
  pruefe_Modifikationszeit (BUCHHALTBARKEIT);
  pruefe_BOM (d);

  return d;
}


// entferne Freiraum und Zeilenumbrüche am Ende des Strings
static void
bereinige_Ende (char *s)
{
  if (!s || !*s)
    return;

  size_t l = strlen (s) - 1;
  while (l && isspace (s[l]))
    --l;

  s[l + 1] = '\0';
}


// eine Leerzeile besteht nur aus Leerzeichen
static bool
Leerzeile (const char *s)
{
  while (*s && isspace (*s))
    ++s;

  return (*s == '\0');
}


// eine Trennzeile besteht nur aus Leerzeichen und Sternen
static bool
Trennzeile (const char *s)
{
  while (*s)
    {
      if (*s != '*' && !isspace (*s))
	return nein;
      ++s;
    }

  return ja;
}


// gibt Kapitelnummer zurück, oder UNBEKANNT
static int
Kapitelzeile (const char *z)
{
  int k = UNBEKANNT;

  if (isspace (*z))
    {
      char *e;
      k = strtol (z, &e, 10);

      // Keine Zahl am Anfang oder kein Punkt folgend?
      if (!e || e == z || *e != '.')
	k = UNBEKANNT;
    }

  return k;
}


static void
Absatzanfang (long int Nr)
{
  if (Nr > 0)
    {
      Code (NL "<p id=\"a");
      Zahl (Nr);
      Code ("\">");
    }
  else
    Code (NL "<p>");

  if (Alternativschrift)
    Code (ALTERNATIVTEXTANFANG);

  if (betont)
    Code ("<em>");

  Zeilenende ();
}


static void
Absatzende (void)
{
  if (betont)
    Code ("</em>");

  if (Alternativschrift)
    Code (ALTERNATIVTEXTENDE);

  Code ("</p>" NL);
}


// ermittle Buchtitel und Autor - oder Umleitungsanweisung
static void
Anfangszeile (Datei d)
{
  char Zeile[ZEILENBUFFER];

  // Leerzeilen ignorieren
  do
    {
      if (!lies_Zeile (d, Zeile, sizeof (Zeile)))
	Fehler (Verfuegbarkeitsfehler);
    }
  while (Leerzeile (Zeile));

  char *z = Textanfang (Zeile);

  // Umleitungsdatei?
  if (*z == '>')
    {
      char *Ziel;

      Ziel = Textanfang (z + 1);
      bereinige_Ende (Ziel);

      if (!*Ziel)
	Fehler (Verfuegbarkeitsfehler);

      url_dekodieren (Ziel);

      // Sicherheitsüberprüfungen
      if (!strcmp (Ziel, Kurzname) || strchr (Ziel, '/'))
	Fehler (Scheisse);

      Umleitung (Ziel);
    }

  char *p = strchr (z, ':');
  if (p)
    {
      *p = '\0';
      kopiere (Autor, sizeof (Autor), z);
      bereinige_Ende (Autor);

      ++p;
    }
  else				// keine Autorenangabe
    p = z;

  kopiere (Buchtitel, sizeof (Buchtitel), Textanfang (p));
  bereinige_Ende (Buchtitel);
}


// gibt Nummer des nächsten Kapitels zurück, oder UNBEKANNT
static int
naechstes_Kapitel (Datei d)
{
  Stelle aktuelles_Kapitel;

  if (Stelle_merken (d, &aktuelles_Kapitel))
    Fehler (Scheisse);

  int naechstes = UNBEKANNT;
  char Zeile[ZEILENBUFFER];
  while (naechstes < 0 && lies_Zeile (d, Zeile, sizeof (Zeile)))
    naechstes = Kapitelzeile (Zeile);

  if (Stelle_wiederherstellen (d, &aktuelles_Kapitel))
    Fehler (Scheisse);

  return naechstes;
}


static void
Intro (Datei d)
{
  char Zeile[ZEILENBUFFER];
  bool Absatz = nein, Container = nein;

  while (lies_Zeile (d, Zeile, sizeof (Zeile)) && Kapitelzeile (Zeile) < 0)
    {
      if (Leerzeile (Zeile))
	{
	  if (Absatz)
	    {
	      Absatzende ();
	      Absatz = nein;
	    }
	}
      else if (Trennzeile (Zeile))
	{
	  if (Absatz)
	    {
	      Absatzende ();
	      Absatz = nein;
	    }

	  Code (NL "<hr />" NL);
	}
      else
	{
	  if (!Container)
	    {
	      Code ("<div class=\"Intro\">");
	      Container = ja;
	    }

	  if (!Absatz)
	    {
	      Absatzanfang (0L);
	      Absatz = ja;
	    }

	  Text (Zeile);
	}
    }

  if (Absatz)
    Absatzende ();

  if (Container)
    Code ("</div>" NL);
}


static void
Inhaltsverzeichnis (Datei d)
{
  Code (NL "<div class=\"Inhaltsverzeichnis\" id=\"Inhaltsverzeichnis\">" NL
	"<h2>" INHALTSVERZEICHNIS "</h2>" NL NL);

  Dateianfang (d);
  bool gestartet = nein;

  char Zeile[ZEILENBUFFER];
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      int k = Kapitelzeile (Zeile);

      if (k >= 0)
	{
	  bereinige_Ende (Zeile);

	  char *Kapitelname = Textanfang (Zeile);

	  if (!gestartet)
	    {
	      Code ("<ul>" NL);
	      gestartet = ja;
	    }

	  if (k == 0)		// Vorwort
	    {
	      char *n = strchr (Kapitelname, '.');
	      if (n)
		Kapitelname = Textanfang (n + 1);

	      Code ("<li><a rel=\"chapter\" href=\"");
	      if (Kapitel == GESAMTTEXT)
		Code ("#k0\">");
	      else
		Code ("0" XHTMLERWEITERUNG "\">");
	      Text (Kapitelname);
	      Code ("</a></li>" NL);
	    }
	  else
	    {
	      Code ("<li><a rel=\"chapter\" href=\"");
	      if (Kapitel == GESAMTTEXT)
		Code ("#k");
	      Zahl (k);
	      if (Kapitel != GESAMTTEXT)
		Code (XHTMLERWEITERUNG);
	      Code ("\">");
	      Text (Kapitelname);
	      Code ("</a></li>" NL);
	    }
	}
    }

  if (gestartet)
    Code ("</ul>" NL);

  Code ("</div>" NL NL);
}


static noreturn void
zeige_Inhaltstext (void)
{
  char Zeile[ZEILENBUFFER];

  Datei d = oeffne_Text ();
  Anfangszeile (d);

  HTTP_Links (UNBEKANNT, naechstes_Kapitel (d));
  Textkopf (BUCHHALTBARKEIT);

  // erstmal alles bis einschließlich der ersten Kapitelzeile
  Dateianfang (d);
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      Klartext (Zeile);
      if (Kapitelzeile (Zeile) >= 0)
	break;
    }

  // dann nur noch die Kapitelzeilen
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      if (Kapitelzeile (Zeile) >= 0)
	Klartext (Zeile);
    }

  schliesse_Datei (d);
  fertig ();
}


static inline void
Software (void)
{
#ifdef SOFTWARE
  Code ("<li><a href=\"" SOFTWAREURL "\">");
  Text (SOFTWARE);
  Code ("</a></li>" NL);
#endif
}


static void
Titelanzeige (void)
{
  Code ("<div class=\"Kopf\">" NL);
  Code ("<h1 class=\"Titel\">");
  Text (Buchtitel);
  Code ("</h1>" NL);

  if (*Autor)
    {
      Code ("<h2 class=\"Autor\">");
      Text (Autor);
      Code ("</h2>" NL);
    }
  Code ("</div>" NL NL);
}


static noreturn void
zeige_Inhalt (void)
{
  Datei d = oeffne_Text ();
  Anfangszeile (d);

  HTTP_Links (UNBEKANNT, naechstes_Kapitel (d));
  Kopf (Buchtitel, (BUCHHALTBARKEIT));

  Code ("<div id=\"Kopf\">" NL "<ul>" NL);
  Hauptlinks ();

  /*
     Code ("<li><a href=\"../index" XHTMLERWEITERUNG "\">");
     Text (INDEX);
     Code ("</a></li>" NL);
   */

#ifdef ALTERNATIVE
  Code ("<li><a href=\"" ALTERNATIVEURL);
  urlkodiert (Kurzname);
  Code ("/index" XHTMLERWEITERUNG "\" " ALTERNATIVER_LINK ">");
  Text (ALTERNATIVE);
  Code ("</a></li>" NL);
#endif

  Code ("</ul>" NL "</div>" NL NL);

  Titelanzeige ();

  Intro (d);
  Inhaltsverzeichnis (d);

  Code ("<div id=\"Fuss\">" NL "<ul>" NL);

#if defined (ALTERNATIVEXHTML) && defined (ALTERNATIVEURL)
  Code ("<li><a class=\"download\" href=\"" ALTERNATIVEURL);
  urlkodiert (Kurzname);
  Code (XHTMLERWEITERUNG "\" " ALTERNATIVER_LINK ">");
  Text (ALTERNATIVEXHTML);
  Code ("</a></li>" NL);
#endif

#ifdef XHTMLDATEI
  Code ("<li><a class=\"download\" href=\"../");
  urlkodiert (Kurzname);
  Code (XHTMLERWEITERUNG
	"\" " ALTERNATIVER_LINK " type=\"application/xhtml+xml\">");
  Text (XHTMLDATEI);
  Code ("</a></li>" NL);
#endif

  Code ("<li><a class=\"download\" href=\"../");
  urlkodiert (Kurzname);
  Code (TEXTERWEITERUNG "\" " ALTERNATIVER_LINK " type=\"text/plain\">");
  Text (TEXTDATEI);
  Code ("</a></li>" NL);

#ifdef TEXTVERSION
  Code ("<li><a href=\"index" TEXTERWEITERUNG
	"\" " ALTERNATIVER_LINK " type=\"text/plain\">");
  Text (TEXTVERSION);
  Code ("</a></li>" NL);
#endif

#ifdef SOFTWARE
  Software ();
#endif

  Code ("</ul>" NL "</div>" NL);
  schliesse_Datei (d);

  Ende ();
}


static void
Kapitelnavigation (int vorheriges, int naechstes)
{
  Code ("<div class=\"Kapitelnavigation\">" NL);

  if (vorheriges >= 0)
    {
      Code ("<a rel=\"prev chapter\" class=\"Navigation\" href=\"");
      Zahl (vorheriges);
      Code (XHTMLERWEITERUNG "\" style=\"float:left\">\u2190</a>" NL);
    }

  if (naechstes >= 0)
    {
      Code ("<a rel=\"next chapter\" class=\"Navigation\" href=\"");
      Zahl (naechstes);
      Code (XHTMLERWEITERUNG "\" style=\"float:right\">\u2192</a>" NL);
    }

  Code ("<a rel=\"contents\" class=\"Inhaltsverzeichnis\" href=\"index"
	XHTMLERWEITERUNG "\">");
  Text (INHALTSVERZEICHNIS);
  Code ("</a>" NL "</div>" NL);
}


static int
Kapitelinhalt (Datei d, const char *Titel, char *Zeile)
{
  Code (NL "<div class=\"Kapitel\" id=\"k");
  Zahl (Kapitel);
  Code ("\">" NL "<h2>");
  Text (Titel);
  Code ("</h2>" NL);

  // war die letzte Zeile eine Leerzeile?
  bool war_leer = ja;
  int k = UNBEKANNT;

  while (lies_Zeile (d, Zeile, ZEILENBUFFER)
	 && (k = Kapitelzeile (Zeile)) < 0)
    {
      if (Leerzeile (Zeile))
	{
	  if (!war_leer)
	    {
	      Absatzende ();
	      war_leer = ja;
	    }
	}
      else if (Trennzeile (Zeile))
	{
	  if (!war_leer)
	    {
	      Absatzende ();
	      war_leer = ja;
	    }

	  Code (NL "<hr />" NL);
	}
      else
	{
	  static long int AbsatzNr;

	  if (war_leer)
	    Absatzanfang (++AbsatzNr);

	  Text (Zeile);
	  war_leer = nein;
	}
    }

  if (!war_leer)
    {
      Absatzende ();
      war_leer = ja;
    }

  Code ("</div>" NL);

  return k;
}


// entfernt Leerzeichen und evtl Kapitelnummer 0.
static char *
Titelanfang (const char *Titel)
{
  // Vorwort ohne Kapitelnummer
  if (Kapitel == 0)
    Titel += strspn (Titel, "0. \t\v\f\n\r");
  else
    Titel += strspn (Titel, " \t\v\f\n\r");

  return (char *) Titel;
}


static noreturn void
Kapitelanzeige (Datei d, const char *Titel, int vorheriges, int naechstes)
{
  Titel = Titelanfang (Titel);
  Kopf (Titel, (BUCHHALTBARKEIT));

  Code ("<div id=\"Kopf\">" NL "<ul>" NL);
  Hauptlinks ();

#ifdef ALTERNATIVE
  Code ("<li><a href=\"" ALTERNATIVEURL);
  urlkodiert (Kurzname);
  Byte ('/');
  Zahl (Kapitel);
  Code (XHTMLERWEITERUNG "\" " ALTERNATIVER_LINK ">");
  Text (ALTERNATIVE);
  Code ("</a></li>" NL);
#endif

  Code ("</ul>" NL NL);

  Kapitelnavigation (vorheriges, naechstes);

  Code ("</div>" NL NL);

  Code ("<h1 class=\"Titel\">");
  Text (Buchtitel);
  Code ("</h1>" NL NL);

  char Zeile[ZEILENBUFFER];
  Kapitelinhalt (d, Titel, Zeile);

  Code (NL "<div id=\"Fuss\">" NL);
  Kapitelnavigation (vorheriges, naechstes);

  Code (NL "<ul>" NL);

  if (naechstes >= 0)
    {
      Code ("<li><a rel=\"next chapter\" class=\"weiter\" href=\"");
      Zahl (naechstes);
      Code (XHTMLERWEITERUNG "\">");
      bereinige_Ende (Zeile);
      Text (Textanfang (Zeile));
      Code ("</a></li>" NL);
    }

#ifdef TEXTVERSION
  Code ("<li><a href=\"");
  Zahl (Kapitel);
  Code (TEXTERWEITERUNG "\" " ALTERNATIVER_LINK " type=\"text/plain\">");
  Text (TEXTVERSION);
  Code ("</a></li>" NL);
#endif

  Code ("</ul>" NL "</div>" NL);

  schliesse_Datei (d);
  Ende ();
}


static noreturn void
zeige_Kapiteltext (Datei d, const char *Titel)
{
  Textkopf (BUCHHALTBARKEIT);

  Klartext (Autor);		// auch wenn nicht angegeben
  Code (": ");
  Klartext (Buchtitel);
  Code (NL NL);
  Klartext (Titel);

  char Zeile[ZEILENBUFFER];
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      Klartext (Zeile);

      // nächste Kapitelzeile wird bewusst noch mitgesendet
      if (Kapitelzeile (Zeile) >= 0)
	break;
    }

  schliesse_Datei (d);
  fertig ();
}


static noreturn void
zeige_Kapitel (void)
{
  char Zeile[ZEILENBUFFER];
  int aktuelles, vorheriges;

  aktuelles = vorheriges = UNBEKANNT;

  Datei d = oeffne_Text ();
  Anfangszeile (d);

  // suche Kapitelanfang
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      aktuelles = Kapitelzeile (Zeile);
      if (aktuelles >= 0)
	{
	  if (aktuelles == Kapitel)
	    break;
	  else
	    vorheriges = aktuelles;
	}
    }

  if (aktuelles != Kapitel)
    Fehler (Kapitelfehler);

  int naechstes = naechstes_Kapitel (d);

  HTTP_Links (vorheriges, naechstes);

  if (Format == TEXT)
    zeige_Kapiteltext (d, Zeile);
  else
    {
      bereinige_Ende (Zeile);
      Kapitelanzeige (d, Zeile, vorheriges, naechstes);
    }
}


// sende reine Textdaten, evtl. mit Filter
static noreturn void
sende_Textdaten (void)
{
  Datei d = oeffne_Text ();
  Anfangszeile (d);
  Dateianfang (d);
  Textkopf (BUCHHALTBARKEIT);

  char Zeile[ZEILENBUFFER];
  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    Klartext (Zeile);

  schliesse_Datei (d);
  fertig ();
}


// sende Gesamtausgabe als XHTML
static noreturn void
sende_xhtml (void)
{
  Datei d = oeffne_Text ();
  Anfangszeile (d);

  Kopf (Buchtitel, BUCHHALTBARKEIT);
  Titelanzeige ();
  Intro (d);
  Inhaltsverzeichnis (d);

  // suche ersten Kapitelanfang
  Dateianfang (d);
  char Zeile[ZEILENBUFFER];
  while (Kapitel < 0 && lies_Zeile (d, Zeile, sizeof (Zeile)))
    Kapitel = Kapitelzeile (Zeile);

  while (Kapitel >= 0)
    {
      char *Titel = Titelanfang (Zeile);
      bereinige_Ende (Titel);
      Kapitel = Kapitelinhalt (d, Textanfang (Titel), Zeile);
    }

  schliesse_Datei (d);
  Ende ();
}


static void
Index_Start (void)
{
  Kopf (INDEX, (INDEXHALTBARKEIT));
  Code ("<div id=\"Kopf\">" NL "<ul>" NL);
  Hauptlinks ();

#ifdef ANDERE
  Code ("<li><a href=\"" ANDEREURL "\">");
  Text (ANDERE);
  Code ("</a></li>" NL);
#endif

#ifdef ALTERNATIVE
  Code ("<li><a href=\"" ALTERNATIVEURL "index" XHTMLERWEITERUNG
	"\" " ALTERNATIVER_LINK ">");
  Text (ALTERNATIVE);
  Code ("</a></li>" NL);
#endif

  Code ("</ul>" NL "</div>" NL NL);
  Code ("<div class=\"Index\">" NL "<h1>");
  Text (INDEX);
  Code ("</h1>" NL NL);

  Code ("<ul>" NL);
}


static noreturn void
Index_Ende (void)
{
  Code ("</ul>" NL "</div>" NL);

#if defined(SOFTWARE) || defined(TEXTVERSION)
  Code ("<div id=\"Fuss\">" NL "<ul>" NL);

#ifdef TEXTVERSION
  Code ("<li><a href=\"");
  Code (Kurzname);
  Code (TEXTERWEITERUNG "\" " ALTERNATIVER_LINK " type=\"text/plain\">");
  Text (TEXTVERSION);
  Code ("</a></li>" NL);
#endif

#ifdef SOFTWARE
  Software ();
#endif

  Code ("</ul>" NL "</div>" NL);
#endif

  Ende ();
}


static void
Indexerstellung (DIR * d, void (*Ausgabe) (const char *, const char *))
{
  struct dirent *Eintrag;

  while (NULL != (Eintrag = readdir (d)))
    {
      char *Name = Eintrag->d_name;
      char *Endung = strrchr (Name, '.');
      if (Endung && Endung != Name && !strcmp (Endung, DATEIERWEITERUNG))
	{
	  *Endung = '\0';
	  if (strcmp (Name, "index"))
	    Ausgabe (Name, NULL);
	}
    }
}


#define Indexname(n)  (!strncmp ((n), "index", 5))


static void
Rohindexeintrag (const char *Name, const char *Titel)
{
  urlkodiert (Name);

  if (Titel && *Titel)
    {
      Byte ('\t');
      Klartext (Titel);
    }

  Zeilenende ();
}


static void
Indexeintrag (const char *Name, const char *Titel)
{
  bool Verzeichnis, Link;

  Link = (strpbrk (Name, ":/") != NULL);
  Verzeichnis = Link ? nein : Indexname (Name);

  Code ("<li class=\"");
  Code (Link ? "Link" : Verzeichnis ? "Verzeichnis" : "Buch");
  Code ("\"><a href=\"");

  if (Link)
    {
      Code (Name);
      Code ("\" " EXTERNER_LINK ">");
    }
  else
    {
      urlkodiert (Name);
      if (!Verzeichnis)
	Code ("/index");
      Code (XHTMLERWEITERUNG "\">");
    }

  Text ((Titel && *Titel) ? Titel : Name);
  Code ("</a></li>" NL);
}


static noreturn void
Verzeichnisindex (void)
{
  DIR *d = opendir (".");
  if (!d)
    Fehler (Scheisse);

  struct stat status;
  if (stat (".", &status) == 0 && status.st_mtime <= jetzt)
    Modifikationszeit = status.st_mtime;

  pruefe_Modifikationszeit (INDEXHALTBARKEIT);

  if (Format == TEXT)
    {
      Textkopf (INDEXHALTBARKEIT);
      Indexerstellung (d, &Rohindexeintrag);
      closedir (d);
    }
  else
    {
      Index_Start ();
      Indexerstellung (d, &Indexeintrag);
      closedir (d);
      Index_Ende ();
    }

  fertig ();
}


static noreturn void
Indextext (Datei d)
{
  char Zeile[ZEILENBUFFER];

  Textkopf (INDEXHALTBARKEIT);

  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      // nur Zeilen, die nicht mit Leerzeichen anfangen,
      // oder leer sind
      if (!isspace (*Zeile))
	Klartext (Zeile);
    }

  schliesse_Datei (d);
  fertig ();
}


static noreturn void
zeige_Indexdatei (Datei d)
{
  char Zeile[ZEILENBUFFER];

  Index_Start ();

  while (lies_Zeile (d, Zeile, sizeof (Zeile)))
    {
      if (!Leerzeile (Zeile))
	{
	  bereinige_Ende (Zeile);
	  char *Name = Zeile;
	  char *Titel = strpbrk (Name, " \t");

	  // fängt mit Leerzeichen an?
	  if (Titel == Name)
	    continue;

	  if (Titel)
	    {
	      *Titel = '\0';
	      Titel = Textanfang (Titel + 1);
	    }
	  else
	    Titel = Name;

	  url_dekodieren (Name);
	  Indexeintrag (Name, Titel);
	}
    }

  schliesse_Datei (d);
  Index_Ende ();
}


static noreturn void
zeige_Index (void)
{
  size_t Kurznamenlaenge = strlen (Kurzname);
  char Pfad[Kurznamenlaenge + sizeof (DATEIERWEITERUNG)];

  strcpy (Pfad, Kurzname);
  strcpy (Pfad + Kurznamenlaenge, DATEIERWEITERUNG);

  Datei d = oeffne_Datei (Pfad);
  if (!d)
    {
      if (!strcmp ("index", Kurzname))
	Verzeichnisindex ();
      else
	Fehler (Buchfehler);
    }

  ermittle_Dateimodifikationszeit (d);
  pruefe_Modifikationszeit (INDEXHALTBARKEIT);
  pruefe_BOM (d);

  if (Format == TEXT)
    Indextext (d);
  else
    zeige_Indexdatei (d);
}


/*
Diese Funktion setzt die Variablen Kurzname, Kapitel und Format,
oder veranlasst eine Fehlermeldung oder Umleitung.
*/
static enum Anfragetyp
Pfad_auswerten (const char *Verzeichnis, const char *Pfad)
{
  Kapitel = UNBEKANNT;
  Format = Formatlos;

  if (!Pfad || !*Pfad || !strcmp ("/", Pfad))
    Umleitung (NULL);		// Buchindex

  kopiere (Kurzname, sizeof (Kurzname), Pfad + 1);
  char *Endung = strrchr (Kurzname, '.');
  if (Endung && !strcmp (Endung, TEXTERWEITERUNG))
    {
      *Endung = '\0';
      Format = TEXT;
    }
  else if (Endung && !strcmp (Endung, XHTMLERWEITERUNG))
    {
      *Endung = '\0';
      Format = XHTML;
    }

  /*
     Fehlermeldungen möglichst erst erzeugen, wenn klar ist,
     ob eine reine Textanfrage vorliegt.
   */

#ifdef HTTP
  if (!*Host)			// Der Host-Eintrag ist ab HTTP/1.1 Pflicht
    Fehler (Anfragefehler);
#endif

  if (chdir (Verzeichnis))
    Fehler (Scheisse);

  // nichts außer Erweiterung?
  if (!*Kurzname)
    Fehler (Buchfehler);

  // Sicherheitsabfrage
  if (strstr (Pfad, "/../"))
    Fehler (Buchfehler);
  else if (Anfragemethode == unbekannte_Methode)
    Fehler (Unterstuetzungsfehler);
  else if (Anfragemethode > maximale_Anfragemethode)
    Fehler (Methodenfehler);

  // andere Datei, zB. lesen.css
  if (Endung && *Endung && !strchr (Endung, '/'))
    return Dateianfrage;

  if (Indexname (Kurzname))
    {
      if (strchr (Kurzname, '/'))
	Fehler (Kapitelfehler);
      else if (Formatlos == Format)
	Umleitung (NULL);
      else
	return Indexanfrage;
    }

  // bis zum Schrägstrich
  char *p = strchr (Kurzname, '/');
  if (p)
    *p = '\0';
  else if (Format == TEXT)	// ohne Schrägstrich
    {
      Kapitel = GESAMTTEXT;
      return Gesamttextanfrage;
    }
  else if (Format == XHTML)	// ohne Schrägstrich
    {
      Kapitel = GESAMTTEXT;
      return Gesamtxhtmlanfrage;
    }
  else
    Umleitung (Kurzname);

  // Kapitel ermitteln
  p = strchr (Pfad + 1, '/');
  if (p)
    {
      ++p;
      if (!*p)			// endet auf Schrägstrich
	Umleitung (Kurzname);
      else if (isdigit (*p))
	{
	  char *e;
	  Kapitel = strtol (p, &e, 10);
	  if (*e == '\0')
	    Umleitung (Kurzname);
	  else if (Formatlos == Format)	// unbekannte Endung
	    Fehler (Kapitelfehler);

	  return Kapitelanfrage;
	}
      else if (!Indexname (p))
	Fehler (Kapitelfehler);
      else if (Formatlos == Format)
	Umleitung (Kurzname);
    }

  return Inhaltanfrage;
}


#ifdef HTTP

// NULL für unbekannten Typ
static noreturn void
sende_Datei (const char *Typ)
{
  Datei d;

  d = oeffne_Binaerdatei (Kurzname);
  if (!d)
    Fehler (errno == EACCES ? Zugriffsfehler : Buchfehler);

  off_t Laenge = 0;
  struct stat status;
  if (!fstat (fileno (d), &status))
    {
      Laenge = status.st_size;
      if (status.st_mtime <= jetzt)
	{
	  Modifikationszeit = status.st_mtime;
	  pruefe_Modifikationszeit (0);
	}
    }

  HTTP_Anfang (NULL);
  HTTP_Zeitangabe ("Last-Modified", Modifikationszeit);

  if (Laenge)
    {
      Code ("Content-Length: ");
      Zahl (Laenge);
      Zeilenende ();
    }

  if (Typ && *Typ)
    Content_Type (Typ);

  HTTP_Ende ();

  int c;
  while ((c = lies_Byte (d)) != EOF)
    Byte (c);

  schliesse_Datei (d);
  fertig ();
}


static enum Anfragetyp
Anfrageauswertung (int argc, char *argv[])
{
  char Zeile[ZEILENBUFFER];

  if (!lies_Zeile (stdin, Zeile, sizeof (Zeile)) || Leerzeile (Zeile))
    Fehler (Anfragefehler);

  char *p = strtok (Zeile, "? \t\r\n");
  Anfragemethode = Methodentyp (p);

  // Pfad/URL
  p = strtok (NULL, "? \t\r\n");

  if (!p)
    p = "";
  else
    {
      // volle URI? :-(
      char *Hostname = strstr (p, "://");
      if (Hostname)
	{
	  Hostname += 3;
	  size_t l = strcspn (Hostname, "/ \t?\r\n");

	  if (l < sizeof (Host))
	    {
	      memcpy (Host, Hostname, l);
	      Host[l] = '\0';
	    }

	  p = Hostname + l;
	}
    }

  // Erstmal zwischenspeichern;
  // Fehlermelungen kommen später, wenn die Anfrage komplett gelesen wurde
  char Pfad[strlen (p) + 1];
  strcpy (Pfad, p);

  // Rest der Anfrage lesen
  while (lies_Zeile (stdin, Zeile, sizeof (Zeile)) && !Leerzeile (Zeile))
    {
      if (!strncasecmp ("If-Modified-Since:", Zeile, 18))
	analysiere_Zeitangabe (&Cachezeit, Zeile + 18);
      // Die Host-Angabe in der URI hat Vorrang
      else if (!*Host && !strncasecmp ("Host:", Zeile, 5))
	{
	  char *Hostname = Textanfang (Zeile + 5);
	  bereinige_Ende (Hostname);
	  if (strlen (Hostname) < sizeof (Host))
	    strcpy (Host, Hostname);
	  else
	    *Host = '\0';
	}
    }

  if (*Pfad)
    url_dekodieren (Pfad);
  else
    *Host = '\0';

  return Pfad_auswerten ((argc >= 2) ? argv[1] : VERZEICHNIS, Pfad);
}

#else
// CGI

#define DIREKTAUFRUFMELDUNG  DIREKTAUFRUF "\a\n"

static enum Anfragetyp
Anfrageauswertung (int argc, char *argv[])
{
  (void) argc;
  (void) argv;

  const char *request_method = getenv ("REQUEST_METHOD");
  if (!request_method)
    {
      // keine CGI-Umgebung
      write (STDERR_FILENO, DIREKTAUFRUFMELDUNG,
	     sizeof (DIREKTAUFRUFMELDUNG) - 1);
      fertig ();
    }

  Anfragemethode = Methodentyp (request_method);
  analysiere_Zeitangabe (&Cachezeit, getenv ("HTTP_IF_MODIFIED_SINCE"));

  // PATH_INFO ist bereits url-dekodiert
  return Pfad_auswerten (VERZEICHNIS, getenv ("PATH_INFO"));
}

#endif


int
main (int argc, char *argv[])
{
  time (&jetzt);

  switch (Anfrageauswertung (argc, argv))
    {
    case Indexanfrage:
      zeige_Index ();
      break;

    case Inhaltanfrage:
      if (Format == TEXT)
	zeige_Inhaltstext ();
      else
	zeige_Inhalt ();
      break;

    case Kapitelanfrage:
      zeige_Kapitel ();
      break;

    case Gesamtxhtmlanfrage:
      sende_xhtml ();
      break;

    case Gesamttextanfrage:
      sende_Textdaten ();
      break;

    case Dateianfrage:
#ifdef HTTP
      if (chdir ((argc >= 3) ? argv[2] : WEBVERZEICHNIS))
	Fehler (errno == EACCES ? Zugriffsfehler : Buchfehler);

      sende_Datei (Medientyp (Kurzname));
#else
      Fehler (Buchfehler);
#endif
      break;
    }

  return 0;
}
