/*
 * Medientypen
 * Copyright © 2015,2017 Andreas K. Förster
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Medientypen.h"

#include <string.h>
#include <strings.h>
#include <stdlib.h>

#define ZEICHENKODIERUNG "UTF-8"

struct Medieneintrag
{
  const char *Endung, *Typ;
};

// Endungen müssen in alphabetischer Reihefolge sein!
// Endungen sind ohne Unterscheidung von Groß- und Kleinschreibung.
static const struct Medieneintrag Medientypen[] = {
  {"asc", "text/plain;charset=" ZEICHENKODIERUNG},
  {"atom", "application/atom+xml"},
  {"au", "audio/basic"},
  {"bz2", "application/x-bzip"},
  {"cbz", "application/x-cbz"},
  {"css", "text/css;charset=" ZEICHENKODIERUNG},
  {"djv", "image/vnd.djvu"},
  {"djvu", "image/vnd.djvu"},
  {"eps", "application/postscript"},
  {"epub", "application/epub+zip"},
  {"flac", "audio/flac"},
  {"gif", "image/gif"},
  {"gz", "application/gzip"},
  {"htm", "text/html;charset=" ZEICHENKODIERUNG},
  {"html", "text/html;charset=" ZEICHENKODIERUNG},
  {"ico", "image/vnd.microsoft.icon"},
  {"jpeg", "image/jpeg"},
  {"jpg", "image/jpeg"},
  {"js", "application/javascript"},
  {"json", "application/json"},
  {"key", "application/pgp-keys"},
  {"m3u", "audio/mpegurl"},
  {"m4a", "audio/mp4"},
  {"mka", "audio/x-matroska"},
  {"mkv", "video/x-matroska"},
  {"mp1", "audio/mpeg"},
  {"mp2", "audio/mpeg"},
  {"mp3", "audio/mpeg"},
  {"mp4", "video/mp4"},
  {"mpega", "audio/mpeg"},
  {"mpga", "audio/mpeg"},
  {"oga", "audio/ogg"},
  {"ogg", "audio/ogg"},
  {"ogv", "video/ogg"},
  {"opus", "audio/ogg"},
  {"otf", "application/font-sfnt"},
  {"pdf", "application/pdf"},
  {"pls", "audio/x-scpls"},
  {"png", "image/png"},
  {"ps", "application/postscript"},
  {"rss", "application/x-rss+xml"},
  {"sig", "application/pgp-signature"},
  {"snd", "audio/basic"},
  {"svg", "image/svg+xml"},
  {"svgz", "image/svg+xml"},
  {"tar", "application/x-tar"},
  {"text", "text/plain;charset=" ZEICHENKODIERUNG},
  {"tif", "image/tiff"},
  {"tiff", "image/tiff"},
  {"torrent", "application/x-bittorrent"},
  {"ttf", "application/font-sfnt"},
  {"txt", "text/plain;charset=" ZEICHENKODIERUNG},
  {"wav", "audio/wave"},
  {"wave", "audio/wave"},
  {"webm", "video/webm"},
  {"woff", "application/font-woff"},
  {"xht", "application/xhtml+xml;charset=" ZEICHENKODIERUNG},
  {"xhtml", "application/xhtml+xml;charset=" ZEICHENKODIERUNG},
  {"xml", "application/xml;charset=" ZEICHENKODIERUNG},
  {"xsl", "application/xslt+xml;charset=" ZEICHENKODIERUNG},
  {"xslt", "application/xslt+xml;charset=" ZEICHENKODIERUNG},
  {"xz", "application/x-xz"},
  {"zip", "application/zip"}
};

/*
Diese Liste ist keineswegs vollständig, und kann gerne
noch erweitert oder auch gekürzt werden.

Die mehrfach auftretenden Typen werden von den meisten
Compilern nur einmal gespeichert. Man sollte nur darauf
achten, die wirklich immer gleich zu schreiben.
*/

static int
Medienvergleich (const void *s1, const void *s2)
{
  const struct Medieneintrag *a = s1, *b = s2;
  return strcasecmp (a->Endung, b->Endung);
}


extern const char *
Medientyp (const char *Dateiname)
{
  const size_t Groesse = sizeof (Medientypen[0]);
  const size_t Anzahl = sizeof (Medientypen) / Groesse;
  struct Medieneintrag Suche, *Ergebnis;

  Ergebnis = NULL;
  const char *Endung = strrchr (Dateiname, '.');

  if (Endung && !strchr (Endung, '/'))
    {
      // Punkt überspringen
      Suche.Endung = Endung + 1;

      Ergebnis =
	bsearch (&Suche, Medientypen, Anzahl, Groesse, &Medienvergleich);
    }

  return Ergebnis ? Ergebnis->Typ : NULL;
}
