/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2016,2018 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

// Referenz: RFC 3986


#include "akfnetz.h"
#include <string.h>

static const char *
Anfang (const char *URL)
{
  const char *a;

  if (!URL)
    return NULL;

  a = strstr (URL, "://");
  if (a)
    a += 3;
  else
    a = URL;

  return a;
}


static const char *
Hostanfang (const char *URL)
{
  const char *a, *e;

  a = Anfang (URL);
  if (!a)
    return NULL;

  e = strpbrk (a, "@/?# \t\r\n");

  // evtl. Zugangsdaten ueberspringen
  if (e && *e == '@')
    a = e + 1;

  return a;
}


// speichert Teilstring in Ziel
static char *
speichere (char *Ziel, size_t Laenge, const char *Wert, size_t l)
{
  *Ziel = '\0';
  if (l >= Laenge)
    return NULL;

  memcpy (Ziel, Wert, l);
  Ziel[l] = '\0';

  return Ziel;
}


// Gibt Hostnamen und evtl. Port aus
extern char *
akfnetz_URL_Host (char *Host, size_t Laenge, const char *URL)
{
  const char *Name;

  if (!Host || !Laenge)
    return NULL;

  *Host = '\0';
  Name = Hostanfang (URL);
  if (!Name)
    return NULL;

  return speichere (Host, Laenge, Name, strcspn (Name, "/?# \t\r\n"));
}


extern char *
akfnetz_URL_Hostname (char *Name, size_t Laenge, const char *URL)
{
  const char *a;
  size_t l;

  if (!Name || !Laenge)
    return NULL;

  *Name = '\0';

  a = Hostanfang (URL);
  if (!a)
    return NULL;

  if (*a != '[')
    l = strcspn (a, ":/?# \t\r\n");
  else
    {
      // IPv6-Adresse
      ++a;
      l = strcspn (a, "]");
    }

  return speichere (Name, Laenge, a, l);
}


extern char *
akfnetz_URL_Port (char *Name, size_t Laenge, const char *URL)
{
  const char *a;
  size_t l;

  if (!Name || !Laenge)
    return NULL;

  *Name = '\0';

  a = Hostanfang (URL);
  if (!a)
    return NULL;

  if (*a != '[')
    a = strpbrk (a, ":/?# \t\r\n");
  else
    {
      a = strchr (a, ']');
      if (a)
	++a;
    }

  if (!a || *a != ':')
    return NULL;

  ++a;
  l = 0;
  while ('0' <= a[l] && a[l] <= '9')
    l++;

  return speichere (Name, Laenge, a, l);
}


extern const char *
akfnetz_URL_Pfad (const char *URL)
{
  const char *a;

  a = Anfang (URL);
  if (!a)
    return NULL;

  return strpbrk (a, "/?#");
}


extern char *
akfnetz_URL_Zugang (char *Zugang, size_t Laenge, const char *URL)
{
  const char *Name;
  size_t l;

  if (!Zugang || !Laenge)
    return NULL;

  *Zugang = '\0';

  Name = Anfang (URL);
  if (!Name)
    return NULL;

  l = strcspn (Name, "/@?# \t\r\n");

  if (Name[l] != '@')
    return NULL;

  return speichere (Zugang, Laenge, Name, l);
}
