/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2018,2019 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

// RFC 1928

#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "akfnetz.h"

#define SOCKSVERSION 5

#define A_VERBINDUNG 1
#define A_ANBINDUNG 2
#define A_UDP 3

#define ADR_IPv4 1
#define ADR_NAME 3
#define ADR_IPv6 4

static int
Namensignorierung (int f)
{
  unsigned char l;
  char Name[256];

  if (akfnetz_Eingabe (f, &l, 1) < 0 || akfnetz_Eingabe (f, &Name, l) < 0)
    return -1;

  return 0;
}


static int
Adresse (int f, const void *s)
{
  const struct sockaddr_in *s4;
  const struct sockaddr_in6 *s6;
  size_t l;
  unsigned char b[256];

  s4 = s;
  // Adressfamilie und Port sind in beiden an gleicher Stelle

  b[0] = SOCKSVERSION;
  b[1] = A_VERBINDUNG;
  b[2] = 0;			// reserviert
  l = 3;

  switch (s4->sin_family)
    {
    case AF_INET:
      b[l++] = ADR_IPv4;
      memcpy (b + l, &s4->sin_addr.s_addr, 4);
      l += 4;
      break;

    case AF_INET6:
      b[l++] = ADR_IPv6;
      s6 = s;
      memcpy (b + l, &s6->sin6_addr.s6_addr, 16);
      l += 16;
      break;

    default:
      errno = EAFNOSUPPORT;
      return -1;
    }

  memcpy (b + l, &s4->sin_port, 2);
  l += 2;

  return akfnetz_Ausgabe (f, b, l);
}


static int
Domainname (int f, const char *Host, in_port_t Port)
{
  size_t l, hl;
  unsigned char b[4 + 256 + 2];

  b[0] = SOCKSVERSION;
  b[1] = A_VERBINDUNG;
  b[2] = 0;			// reserviert
  b[3] = ADR_NAME;
  l = 4;

  hl = strlen (Host);
  if (hl > 255)
    {
      errno = EMSGSIZE;
      return -1;
    }

  b[l++] = (unsigned char) hl;
  memcpy (b + l, Host, hl);
  l += hl;

  memcpy (b + l, &Port, 2);
  l += 2;

  return akfnetz_Ausgabe (f, b, l);
}


static int
Antwort (int f)
{
  in_port_t Port;
  unsigned char b[4];
  unsigned char Adresse[16];

  if (akfnetz_Eingabe (f, &b, 4) < 0 || b[0] != SOCKSVERSION)
    return -1;

  switch (b[1])
    {
    case 0:
      // Alles klar
      break;

    case 2:
    case 5:
      errno = ECONNREFUSED;
      break;

    case 1:
    case 3:
      errno = ENETUNREACH;
      break;

    case 4:
      errno = EHOSTUNREACH;
      break;

    case 6:
      errno = ETIMEDOUT;
      break;

    case 8:
      errno = EAFNOSUPPORT;
      break;

    default:
      errno = ECONNREFUSED;
      break;
    }

  if (b[1])
    return -1;

  // Adresse lesen & ignorieren
  switch (b[3])
    {
    case ADR_IPv4:
      if (akfnetz_Eingabe (f, &Adresse, 4) < 0)
	return -1;
      break;

    case ADR_IPv6:
      if (akfnetz_Eingabe (f, &Adresse, 16) < 0)
	return -1;
      break;

    case ADR_NAME:
      if (Namensignorierung (f) < 0)
	return -1;
      break;

    default:
      errno = ENOTSUP;
      return -1;
    }

  // Port lesen & ignorieren
  if (akfnetz_Eingabe (f, &Port, 2) < 0)
    return -1;

  return 0;
}


// RFC 1929
static int
Zugangsdaten (int f, const char *Name, const char *Passwort)
{
  size_t l, nl, pl;
  unsigned char g[512 + 4];

  nl = strlen (Name);
  pl = strlen (Passwort);

  if (nl > 255 || pl > 255)
    {
      errno = EMSGSIZE;
      return -1;
    }

  l = 0;
  g[l++] = 1;

  g[l++] = (unsigned char) nl;
  memcpy (g + l, Name, nl);
  l += nl;

  g[l++] = (unsigned char) pl;
  memcpy (g + l, Passwort, pl);
  l += pl;

  if (akfnetz_Ausgabe (f, &g, l) < 0)
    return -1;

  if (akfnetz_Eingabe (f, &g, 2) < 0)
    return -1;

  if (g[1])
    {
      errno = ECONNREFUSED;
      return -1;
    }

  return 0;
}


static int
Anmeldung (int f, const char *Name, const char *Passwort)
{
  size_t l;
  unsigned char g[80];

  if (!Name || !*Name || !Passwort || !*Passwort)
    Name = Passwort = NULL;

  l = 0;
  g[l++] = SOCKSVERSION;

  if (Name)
    {
      g[l++] = 2;		// Anzahl Autorisierungsmethoden
      g[l++] = 2;		// Name / Passwort
    }
  else
    g[l++] = 1;			// Anzahl Autorisierungsmethoden

  g[l++] = 0;			// ohne Autorisierung

  if (akfnetz_Ausgabe (f, &g, l) < 0)
    return -1;

  if (akfnetz_Eingabe (f, &g, 2) < 0)
    return -1;

  if (g[0] != SOCKSVERSION || g[1] == 0xFF)
    {
      errno = ECONNREFUSED;
      return -1;
    }

  // ohne Zugangsdaten erlaubt
  if (g[1] == 0)
    return 0;

  if (g[1] == 2 && Name)
    return Zugangsdaten (f, Name, Passwort);

  errno = ECONNREFUSED;
  return -1;
}


extern int
akfnetz_socks5_Verbindung (int proxy,
			   const char *Name, const char *Passwort,
			   const void *sockaddr)
{
  if (!sockaddr)
    {
      errno = EDESTADDRREQ;
      return -1;
    }

  if (Anmeldung (proxy, Name, Passwort) < 0 || Adresse (proxy, sockaddr) < 0
      || Antwort (proxy) < 0)
    return -1;

  return proxy;
}


extern int
akfnetz_socks5h_Verbindung (int proxy,
			    const char *Name, const char *Passwort,
			    const char *Host, int Port)
{
  if (!Host || !*Host)
    {
      errno = EDESTADDRREQ;
      return -1;
    }

  if (0 >= Port || Port > 0xFFFF)
    {
      errno = EDOM;
      return -1;
    }

  if (Anmeldung (proxy, Name, Passwort) < 0
      || Domainname (proxy, Host, htons ((in_port_t) Port)) < 0
      || Antwort (proxy) < 0)
    return -1;

  return proxy;
}
