/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2018,2019 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "akfnetz.h"

#define A_VERBINDUNG 1
#define A_ANBINDUNG 2
#define A_AKZEPTIERT 90
#define A_ABGELEHNT 91
#define A_KEIN_INETD 92
#define A_UNAUTORISIERT 93

struct Anfrage
{
  unsigned char Version, Status;
  in_port_t Port;
  in_addr_t Adresse;
};


// String mit Terminator ausgeben
static inline int
Stringausgabe (int f, const char *s)
{
  return akfnetz_Ausgabe (f, s, strlen (s) + 1);
}


static int
anfragen (int f, const char *Id, in_addr_t Adresse, in_port_t Port)
{
  struct Anfrage a;

  a.Version = 4;
  a.Status = A_VERBINDUNG;
  a.Port = Port;
  a.Adresse = Adresse;

  if (akfnetz_Ausgabe (f, &a, sizeof (a)) < 0)
    return -1;

  if (Id && *Id)
    return Stringausgabe (f, Id);
  else
    return akfnetz_Ausgabe (f, "", 1);
}


static int
Antwort (int f)
{
  struct Anfrage a;

  if (akfnetz_Eingabe (f, &a, sizeof (a)) < 0)
    return -1;

  if (a.Status != A_AKZEPTIERT)
    {
      errno = ECONNREFUSED;
      return -1;
    }

  return 0;
}


extern int
akfnetz_socks4_Verbindung (int proxy, const char *Id, const void *sockaddr)
{
  const struct sockaddr_in *s;

  if (!sockaddr)
    {
      errno = EDESTADDRREQ;
      return -1;
    }

  s = sockaddr;

  if (s->sin_family != AF_INET)
    {
      errno = EAFNOSUPPORT;
      return -1;
    }

  // Adressbereich reserviert fuer 4a
  if (!(s->sin_addr.s_addr & htonl (0xFFFFFF00)))
    {
      errno = EPERM;
      return -1;
    }

  if (anfragen (proxy, Id, s->sin_addr.s_addr, s->sin_port) < 0
      || Antwort (proxy) < 0)
    return -1;

  return proxy;
}


extern int
akfnetz_socks4a_Verbindung (int proxy, const char *Id,
			    const char *Host, int Port)
{
  if (!Host || !*Host)
    {
      errno = EDESTADDRREQ;
      return -1;
    }

  if (0 >= Port || Port > 0xFFFF)
    {
      errno = EDOM;
      return -1;
    }

  // bewusst ungueltige Adresse
  if (anfragen (proxy, Id, htonl (1), htons ((in_port_t) Port)) < 0
      || Stringausgabe (proxy, Host) < 0 || Antwort (proxy) < 0)
    return -1;

  return proxy;
}
