/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2019-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <stdbool.h>
#include <uchar.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include "akfnetz.h"

#define TEXTRAND 70
#define TRENNZEILENLAENGE 72
#define ARGUMENTGROESSE 10239

#define LISTENPUNKTSYMBOL "*"
#define UNGUELTIG u8"\uFFFD"
#define LEERZEICHEN ' '

// Terminal-Steuersequenzen
#define CSI "\033["
#define UEBERSCHRIFT CSI "1;4m"
#define BETONT CSI "4m"
#define FETT CSI "1m"
#define NORMALSCHRIFT CSI "m"

// schaltet zwischen Ziffernwert und ASCII um (beide Richtungen)
#define Ziffer(x) ((x) ^ 0x30)

#pragma GCC poison  puts fputc

struct Eigenschaften
{
  bool Umbruch, ausgeblendet, formatiert, in_Link;
  int Schalter;
  long int Position, Zitatebene, Einzug, Listennummer, Linknummer;
  char *Argumente;
  FILE *Linkliste;
  char Ausblendname[10];
};


static void Aktion (const char *, FILE *, struct Eigenschaften *);
static void Tag (FILE *, FILE *, struct Eigenschaften *);
static bool Tagende (FILE *, FILE *, const char *, int,
		     struct Eigenschaften *);
static bool Kommentar (FILE *);
static bool CDATA (FILE *, FILE *, struct Eigenschaften *);
static void neue_Zeile (FILE *, struct Eigenschaften *);
static void Entitaet (FILE *, FILE *, struct Eigenschaften *);
static void Linkanfang (FILE *, struct Eigenschaften *);
static void Linkende (FILE *, struct Eigenschaften *);
static bool relativer_Link (const char *);
static void Zitateinleitung (FILE *, struct Eigenschaften *);
static void Absatz (FILE *, struct Eigenschaften *);
static void Trennzeile (FILE *, struct Eigenschaften *);
static void Listenelement (FILE *, struct Eigenschaften *);
static void Zeichenausgabe (int, FILE *, struct Eigenschaften *);
static void Unicode (char32_t, FILE *);
static void Bild (FILE *, struct Eigenschaften *);
static const char *suche (const char *, const char *);
static const char *Attribut (const char *, const char *);
static void Attributwertausgabe (FILE *, struct Eigenschaften *,
				 const char *);
static void wiederhole (FILE *, int, int);
inline static bool UTF8_Startbyte (unsigned char);
inline static bool geschaltet (struct Eigenschaften *, int);
inline static bool Terminalausgabe (struct Eigenschaften *);


// s ist der Tagname kleingeschrieben
static void
Aktion (const char *s, FILE * aus, struct Eigenschaften *e)
{
  // fprintf (aus, "<%s %s>", s, e->Argumente);

  if (!strcmp ("p", s) || !strcmp ("div", s) || !strcmp ("tr", s)
      || !strcmp ("main", s) || !strcmp ("address", s)
      || !strcmp ("article", s) || !strcmp ("section", s)
      || !strcmp ("aside", s) || !strcmp ("br", s) || !strcmp ("dt", s)
      || !strcmp ("figure", s) || !strcmp ("figcaption", s))
    Absatz (aus, e);
  else if (s[0] == 'h' && s[1] >= '1' && s[1] <= '6' && !s[2])
    {
      Absatz (aus, e);
      if (Terminalausgabe (e) && !e->ausgeblendet)
	fputs (UEBERSCHRIFT, aus);
      else			// atx/Markdown
	{
	  int Ebene = Ziffer (s[1]);
	  wiederhole (aus, '#', Ebene);
	  putc (LEERZEICHEN, aus);
	  e->Position += Ebene + 1;
	}
    }
  else if (s[0] == '/' && s[1] == 'h' && s[2] >= '1' && s[2] <= '6' && !s[3])
    {
      if (Terminalausgabe (e) && !e->ausgeblendet)
	fputs (NORMALSCHRIFT, aus);
      Absatz (aus, e);
    }
  else if (!strcmp ("a", s))
    Linkanfang (aus, e);
  else if (!strcmp ("/a", s))
    Linkende (aus, e);
  else if (!strcmp ("blockquote", s))
    {
      ++e->Zitatebene;
      Absatz (aus, e);
    }
  else if (!strcmp ("/blockquote", s))
    {
      if (e->Zitatebene > 0)
	--e->Zitatebene;
    }
  else if (!strcmp ("hr", s))
    Trennzeile (aus, e);
  else if (!strcmp ("dd", s))
    {
      if (e->Einzug <= 70)
	e->Einzug += 4;
      Absatz (aus, e);
    }
  else if (!strcmp ("/dd", s))
    {
      if (e->Einzug >= 4)
	e->Einzug -= 4;
      Absatz (aus, e);
    }
  else if (!strcmp ("ol", s))
    e->Listennummer = 1;
  else if (!strcmp ("ul", s) || !strcmp ("/ol", s))
    e->Listennummer = 0;
  else if (!strcmp ("li", s))
    Listenelement (aus, e);
  else if (!e->ausgeblendet && (!strcmp ("/td", s) || !strcmp ("/th", s)))
    putc ('\t', aus);
  else if (!strcmp ("pre", s))
    {
      e->formatiert = true;
      Absatz (aus, e);
    }
  else if (!strcmp ("/pre", s))
    e->formatiert = false;
  else if (!e->ausgeblendet
	   && (!strcmp ("em", s) || !strcmp ("i", s) || !strcmp ("u", s)
	       || !strcmp ("mark", s)))
    {
      if (Terminalausgabe (e))
	fputs (BETONT, aus);
      else
	{
	  putc ('_', aus);
	  ++e->Position;
	}
    }
  else if (!e->ausgeblendet
	   && (!strcmp ("/em", s) || !strcmp ("/i", s) || !strcmp ("/u", s)
	       || !strcmp ("/mark", s)))
    {
      if (Terminalausgabe (e))
	fputs (NORMALSCHRIFT, aus);
      else
	{
	  putc ('_', aus);
	  ++e->Position;
	}
    }
  else if (!e->ausgeblendet && (!strcmp ("strong", s) || !strcmp ("b", s)))
    {
      if (Terminalausgabe (e))
	fputs (FETT, aus);
      else
	{
	  fputs ("__", aus);
	  e->Position += 2;
	}
    }
  else if (!e->ausgeblendet && (!strcmp ("/strong", s) || !strcmp ("/b", s)))
    {
      if (Terminalausgabe (e))
	fputs (NORMALSCHRIFT, aus);
      else
	{
	  fputs ("__", aus);
	  e->Position += 2;
	}
    }
  else if (!e->ausgeblendet && (!strcmp ("code", s) || !strcmp ("/code", s)))
    {
      putc ('`', aus);
      ++e->Position;
    }
  else if (!e->ausgeblendet &&
	   (!strcmp ("head", s) || !strcmp ("title", s)
	    || !strcmp ("script", s) || !strcmp ("form", s)
	    || !strcmp ("style", s) || !strcmp ("del", s)
	    || !strcmp ("nav", s) || !strcmp ("svg", s)
	    || !strcmp ("header", s) || !strcmp ("footer", s)))
    {
      strcpy (e->Ausblendname, s);
      e->ausgeblendet = true;
    }
  else if (e->ausgeblendet && s[0] == '/' && !strcmp (e->Ausblendname, s + 1))
    e->ausgeblendet = false;
  else if (!strcmp ("img", s))
    Bild (aus, e);
  else if ((!strcmp ("meta", s) && suche (e->Argumente, "charset"))
	   || (!strcmp ("?xml", s) && strstr (e->Argumente, "encoding")))
    {
      if (suche (e->Argumente, "UTF-8"))
	e->Schalter |= AKFNETZ_SC_UTF8;
    }

  /*
     Viele Tags werden bewusst ignoriert.
     Zum Beispiel bei noscript, object, audio, video
     soll der Inhalt angezeigt und nicht ausgeblendet werden.
   */
}


inline static bool
geschaltet (struct Eigenschaften *e, int x)
{
  return ((e->Schalter & x) != 0);
}


inline static bool
Terminalausgabe (struct Eigenschaften *e)
{
  return geschaltet (e, AKFNETZ_SC_TERMINAL);
}


static void
Tag (FILE * ein, FILE * aus, struct Eigenschaften *e)
{
  size_t l;
  int z;
  bool selbstschliessend;
  char Name[40];

  e->Argumente[0] = '\0';
  l = 0;

  // Zeichen zum Verschieben frei lassen
  while (l < (sizeof (Name) - 2))
    {
      z = getc (ein);

      // das erste Zeichen auf jeden Fall uebernehmen
      if (l > 0 && !isalnum (z))
	break;

      Name[l++] = (char) tolower (z);
    }

  Name[l] = '\0';

  selbstschliessend = Tagende (ein, aus, Name, z, e);

  if (!l)
    return;

  Aktion (Name, aus, e);

  if (selbstschliessend)
    {
      e->Argumente[0] = '\0';
      memmove (Name + 1, Name, l + 1);
      Name[0] = '/';
      Aktion (Name, aus, e);
    }
}


static bool
Tagende (FILE * ein, FILE * aus, const char *Name, int z,
	 struct Eigenschaften *e)
{
  int a = z;
  size_t l;

  while (isspace (z))
    z = getc (ein);

  l = 0;
  while (z != '>' && z != EOF)
    {
      if (l < ARGUMENTGROESSE)
	e->Argumente[l++] = (char) z;

      if (Name[0] == '!' && !Name[1])
	{
	  if (l == 2 && !memcmp ("--", e->Argumente, 2))
	    return Kommentar (ein);

	  if (l == 7 && !memcmp ("[CDATA[", e->Argumente, 7))
	    return CDATA (ein, aus, e);
	}

      a = z;
      z = getc (ein);
    }

  e->Argumente[l] = '\0';

  // selbstschliessendes Tag?
  return (a == '/' && z == '>');
}


static bool
Kommentar (FILE * ein)
{
  int z1, z2, z3;

  z1 = z2 = z3 = 0;

  while (z3 != '-' || z2 != '-' || z1 != '>')
    {
      z3 = z2;
      z2 = z1;
      z1 = getc (ein);

      if (z1 == EOF)
	break;
    }

  return false;
}


static bool
CDATA (FILE * ein, FILE * aus, struct Eigenschaften *e)
{
  size_t l = 0;
  int z1, z2, z3;

  e->Argumente[0] = '\0';
  z2 = z3 = 0;

  z1 = getc (ein);
  while (z3 != ']' || z2 != ']' || z1 != '>')
    {
      if (l >= ARGUMENTGROESSE)
	{
	  if (!e->ausgeblendet)
	    {
	      fwrite (e->Argumente, l, 1, aus);
	      e->Position += l;
	    }
	  l = 0;
	  // eventueller Darstellungsfehler bewusst in Kauf genommen
	}

      e->Argumente[l++] = z1;

      z3 = z2;
      z2 = z1;
      z1 = getc (ein);

      if (z1 == EOF)
	break;
    }

  if (!e->ausgeblendet && l > 2)
    {
      fwrite (e->Argumente, l - 2, 1, aus);
      e->Position += l - 2;
    }

  return false;
}


static bool
relativer_Link (const char *s)
{
  size_t l;

  // Laenge: Scheme
  l = strspn (s, "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
	      "abcdefghijklmnopqrstuvwxyz0123456789+-.");

  return (!l || s[l] != ':' || !strncasecmp ("javascript", s, 10));
}


// schreibt Linkadressen erstmal in Temporaerdatei
static void
Linkanfang (FILE * aus, struct Eigenschaften *e)
{
  const char *p;
  char z;

  if (!e->Linkliste || e->ausgeblendet)
    return;

  p = Attribut (e->Argumente, "href");
  if (!p)
    return;

  z = *p;
  if (z != '"' && z != '\'')
    return;

  ++p;

  if (relativer_Link (p) || ++(e->Linknummer) <= 0)
    return;

  e->in_Link = true;
  putc ('[', aus);
  ++e->Position;

  fprintf (e->Linkliste, "\n[%ld]: ", e->Linknummer);

  while (*p && *p != z)
    {
      // HTML-Zeichen dekodieren, aber URL-Kodierung belassen!
      if (*p != '&')
	putc (*p, e->Linkliste);
      else			// HTML-Zeichen dekodieren
	{
	  Unicode (akfnetz_Entitaet (++p), e->Linkliste);
	  p = strpbrk (p, ";&'\"<>");
	  if (!p)		// kaputtes HTML
	    return;
	}

      ++p;
    }
}


static void
Linkende (FILE * aus, struct Eigenschaften *e)
{
  if (e->Linkliste && e->in_Link && e->Linknummer > 0)
    e->Position += fprintf (aus, "][%ld]", e->Linknummer);

  e->in_Link = false;
}


static void
Zitateinleitung (FILE * aus, struct Eigenschaften *e)
{
  if (e->Zitatebene > 0)
    {
      wiederhole (aus, '>', e->Zitatebene);
      putc (LEERZEICHEN, aus);
      e->Position += e->Zitatebene + 1;
    }
}

static void
Absatz (FILE * aus, struct Eigenschaften *e)
{
  if (e->Umbruch || e->ausgeblendet)
    return;

  fputs ("\n\n", aus);
  e->Umbruch = true;
  e->Position = 0;

  Zitateinleitung (aus, e);
}


static void
wiederhole (FILE * aus, int c, int n)
{
  for (int i = 0; i < n; ++i)
    putc (c, aus);
}


static void
Trennzeile (FILE * aus, struct Eigenschaften *e)
{
  if (e->ausgeblendet)
    return;

  if (!e->Umbruch)
    fputs ("\n\n", aus);

  wiederhole (aus, '-', TRENNZEILENLAENGE);

  fputs ("\n\n", aus);
  e->Position = 0;
  e->Umbruch = true;
}


static void
Listenelement (FILE * aus, struct Eigenschaften *e)
{
  if (e->ausgeblendet)
    return;

  if (!e->Umbruch)
    putc ('\n', aus);

  e->Position = 0;

  if (e->Listennummer)
    e->Position = fprintf (aus, "\n%ld. ", e->Listennummer++);
  else
    {
      fputs ("\n" LISTENPUNKTSYMBOL " ", aus);
      e->Position = 2;
    }

  // kein weiterer Umbruch, auch wenn <p> folgt
  e->Umbruch = true;
}


static void
neue_Zeile (FILE * aus, struct Eigenschaften *e)
{
  putc ('\n', aus);
  e->Position = 0;
  Zitateinleitung (aus, e);
}


static void
Unicode (char32_t z, FILE * aus)
{
  size_t l;
  char u[5];

  l = akfnetz_UTF8 (u, z);
  fwrite (u, l, 1, aus);
}


#if 0
// Locale-abhaengig
static void
Unicode (char32_t z, FILE * aus)
{
  size_t l;
  char u[MB_LEN_MAX];

  l = c32rtomb (u, z, NULL);

  if (l == (size_t) (-1))
    fputs (UNGUELTIG, aus);
  else
    fwrite (u, l, 1, aus);
}
#endif


inline static bool
UTF8_Startbyte (unsigned char z)
{
  return (z < 0x80 || z > 0xC0);
}


static void
Zeichenausgabe (int c, FILE * aus, struct Eigenschaften *e)
{
  unsigned char z;

  if (e->ausgeblendet)
    return;

  z = (unsigned char) c;

  // keine Steuerzeichen
  if (z < 0x20)
    return;

  if (e->Position == 0 && e->Einzug > 0)
    {
      wiederhole (aus, LEERZEICHEN, e->Einzug);
      e->Position += e->Einzug;
    }

  if (z < 0x80 || geschaltet (e, AKFNETZ_SC_UTF8))
    {
      // ASCII oder UTF-8
      putc (z, aus);
      if (UTF8_Startbyte (z))
	++e->Position;
      // FIXME: weiches Trennzeichen als UTF-8?
    }
  else if (z == 0xAD && !geschaltet (e, AKFNETZ_SC_LANGZEILEN))
    {
      // weiches Trennzeichen, 8-Bit
      // Bei Langzeilen als solches ausgeben, sonst nicht
      if (e->Position > TEXTRAND)
	{
	  putc ('-', aus);
	  neue_Zeile (aus, e);
	}
    }
  else
    {
      // ISO-8859-1 / Codepage 1252
      Unicode (akfnetz_cp1252 (z), aus);
      ++e->Position;
    }

  e->Umbruch = false;
}


static void
Entitaet (FILE * ein, FILE * aus, struct Eigenschaften *e)
{
  size_t l;
  int z;
  char Name[40];

  l = 0;

  while (l < (sizeof (Name) - 1))
    {
      z = getc (ein);
      if (z == ';')
	break;

      Name[l++] = (char) z;

      if (z != '#' && !isalnum (z))
	break;
    }

  Name[l] = '\0';

  if (e->ausgeblendet)
    return;

  if (e->Umbruch && e->Einzug > 0)
    {
      wiederhole (aus, LEERZEICHEN, e->Einzug);
      e->Position += e->Einzug;
    }

  e->Umbruch = false;

  if (z != ';')			// kaputtes HTML
    {
      putc ('&', aus);
      fwrite (Name, l, 1, aus);
      e->Position += l + 1;
      return;
    }

  char32_t Zeichen = akfnetz_Entitaet (Name);

  // weiches Trennzeichen?
  if (Zeichen == 0xAD && !geschaltet (e, AKFNETZ_SC_LANGZEILEN))
    {
      if (e->Position > TEXTRAND)
	{
	  putc ('-', aus);
	  neue_Zeile (aus, e);
	}
    }
  else
    {
      Unicode (Zeichen, aus);
      ++e->Position;
    }
}


static void
Bild (FILE * aus, struct Eigenschaften *e)
{
  const char *alt, *src;

  if (e->ausgeblendet)
    return;

  alt = Attribut (e->Argumente, "alt");
  src = Attribut (e->Argumente, "src");
  if (!alt || !*alt || !src)
    return;

  if (geschaltet (e, AKFNETZ_SC_GRAFIK))
    {
      fputs ("![", aus);
      Attributwertausgabe (aus, e, alt);
      fputs ("](", aus);
      Attributwertausgabe (aus, e, src);
      putc (')', aus);
    }
  else
    {
      putc ('{', aus);
      Attributwertausgabe (aus, e, alt);
      putc ('}', aus);
    }
}


static const char *
suche (const char *gesamt, const char *Ausdruck)
{
  if (!Ausdruck || !*Ausdruck || !gesamt)
    return gesamt;

  size_t l = strlen (Ausdruck);
  char a = tolower (*Ausdruck);

  for (const char *p = gesamt; *p; ++p)
    {
      if (tolower (*p) == a && !strncasecmp (p, Ausdruck, l))
	return p;
    }

  return NULL;
}


// suche Attribut
static const char *
Attribut (const char *gesamt, const char *Ausdruck)
{
  const char *p;

  p = suche (gesamt, Ausdruck);
  if (!p)
    return NULL;

  p += strlen (Ausdruck);
  while (isspace (*p))
    ++p;

  if (*p != '=')
    return NULL;

  ++p;

  while (isspace (*p))
    ++p;

  return p;
}


static void
Attributwertausgabe (FILE * aus, struct Eigenschaften *e, const char *s)
{
  char g;

  if (!s || e->ausgeblendet)
    return;

  // Wert muss in Anfuehrungsstrichen stehen
  g = *s;
  if (g != '"' && g != '\'')
    return;

  ++s;

  while (*s && *s != g)
    {
      if (isspace (*s))
	{
	  if (e->Position > TEXTRAND
	      && !geschaltet (e, AKFNETZ_SC_LANGZEILEN))
	    neue_Zeile (aus, e);
	  else
	    putc (LEERZEICHEN, aus);

	  while (isspace (*s))
	    ++s;

	  continue;
	}

      if ((unsigned char) *s == 0xAD && !geschaltet (e, AKFNETZ_SC_UTF8)
	  && !geschaltet (e, AKFNETZ_SC_LANGZEILEN))
	{
	  if (e->Position > TEXTRAND)
	    {
	      putc ('-', aus);
	      neue_Zeile (aus, e);
	    }
	  continue;
	}

      if (*s != '&')
	{
	  Zeichenausgabe (*s, aus, e);
	  ++e->Position;
	}
      else			// Entitaet
	{
	  char *Ende = strchr (s, ';');

	  if (!Ende)		// kaputtes HTML
	    {
	      putc ('&', aus);
	      ++s;
	      continue;
	    }

	  char32_t Zeichen = akfnetz_Entitaet (s);
	  if (Zeichen == 0xAD && !geschaltet (e, AKFNETZ_SC_LANGZEILEN))
	    {
	      if (e->Position > TEXTRAND)
		{
		  putc ('-', aus);
		  neue_Zeile (aus, e);
		}
	    }
	  else
	    {
	      Unicode (Zeichen, aus);
	      s = Ende;
	      ++e->Position;
	    }
	}

      ++s;
    }
}


extern int
akfnetz_scrape (FILE * ein, FILE * aus, int Linkignoranz, int Schalter)
{
  size_t Linkpuffergroesse = 0;
  char *Linkpuffer = NULL;
  int c;
  bool Freiraum = true;
  struct Eigenschaften e;

  e.Schalter = Schalter;
  e.in_Link = e.ausgeblendet = e.formatiert = false;
  e.Umbruch = true;
  e.Position = 0;
  e.Einzug = 0;
  e.Listennummer = 0;
  e.Zitatebene = 0;
  e.Linkliste = NULL;
  e.Linknummer = 0 - Linkignoranz;
  e.Ausblendname[0] = '\0';

  e.Argumente = malloc (ARGUMENTGROESSE + 1);
  if (!e.Argumente)
    return -1;

  if (geschaltet (&e, AKFNETZ_SC_LINKS))
    {
      e.Linkliste = open_memstream (&Linkpuffer, &Linkpuffergroesse);
      // bei Fehler keine Liste
    }

  while ((c = getc (ein)) != EOF)
    switch (c)
      {
      case '<':
	Tag (ein, aus, &e);
	break;


	// all diese Zeichen werden als Leerzeichen interpretiert
      case LEERZEICHEN:
      case '\r':
      case '\n':
      case '\t':
      case '\v':
      case '\f':
	if (!e.ausgeblendet)
	  {
	    if (e.formatiert)
	      {
		putc (c, aus);
		++e.Position;
	      }
	    else if (e.Position > TEXTRAND
		     && !geschaltet (&e, AKFNETZ_SC_LANGZEILEN))
	      neue_Zeile (aus, &e);
	    else if (!Freiraum && !e.Umbruch)
	      {
		putc (LEERZEICHEN, aus);
		++e.Position;
	      }

	    Freiraum = true;
	  }
	break;


	// Zeichen nach denen Umbruch erlaubt ist (nur ASCII)
      case '-':
      case '/':
	if (!e.ausgeblendet)
	  {
	    Zeichenausgabe (c, aus, &e);
	    if (e.Position > TEXTRAND
		&& !geschaltet (&e, AKFNETZ_SC_LANGZEILEN))
	      neue_Zeile (aus, &e);
	  }
	break;


      case 0x7F:		// DELETE
	break;


      case '&':
	Entitaet (ein, aus, &e);
	if (!e.ausgeblendet)
	  {
	    Freiraum = false;
	    ++e.Position;
	  }
	break;


      default:
	if (!e.ausgeblendet)
	  {
	    Zeichenausgabe (c, aus, &e);
	    Freiraum = false;
	  }
	break;
      }


  if (e.Linkliste)
    {
      // erst schliessen, damit Puffer aktualisiert wird
      fclose (e.Linkliste);
      e.Linkliste = NULL;

      e.ausgeblendet = false;

      if (e.Linknummer)
	{
	  // keine automatischen Zeilenumbrueche
	  Trennzeile (aus, &e);
	  fwrite (Linkpuffer, 1, Linkpuffergroesse, aus);
	}

      free (Linkpuffer);
    }

  free (e.Argumente);

  fputs ("\n\n", aus);

  return 0;
}
