/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2024-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <sys/socket.h>
#include "akfnetz.h"

// faengt String s mit der Konstanten c an?
#define Praefix(s,c) (strncasecmp (s, "" c, sizeof(c)-1) == 0)

static int Socks5h_Verbindung (int Proxy, const char *proxy_url,
			       const char *Host, int Port);

extern int
akfnetz_Proxy_Verbindung (const char *proxy_url, const char *url)
{
  char *Port;
  char Host[256], Portpuffer[8];

  if (!url || !akfnetz_URL_Hostname (Host, sizeof (Host), url))
    {
      errno = EADDRNOTAVAIL;
      return -1;
    }

  Port = akfnetz_URL_Port (Portpuffer, sizeof (Portpuffer), url);
  if (!Port)
    Port = akfnetz_Standardport (url);

  return akfnetz_Proxy_Verbindung_Host (proxy_url, Host, Port);
}


extern int
akfnetz_Proxy_Verbindung_Host (const char *proxy_url,
			       const char *Host, const char *Port)
{
  int v, Proxy;
  struct sockaddr_storage s;

  if (!proxy_url || !*proxy_url)
    return akfnetz_Serververbindung (Host, Port, AF_UNSPEC);

  // Verbindung zu Proxy herstellen
  Proxy = akfnetz_URL_Verbindung (proxy_url, AF_UNSPEC);
  if (Proxy < 0)
    return -1;

  // Zieladresse uebermitteln
  v = -1;

  if (Praefix (proxy_url, "socks4:") || Praefix (proxy_url, "socks:"))
    v = akfnetz_socks4_Verbindung (Proxy, getenv ("LOGNAME"),
				   akfnetz_IP_Adresse (&s, Host, Port,
						       AF_INET));

  else if (Praefix (proxy_url, "socks4a:"))
    v = akfnetz_socks4a_Verbindung (Proxy, getenv ("LOGNAME"),
				    Host, atoi (Port));

  else if (Praefix (proxy_url, "socks5:") || Praefix (proxy_url, "socks5h:"))
    v = Socks5h_Verbindung (Proxy, proxy_url, Host, atoi (Port));

  if (v < 0)
    close (Proxy);

  return v;
}


static int
Socks5h_Verbindung (int Proxy, const char *proxy_url,
		    const char *Host, int Port)
{
  char *Name, *Passwort;
  char Zugang[800];

  Name = Passwort = NULL;

  if (akfnetz_URL_Zugang (Zugang, sizeof (Zugang), proxy_url))
    {
      char *p;

      akfnetz_url_dekodieren (Zugang);

      p = strchr (Zugang, ':');
      if (p)
	{
	  *p = '\0';
	  Name = Zugang;
	  Passwort = p + 1;
	}
    }

  return akfnetz_socks5h_Verbindung (Proxy, Name, Passwort, Host, Port);
}
