/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <stdbool.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include "akfnetz.h"

#define ADR(a1, a2, a3, a4)  (htonl((a1)<<24|(a2)<<16|(a3)<<8|(a4)))

/*
Verbindung von privaten Netz-Adressen?

privat sind:
127.0.0.0/8  loopback

gemaess RFC-1918:
192.168.0.0/16
172.16.0.0/12
10.0.0.0/8

RFC-3927 (Link-Local):
169.254.0.0/16
*/
static bool
privates_IPv4 (in_addr_t a)
{
  return (bool) ((a & ADR (255, 0, 0, 0)) == ADR (127, 0, 0, 0)
		 || (a & ADR (255, 255, 0, 0)) == ADR (192, 168, 0, 0)
		 || (a & ADR (255, 240, 0, 0)) == ADR (172, 16, 0, 0)
		 || (a & ADR (255, 0, 0, 0)) == ADR (10, 0, 0, 0)
		 || (a & ADR (255, 255, 0, 0)) == ADR (169, 254, 0, 0));
}


extern bool
akfnetz_privates_Netz (const void *s)
{
  switch (((struct sockaddr *) s)->sa_family)
    {
    case AF_INET:
      return privates_IPv4 (((struct sockaddr_in *) s)->sin_addr.s_addr);


    case AF_INET6:
      {
	const struct in6_addr *a6 = &((struct sockaddr_in6 *) s)->sin6_addr;

	/*
	   FC00::/7     Unique Local Unicast (RFC-4193)
	   ::1          localhost
	   FE80::/10    Link-Local
	 */
	if ((a6->s6_addr[0] & 0xFE) == 0xFC
	    || IN6_IS_ADDR_LOOPBACK (a6) || IN6_IS_ADDR_LINKLOCAL (a6))
	  return true;

	if (IN6_IS_ADDR_V4MAPPED (a6))
	  {
	    in_addr_t a;
	    memcpy (&a, a6->s6_addr + (16 - 4), 4);
	    return privates_IPv4 (a);
	  }

	return false;
      }


    case AF_UNIX:		// immer lokal
      return true;
    }

  // im Zweifel abblocken
  return false;
}
