/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfwebserver - Transportschichten
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _POSIX_C_SOURCE 200112L
#define _XOPEN_SOURCE 600
#define _ISOC99_SOURCE

#define _DEFAULT_SOURCE
#define _GNU_SOURCE
#define _BSD_SOURCE
#define _NETBSD_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "akfnetz.h"
#include "akfwebserver.h"

static char *
akfnetz_Hostname (struct sockaddr *s, socklen_t l)
{
  char Name[1025];

  if (getnameinfo (s, l, Name, sizeof (Name), NULL, 0, NI_NAMEREQD))
    return NULL;

  return strdup (Name);
}


// setzt http.Klientenadresse, http.Klientenport und evtl. http.Klientenname
static void
Klientenadresse (void *Klient, socklen_t Groesse)
{
  const struct sockaddr_in *s4 = Klient;
  const struct sockaddr_in6 *s6 = Klient;

  if (http.Klientenname)
    {
      free (http.Klientenname);
      http.Klientenname = NULL;
    }

  switch (((struct sockaddr *) Klient)->sa_family)
    {
    case AF_INET:
      inet_ntop (AF_INET, &s4->sin_addr,
		 http.Klientenadresse, sizeof (http.Klientenadresse));

      http.Klientenport = ntohs (s4->sin_port);

      if (Einstellung.Hostname)
	http.Klientenname = akfnetz_Hostname (Klient, Groesse);
      break;


    case AF_INET6:
      akfnetz_Adressentext (AF_INET6, &s6->sin6_addr,
			    http.Klientenadresse,
			    sizeof (http.Klientenadresse));

      http.Klientenport = ntohs (s6->sin6_port);

      if (Einstellung.Hostname)
	http.Klientenname = akfnetz_Hostname (Klient, Groesse);
      break;


    case AF_UNIX:
    default:
      // nichts tun
      break;
    }
}


static void
Verbindungseinstellungen (int v)
{
  struct timeval z;

  // Auszeit setzen
  z.tv_sec = ZEITLIMIT;
  z.tv_usec = 0;

  // Fehler ignorieren (v muss nicht unbedingt ein Socket sein)
  setsockopt (v, SOL_SOCKET, SO_RCVTIMEO, &z, sizeof (z));
  setsockopt (v, SOL_SOCKET, SO_SNDTIMEO, &z, sizeof (z));

#ifdef TCP_CORK
  // moeglichst keine unvollstaendigen Frames senden
  int i = true;
  setsockopt (v, IPPROTO_TCP, TCP_CORK, &i, sizeof (i));
#endif
}


static int
Verbindungsverarbeitung (int Verbindung)
{
  // Kindprozess

  signal (SIGALRM, SIG_DFL);
  signal (SIGINT, SIG_DFL);
  signal (SIGHUP, SIG_DFL);
  signal (SIGTERM, SIG_DFL);
  signal (SIGPIPE, SIG_DFL);

  // noch behandelt: SIGCHLD, SIGUSR1, SIGUSR2

  Verbindungseinstellungen (Verbindung);

  // zwei Dateistroeme fuer unabhaengige Puffer
  FILE *ein, *aus;
  aus = NULL;
  ein = fdopen (Verbindung, "rb");
  if (!ein)
    {
      http_ueberlastet (Verbindung);
      close (Verbindung);
      return 127;
    }

  int n = dup (Verbindung);
  fcntl (n, F_SETFD, FD_CLOEXEC);
  aus = fdopen (n, "wb");
  if (!aus)
    {
      http_ueberlastet (Verbindung);
      fclose (ein);
      close (n);
      return 127;
    }


  akfnetz_HTTP_Anfrage (ein, aus);

  fclose (aus);
  fclose (ein);

  return EXIT_SUCCESS;
}


extern noreturn void
akfnetz_Verbindungsannahme (int Lauscher)
{
  struct sockaddr_storage Klient;
  socklen_t Klientengroesse;
  int Verbindung;
  pid_t Prozess;

  while (true)
    {
      do
	{
	  Klientengroesse = sizeof (Klient);
	  Verbindung = accept (Lauscher,
			       (struct sockaddr *) &Klient, &Klientengroesse);
	}
      while (Verbindung < 0);

      // Zugriffe von nicht privaten Adressen abweisen
      if (!Einstellung.offen && !akfnetz_privates_Netz (&Klient))
	{
	  close (Verbindung);
	  if (redsam (2))
	    akfnetz_Logbuch ("* %s\n\n", Deutsch == Systemsprache ?
			     "Verbindung abgewiesen" :
			     Esperanto == Systemsprache
			     ? "Konekto rifuzis" : "Connection refused");
	  continue;
	}

      fcntl (Verbindung, F_SETFD, FD_CLOEXEC);

      // Kindprozess erstellen
      Prozess = fork ();

      switch (Prozess)
	{
	case 0:		// Kindprozess
	  close (Lauscher);
	  Klientenadresse (&Klient, Klientengroesse);
	  _exit (Verbindungsverarbeitung (Verbindung));
	  break;


	case -1:		// Fehler
	  http_ueberlastet (Verbindung);
	  break;


	default:
	  if (redsam (2))
	    akfnetz_Logbuch ("* Prozess: %ld\n", (long) Prozess);
	  break;
	}

      close (Verbindung);
    }				// Schleife

  // wird nie erreicht
  abort ();
}


extern int
akfnetz_inetd (void)
{
  struct sockaddr_storage s;
  socklen_t l;

  // root ist verboten!
  if (!geteuid ())
    return EXIT_FAILURE;

  // Port ermitteln
  l = sizeof (s);
  if (!getsockname (STDIN_FILENO, (struct sockaddr *) &s, &l)
      && (s.ss_family == AF_INET || s.ss_family == AF_INET6))
    {
      Einstellung.Port = ntohs (((struct sockaddr_in *) &s)->sin_port);
      if (!Einstellung.Serverport)
	Einstellung.Serverport = Einstellung.Port;
    }
  else
    Einstellung.Port = 0;

  // Klientendaten ermitteln
  l = sizeof (s);
  if (!getpeername (STDIN_FILENO, (struct sockaddr *) &s, &l))
    Klientenadresse (&s, l);

  // evtl Loglevel verringern
  if (fileno (Einstellung.Logbuch) <= STDERR_FILENO)
    Einstellung.Loglevel = 0;

  // Ein- und Ausgabe sollten sich auf den selben Socket beziehen
  Verbindungseinstellungen (STDIN_FILENO);

  akfnetz_HTTP_Anfrage (stdin, stdout);

  return EXIT_SUCCESS;
}
