/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * HTTP-Server
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
  Falls der Klient die Verbindung schliesst, waehrend der Server noch sendet,
  wird der Prozess mit einem SIGPIPE beendet.  Das ist hier durchaus
  erwuenscht, da eh fuer jede Verbindung ein neuer Prozess aufgemacht wird.
*/

#define _POSIX
#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 700

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <time.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "akfnetz.h"
#include "akfwebserver.h"

#define INDEX_XHTML "index.xhtml"
#define INDEX_HTML "index.html"
#define INDEX_CGI "index.cgi"
#define KOPFZEILENLAENGE 8192
// RFC-7230 3.1.1 legt als Minimum fuer Anfrage-Zeilen 8000 Octets fest

#define GZ_ERWEITERUNG ".gz~"
#define unkomprimiert NULL
#define Kleinbuchstabe(z) (('A'<=(z)&&(z)<='Z')?(z)^0x20:(z))

// Dies ist ein Kindprozess: _exit verwenden, wenn ueberhaupt
#pragma GCC poison  exit

#ifndef O_BINARY
#define O_BINARY 0
#endif


struct HTTP_Angaben http;

static inline noreturn void Verbindungsabbruch (void);
static inline bool verborgen (void);
static enum Zustand Serververzeichnis (void);
static enum Zustand lies_Anfrage (FILE *);
static void CGI_ermitteln (void);
static enum Zustand Anfragebearbeitung (FILE *, FILE *);
static enum Zustand Anfrage_analysieren (char *);
static enum Zustand Anfrageinhalt (FILE *, FILE *, int);
static void aufraeumen (void);
static void http_ContentRange (FILE *, intmax_t, intmax_t);
static bool Berechtigungspruefung (void);
static bool unberechtigt (void);
static bool Personenberechtigung (void);
static enum Zustand Erweiterungsbearbeitung (void);
static enum Zustand sende_Datei (FILE *, const char *, const char *,
				 const char *);
static enum Zustand sende_komprimiert (FILE *, const char *);
static enum Zustand senden (FILE * aus);
static enum Zustand Indexdateien (FILE *);
static enum Zustand Verzeichnis (FILE *);
static enum Zustand Indexer (FILE *);
static enum Zustand Optionen (FILE *);
static enum Zustand Trace (FILE *);
static enum Zustand entfernen (FILE *);
static enum Zustand empfangen (FILE *, FILE *);
static void http_Zeitangabe (FILE *, const char *, time_t);
static void bereinige_Ende (char *);
static bool Proxyprotokoll1 (char *);


extern void
akfnetz_HTTP_Anfrage (FILE *ein, FILE *aus)
{
  enum Zustand f;

  if (redsam (2))
    Logbuch_Verbindung ();

  http.Inhalt = -1;
  http.Status = 0;
  http.Antwortlaenge = 0;
  http.Zeitpunkt = (time_t) (-1);
  http.autorisiert = false;
  http.ip_gesetzt = false;

  // falls bereits bei der Anfrage Fehler auftauchen:
  http.Version = 1;
  http.Unterversion = 1;

  do
    {
      f = lies_Anfrage (ein);
      if (Fehlerfrei == f)
	f = Anfragebearbeitung (aus, ein);

      // ein Fehler kann hier schon behandelt worden sein
      if (f > Persistenzfehler)
	http_Webfehler (aus, f);

      // fuer HTTP/0.9 unbedingt schliessen
      if (http.Version == 0)
	f = Persistenzfehler;

      // "Common Log Format"
      if (Einstellung.Loglevel == 1 && http.Status)
	Logbucheintrag ();

      fflush (aus);
      aufraeumen ();
    }
  while (http.komplett && f != Persistenzfehler);

  if (redsam (2))
    akfnetz_Logbuch ("* %s\n\n", Deutsch == Systemsprache ?
		     "Verbindung geschlossen" : "Connection closed");
}


static enum Zustand
Anfragebearbeitung (FILE *aus, FILE *ein)
{
  enum Zustand f;

  // Achtung: nicht das Arbeitsverzeichnis aendern!
  // Bei folgenden Bedingungen ist die Reihenfolge von Bedeutung!

  do
    {
      http.Dateigroesse = 0;
      f = Fehlerfrei;

      CGI_ermitteln ();

      if (NUTZE_CGI && !http.cgi && Einstellung.Erweiterungen)
	{
	  f = Erweiterungsbearbeitung ();
	  if (f != Fehlerfrei)
	    return f;
	}

      if (http.Version > 1)
	f = Versionsfehler;
      else if (GET > http.Methode || http.Methode > DELETE)
	f = Unterstuetzungsfehler;
      else if (http.Version == 0 && http.Methode != GET)
	f = Anfragefehler;	// HTTP/0.9 kennt nur GET
      else if ((http.Version == 1 && http.Unterversion >= 1) && !http.Host)
	f = Anfragefehler;	// HTTP/1.1 muss Host angegeben werden
      else if (TRACE == http.Methode)
	f = Trace (aus);	// nur ohne Inhalt erlaubt, ohne Zugangsdaten
      else if (http.Methode != PUT
	       && (f = Anfrageinhalt (ein, aus, -1)) != Fehlerfrei)
	break;
      else if (Einstellung.Wartungsmodus)
	f = Betriebsfehler;
      else if ((Einstellung.Schutzbereich || Einstellung.Schreibschutz)
	       && unberechtigt ())
	f = Berechtigungsfehler;
      else if (!Personenberechtigung ())
	f = (http.autorisiert || !Einstellung.Zugang)
	  ? Zugriffsfehler : Berechtigungsfehler;
      else if (verborgen ())
	f = (http.Methode == PUT) ? Zugriffsfehler : Suchfehler;
      else if (OPTIONS == http.Methode)
	f = Optionen (aus);
      else if (http.cgi)	// auch bei deaktiviertem CGI!
	f = akfnetz_cgi (aus, NULL);
      else if (POST == http.Methode)
	f = Methodenfehler;	// POST nur bei CGI
      else if (DELETE == http.Methode)
	f = entfernen (aus);
      else if (PUT == http.Methode)
	f = empfangen (aus, ein);	// ruft Anfrageinhalt ab
      else if (http.Pfad[http.Pfadlaenge - 1] == '/')
	f = Verzeichnis (aus);
      else
	f = senden (aus);
    }
  while (f == Neuanforderung);

  // Schutzbereich blockiert?
  if (f == Berechtigungsfehler && Einstellung.Schutzbereichsblockierung)
    f = Zugriffsfehler;

  return f;
}


static void
aufraeumen (void)
{
  if (http.Inhalt >= 0)
    {
      close (http.Inhalt);
      http.Inhalt = -1;
    }

  if (http.Anfragezeile)
    free (http.Anfragezeile);
  if (http.URI)
    free (http.URI);
  if (http.Pfad)
    free (http.Pfad);
  if (http.Originalpfad)
    free (http.Originalpfad);
  if (http.Kopf)
    akfnetz_Kopffreigabe (http.Kopf);

  http.Anfragezeile = http.URI = http.Pfad = http.Originalpfad = NULL;
  http.Host = http.cgi = http.Extrapfad = http.Anfrage = NULL;
  http.Kopf = NULL;
  http.Methode = akfnetz_unbekannte_Methode;
  http.Status = 0;
  http.Pfadlaenge = http.CGIlaenge = 0;
  http.Inhaltslaenge = http.Antwortlaenge = 0;
  http.Zeitpunkt = (time_t) (-1);
  http.autorisiert = false;

  // http.komplett, http.Virtuellname: noch benoetigt
}


static inline bool
verborgen (void)
{
  const char *p = http.Pfad;
  size_t l = http.Pfadlaenge;

  /*
     Keine Backup-Dateien finden.
     Keine Dateien oder Verzeichnisse finden, die mit ".ht" beginnen. (404)
     Pfad darf nicht mit mehreren '/' beginnen
   */

  return (!memcmp (p, "//", 2)
	  || strstr (p, "/.ht")
	  || strstr (p, "/../")
	  || (l > 1 && p[l - 1] == '~')
	  || (l > 4 && !strcasecmp (".bak", p + l - 4)));
}


// enthaelt die Anfrage ein CGI-Element?
// auch bei deaktiviertem CGI ermitteln!
static void
CGI_ermitteln (void)
{
  //assert (*http.Pfad == '/');

  // Verzeichnis cgi-bin
  if (!strncmp (http.Pfad, "/cgi-bin/", 9))
    http.cgi = http.Pfad + 9;
  else if (Einstellung.cgiDatei)	// bennante CGI-Dateien
    {
      char *p = http.Pfad + 1;

      for (struct akfnetz_Liste * d = Einstellung.cgiDatei; d; d = d->weiter)
	{
	  const char *e = d->Eintrag;
	  size_t l = d->Laenge;

	  if (!strncmp (p, e, l) && (p[l] == '\0' || p[l] == '/'))
	    {
	      char *s = strrchr (e, '/');
	      if (s)
		p += (s - e + 1);

	      http.cgi = p;
	      break;
	    }
	}
    }

  if (!http.cgi)		// Dateiendung .cgi
    {
      char *prg = strstr (http.Pfad, ".cgi");
      if (prg)
	{
	  while (*prg != '/')
	    --prg;

	  http.cgi = prg + 1;
	}
    }

  if (http.cgi)
    {
      // http.Pfad & http.cgi enthaelt keinen Query-String oder Fragment
      http.CGIlaenge = strcspn (http.cgi, "/");
      if (http.cgi[http.CGIlaenge])
	http.Extrapfad = http.cgi + http.CGIlaenge;
    }
}


// Status: 201 Created
static enum Zustand
erstellt (FILE *aus)
{
  enum Zustand f;
  size_t Hostlaenge, URI_Laenge, Adresslaenge, Inhaltsgroesse;
  char *Inhalt;

  Inhalt = NULL;
  Inhaltsgroesse = 0;

  Hostlaenge = http.Host ? strlen (http.Host) : 0;
  URI_Laenge = strcspn (http.URI, "?#");

  Adresslaenge = 7 + Hostlaenge + URI_Laenge;
  char Adresse[Adresslaenge + 1];
  *Adresse = '\0';

  // FIXME: HTTPS, allgemeine Funktion fuer Basisadresse
  if (http.Host)
    {
      strcpy (Adresse, "http://");
      strcpy (Adresse + 7, http.Host);
      strncpy (Adresse + 7 + Hostlaenge, http.URI, URI_Laenge);
      Adresse[Adresslaenge] = '\0';

      FILE *d = open_memstream (&Inhalt, &Inhaltsgroesse);
      if (!d)
	return Serverfehler;

      akfnetz_html_Linkseite (d, Adresse);

      if (fclose (d))
	return Serverfehler;
    }

  http_Anfang (aus, 201, "Created");

  if (http.Host)
    http_Kopf (aus, "Location", Adresse);

  http_Kopf (aus, "Content-Type", HTML);
  http_ContentLength (aus, Inhaltsgroesse);
  f = http_Dauerverbindung (aus);
  http_Ende (aus);

  if (Inhaltsgroesse && http.Methode != HEAD)
    fwrite (Inhalt, 1, Inhaltsgroesse, aus);

  free (Inhalt);
  return f;
}


// Status: 304 Not Modified
static enum Zustand
Unveraendertmeldung (FILE *aus, time_t Modifikationszeit)
{
  enum Zustand f;

  http_Anfang (aus, 304, "Not Modified");
  http_Zeitangabe (aus, "Last-Modified", Modifikationszeit);
  f = http_Dauerverbindung (aus);
  http_Ende (aus);

  // hier ist kein Inhalt erlaubt
  return f;
}


// stille Erfolgsmeldung
// Status: 204 Done
static enum Zustand
Erfolgsmeldung (FILE *aus)
{
  enum Zustand f;

  // offiziell "No Content", aber das ist eher verwirrend
  http_Anfang (aus, 204, "Done");
  f = http_Dauerverbindung (aus);
  http_Ende (aus);

  // hier ist kein Inhalt erlaubt
  return f;
}


// schreibt Kopfzeile zensiert ins Logbuch
static void
zensiert (const char *z)
{
  size_t l = strcspn (z, ":");
  char Zeile[l + 1];
  memcpy (Zeile, z, l);
  Zeile[l] = '\0';
  akfnetz_Logbuch ("< %s: ################\n", Zeile);
}


// schreibe Namen in Logbuch und speichere ihn
static void
Nameneintrag (void)
{
  if (!Einstellung.Datenschutz && redsam (2))
    {
      char *von = akfnetz_Kopfeintrag (http.Kopf, "From");
      if (von)
	akfnetz_Logbuch (Deutsch == Systemsprache
			 ? "* Kontakt: %s\n" : "* contact: %s\n", von);
    }

  *http.Name = '\0';
  http.Namenslaenge = 0;

  char *Berechtigung = akfnetz_Kopfeintrag (http.Kopf, "Authorization");

  if (!Berechtigung || !Praefix (Berechtigung, "Basic "))
    return;

  size_t l;
  l = akfnetz_base64_Teilbereich (http.Name, sizeof (http.Name) - 1,
				  Berechtigung + 6, ':');
  http.Name[l] = '\0';
  http.Namenslaenge = l;

  if (l && !Einstellung.Datenschutz && redsam (2))
    akfnetz_Logbuch ("* Name: %s\n", http.Name);

  // Der Zeichensatz ist nicht festgelegt,
  // und bei verschiedenen Browsern tatsaechlich verschieden :-(
}


// Proxy-Protokoll, Version 1 (HAProxy, stunnel)
// veraendert eventuell String!
static bool
Proxyprotokoll1 (char *Zeile)
{
  char *s, *sz;

  if (*Zeile != 'P' || memcmp ("PROXY ", Zeile, 6))
    return false;

  if (redsam (2))
    akfnetz_Logbuch ("\n< %s", Zeile);

  if (http.ip_gesetzt)
    return true;		// nicht nochmal setzen!

  sz = NULL;

  // PROXY
  (void) strtok_r (Zeile, " ", &sz);

  // UNKNOWN oder TCP4 oder TCP6
  s = strtok_r (NULL, " ", &sz);
  if (!s || !memcmp ("UNKNOWN", s, 7))
    return true;
  else if (memcmp ("TCP", s, 3) || (s[3] != '4' && s[3] != '6'))
    Verbindungsabbruch ();

  if (http.Klientenname)
    {
      free (http.Klientenname);
      http.Klientenname = NULL;
    }

  http.Klientenport = 0;

  // Klientenadresse
  s = strtok_r (NULL, " ", &sz);
  if (s)
    strncpy (http.Klientenadresse, s, sizeof (http.Klientenadresse) - 1);

  // Serveradresse ignorieren
  (void) strtok_r (NULL, " ", &sz);

  // Klientenport
  s = strtok_r (NULL, " ", &sz);
  if (s)
    http.Klientenport = (unsigned short int) strtoul (s, NULL, 10);

  // Serverport ignorieren
  // (void) strtok_r (NULL, " ", &sz);

  http.ip_gesetzt = true;

  return true;
}


// X-Forwarded-For
// falls die Anfrage durch einen reverse Proxy weitergeleitet wurde
static void
Weiterleitungsverarbeitung (void)
{
  char *p;

  if (http.ip_gesetzt)
    return;

  p = akfnetz_Kopfeintrag (http.Kopf, "X-Forwarded-For");
  if (!p)
    return;

  // TODO: "Forwarded" unterstuetzen (RFC 7239)

  size_t l = strcspn (p, ", ");
  if (!l || l >= sizeof (http.Klientenadresse))
    return;

  if (!strncasecmp (p, "::FFFF:", 7))
    {
      p += 7;
      l -= 7;
    }

  memcpy (http.Klientenadresse, p, l);
  http.Klientenadresse[l] = '\0';

  http.Klientenport = 0;
  http.ip_gesetzt = true;

  if (http.Klientenname)
    {
      free (http.Klientenname);
      http.Klientenname = NULL;
    }
}


static void
Kopfprotokollierung (void)
{
  for (int i = 0; http.Kopf[i]; ++i)
    {
      char *z = http.Kopf[i];

      if (!Praefix (z, "Authorization:")
	  && !Praefix (z, "Proxy-Authorization:"))
	akfnetz_Logbuch ("< %s\n", z);
      else
	zensiert (z);
    }
}


// virtuelles Hosting: Verzeichniswechsel
static enum Zustand
Serververzeichnis (void)
{
  size_t l;
  char *v;

  if (!http.Host)
    return Anfragefehler;

  if (http.Virtuellname)
    {
      free (http.Virtuellname);
      http.Virtuellname = NULL;
    }

  l = strcspn (http.Host, ":/#? \t");
  v = malloc (l + 1);
  if (!v)
    return Scheisse;

  for (size_t i = 0; i < l; ++i)
    v[i] = Kleinbuchstabe (http.Host[i]);

  v[l] = '\0';

  if (chdir (v) < 0)
    {
      free (v);
      return Zugriffsfehler;
    }

  http.Virtuellname = v;

  return Fehlerfrei;
}


static enum Zustand
lies_Anfrage (FILE *d)
{
  char Zeile[KOPFZEILENLAENGE];

  // wurde Anfrage komplett gelesen?
  http.komplett = false;

lesen:

  // Anfragezeile
  if (!fgets (Zeile, sizeof (Zeile), d))
    return Persistenzfehler;

  if (!*Zeile)
    return Anfragefehler;

  // Muell oder verschluesselte Daten? Erste 4 Zeichen nicht lesbar?
  for (int i = 0; i < 4; ++i)
    {
      register signed char c;
      c = (signed char) Zeile[i];
      if (c < 0x20)
	return Unterstuetzungsfehler;
    }

  size_t Laenge = strlen (Zeile);

  // zu lange Zeile?
  if (Zeile[--Laenge] != '\n')
    return Adressfehler;

  if (Zeile[Laenge - 1] == '\r')
    --Laenge;

  Zeile[Laenge] = '\0';

  // Proxy Protokoll, Version 1? (HAProxy, stunnel)
  if (Proxyprotokoll1 (Zeile))
    goto lesen;			// Zeile nun kaputt

  if (redsam (2))
    akfnetz_Logbuch ("\n< %s\n", Zeile);

  // kein strdup, da Laenge bekannt ist
  http.Anfragezeile = malloc (Laenge + 1);
  if (http.Anfragezeile)
    memcpy (http.Anfragezeile, Zeile, Laenge + 1);

  // die Zeile wird hier auseinandergenommen
  enum Zustand f = Anfrage_analysieren (Zeile);
  if (f != Fehlerfrei)
    return f;

  if (http.Version != 0)
    {
      http.Kopf = akfnetz_Kopflesen (d);
      if (redsam (2))
	Kopfprotokollierung ();

      http.Host = akfnetz_Kopfeintrag (http.Kopf, "Host");
      Weiterleitungsverarbeitung ();

      if (Einstellung.virtuell && !http.Virtuellname)
	{
	  f = Serververzeichnis ();
	  if (f != Fehlerfrei)
	    return f;
	}
    }

  if (redsam (2))
    akfnetz_Logbuch ("\n");

  http.onion =
    (http.Host && !strcasecmp (".onion", http.Host + strlen (http.Host) - 6));

  Nameneintrag ();

  return Fehlerfrei;
}


static enum Zustand
Anfrage_analysieren (char *Anfrage)
{
  char *sz;

  http.Methode = akfnetz_Methodentyp (strtok_r (Anfrage, " ", &sz));

  // Pfad und Anfrage-String
  char *p = strtok_r (NULL, " ", &sz);
  if (!p)
    return Anfragefehler;

  // eventuell Fragment entfernen
  char *Fragment = strchr (p, '#');
  if (Fragment)
    *Fragment = '\0';

  http.URI = strdup (p);

  // absolute URL ist erlaubt :-(
  if (Praefix (p, "http:"))
    p += 5;
  else if (Praefix (p, "https:"))
    p += 6;

  // Authority ueberspringen
  if (!memcmp (p, "//", 2))
    p = strpbrk (p + 2, "/?");

  /*
     Auch wenn eine absolute URL angegeben ist, oder ein fremder
     Host-Eintrag, auf keinen Fall Inhalte von fremden Servern
     weiterleiten!  Das koennte missbraucht werden!
     Dann hat man naemlich einen anonymisierenden Proxy.
   */

  char *q = p ? strchr (p, '?') : NULL;
  if (q)
    {
      *q = '\0';
      http.Anfrage = strchr (http.URI, '?') + 1;
    }

  if (p && *p)
    akfnetz_url_dekodieren (p);
  else				// absolute URL ohne Pfad
    p = "/";

  http.Pfadlaenge = strlen (p);
  http.Pfad = malloc (http.Pfadlaenge + 1);
  if (http.Pfad)
    memcpy (http.Pfad, p, http.Pfadlaenge + 1);
  else
    http.Pfadlaenge = 0;

  http.Version = http.Unterversion = 1;

  p = strtok_r (NULL, " ", &sz);

  // HTTP/0.9?
  if (!p)
    {
      http.Version = 0;
      http.Unterversion = 9;

      // virtuelles Hosting braucht einen Hostnamen
      if (Einstellung.virtuell)
	return Versionsfehler;

      return Fehlerfrei;
    }

  if (!Praefix (p, "HTTP/"))
    return Anfragefehler;

  http.Version = (unsigned short) strtoul (p + 5, &p, 10);
  if (*p == '.')
    http.Unterversion = (unsigned short) strtoul (p + 1, NULL, 10);
  else
    http.Unterversion = 0;

  return Fehlerfrei;
}


// Wenn Pfad mit registrierter Erweiterung endet, dann Aktion vor Pfad setzen
// Pfad als Originalpfad speichern
static enum Zustand
Erweiterungsbearbeitung (void)
{
  size_t Erweiterungslaenge = 0;
  struct akfnetz_Liste *e;

  for (e = Einstellung.Erweiterungen; e; e = e->weiter)
    {
      Erweiterungslaenge = strcspn (e->Eintrag, "=");

      // endet Pfad mit der Erweiterung?
      if (Erweiterungslaenge
	  && Erweiterungslaenge < http.Pfadlaenge
	  && http.Pfad[http.Pfadlaenge - Erweiterungslaenge - 1] == '.'
	  && !strncasecmp (http.Pfad + http.Pfadlaenge - Erweiterungslaenge,
			   e->Eintrag, Erweiterungslaenge))
	break;			// gefunden
    }

  // ohne verknuepfte Erweiterung
  if (!e)
    return Fehlerfrei;

  // ist die Datei vorhanden und lesbar?
  if (access (http.Pfad + 1, R_OK) < 0)
    return ((errno == EACCES) ? Zugriffsfehler : Suchfehler);

  // Pfad aendern
  size_t Aktionslaenge = e->Laenge - (Erweiterungslaenge + 1);
  const char *Aktion = e->Eintrag + Erweiterungslaenge + 1;
  if (!Aktionslaenge || *Aktion != '/')
    return Zugriffsfehler;

  char *Neupfad = malloc (Aktionslaenge + http.Pfadlaenge + 1);
  if (!Neupfad)
    return Scheisse;

  memcpy (Neupfad, Aktion, Aktionslaenge);
  memcpy (Neupfad + Aktionslaenge, http.Pfad, http.Pfadlaenge + 1);

  if (!http.Originalpfad)
    http.Originalpfad = http.Pfad;
  else
    free (http.Pfad);

  http.Pfad = Neupfad;
  http.Pfadlaenge += Aktionslaenge;

  CGI_ermitteln ();

  return Fehlerfrei;
}


extern void
http_Kopf (FILE *aus, const char *Name, const char *Eintrag)
{
  if (http.Version == 0)
    return;

  fprintf (aus, "%s: %s\r\n", Name, Eintrag);
  if (redsam (2))
    akfnetz_Logbuch ("> %s: %s\n", Name, Eintrag);
}


static void
http_Zeitangabe (FILE *aus, const char *Name, time_t Zeit)
{
  char z[30];
  struct tm tm;

  if (http.Version == 0)
    return;

  if (Zeit == (time_t) (-1) || Zeit < 0)
    return;

  if (akfnetz_http_Zeitstring (z, sizeof (z), gmtime_r (&Zeit, &tm)))
    http_Kopf (aus, Name, z);
}


extern void
http_ContentLength (FILE *aus, uintmax_t Laenge)
{
  http.Antwortlaenge = Laenge;

  if (http.Version == 0)
    return;

  fprintf (aus, "Content-Length: %ju\r\n", Laenge);
  if (redsam (2))
    akfnetz_Logbuch ("> Content-Length: %ju\n", Laenge);
}


extern enum Zustand
http_Dauerverbindung (FILE *aus)
{
  if (http.Version == 0)
    return Persistenzfehler;

  if (!http.komplett
      || (http.Version == 1 && http.Unterversion == 0)
      || akfnetz_Tokensuche (http.Kopf, "Connection", "close"))
    {
      http_Kopf (aus, "Connection", "close");
      return Persistenzfehler;
    }

  http_Kopf (aus, "Connection", "keep-alive");
  return Fehlerfrei;
}


static void
http_ContentRange (FILE *aus, intmax_t von, intmax_t bis)
{
  char s[80];

  if (http.Dateigroesse > 0)
    snprintf (s, sizeof (s), "bytes %jd-%jd/%jd", von, bis,
	      (intmax_t) http.Dateigroesse);
  else
    snprintf (s, sizeof (s), "bytes %jd-%jd/*", von, bis);

  /* */
  http_Kopf (aus, "Content-Range", s);
}


extern void
http_Methodenerlaubnis (FILE *aus)
{
  char s[80];

  strcpy (s, "GET,HEAD,OPTIONS");

  if (Einstellung.Trace)
    strcat (s, ",TRACE");

  if (http.cgi || !http.Pfad || !strcmp (http.Pfad, "*"))
    strcat (s, ",POST");

  if (Einstellung.beschreibbar && !http.cgi)
    strcat (s, ",PUT,DELETE");

  http_Kopf (aus, "Allow", s);
}


// Gibt die aktuelle Zeit zurueck
extern time_t
http_Anfang (FILE *aus, int Codenr, const char *Beschreibung)
{
  time_t jetzt;

  http.Zeitpunkt = time (&jetzt);
  http.Status = (unsigned short int) Codenr;

  if (http.Version == 0)
    return jetzt;

  fprintf (aus, PROTOKOLL " %03d %s\r\n", Codenr, Beschreibung);

  if (redsam (2))
    {
      if (Deutsch == Systemsprache)
	akfnetz_Logbuch ("> " PROTOKOLL " %03d %s (%s)\n",
			 Codenr, Beschreibung,
			 akfnetz_Statusmeldung (Codenr));
      else
	akfnetz_Logbuch ("> " PROTOKOLL " %03d %s\n", Codenr, Beschreibung);
    }

  if (Codenr < 200)
    return jetzt;

  http_Zeitangabe (aus, "Date", jetzt);

#ifdef SERVER_SOFTWARE
  if (Einstellung.Signatur && !http.onion)
    http_Kopf (aus, "Server", SERVER_SOFTWARE);
#endif

  // zusaetzliche Kopfzeilen ausgeben
  for (struct akfnetz_Liste * e = Einstellung.Kopfzeilen; e; e = e->weiter)
    {
      fprintf (aus, "%s\r\n", e->Eintrag);
      if (redsam (2))
	akfnetz_Logbuch ("> %s\n", e->Eintrag);
    }

  if (Einstellung.Onion_Location && !http.onion)
    {
      fprintf (aus, "Onion-Location: %s%s\r\n",
	       Einstellung.Onion_Location, http.URI);
      if (redsam (2))
	akfnetz_Logbuch ("> Onion-Location: %s%s\n",
			 Einstellung.Onion_Location, http.URI);
    }

  return jetzt;
}


extern void
http_Ende (FILE *aus)
{
  // Leerzeile beendet den Kopf
  if (http.Version != 0)
    fputs ("\r\n", aus);
}


static bool
unveraendert (time_t Modifikationszeit)
{
  struct tm Dateizeit, Vergleichszeit;
  char *z;

  z = akfnetz_Kopfeintrag (http.Kopf, "If-Unmodified-Since");
  if (!z)
    return true;

  akfnetz_analysiere_http_Zeit (&Vergleichszeit, z);
  gmtime_r (&Modifikationszeit, &Dateizeit);

  return (akfnetz_Zeitenvergleich (&Dateizeit, &Vergleichszeit) <= 0);
}


static bool
veraendert (time_t Modifikationszeit)
{
  struct tm Dateizeit, Vergleichszeit;
  char *z;

  z = akfnetz_Kopfeintrag (http.Kopf, "If-Modified-Since");
  if (!z)
    return true;

  akfnetz_analysiere_http_Zeit (&Vergleichszeit, z);
  gmtime_r (&Modifikationszeit, &Dateizeit);

  return (akfnetz_Zeitenvergleich (&Dateizeit, &Vergleichszeit) > 0);
}


// true wenn Bereich gesendet werden soll, ansonsten alles nochmal senden
static bool
Bereichsbedingung (time_t Modifikationszeit)
{
  struct tm Dateizeit, Vergleichszeit;
  char *z;

  z = akfnetz_Kopfeintrag (http.Kopf, "If-Range");
  if (!z)
    return true;

  // das koennte auch ein ETag sein!
  if (!akfnetz_analysiere_http_Zeit (&Vergleichszeit, z))
    return false;

  gmtime_r (&Modifikationszeit, &Dateizeit);

  // Zeit muss genau uebereinstimmen
  return (akfnetz_Zeitenvergleich (&Dateizeit, &Vergleichszeit) == 0);
}


static int
Bereichsauswertung (char *Bereich, off_t *v, off_t *b)
{
  off_t von, bis;
  char *p = Bereich + 6;

  if (*p != '-')
    {
      von = (off_t) strtoimax (p, &p, 10);
      ++p;
      bis = (off_t) strtoimax (p, &p, 10);
      if (!bis)
	bis = http.Dateigroesse - 1;
    }
  else				// Suffix
    {
      ++p;
      von = 0;
      bis = http.Dateigroesse - 1;
      off_t Suffix = (off_t) strtoimax (p, &p, 10);
      if (Suffix < http.Dateigroesse)
	von = http.Dateigroesse - Suffix;
    }

  if (bis >= http.Dateigroesse)
    bis = http.Dateigroesse - 1;

  *v = von;
  *b = bis;

  if (von >= http.Dateigroesse || bis < von)
    return -1;

  return 0;
}


// Methode GET
static enum Zustand
sende_Datei (FILE *aus, const char *Datei, const char *Typ,
	     const char *Komprimierung)
{
  enum Zustand f = Fehlerfrei;
  struct stat status;

  if (stat (Datei, &status) < 0)
    return (errno == EACCES) ? Zugriffsfehler : Suchfehler;

  if (S_ISDIR (status.st_mode))
    return Verzeichnisfehler;

  // Es muss eine regulaere Datei sein
  if (!S_ISREG (status.st_mode))
    return Zugriffsfehler;

  // Dateityp verboten? ("")
  if (Typ && !*Typ)
    return Zugriffsfehler;

  // If-Unmodified-Since
  if (!unveraendert (status.st_mtime))
    return Voraussetzungsfehler;

  // If-Modified-Since
  if (!veraendert (status.st_mtime))
    return Unveraendertmeldung (aus, status.st_mtime);

  off_t Datenmenge, von, bis;
  Datenmenge = http.Dateigroesse = status.st_size;
  von = bis = 0;

  /*
     TODO: Einschraenkung: wenn mehrere Bereiche angefordert werden,
     wird nur der erste ausgeliefert.
   */

  char *Bereich = NULL;

  if (!Komprimierung && http.Methode <= HEAD)
    {
      Bereich = akfnetz_Kopfeintrag (http.Kopf, "Range");
      if (Bereich && (!Praefix (Bereich, "bytes=")
		      || !Bereichsbedingung (status.st_mtime)))
	Bereich = NULL;

      if (Bereich)
	{
	  if (Bereichsauswertung (Bereich, &von, &bis) < 0)
	    return Bereichsfehler;

	  Datenmenge = bis - von + 1;
	}
    }

  int d = open (Datei, O_RDONLY | O_BINARY | O_NOCTTY);
  if (d < 0)
    return (errno == EACCES) ? Zugriffsfehler : Suchfehler;

  time_t jetzt;
  if (!Bereich)
    {
      jetzt = http_Anfang (aus, 200, "OK");
      http_Kopf (aus, "Accept-Ranges", "bytes");
    }
  else
    {
      jetzt = http_Anfang (aus, 206, "Partial Content");
      http_ContentRange (aus, von, bis);
    }

  http_Kopf (aus, "Content-Type", Typ);
  http_Zeitangabe (aus, "Last-Modified", MIN (status.st_mtime, jetzt));
  if (Einstellung.gzip)
    http_Kopf (aus, "Vary", "Accept-Encoding");
  if (Komprimierung)
    http_Kopf (aus, "Content-Encoding", Komprimierung);
  http_ContentLength (aus, Datenmenge);

  // Damit Web Key Directory (WKD) funktioniert (benoetigt auch HTTPS)
  if (!strncmp ("/.well-known/openpgpkey/", http.Pfad, 24))
    http_Kopf (aus, "Access-Control-Allow-Origin", "*");

  f = http_Dauerverbindung (aus);
  http_Ende (aus);

  if (http.Methode != HEAD && Dateiinhalt (aus, d, von, Datenmenge) < 0)
    f = Persistenzfehler;

  close (d);

  return f;
}


static enum Zustand
sende_komprimiert (FILE *aus, const char *Typ)
{
  size_t l = http.Pfadlaenge - 1;
  char Pfad[l + sizeof (GZ_ERWEITERUNG)];

  memcpy (Pfad, http.Pfad + 1, l);
  memcpy (Pfad + l, GZ_ERWEITERUNG, sizeof (GZ_ERWEITERUNG));

  return sende_Datei (aus, Pfad, Typ, "gzip");
}


// Methode GET
static enum Zustand
senden (FILE *aus)
{
  enum Zustand f;
  const char *Typ;

  Typ = akfnetz_Medientyp (http.Pfad);

  if (Einstellung.gzip
      && !akfnetz_Kopfeintrag (http.Kopf, "Range")
      && akfnetz_Tokensuche (http.Kopf, "Accept-Encoding", "gzip"))
    {
      f = sende_komprimiert (aus, Typ);
      if (f != Suchfehler)
	return f;
    }

  f = sende_Datei (aus, http.Pfad + 1, Typ, unkomprimiert);

  // eventuell interne Ressource?
  if (Suchfehler == f)
    {
      if (!strcmp ("/.well-known/akfnetz/Fehler.css", http.Pfad))
	f = http_sende_Fehlerstil (aus);
    }

  return f;
}


static enum Zustand
Indexdateien (FILE *aus)
{
  enum Zustand f;
  char Indexpfad[http.Pfadlaenge - 1 + sizeof (INDEX_XHTML) - 1
		 + sizeof (GZ_ERWEITERUNG)];

  memcpy (Indexpfad, http.Pfad + 1, http.Pfadlaenge - 1);

  char *Datei = Indexpfad + http.Pfadlaenge - 1;

  // index.cgi
  memcpy (Datei, INDEX_CGI, sizeof (INDEX_CGI));
  if (!access (Indexpfad, F_OK))
    return akfnetz_cgi (aus, INDEX_CGI);

  bool gzip = (Einstellung.gzip
	       && !akfnetz_Kopfeintrag (http.Kopf, "Range")
	       && akfnetz_Tokensuche (http.Kopf, "Accept-Encoding", "gzip"));

  // index.xhtml
  memcpy (Datei, INDEX_XHTML, sizeof (INDEX_XHTML));
  if (gzip)
    {
      memcpy (Datei + sizeof (INDEX_XHTML) - 1, GZ_ERWEITERUNG,
	      sizeof (GZ_ERWEITERUNG));
      f = sende_Datei (aus, Indexpfad, XHTML, "gzip");
      if (f != Suchfehler)
	return f;
    }

  Datei[sizeof (INDEX_XHTML) - 1] = '\0';
  f = sende_Datei (aus, Indexpfad, XHTML, unkomprimiert);
  if (f != Suchfehler)
    return f;

  // oder index.html
  memcpy (Datei, INDEX_HTML, sizeof (INDEX_HTML));
  if (gzip)
    {
      memcpy (Datei + sizeof (INDEX_HTML) - 1, GZ_ERWEITERUNG,
	      sizeof (GZ_ERWEITERUNG));
      f = sende_Datei (aus, Indexpfad, HTML, "gzip");
      if (f != Suchfehler)
	return f;
    }

  Datei[sizeof (INDEX_HTML) - 1] = '\0';
  f = sende_Datei (aus, Indexpfad, HTML, unkomprimiert);

  /*
     Wenn diese Dateien existieren, aber nicht lesbar sind,
     wird ein Fehler ausgegeben und nicht weitergemacht.
     So soll's sein! Nicht aendern.
   */

  return f;
}


// Methode GET
static enum Zustand
Verzeichnis (FILE *aus)
{
  enum Zustand f;

  if (!*http.Pfad)
    return Suchfehler;

  f = Indexdateien (aus);
  if (f != Suchfehler)
    return f;

  // kein automatischer Index in /cgi-bin, /.well-known
  if (!strncmp (http.Pfad, "/cgi-bin/", 9)
      || !strncmp (http.Pfad, "/.well-known/", 13))
    return Zugriffsfehler;

  // Ist das Verzeichnis lesbar?
  if (access (http.Pfad[1] ? http.Pfad + 1 : ".", R_OK | X_OK) < 0)
    return (errno == EACCES) ? Zugriffsfehler : Suchfehler;

  return Indexer (aus);
}


// Index automatisch erstellen
static enum Zustand
Indexer (FILE *aus)
{
  if (NUTZE_INDEXER)
    {
      FILE *d;
      char *Inhalt;
      size_t Inhaltsgroesse;
      enum Zustand f;

      d = open_memstream (&Inhalt, &Inhaltsgroesse);
      if (!d)
	return Serverfehler;

      char *Sprachakzeptanz =
	akfnetz_Kopfeintrag (http.Kopf, "Accept-Language");

      akfnetz_html_Verzeichnis (d, http.Pfad + 1, http.Pfad,
				 http.Pfad[1] != '\0', http.Anfrage,
				 Sprachakzeptanz,
				 http.autorisiert ? http.Name : NULL);

      if (fclose (d))
	return Serverfehler;

      http_Anfang (aus, 200, "OK");
      http_Kopf (aus, "Cache-Control", "no-cache");
      http_Kopf (aus, "Pragma", "no-cache");
      http_Kopf (aus, "Content-Type", "text/html; charset=UTF-8");
      http_ContentLength (aus, Inhaltsgroesse);
      http_Kopf (aus, "Vary", "Accept-Language");
      f = http_Dauerverbindung (aus);
      http_Ende (aus);

      if (http.Methode != HEAD)
	fwrite (Inhalt, 1, Inhaltsgroesse, aus);

      free (Inhalt);

      return f;
    }
  else
    {
      return Zugriffsfehler;
      // Nicht Unterstuetzungsfehler oder gar Verzeichnisfehler!
    }
}


// entferne Freiraum am Ende des Strings
static void
bereinige_Ende (char *s)
{
  if (!s || !*s)
    return;

  size_t l = strlen (s) - 1;
  while (l && s[l] <= ' ')
    --l;

  s[l + 1] = '\0';
}


static bool
Dateiberechtigung (const char *b)
{
  bool okay = false;
  FILE *d;
  char Zeile[8192];

  d = fopen (".htZugang", "r");
  if (!d)
    return false;

  while (fgets (Zeile, sizeof (Zeile), d))
    {
      char *s = Zeile;

      while (*s && *s <= ' ')
	++s;

      if (!*s || *s == '#')
	continue;

      bereinige_Ende (s);

      if (!*s)
	continue;

      // Wenn ':' am Anfang, dann ist der Rest bereits kodiert
      if (*s == ':')
	{
	  if (!strcmp (b, s + 1))
	    {
	      okay = true;
	      break;
	    }
	}
      else if (s[http.Namenslaenge] == ':'
	       && !strncmp (s, http.Name, http.Namenslaenge))
	{			// Kodieren
	  size_t sl = strlen (s);
	  size_t pl = akfnetz_base64_Platz (sl);
	  char Puffer[pl];

	  akfnetz_base64_kodiert (Puffer, pl, s, sl, AKFNETZ_BASE64);
	  if (!strcmp (b, Puffer))
	    {
	      okay = true;
	      break;
	    }
	}
    }

  fclose (d);
  return okay;
}


static bool
Berechtigungspruefung (void)
{
  char *b;
  struct akfnetz_Liste *e;
  size_t l;

  http.autorisiert = false;
  b = akfnetz_Kopfeintrag (http.Kopf, "Authorization");

  // Basic-Authorization
  if (!b || !Praefix (b, "Basic "))
    return false;

  // Anfang der Zugangsdaten
  b += 6;
  l = strlen (b);

  // feste Zugangsdaten
  for (e = Einstellung.Zugang; e; e = e->weiter)
    if (e->Laenge == l && !strcmp (b, e->Eintrag))
      {
	http.autorisiert = true;
	return true;
      }

  // pruefe .htZugang
  if (Dateiberechtigung (b))
    {
      http.autorisiert = true;
      return true;
    }

  // bei falschen Zugangsdaten verzoegern,
  // um Angriffe zu erschweren
  sleep (STRAFSEKUNDEN);

  return false;
}


static bool
unberechtigt (void)
{
  // bei Schreibschutz sind nur Schreibvorgaenge geschuetzt
  if (Einstellung.Schreibschutz
      && http.Methode != PUT && http.Methode != DELETE
      && http.Methode != PATCH && http.Methode <= PROPFIND)
    return false;

  // ausserhalb des Schutzbereiches?
  // auch der Extrapfad darf nicht im Schutzbereich liegen
  if (Einstellung.Schutzbereich && Einstellung.Schutzbereich[1])
    {
      size_t l = Einstellung.Schutzbereichslaenge;

      if ((l >= http.Pfadlaenge
	   || http.Pfad[l] != '/'
	   || strncmp (http.Pfad, Einstellung.Schutzbereich, l))
	  &&
	  (!http.Extrapfad
	   || strncmp (http.Extrapfad, Einstellung.Schutzbereich, l)
	   || http.Extrapfad[l] != '/'))
	return false;
    }

  if (Einstellung.Schutzbereichsblockierung)
    return true;

  return !Berechtigungspruefung ();
}


// Verzeichnisnamen mit "~" am Anfang sind nur fuer berechtigte Personen
static bool
Personenberechtigung (void)
{
  char *p;
  size_t l;

  // Es wird nur das erste Auftauchen von "/~" untersucht
  // http.Pfad ist bereits dekodiert
  p = strstr (http.Pfad, "/~");

  if (!p)
    return true;		// kein personalisiertes Verzeichnis

  if (!http.autorisiert && !Berechtigungspruefung ())
    return false;

  p += 2;
  l = http.Namenslaenge;

  return (p[l] == '/' && strncmp (p, http.Name, l) == 0);
}


// Methode OPTIONS
static enum Zustand
Optionen (FILE *aus)
{
  enum Zustand f = Fehlerfrei;

  http_Anfang (aus, 200, "OK");
  http_Methodenerlaubnis (aus);
  if (!http.cgi)
    http_Kopf (aus, "Accept-Ranges", "bytes");
  http_Kopf (aus, "Content-Length", "0");	// Vorschrift
  f = http_Dauerverbindung (aus);
  http_Ende (aus);

  return f;
}


// Methode TRACE
static enum Zustand
Trace (FILE *aus)
{
  FILE *d;
  char *Inhalt;
  size_t Inhaltsgroesse;

  if (!Einstellung.Trace)
    return Methodenfehler;

  // bei TRACE ist kein Inhalt erlaubt
  if (akfnetz_Kopfeintrag (http.Kopf, "Transfer-Encoding")
      || akfnetz_Kopfeintrag (http.Kopf, "Content-Length"))
    return Anfragefehler;

  d = open_memstream (&Inhalt, &Inhaltsgroesse);
  if (!d)
    return Serverfehler;

  fprintf (d, "%s\r\n", http.Anfragezeile);

  for (int i = 0; http.Kopf[i]; ++i)
    {
      char *z = http.Kopf[i];

      // schuetzt u.A. gegen Cross Site Tracing Angriff
      if (!Praefix (z, "Authorization:")
	  && !Praefix (z, "Proxy-Authorization:") && !Praefix (z, "Cookie:"))
	fprintf (d, "%s\r\n", z);
    }

  fputs ("\r\n", d);

  if (fclose (d))
    return Serverfehler;

  http_Anfang (aus, 200, "OK");
  http_Kopf (aus, "Content-Type", "message/http");
  http_ContentLength (aus, Inhaltsgroesse);
  enum Zustand f = http_Dauerverbindung (aus);
  http_Ende (aus);

  fwrite (Inhalt, 1, Inhaltsgroesse, aus);
  free (Inhalt);

  return f;
}


// Kopfzeile: Expect
static enum Zustand
Erwartung (FILE *aus)
{
  char *e;

  e = akfnetz_Kopfeintrag (http.Kopf, "Expect");
  if (e && !strcasecmp (e, "100-continue"))
    {
      if (Einstellung.Wartungsmodus)
	return Betriebsfehler;

      // wenn kleiner als HTTP/1.1, ignorieren
      if (http.Version < 1 || http.Unterversion < 1)
	return Fehlerfrei;

      fputs (PROTOKOLL " 100 Continue\r\n\r\n", aus);
      fflush (aus);

      if (redsam (2))
	{
	  if (Deutsch == Systemsprache)
	    akfnetz_Logbuch ("> " PROTOKOLL " 100 Continue (Weitermachen)\n");
	  else
	    akfnetz_Logbuch ("> " PROTOKOLL " 100 Continue\n");
	}
    }
  else if (e && *e)
    return Erwartungsfehler;	// andere Erwartungen werden nicht unterstuetzt

  return Fehlerfrei;
}


// Fuer gestueckelte ("chunked") Uebertragung
static int
Stueckspeicherung (void *Puffer, size_t Laenge, void *Daten)
{
  int *fd = Daten;
  return akfnetz_Ausgabe (*fd, Puffer, Laenge);
}


static enum Zustand
lies_gestueckelt (FILE *ein, FILE *aus, int d, intmax_t *Laenge)
{
  enum Zustand f;

  f = Erwartung (aus);
  if (f != Fehlerfrei)
    return f;

  if (d < 0)
    *Laenge = akfnetz_gestueckelt (ein, NULL, NULL);	// ignorieren
  else
    *Laenge = akfnetz_gestueckelt (ein, &Stueckspeicherung, &d);

  // Trailer ignorieren
  char z[KOPFZEILENLAENGE];
  do
    if (!fgets (z, sizeof (z), ein))
      break;
  while (*z);

  return f;
}


static enum Zustand
lies_Laenge (FILE *ein, FILE *aus, int d, intmax_t Laenge)
{
  enum Zustand f;
  uintmax_t Rest;
  char Puffer[8192];

  f = Erwartung (aus);
  if (f != Fehlerfrei)
    return f;

  Rest = Laenge;
  while (Rest > 0)
    {
      size_t n;

      n = fread (Puffer, sizeof (char), MIN (sizeof (Puffer), Rest), ein);

      if (!n)			// vorzeitiger Abbruch?
	return Anfragefehler;

      if (f == Fehlerfrei && d >= 0 && akfnetz_Ausgabe (d, Puffer, n) < 0)
	{
	  if (redsam (2))
	    akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));

	  return Serverfehler;
	}

      if (n <= Rest)
	Rest -= n;
    }

  return f;
}


// erzeugt Temporaerdatei und belegt evtl. schonmal Speicher vor
static int
Temporaerdatei (intmax_t Laenge)
{
  int d;

  d = akfnetz_Temporaerdatei (Einstellung.Tempverzeichnis);
  if (d < 0)
    return -1;

  // Speicher eventuell schonmal vorbelegen
  if (Laenge <= 0 || posix_fallocate (d, 0, Laenge) != ENOSPC)
    return d;

  // zu gross? -> nochmal auf /var/tmp versuchen
  close (d);

  d = akfnetz_Temporaerdatei ("/var/tmp");
  if (d < 0)
    return -1;

  if (posix_fallocate (d, 0, Laenge) == ENOSPC)
    {
      close (d);
      return -1;
    }

  return d;
}


/*
Lies Anfrageinhalt, wenn Transfer-Encoding oder Content-Length angegeben ist.
Wenn Ausgabe<0 ist, wird eine Temporaerdatei angelegt und http.Inhalt gesetzt.
Inhalte machen nur bei wenigen Methoden Sinn, sind aber bei vielen erlaubt.
Persistenzfehler wenn Schreib- oder Lese-Fehler auftreten
*/
static enum Zustand
Anfrageinhalt (FILE *ein, FILE *aus, int Ausgabe)
{
  enum Zustand f = Fehlerfrei;
  int d = Ausgabe;

  http.Inhalt = -1;

  // HTTP/0.9 kann nur einfaches GET
  if (http.Version == 0)
    return Fehlerfrei;

  intmax_t Laenge = akfnetz_Kopfzahl (http.Kopf, "Content-Length");

  // wenn keine Laenge angegeben waere, waere der Wert -1
  if (Laenge == 0)
    return Fehlerfrei;

  // keine Unterstuetzung fuer grosse Dateien?
  if (sizeof (off_t) == sizeof (int_least32_t) && Laenge > INT_LEAST32_MAX)
    return Platzfehler;

  // POST noch auf Adressraum begrenzen, da hochladen.cgi begrenzt ist
  if (POST == http.Methode &&
      sizeof (size_t) == sizeof (uint_least32_t) && Laenge > 0x7ffff000)
    return Platzfehler;

  if (POST == http.Methode && (d = Temporaerdatei (Laenge)) < 0)
    return Platzfehler;
  else if (PUT == http.Methode
	   && Laenge > 0 && posix_fallocate (d, 0, Laenge) == ENOSPC)
    return Platzfehler;

  // Transferkodierung
  // muss entweder "chunked" oder "identity" sein und nichts anderes
  char *TK = akfnetz_Kopfeintrag (http.Kopf, "Transfer-Encoding");

  if (TK && (!*TK || !strcasecmp (TK, "identity")))
    TK = NULL;

  if (TK && !strcasecmp (TK, "chunked"))
    f = lies_gestueckelt (ein, aus, d, &Laenge);
  else if (TK)
    f = Unterstuetzungsfehler;	// nicht unterstuetzte Kodierung
  else if (Laenge > 0)
    f = lies_Laenge (ein, aus, d, Laenge);

  if (f != Fehlerfrei && d >= 0)
    close (d);
  else if (Fehlerfrei == f && d >= 0)
    {
      // fertig geschrieben - an den Anfang springen
      lseek (d, 0, SEEK_SET);
      fcntl (d, F_SETFD, FD_CLOEXEC);

      if (Ausgabe < 0)
	{
	  http.Inhalt = d;
	  http.Inhaltslaenge = Laenge;
	}
    }

  if (Fehlerfrei == f)
    http.komplett = true;

  return f;
}


// Datei verschieben
static int
Sicherheitskopie (const char *Name)
{
  char neu[strlen (Name) + 6];

  for (unsigned int nr = 0; nr < 100; ++nr)
    {
      snprintf (neu, sizeof (neu), "%s~%.2u~", Name, nr);
      if (access (neu, F_OK) < 0)
	return rename (Name, neu);
    }

  // zu viele Sicherheitskopien
  errno = EEXIST;
  return -1;
}


// Methode DELETE
static enum Zustand
entfernen (FILE *aus)
{
  if (!Einstellung.beschreibbar)
    return Methodenfehler;

  if (http.cgi)
    return Zugriffsfehler;

  char *Datei = http.Pfad + 1;
  if (!*Datei)			// Hauptverzeichnis
    return Zugriffsfehler;

  struct stat status;
  if (lstat (Datei, &status) < 0)
    return (errno == EACCES) ? Zugriffsfehler : Suchfehler;

  // Es muss eine regulaere Datei sein, und Schreibrechte haben
  if (!S_ISREG (status.st_mode) || access (Datei, W_OK) < 0)
    return Zugriffsfehler;

  // If-Unmodified-Since
  if (!unveraendert (status.st_mtime))
    return Voraussetzungsfehler;

  // If-Modified-Since
  if (!veraendert (status.st_mtime))
    return Unveraendertmeldung (aus, status.st_mtime);

  if (Sicherheitskopie (Datei) < 0)
    return Zugriffsfehler;

  return Erfolgsmeldung (aus);
}


// Methode PUT
static enum Zustand
empfangen (FILE *aus, FILE *ein)
{
  if (!Einstellung.beschreibbar)
    return Methodenfehler;

  if (http.cgi)
    return Zugriffsfehler;

  // Hier keine Teiluebertragung akzeptieren.
  // Diese Ueberpruefung wird vom Standard gefordert.
  if (akfnetz_Kopfeintrag (http.Kopf, "Content-Range"))
    return Anfragefehler;

  char *Name = http.Pfad + 1;
  if (!*Name)			// Hauptverzeichnis
    return Zugriffsfehler;

  bool neu = false;

  struct stat status;
  if (lstat (Name, &status) < 0)
    {
      if (errno == EACCES)
	return Zugriffsfehler;
      else
	neu = true;
    }

  if (!neu)
    {
      // Es muss eine regulaere Datei sein, und Schreibrechte haben
      if (!S_ISREG (status.st_mode) || access (Name, W_OK) < 0)
	return Zugriffsfehler;

      // If-Unmodified-Since
      if (!unveraendert (status.st_mtime))
	return Voraussetzungsfehler;

      // If-Modified-Since
      if (!veraendert (status.st_mtime))
	return Unveraendertmeldung (aus, status.st_mtime);

      if (Sicherheitskopie (Name) < 0)
	return Zugriffsfehler;
    }

  // TODO evtl. benoetigte Unterverzeichnisse erstellen?

  // erstmal ohne Leserechte erstellen, solange es unvollstaendig ist
  int Datei = open (Name, O_WRONLY | O_CREAT | O_EXCL | O_BINARY, S_IWUSR);
  if (Datei < 0)
    return Zugriffsfehler;

  // Inhalt erst lesen, nachdem all diese Tests durchgefuehrt wurden
  // erst dann soll er 100 Continue schicken
  enum Zustand f = Anfrageinhalt (ein, aus, Datei);
  if (f != Fehlerfrei)
    {
      // Persistenzfehler ist hier auch wirklich ein Fehler!
      close (Datei);
      return f;
    }

  // Leserechte vergeben und schliessen
  if (fchmod (Datei, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH) < 0
      || close (Datei) < 0)
    return Serverfehler;

  if (neu)
    return erstellt (aus);

  // bestehende Datei ersetzt = Erfolg
  return Erfolgsmeldung (aus);
}


static inline noreturn void
Verbindungsabbruch (void)
{
  _exit (1);
}
