/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2019,2023,2024 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// FIXME: das muss komplett ueberarbeitet werden!

#define _POSIX_C_SOURCE 200112L
#define _XOPEN_SOURCE 600

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <stdbool.h>
#include <stdint.h>
#include <unistd.h>
#include <termios.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <time.h>
#include <netdb.h>

#include "akfnetz.h"
#include "httpclient.h"

#define DATEIPUFFER 8192
#define INDEX "index"
#define ZEITLIMIT 10
#define NETRC "/.netrc"
#define SEP " \t\v\f\r\n"

#ifndef O_BINARY
#define O_BINARY 0
#endif

// faengt String s mit der Konstanten c an?
#define Praefix(s,c) (strncasecmp (s, "" c "", sizeof(c)-1) == 0)

bool deutsch, terminal;


static void
Warnung (int Fehlernr, const char *Ursache)
{
  if (Ursache && *Ursache)
    fputs (Ursache, stderr);

  if (Fehlernr)
    fprintf (stderr, ": %s", Fehlermeldung (Fehlernr));

  putc ('\n', stderr);
}


static int
Proxyermittlung (struct akfnetz_Clientdaten *vb)
{
  const char *url;

  url = vb->HTTP_Proxy;

  if (!url)
    vb->Proxyart = Proxy_kein;
  else if (Praefix (url, "http:"))
    vb->Proxyart = Proxy_http;
  else if (Praefix (url, "socks4a:"))
    vb->Proxyart = Proxy_socks4a;
  else if (Praefix (url, "socks4:") || Praefix (url, "socks:"))
    vb->Proxyart = Proxy_socks4;
  else if (Praefix (url, "socks5:") || Praefix (url, "socks5h:"))
    vb->Proxyart = Proxy_socks5;
  else
    {
      errno = ENOSYS;
      return -1;
    }

  return 0;
}


static char *
var (const char *n)
{
  char *e = getenv (n);

  if (e && !*e)
    e = NULL;

  return e;
}


static void
Umgebungsvariablen (struct akfnetz_Clientdaten *vb)
{
  char *p;

  if (!(p = var ("http_proxy")) && !(p = var ("ALL_PROXY")))
    p = var ("all_proxy");

  if (p)
    vb->HTTP_Proxy = p;

/*
  Manche Server koennen ausgetrickst werden, dass sie
  bei CGI oder PHP oAe. die Umgebungsvariable HTTP_PROXY
  (grossgeschrieben) erzeugen.  Darum wird sie hier nicht erlaubt.
  Siehe <https://httpoxy.org/>.
*/

  p = var ("AKFUSERAGENT");
  if (p)
    vb->Agent = p;

  p = var ("AKFAGENTMAIL");
  if (p)
    vb->Mail = p;

  p = var ("HTTPSOCKET");
  if (p)
    vb->Socketpfad = p;

  deutsch = akfnetz_deutschsprachig ();
}


extern void
akfnetz_Clientinitialisierung (struct akfnetz_Clientdaten *vb)
{
  if (!vb)
    return;

  memset (vb, 0, sizeof (*vb));

  vb->Kopf = NULL;
  vb->Laenge = -1;
  vb->Methode = GET;
  vb->Socketpfad = NULL;
  vb->Adressfamilie = AF_UNSPEC;
  vb->Proxyart = Proxy_kein;

  terminal = (bool) isatty (STDERR_FILENO);
  Umgebungsvariablen (vb);
}


static void
Clientfreigabe (struct akfnetz_Clientdaten *vb)
{
  free (vb->Adresse);
  free (vb->Zugang);
  akfnetz_Kopffreigabe (vb->Kopf);
  vb->Adresse = vb->Zugang = NULL;
  vb->Kopf = NULL;
}


extern int
akfnetz_neue_Adresse (struct akfnetz_Clientdaten *vb, const char *neu)
{
  if (!vb)
    return -1;

  if (vb->Adresse)
    {
      free (vb->Adresse);
      vb->Adresse = NULL;
    }

  if (!neu || !*neu || strpbrk (neu, " \t\v\b\a\r\n"))
    return -1;

  size_t l = strcspn (neu, "#");
  vb->Adresse = malloc (l + 1);
  if (!vb->Adresse)
    return -1;

  memcpy (vb->Adresse, neu, l);
  vb->Adresse[l] = '\0';

  return 0;
}


static FILE *
neuverbinden (struct akfnetz_Clientdaten *vb, FILE * Verbindung)
{
  // die HTTP-Proxy-Adresse wechselt nie
  if (Verbindung && (vb->Proxyart == Proxy_http || vb->Socketpfad))
    return Verbindung;

  if (Verbindung)
    fclose (Verbindung);

  // ohne Proxy ist nur http: erlaubt
  // (ein HTTP-Proxy koennte aber ein Gateway fuer andere Protokolle sein)
  if (vb->Proxyart != Proxy_http && !vb->Socketpfad
      && !Praefix (vb->Adresse, "http:"))
    {
      errno = EPROTONOSUPPORT;
      return NULL;
    }

  int v;
  if (vb->Socketpfad)
    v = akfnetz_Lokalverbindung (vb->Socketpfad);
  else if (vb->Proxyart >= Proxy_socks4)
    v = akfnetz_Proxy_Verbindung (vb->HTTP_Proxy, vb->Adresse);
  else
    {
      char *a = vb->Proxyart == Proxy_http ? vb->HTTP_Proxy : vb->Adresse;
      if (!a)
	return NULL;

      v = akfnetz_URL_Verbindung (a, vb->Adressfamilie);
    }

  if (v < 0)
    return NULL;

  // Auszeit setzen - er macht evtl. zwei Versuche
  struct timeval Auszeit;
  Auszeit.tv_sec = 30;
  Auszeit.tv_usec = 0;

  setsockopt (v, SOL_SOCKET, SO_RCVTIMEO, &Auszeit, sizeof (Auszeit));
  setsockopt (v, SOL_SOCKET, SO_SNDTIMEO, &Auszeit, sizeof (Auszeit));

  return fdopen (v, "r+b");
}


static void
Serverstatus (int Status, char **Kopf)
{
  if (deutsch || !Kopf || !Kopf[0])
    fprintf (stderr, "Status: %03d %s\n", Status,
	     akfnetz_Statusmeldung (Status));
  else
    fprintf (stderr, "Status: %s\n", strchr (Kopf[0], ' ') + 1);
}


static void
Fortschritt (intmax_t uebertragen, intmax_t gesamt)
{
  int Prozent;

  Prozent = (int) (uebertragen * 100 / gesamt);

  fprintf (stderr, "\r%16jd Bytes, %3d%%", uebertragen, Prozent);
  fflush (stderr);
}


static int
abwarten (int fd)
{
  int r;

  do
    {
      struct timeval Zeitlimit;
      Zeitlimit.tv_sec = ZEITLIMIT;
      Zeitlimit.tv_usec = 0;

      fd_set lesen;
      FD_ZERO (&lesen);
      FD_SET (fd, &lesen);

      r = select (fd + 1, &lesen, NULL, NULL, &Zeitlimit);
    }
  while (r < 0 && errno == EINTR);

  // Zeitueberschreitung oder Fehler?
  if (r <= 0)
    return -1;

  return 0;
}


// aus darf NULL sein
static off_t
Datentransfer (struct akfnetz_Clientdaten *vb, FILE * ein, FILE * aus)
{
  char Puffer[DATEIPUFFER];
  size_t n;
  off_t gesamt = 0;

  // <0 fuer unbekannt
  if (vb->Laenge == 0)
    return 0;

  if (!vb->still && aus && vb->Laenge > DATEIPUFFER)
    fprintf (stderr, "%16jd Bytes\n", vb->Laenge);

  do
    {
      size_t l = sizeof (Puffer);
      if (0 < vb->Laenge && vb->Laenge - gesamt < (off_t) sizeof (Puffer))
	l = vb->Laenge - gesamt;

      n = fread (Puffer, 1, l, ein);

      if ((!n && ferror (ein)) || (n && aus && !fwrite (Puffer, 1, n, aus)))
	{
	  if (!vb->still && aus)
	    putc ('\n', stderr);
	  return 0;
	}

      gesamt += n;
      if (!vb->still && terminal && aus && vb->Laenge > DATEIPUFFER)
	Fortschritt (gesamt, vb->Laenge);
    }
  while (n && (gesamt < vb->Laenge || vb->Laenge <= 0));

  if (!vb->still && aus)
    putc ('\n', stderr);

  return gesamt;
}


static void
Dateneinlesen (FILE * ein, struct akfnetz_Text *sp, intmax_t Laenge)
{
  char Puffer[DATEIPUFFER];
  size_t n;
  intmax_t Rest = Laenge;

  while (Rest > 0 || Laenge <= 0)
    {
      size_t l = sizeof (Puffer);
      if (0 < Rest && Rest < (off_t) sizeof (Puffer))
	l = Rest;

      n = fread (Puffer, 1, l, ein);

      if (!n)
	break;

      // FIXME
      akfnetz_hinzufuegen (sp, Puffer, n);
      Rest -= n;
    }

  if (ferror (ein))
    sp->Laenge = 0;
}


static void
Kopfzeile (FILE * d, const char *Name, const char *Wert)
{
  if (Wert && *Wert)
    fprintf (d, "%s: %s\r\n", Name, Wert);
}


static void
Kopfzeilenzahl (FILE * d, const char *Name, intmax_t Zahl)
{
  fprintf (d, "%s: %jd\r\n", Name, Zahl);
}


static void
Datensendung (struct akfnetz_Clientdaten *vb, FILE * aus, FILE * ein)
{
  time_t Startzeit;

  time (&Startzeit);
  Datentransfer (vb, ein, aus);
  vb->Laenge = -1;

  if (!vb->still)
    akfnetz_Dauer (stderr, Startzeit, deutsch);
}


static void
Aktualisierung (struct akfnetz_Clientdaten *vb, FILE * v)
{
  struct stat s;
  struct tm t;
  char Zeit[30];

  if (stat (vb->Ausgabename, &s) < 0)
    {
      vb->aktualisiere = false;
      return;
    }

  akfnetz_http_Zeitstring (Zeit, sizeof (Zeit), gmtime_r (&s.st_mtime, &t));
  Kopfzeile (v, "If-Modified-Since", Zeit);
}


static int
Statuscode (char **k)
{
  char *p;

  if (!k || !*k)
    return 901;			// Verbindung zum Server fehlgeschlagen

  // erste Zeile ist Antwort, wie "HTTP/1.1 200 OK"
  p = strchr (*k, ' ');
  if (!p || !Praefix (*k, "HTTP/"))
    return 902;			// Server-Antwort nicht verstanden

  return (int) strtol (p, NULL, 10);
}


// endet die Adresse mit einem Schraegstrich?
static inline bool
Verzeichnisadresse (const char *Adresse)
{
  return (Adresse && *Adresse && Adresse[strlen (Adresse) - 1] == '/');
}


static FILE *
Eingabeoeffnung (struct akfnetz_Clientdaten *vb, const char *Eingabename,
		 struct stat *status)
{
  if (!Eingabename)
    return NULL;

  if (!vb->still)
    fprintf (stderr, deutsch ? "sende Datei: %s\n" : "sending file: %s\n",
	     Eingabename);

  if (stat (Eingabename, status) < 0)
    return NULL;

  // Es muss eine regulaere Datei sein
  if (!S_ISREG (status->st_mode))
    {
      errno = S_ISDIR (status->st_mode) ? EISDIR : EINVAL;
      return NULL;
    }

  vb->Laenge = status->st_size;

  return fopen (Eingabename, "rb");
}


// Angaben fuer zu uebertragende Datei bei PUT
static void
Eingabedateiangaben (FILE * v, time_t z, intmax_t Laenge)
{
  struct tm t;
  char Zeit[30];

  // Aenderungszeit der lokalen Datei
  akfnetz_http_Zeitstring (Zeit, sizeof (Zeit), gmtime_r (&z, &t));

  // nur senden, wenn neuer als Datei auf dem Server
  Kopfzeile (v, "If-Unmodified-Since", Zeit);
  Kopfzeile (v, "Last-Modified", Zeit);
  Kopfzeilenzahl (v, "Content-Length", Laenge);
  Kopfzeile (v, "Expect", "100-continue");
}


static void
Anfragezeile (struct akfnetz_Clientdaten *vb, FILE * v,
	      const char *Eingabename)
{
  fputs (akfnetz_Methodenname[vb->Methode], v);
  putc (' ', v);

  if (!vb->Adresse)
    putc (vb->Methode == OPTIONS ? '*' : '/', v);
  else if (vb->Proxyart == Proxy_http || vb->Socketpfad)
    fputs (vb->Adresse, v);
  else
    {
      const char *Pfad = akfnetz_URL_Pfad (vb->Adresse);
      if (!Pfad || *Pfad != '/')	// nur Anfrage oder Fragment?
	putc (vb->Methode == OPTIONS ? '*' : '/', v);
      if (Pfad)
	fputs (Pfad, v);
    }

  // PUT: Dateinamen anhaengen
  if (PUT == vb->Methode && Verzeichnisadresse (vb->Adresse))
    {
      // Pfad entfernen
      char *p = strrchr (Eingabename, '/');
      akfnetz_urlkodiert (v, p ? p + 1 : Eingabename, "()");
    }

  fputs (" HTTP/1.1\r\n", v);
}


static FILE *
Anfragestellung (struct akfnetz_Clientdaten *vb, FILE * v,
		 const char *Eingabename)
{
  FILE *ein = NULL;
  struct stat Eingabeinfos;
  char Puffer[1024];

  if (PUT == vb->Methode)
    {
      ein = Eingabeoeffnung (vb, Eingabename, &Eingabeinfos);
      if (!ein)
	{
	  Warnung (errno, Eingabename);
	  return NULL;		// naechste Datei versuchen
	}
    }

  Anfragezeile (vb, v, Eingabename);

  // Host: muss immer angegeben sein, auch wenn er leer ist
  if (akfnetz_URL_Host (Puffer, sizeof (Puffer), vb->Adresse))
    fprintf (v, "Host: %s\r\n", Puffer);

  Kopfzeile (v, "Referer", vb->Referrer);	// Schreibfehler beabsichtigt!
  Kopfzeile (v, "User-Agent", vb->Agent);
  Kopfzeile (v, "From", vb->Mail);
  //fputs ("Connection: close\r\n", v);

  Kopfzeile (v, "Accept-Encoding",
	     (vb->gzip) ? "gzip, compress" : "identity");

  if (vb->aktualisiere)
    Aktualisierung (vb, v);

  if (vb->Proxyart == Proxy_http && vb->Methode != TRACE
      && akfnetz_URL_Zugang (Puffer, sizeof (Puffer), vb->HTTP_Proxy))
    akfnetz_Authorization_Basic (v, "Proxy-Authorization",
				 akfnetz_url_dekodieren (Puffer));

  if (vb->Zugang && vb->Methode != TRACE)
    akfnetz_Authorization_Basic (v, "Authorization", vb->Zugang);

  if (PUT == vb->Methode)
    Eingabedateiangaben (v, Eingabeinfos.st_mtime, vb->Laenge);

  // Anfrage abschliessen
  fputs ("\r\n", v);

  return ein;
}


// liest Antwort und schickt evtl. noch Eingabedatei
static int
Antwortlesen (struct akfnetz_Clientdaten *vb, FILE * v, FILE * ein)
{
  int Status;

  Status = 100;			// weitermachen

  while (Status < 200)
    {
      // Die Anfrage muss vollstaendig sein bevor die Antwort kommt
      fflush (v);

      // das abwarten ist nur da, falls der Server keinen Status 100 sendet
      if (vb->Methode != PUT || abwarten (fileno (v)) == 0)
	{
	  akfnetz_Kopffreigabe (vb->Kopf);
	  vb->Kopf = akfnetz_Kopflesen (v);
	  Status = Statuscode (vb->Kopf);
	}

      // Daten senden, falls Status 100 empfangen, oder Zeitlimit ueberschritten
      if (PUT == vb->Methode && Status == 100 && vb->Laenge > 0)
	Datensendung (vb, v, ein);
    }

  return Status;
}


static int
anfragen (struct akfnetz_Clientdaten *vb, FILE * v, const char *Eingabename)
{
  vb->Laenge = -1;

  FILE *ein = Anfragestellung (vb, v, Eingabename);
  if (PUT == vb->Methode && !ein)
    return -1;			// naechste Eingabedatei versuchen

  int Status = Antwortlesen (vb, v, ein);

  if (ein)
    fclose (ein);

  return Status;
}


// eventuell eine passende Dateiendung anhaengen
static void
Dateiendung (struct akfnetz_Clientdaten *vb, char *n, size_t l)
{
  char *Typ;

  Typ = akfnetz_Kopfeintrag (vb->Kopf, "Content-Type");

  if (l >= 6 && Praefix (Typ, "text/html"))
    memcpy (n, ".html", 6);
  else if (l >= 7 && Praefix (Typ, "application/xhtml+xml"))
    memcpy (n, ".xhtml", 7);
  else if (l >= 5 && (Praefix (Typ, "application/xml")
		      || Praefix (Typ, "text/xml")))
    memcpy (n, ".xml", 5);
  else if (l >= 5 && Praefix (Typ, "text/plain"))
    memcpy (n, ".txt", 5);
}


static char *
URL_Dateiname (struct akfnetz_Clientdaten *vb, char *Dateiname, size_t Laenge)
{
  const char *Pfad;
  char *Name;

  Pfad = akfnetz_URL_Pfad (vb->Adresse);
  Name = Pfad ? strrchr (Pfad, '/') : NULL;

  if (!Name || !*Name || !Name[1])
    Name = INDEX;
  else
    ++Name;

  size_t l = strcspn (Name, "?# \t");
  if (l >= Laenge)
    l = Laenge - 1;

  if (l == 0)
    {
      Name = INDEX;
      l = sizeof (INDEX) - 1;
    }

  memcpy (Dateiname, Name, l);
  Dateiname[l] = '\0';

  // eventuell eine passende Dateiendung hinzufuegen
  if (!memchr (Dateiname, '.', l))
    Dateiendung (vb, Dateiname + l, Laenge - l);

  akfnetz_url_dekodieren (Dateiname);

  // Sicherheitsmassnahme
  // Schraegstriche durch Unterstriche ersetzen
  char *p;
  while ((p = strchr (Dateiname, '/')) != NULL)
    *p = '_';

  return Dateiname;
}


// Datei verschieben
static int
Sicherheitskopie (const char *Name)
{
  size_t l = strlen (Name);
  char n[l + 2];
  memcpy (n, Name, l);
  memcpy (n + l, "~", 2);

  return rename (Name, n);
}


static void
Komprimierungsname (struct akfnetz_Clientdaten *vb, char *Dateiname,
		    size_t Groesse)
{
  size_t Laenge;
  char *Kodierung;

  Laenge = strlen (Dateiname);
  Kodierung = akfnetz_Kopfeintrag (vb->Kopf, "Content-Encoding");
  if (Kodierung
      && (!strcasecmp (Kodierung, "gzip")
	  || !strcasecmp (Kodierung, "x-gzip")) && Laenge < Groesse - 3)
    memcpy (Dateiname + Laenge, ".gz", 4);
  else if (Kodierung
	   && (!strcasecmp (Kodierung, "compress")
	       || !strcasecmp (Kodierung, "x-compress"))
	   && Laenge < Groesse - 2)
    memcpy (Dateiname + Laenge, ".Z", 3);
}


static FILE *
Dateierstellung (struct akfnetz_Clientdaten *vb)
{
  char Dateiname[256];

  if (vb->Ausgabename && strlen (vb->Ausgabename) < sizeof (Dateiname))
    strcpy (Dateiname, vb->Ausgabename);
  else
    URL_Dateiname (vb, Dateiname, sizeof (Dateiname));

  // fuegt gegebenenfalls .gz oder .Z an
  Komprimierungsname (vb, Dateiname, sizeof (Dateiname));

  if (!vb->still)
    {
      if (vb->aktualisiere)
	fprintf (stderr, deutsch ? "\naktualisiere Datei: %s\n"
		 : "\nupdating file: %s\n", Dateiname);
      else
	fprintf (stderr, deutsch ? "\nerstelle Datei: %s\n"
		 : "\ncreating file: %s\n", Dateiname);
    }

  if (vb->aktualisiere && Sicherheitskopie (Dateiname) < 0)
    return NULL;

  // In C99 kennt fopen den Modifizierer "x" noch nicht.
  FILE *a = fopen (Dateiname, "wbx");
  if (!a)
    Warnung (errno, Dateiname);

  return a;
}


static int
Inhalt (struct akfnetz_Clientdaten *vb, FILE * Verbindung)
{
  FILE *Ausgabe;
  time_t Startzeit;

  time (&Startzeit);

  Ausgabe = stdout;
  if (!vb->Ausgabename || strcmp (vb->Ausgabename, "-") != 0)
    {
      Ausgabe = Dateierstellung (vb);
      if (!Ausgabe)
	return -1;
    }

  char *TE = akfnetz_Kopfeintrag (vb->Kopf, "Transfer-Encoding");
  if (TE && !strncasecmp (TE, "chunked", 7))
    vb->Laenge = akfnetz_gestueckelt_Filter (Verbindung, Ausgabe);
  else
    {
      vb->Laenge = akfnetz_Kopfzahl (vb->Kopf, "Content-Length");
      if (vb->Laenge != 0)	// <0 fuer unbekannt
	Datentransfer (vb, Verbindung, Ausgabe);
    }

  if (Ausgabe != stdout && fclose (Ausgabe) == EOF)
    return -1;

  if (!vb->still)
    akfnetz_Dauer (stderr, Startzeit, deutsch);

  return 0;
}


static int
Inhalt_ignorieren (struct akfnetz_Clientdaten *vb, FILE * v)
{
  char *TE = akfnetz_Kopfeintrag (vb->Kopf, "Transfer-Encoding");

  if (TE && !strncasecmp (TE, "chunked", 7))
    vb->Laenge = akfnetz_gestueckelt (v, NULL, NULL);
  else
    {
      vb->Laenge = akfnetz_Kopfzahl (vb->Kopf, "Content-Length");
      if (vb->Laenge != 0)	// kleiner 0 fuer unbekannt
	Datentransfer (vb, v, NULL);
    }

  return 0;
}


// --Links
static void
Linkanzeige (const char *Link, size_t Linklaenge, void *d)
{
  // ignorieren falls nur Fragment oder Query-String
  if (*Link == '#' || *Link == '?')
    return;

  fwrite (Link, sizeof (char), Linklaenge, d);
  putc ('\n', d);
}


// liest HTML-Code und gibt Links aus
// --Links
static int
Linkausgabe (struct akfnetz_Clientdaten *vb, FILE * v)
{
  struct akfnetz_Text html;

  char *Typ = akfnetz_Kopfeintrag (vb->Kopf, "Content-Type");
  if (!Typ
      || (!Praefix (Typ, "text/html")
	  && !Praefix (Typ, "application/xhtml+xml")
	  && !Praefix (Typ, "application/xml") && !Praefix (Typ, "text/xml")))
    {
      // nicht als (X)HTML ausgewiesen
      Inhalt_ignorieren (vb, v);
      return -1;
    }

  char *TE = akfnetz_Kopfeintrag (vb->Kopf, "Transfer-Encoding");
  if (TE && !strncasecmp (TE, "chunked", 7))
    {
      if (!akfnetz_gestueckelt_Text (v, &html))
	return -1;

      vb->Laenge = html.Laenge;
    }
  else
    {
      vb->Laenge = akfnetz_Kopfzahl (vb->Kopf, "Content-Length");
      if (!akfnetz_Texterstellung
	  (&html, vb->Laenge > 0 ? vb->Laenge + 1 : 1024))
	return -1;
      Dateneinlesen (v, &html, vb->Laenge);
    }

  if (html.Laenge > 0)
    akfnetz_Links (html.Inhalt, &Linkanzeige, stdout);

  akfnetz_Textfreigabe (&html);

  return 0;
}


static void
Zugangsloeschung (struct akfnetz_Clientdaten *vb)
{
  if (vb->Zugang)
    {
      free (vb->Zugang);
      vb->Zugang = NULL;
    }
}


static char *
Zugangsdatenimport (const char *Login, const char *Passwort)
{
  if (!Login && !Passwort)
    return NULL;

  if (!Login)
    Login = "";

  if (!Passwort)
    Passwort = "";

  size_t nl = strlen (Login), pl = strlen (Passwort);
  char *Zugang = malloc (nl + pl + 2);
  if (Zugang)
    {
      memcpy (Zugang, Login, nl);
      Zugang[nl] = ':';
      memcpy (Zugang + nl + 1, Passwort, pl + 1);
    }

  return Zugang;
}


// $HOME/.netrc einlesen
static bool
Zugangsdatendatei (struct akfnetz_Text *sp)
{
  char *Heim = var ("HOME");
  if (!Heim)
    return false;

  size_t l = strlen (Heim);
  char Name[l + sizeof (NETRC)];
  memcpy (Name, Heim, l);
  memcpy (Name + l, NETRC, sizeof (NETRC));

  struct stat st;
  if (stat (Name, &st) < 0)
    {
      Warnung (errno, Name);
      return false;
    }

  // .netrc darf nur fuer den Besitzer lesbar sein
  if ((st.st_mode & (S_IRGRP | S_IROTH)) != 0)
    {
      Warnung (EACCES, Name);
      return false;
    }

  FILE *d = fopen (Name, "r");
  if (!d)
    return false;

  Dateneinlesen (d, sp, st.st_size);

  fclose (d);
  return true;
}


// ~/.netrc auswerten
static bool
Zugangsdatei (struct akfnetz_Clientdaten *vb)
{
  char *Login, *Passwort, *sc;
  char Server[256];
  struct akfnetz_Text sp;

  Zugangsloeschung (vb);
  Login = Passwort = NULL;

  if (!akfnetz_URL_Host (Server, sizeof (Server), vb->Adresse)
      || !akfnetz_Texterstellung (&sp, 1024))
    return false;

  if (!Zugangsdatendatei (&sp))
    {
      akfnetz_Textfreigabe (&sp);
      return false;
    }

  char *Token = strtok_r (sp.Inhalt, SEP, &sc);
  char *Wert = strtok_r (NULL, SEP, &sc);
  while (Token && Wert && !Login && !Passwort)
    {
      if (!strcmp (Token, "machine") && !strcmp (Wert, Server))
	{
	  // Maschine gefunden
	  do
	    {
	      Token = strtok_r (NULL, SEP, &sc);

	      if (Token)
		{
		  Wert = strtok_r (NULL, SEP, &sc);
		  if (!strcmp (Token, "login"))
		    Login = Wert;
		  else if (!strcmp (Token, "password"))
		    Passwort = Wert;
		}
	    }
	  while (Token && Wert && strcmp (Token, "machine") != 0
		 && strcmp (Token, "default") != 0);
	}

      Token = strtok_r (NULL, SEP, &sc);
      Wert = strtok_r (NULL, SEP, &sc);
    }

  if (Login || Passwort)
    vb->Zugang = Zugangsdatenimport (Login, Passwort);

  akfnetz_Textfreigabe (&sp);

  return (Login != NULL || Passwort != NULL);
}


static void
Zeilenendeentfernung (char *s)
{
  char *t;
  t = strpbrk (s, "\r\n");
  if (t)
    *t = '\0';
}


// Kennung verdeckt eingeben
static void
Kennungseingabe (char *s, size_t l)
{
  struct termios t, n;
  int in;

  in = fileno (stdin);

  tcgetattr (in, &t);
  memcpy (&n, &t, sizeof (n));
  n.c_lflag &= ~ECHO;
  n.c_lflag |= ECHONL;
  tcsetattr (in, TCSANOW, &n);

  fgets (s, l, stdin);
  tcsetattr (in, TCSANOW, &t);
}


// Zugangsdaten ueber Terminal abfragen
static void
Zugangsdatenabfrage (struct akfnetz_Clientdaten *vb)
{
  char n[512], p[512];

  Zugangsloeschung (vb);

  fputs (deutsch
	 ? "Bitte Zugangsdaten eingeben.\n"
	 "Abbruch durch Angabe eines leeren Namens.\n"
	 : "Please enter your credentials.\n"
	 "Enter an empty name to cancel.\n", stderr);

  fputs ("Name: ", stderr);
  fgets (n, sizeof (n), stdin);
  Zeilenendeentfernung (n);
  if (!*n)
    return;

  fputs (deutsch ? "Kennung: " : "Passphrase: ", stderr);
  Kennungseingabe (p, sizeof (p));
  Zeilenendeentfernung (p);

  vb->Zugang = Zugangsdatenimport (n, p);
}


static void
Infoanzeige (const char *Name, const char *Wert)
{
  if (Name && Wert)
    fprintf (stderr, "%s: %s\n", Name, Wert);
}


static void
Kopfanzeigen (char **Kopf, const char *Name, const char *Eintrag)
{
  if (deutsch)
    Infoanzeige (Name, akfnetz_Kopfeintrag (Kopf, Eintrag));
  else
    Infoanzeige (Eintrag, akfnetz_Kopfeintrag (Kopf, Eintrag));
}


static void
Datumanzeigen (char **Kopf, const char *Einleitung, const char *Eintrag)
{
  char *Wert;

  Wert = akfnetz_Kopfeintrag (Kopf, Eintrag);
  if (Einleitung && Wert)
    {
      struct tm Zeit;

      if (deutsch && akfnetz_analysiere_http_Zeit (&Zeit, Wert))
	{
	  char zt[30];
	  strftime (zt, sizeof (zt),
		    "%d.%m.%Y um %k\u00A0Uhr\u00A0%M UTC", &Zeit);

	  fprintf (stderr, "%s %s, dem %s\n",
		   Einleitung, akfnetz_Wochentag[Zeit.tm_wday], zt);
	}
      else
	fprintf (stderr, "%s: %s\n", Eintrag, Wert);
    }
}


static void
Groessenanzeige (const char *Name, intmax_t Wert)
{
  if (Wert >= 1000)
    {
      char g[10];
      akfnetz_Datengroessenstring (g, sizeof (g), Wert);
      fprintf (stderr, "%s: %s (%jd Bytes)\n", Name, g, Wert);
    }
  else
    fprintf (stderr, "%s: %jd Bytes\n", Name, Wert);
}


static void
IPanzeige (FILE * Verbindung)
{
  struct sockaddr_storage s;
  socklen_t sl;
  char Adresse[INET6_ADDRSTRLEN];

  sl = sizeof (s);
  if (!getpeername (fileno (Verbindung), (struct sockaddr *) &s, &sl)
      && inet_ntop (s.ss_family, akfnetz_Socketadresse (&s),
		    Adresse, sizeof (Adresse)))
    fprintf (stderr, "IP: %s\n", Adresse);

  // Wenn es keine IP-Verbindung ist, wird nichts angezeigt
}


#define Kopfanzeige(a,b)  Kopfanzeigen(vb->Kopf, a,b)
#define Datumsanzeige(a,b)  Datumanzeigen(vb->Kopf, a,b)


static void
Informationsanzeige (struct akfnetz_Clientdaten *vb, FILE * v, int Status)
{
  // Der Proxy-Wert koennte Zugangsdaten enthalten, also nicht anzeigen!
  if (vb->HTTP_Proxy)
    fputs (deutsch ? "Die Anfrage erfolgt ueber einen Proxy.\n"
	   : "This request goes through a proxy.\n", stderr);
  else
    IPanzeige (v);

  Infoanzeige (deutsch ? "Adresse" : "Address", vb->Adresse);
  Serverstatus (Status, vb->Kopf);

  Kopfanzeige ("neue Adresse", "Location");
  Kopfanzeige ("Server", "Server");

  // wenn fehlerfrei, oder unveraendert
  if ((Status < 300 || Status == 304) && vb->Methode != PUT)
    {
      //Datumsanzeige ("abgeschickt am", "Date");
      Kopfanzeige ("eigentliche Adresse", "Content-Location");
      Kopfanzeige ("Typ", "Content-Type");
      Kopfanzeige ("Sprache", "Content-Language");
      Kopfanzeige ("MD5-Pruefwert", "Content-MD5");	// aus Standard entfernt
      Datumsanzeige ("zuletzt geaendert am", "Last-Modified");
      Datumsanzeige ("gueltig bis", "Expires");
      intmax_t l = akfnetz_Kopfzahl (vb->Kopf, "Content-Length");
      if (l >= 0)
	Groessenanzeige (deutsch ?
			 "angegebene Groesse" : "Content-Length", l);
      if (vb->Laenge > 0)
	Groessenanzeige (deutsch ?
			 "uebertragene Groesse" :
			 "transmitted length", vb->Laenge);

      Kopfanzeige ("Verbindung", "Connection");
      Kopfanzeige ("Kodierung", "Content-Encoding");
      Kopfanzeige ("Link", "Link");
      Kopfanzeige ("Transport-Kodierung", "Transfer-Encoding");
      Kopfanzeige ("Via", "Via");
    }

  putc ('\n', stderr);
}


static void
Direktanzeige (char **Kopf)
{
  for (int i = 0; Kopf[i]; ++i)
    {
      fputs (Kopf[i], stderr);
      putc ('\n', stderr);
    }

  putc ('\n', stderr);
}


// Testet, ob Berechtigung vom Typ Basic verlangt wird
static inline bool
Basic_Berechtigung (char **Kopf)
{
  return (akfnetz_Tokensuche (Kopf, "WWW-Authenticate", "Basic") != NULL);
}


// Parameter: Vektor und Anzahl der zu uebertragenden Dateien bei PUT
// Ergebnis: letzter Statuscode, oder -1 fuer Fehler
extern int
akfnetz_Netzaufruf (struct akfnetz_Clientdaten *vb, char **Dateien,
		    int Dateianzahl)
{
  int Status, Altstatus;
  int Dateinummer, Einlogversuche;
  FILE *Verbindung;

  if (!vb)
    return -1;

  Status = Altstatus = 0;
  Einlogversuche = 0;
  Dateinummer = 0;

  if (Proxyermittlung (vb) < 0)
    return -1;

  Verbindung = neuverbinden (vb, NULL);
  if (!Verbindung)
    {
      Clientfreigabe (vb);
      errno = EHOSTUNREACH;
      return -1;
    }

Anfrage:
  Altstatus = Status;
  Status = anfragen (vb, Verbindung,
		     (Dateianzahl > 0) ? Dateien[Dateinummer] : NULL);

  if (Status < 0)
    goto weiter;		// naechste Eingabe-Datei versuchen

  if (Status == 901 && Altstatus < 900)
    {
      // Server antwortet nicht mehr? -> nochmal versuchen
      Verbindung = neuverbinden (vb, Verbindung);
      if (!Verbindung)
	{
	  Status = -1;
	  goto Ende;
	}

      goto Anfrage;
    }

  if (Status == 401 && Einlogversuche < 3 && Basic_Berechtigung (vb->Kopf))
    {
      if ((Einlogversuche != 0 || !Zugangsdatei (vb))
	  && !vb->nicht_interaktiv)
	Zugangsdatenabfrage (vb);

      if (vb->Zugang)
	{
	  if (vb->Methode != HEAD)
	    Inhalt_ignorieren (vb, Verbindung);

	  ++Einlogversuche;
	  goto Anfrage;
	}
    }
  else if (!vb->nicht_umleiten && (Status / 100) == 3)
    {
      // Mit Proxy findet kein Neuverbinden statt
      if (vb->Methode != HEAD && Status != 304)
	Inhalt_ignorieren (vb, Verbindung);

      // zur Sicherheit
      Zugangsloeschung (vb);
      char *n = akfnetz_Kopfeintrag (vb->Kopf, "Location");
      if (n && *n)
	{
	  if (!vb->still)
	    Informationsanzeige (vb, Verbindung, Status);
	  akfnetz_neue_Adresse (vb, n);

	  if (!vb->HTTP_Proxy && !vb->Socketpfad)
	    {
	      Verbindung = neuverbinden (vb, Verbindung);

	      if (!Verbindung)
		{
		  Status = -1;
		  goto Ende;
		}
	    }

	  goto Anfrage;
	}
      // bei 304 (unveraendert) erfolgt keine Umleitung
    }

  // Infos ausgeben
  if (!vb->still)
    {
      if (vb->ungefiltert && vb->Kopf)
	Direktanzeige (vb->Kopf);
      else
	Informationsanzeige (vb, Verbindung, Status);
    }

  if (vb->Methode != HEAD && Status >= 200 && Status != 204 && Status != 304
      && Status < 900)
    {
      int r = 0;

      if (vb->Links && Status < 300)
	r = Linkausgabe (vb, Verbindung);
      else if (Status < 300 && (vb->Methode == GET || vb->Methode == TRACE))
	r = Inhalt (vb, Verbindung);
      else if (vb->Methode != HEAD)
	r = Inhalt_ignorieren (vb, Verbindung);

      if (r < 0)
	{
	  Status = -1;
	  goto Ende;
	}
    }

weiter:
  // eventuell naechste Datei senden
  if (vb->Methode == PUT && ++Dateinummer < Dateianzahl)
    goto Anfrage;

Ende:
  if (Verbindung)
    fclose (Verbindung);

  Clientfreigabe (vb);

  return Status;
}
