/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2024-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// siehe RFC 4266 - The gopher URI Scheme
// unterstuetzt auch finger - die URI dafuer ist nicht genormt

#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include "akfnetz.h"

enum Protokolltyp
{ P_GOPHER, P_FINGER };

static int Anfrage (int Verbindung, const char *url, enum Protokolltyp);

// Unterstuetzt Gopher und Finger
extern int
akfnetz_Gopher_URL (const char *url, const char *Proxy)
{
  int Verbindung;
  enum Protokolltyp Protokoll;

  if (!url || !*url)
    {
      errno = EDESTADDRREQ;
      return -1;
    }

  // Ist es eine Gopher-URL?
  if (!strncasecmp ("gopher://", url, 9))
    Protokoll = P_GOPHER;
  else if (!strncasecmp ("finger://", url, 9))
    Protokoll = P_FINGER;
  else
    {
      errno = EPROTONOSUPPORT;
      return -1;
    }

  Verbindung = akfnetz_Proxy_Verbindung (Proxy, url);
  if (Verbindung < 0)
    return -1;

  return Anfrage (Verbindung, url, Protokoll);
}


static int
Anfrage (int Verbindung, const char *url, enum Protokolltyp Protokoll)
{
  int v = -1;

  // Pfad ermitteln
  const char *pf = akfnetz_URL_Pfad (url);
  if (!pf)
    pf = "";

  // Schraegstriche am Anfang ueberspringen
  while (*pf == '/')
    ++pf;

  // Das erste Zeichen bei Gopher ist der Typ, erstmal ignorieren
  if (Protokoll == P_GOPHER && *pf)
    ++pf;

  if (*pf || Protokoll == P_GOPHER)
    {
      size_t pfl = strlen (pf) + 1;
      char Selektor[pfl];
      memcpy (Selektor, pf, pfl);
      akfnetz_url_dekodieren (Selektor);
      v = akfnetz_Gopher_Selektor (Verbindung, Selektor);
    }
  else				// finger://Benutzer@example.net
    {
      char Benutzer[1024];
      akfnetz_URL_Zugang (Benutzer, sizeof (Benutzer), url);
      char *p = strchr (Benutzer, ':');
      if (p)
	*p = '\0';		// Passwort ignorieren
      akfnetz_url_dekodieren (Benutzer);
      v = akfnetz_Gopher_Selektor (Verbindung, Benutzer);
    }

  if (v != Verbindung)		// Fehler
    close (Verbindung);

  return v;
}
