/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * CGI fuer AKF-Netz-Server
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/*
 * CGI/1.1 (Common Gateway Interface)
 * RFC 3875
 */

#ifndef KEIN_CGI

#define _POSIX
#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 700

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/socket.h>

#include "akfnetz.h"
#include "akfwebserver.h"

#define CGI_ZEITLIMIT (5*60)	// Sekunden

#pragma GCC poison  exit

static inline char *Textanfang (const char *);

// false bei Fehler
static bool
Verzeichniswechsel (const char *v, size_t l)
{
  if (!l)
    return true;

  // wenn der String schon terminiert ist, brauchen wir keine Kopie
  if (v[l] == '\0')
    return !chdir (v);

  // terminierten String erstellen
  char Verzeichnis[l + 1];
  memcpy (Verzeichnis, v, l);
  Verzeichnis[l] = '\0';

  return !chdir (Verzeichnis);
}


static void
Lokalumleitung (char *Ort)
{
  if (redsam (2))
    akfnetz_Logbuch (Deutsch == Systemsprache
		     ? "* Lokale Umleitung -> %s\n"
		     : "* local redirection -> %s\n", Ort);

  http.Anfrage = http.cgi = NULL;
  free (http.URI);

  // Fragment wegignorieren
  char *Fragment = strchr (Ort, '#');
  if (Fragment)
    *Fragment = '\0';

  http.URI = strdup (Ort);

  // Query-String entfernen
  char *q = strchr (Ort, '?');
  if (q)
    *q = '\0';

  free (http.Pfad);

  akfnetz_url_dekodieren (Ort);
  http.Pfadlaenge = strlen (Ort);
  http.Pfad = malloc (http.Pfadlaenge + 1);
  if (http.Pfad)
    memcpy (http.Pfad, Ort, http.Pfadlaenge + 1);
  else
    http.Pfadlaenge = 0;

  q = strchr (http.URI, '?');
  if (q)
    http.Anfrage = q + 1;

  if (POST == http.Methode)
    http.Methode = GET;
}


// HTTP-Kopf ist schon angefangen, aber noch nicht beendet
static void
Umleitungsseite (FILE * aus, const char *Ort)
{
  FILE *d;
  char *Inhalt;
  size_t Inhaltsgroesse;

  d = open_memstream (&Inhalt, &Inhaltsgroesse);
  if (!d)
    {
      http_Ende (aus);
      return;
    }

  akfnetz_html_Linkseite (d, Ort);
  fclose (d);

  // Statuszeile wurde bereits ausgegeben
  http_Kopf (aus, "Content-Type", "text/html");
  http_ContentLength (aus, Inhaltsgroesse);
  http_Ende (aus);

  if (http.Methode != HEAD)
    fwrite (Inhalt, 1, Inhaltsgroesse, aus);

  free (Inhalt);
}


static enum Zustand
kaputt (char **Kopf)
{
  akfnetz_Kopffreigabe (Kopf);
  if (redsam (2))
    akfnetz_Logbuch ("* Das CGI ist kaputt.\n");

  return Serverfehler;
}


// auf Antwort vom CGI warten, aber nicht ewig
static int
abwarten (int ein, pid_t prg)
{
  int r;

  do
    {
      struct timeval Zeitlimit;
      Zeitlimit.tv_sec = CGI_ZEITLIMIT;
      Zeitlimit.tv_usec = 0;

      fd_set lesen;
      FD_ZERO (&lesen);
      FD_SET (ein, &lesen);

      r = select (ein + 1, &lesen, NULL, NULL, &Zeitlimit);
    }
  while (r < 0 && errno == EINTR);

  // Zeitueberschreitung oder Fehler?
  if (r <= 0)
    {
      // jetzt reicht's mir aber mal...
      kill (prg, SIGTERM);
      if (redsam (2))
	{
	  if (Deutsch == Systemsprache)
	    akfnetz_Logbuch ("* Das CGI war mir zu langsam. "
			     "Das hab ich jetzt abgeschossen.\n");
	  else
	    akfnetz_Logbuch ("* The CGI was too slow. I've killed it.\n");
	  // Es ist ein Mord geschehen! Huelfe, Polizei!
	}

      return -1;
    }

  return 0;
}


// handelt es sich um ein NPH-Skript?
static inline bool
NPH_Skript (const char *Befehl)
{
  return Praefix (Befehl, "nph-");
}


// Fuer NPH-Skripte
static enum Zustand
Direktausgabe (FILE * aus, FILE * ein)
{
  if (!ein || !aus)
    return Serverfehler;

  // die Pufferung darf leider nicht nachtraeglich veraendert werden,
  // also umgehen wir den Stream
  fflush (aus);
  int Ausgabe = fileno (aus);

  int c;
  while ((c = getc_unlocked (ein)) != EOF)
    {
      char Zeichen = (char) c;
      ssize_t n;

      do
	n = write (Ausgabe, &Zeichen, sizeof (Zeichen));
      while (n < 0 && errno == EINTR);
    }

  return Persistenzfehler;
}


// Uebertraegt den Inhalt gepuffert
static void
Inhalt (FILE * aus, FILE * ein)
{
  size_t n;
  char Puffer[2048];

  http.Antwortlaenge = 0;

  while ((n = fread (Puffer, sizeof (char), sizeof (Puffer), ein)) != 0)
    {
      fwrite (Puffer, sizeof (char), n, aus);
      http.Antwortlaenge += n;
    }
}


// Uebertraegt den Inhalt blockweise (chunked)
static void
Inhalt_blockweise (FILE * aus, FILE * ein)
{
  size_t n;
  char Puffer[2048];

  http.Antwortlaenge = 0;

  while ((n = fread (Puffer, sizeof (char), sizeof (Puffer), ein)) != 0)
    {
      fprintf (aus, "%zX\r\n", n);
      fwrite (Puffer, sizeof (char), n, aus);
      fputs ("\r\n", aus);

      http.Antwortlaenge += n;
    }

  // letzter Block und leerer Trailer
  fputs ("0\r\n\r\n", aus);
}


static void
Kopfausgabe (FILE * aus, char **cgiKopf)
{
  int i;
  char *k;

  if (http.Version == 0)
    return;

  for (i = 0; (k = cgiKopf[i]) != NULL; ++i)
    {
      // ueberfluessige oder gefaehrliche Angaben rausfiltern
      if (Praefix (k, "Status:")
	  || Praefix (k, "X-CGI-")
	  || Praefix (k, "Server:")
	  || Praefix (k, "Date:")
	  || Praefix (k, "Transfer-Encoding:")
	  || Praefix (k, "Trailer:")
	  || Praefix (k, "Connection:") || Praefix (k, "Proxy-Connection:"))
	continue;

      fprintf (aus, "%s\r\n", k);
      if (redsam (2))
	akfnetz_Logbuch ("> %s\n", k);
    }
}


// wertet CGI-Kopf aus
static enum Zustand
auswerten (FILE * aus, FILE * ein)
{
  int Statusnummer = 200;
  char **cgiKopf;

Anfang:
  if (redsam (2))
    akfnetz_Logbuch ("* CGI\n");

  cgiKopf = akfnetz_Kopflesen (ein);

  // CGI-Kopf darf nicht leer sein, NPH hier abweisen
  if (!cgiKopf || !cgiKopf[0] || Praefix (cgiKopf[0], "HTTP/"))
    return kaputt (cgiKopf);

  char *Status = akfnetz_Kopfeintrag (cgiKopf, "Status");
  char *Ort = akfnetz_Kopfeintrag (cgiKopf, "Location");
  char *Inhaltstyp = akfnetz_Kopfeintrag (cgiKopf, "Content-Type");

  // mindestens eines noetig: Status, Location, Content-Type
  if ((!Status || !*Status) && (!Inhaltstyp || !*Inhaltstyp)
      && (!Ort || !*Ort))
    return kaputt (cgiKopf);

  // Lokale Umleitung
  if (Ort && *Ort == '/')
    {
      // Location muss jetzt einzige Kopfzeile sein
      if (cgiKopf[1] != NULL)
	return kaputt (cgiKopf);

      Lokalumleitung (Ort);
      akfnetz_Kopffreigabe (cgiKopf);
      return Neuanforderung;
    }

  // HTTP-Kopf
  if (Status)
    {
      // Status ermitteln
      char *Meldung;
      Statusnummer = (int) strtol (Status, &Meldung, 10);

      if (100 > Statusnummer || Statusnummer > 999)
	return kaputt (cgiKopf);

      http_Anfang (aus, Statusnummer, Textanfang (Meldung));
    }
  else if (Ort)
    http_Anfang (aus, (Statusnummer = 302), "Found");
  else
    http_Anfang (aus, (Statusnummer = 200), "OK");

  Kopfausgabe (aus, cgiKopf);

  // Umleitung ohne Inhalt => Umleitungsseite generieren
  // entgegen Standard auch mit anderen Kopfzeilen erlaubt
  if (Ort && !Inhaltstyp)
    {
      Umleitungsseite (aus, Ort);
      akfnetz_Kopffreigabe (cgiKopf);
      return Fehlerfrei;
    }

  bool Laenge = (akfnetz_Kopfeintrag (cgiKopf, "Content-Length") != NULL);
  bool blockweise = false;	// chunked

  if (!Laenge && http.Version == 1 && http.Unterversion >= 1)
    blockweise = true;

  // "Connection: close" in der Anfrage(!) respektieren
  // (Das CGI kann das nicht ausloesen)
  if (akfnetz_Tokensuche (http.Kopf, "Connection", "close"))
    blockweise = Laenge = false;

  // per Definition leer?
  bool leer = (Statusnummer == 204 || Statusnummer == 304
	       || Statusnummer < 200);

  if (!leer && !Laenge && !Inhaltstyp)
    {
      http_Kopf (aus, "Content-Length", "0");
      leer = true;
    }

  if (!leer)
    {
      if (blockweise)
	http_Kopf (aus, "Transfer-Encoding", "chunked");
      else if (!Laenge)
	http_Kopf (aus, "Connection", "close");
    }

  http_Ende (aus);
  akfnetz_Kopffreigabe (cgiKopf);

  if (Statusnummer < 200)
    goto Anfang;

  if (HEAD == http.Methode || leer)
    return Fehlerfrei;

  if (blockweise)
    Inhalt_blockweise (aus, ein);
  else
    Inhalt (aus, ein);

  if (!Laenge && !blockweise)
    return Persistenzfehler;

  return Fehlerfrei;
}


static enum Zustand
abarbeiten (FILE * aus, int CGIAusgabe, pid_t prg, const char *Befehl)
{
  if (http.Inhalt >= 0)
    {
      close (http.Inhalt);
      http.Inhalt = -1;
    }

  // die Eingabe ist hier die Ausgabe des CGI-Skriptes
  FILE *Eingabe;
  if (abwarten (CGIAusgabe, prg) < 0
      || !(Eingabe = fdopen (CGIAusgabe, "rb")))
    {
      close (CGIAusgabe);
      return Serverfehler;
    }

  enum Zustand Status;
  if (!NPH_Skript (Befehl))
    Status = auswerten (aus, Eingabe);
  else				// NPH
    Status = Direktausgabe (aus, Eingabe);

  // schliesst auch CGIAusgabe
  fclose (Eingabe);

  return Status;
}


// Teste, ob das Programm vorhanden und ausfuehrbar ist
static int
Befehlstest (const char *Verzeichnis, size_t Verzeichnislaenge,
	     const char *Befehl, size_t Befehllaenge)
{
  char Befehlspfad[Verzeichnislaenge + Befehllaenge + 2];

  if (Verzeichnislaenge)
    {
      memcpy (Befehlspfad, Verzeichnis, Verzeichnislaenge);
      Befehlspfad[Verzeichnislaenge] = '/';
      memcpy (Befehlspfad + Verzeichnislaenge + 1, Befehl, Befehllaenge);
      Befehlspfad[Verzeichnislaenge + 1 + Befehllaenge] = '\0';
    }
  else
    {
      memcpy (Befehlspfad, Befehl, Befehllaenge);
      Befehlspfad[Befehllaenge] = '\0';
    }

  return access (Befehlspfad, R_OK | X_OK);
}


static enum Zustand
Prozesserstellung (FILE * aus,
		   const char *Verzeichnis, size_t Verzeichnislaenge,
		   const char *Befehl, size_t Befehllaenge)
{
  enum Zustand e = Fehlerfrei;
  int Rohr[2];

  if (pipe (Rohr) < 0)
    {
      if (redsam (2))
	akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));
      return Serverfehler;
    }

  fflush (aus);
  // Zur Sicherheit, damit der Pufferinhalt nicht verdoppelt wird

  pid_t prg = fork ();

  switch (prg)
    {
    case 0:			// Kindprozess (CGI-Skript ausfuehren)
      close (Rohr[0]);
      fclose (Einstellung.Logbuch);
      fclose (aus);

      if (Verzeichniswechsel (Verzeichnis, Verzeichnislaenge))
	akfnetz_cgi_ausfuehren (Rohr[1], Befehl, Befehllaenge);

      // Befehl konnte nicht ausgefuehrt werden
      _exit (127);
      break;


    case -1:			// Fehler
      if (redsam (2))
	akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));

      close (Rohr[0]);
      close (Rohr[1]);

      e = Serverfehler;
      break;


    default:			// Elternprozess
      close (Rohr[1]);
      e = abarbeiten (aus, Rohr[0], prg, Befehl);
    }

  return e;
}


extern enum Zustand
akfnetz_cgi (FILE * aus, const char *Befehl)
{
  size_t Befehllaenge;

  // OPTIONS & TRACE wird ausserhalb behandelt
  if (http.Methode > POST)
    return Methodenfehler;

  if (Befehl)
    Befehllaenge = strlen (Befehl);
  else
    {
      Befehl = http.cgi;
      Befehllaenge = http.CGIlaenge;
    }

  // zB. /cgi-bin/
  if (!Befehl || !*Befehl)
    return Zugriffsfehler;

  char *Verzeichnis = http.Pfad + 1;
  size_t Verzeichnislaenge = 0;

  // in /cgi-bin/
  if (Befehl == http.Pfad + 9 && !strncmp (http.Pfad, "/cgi-bin/", 9))
    Verzeichnislaenge = sizeof ("cgi-bin") - 1;
  else				// woanders
    {
      // Dateierweiterung vorhanden, aber nicht aktiviert?
      if (!Einstellung.cgi
	  && Befehllaenge > 4 && !strcmp (Befehl + Befehllaenge - 4, ".cgi"))
	return Zugriffsfehler;

      // Wenn http.cgi gesetzt ist, ist es Teil des Pfades

      if (http.cgi && http.cgi > http.Pfad + 1)
	Verzeichnislaenge = http.cgi - Verzeichnis - 1;
      else if (!http.cgi && http.Pfad[1])	// index.cgi
	Verzeichnislaenge = http.Pfadlaenge - 1;
    }

  if (Befehlstest (Verzeichnis, Verzeichnislaenge, Befehl, Befehllaenge) < 0)
    return (errno == EACCES) ? Zugriffsfehler : Suchfehler;

  return Prozesserstellung (aus, Verzeichnis, Verzeichnislaenge,
			    Befehl, Befehllaenge);
}

#endif
