/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * CGI: hochladen (mittels mmap)
 * Copyright (c) 2018-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As an exception HTML and CSS code from this file may be copied freely,
 * without the result being subject to this license.
 */

// Auf 32-Bit-Systemen liegt das Limit etwas unter 2GB

// siehe RFC 7578

#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 600

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <limits.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>

#include "akfnetz.h"
#include "FehlerStil.h"

// Es kann auch am Server die CGI-Variable UPLOAD gesetzt werden
#ifndef UPLOAD
#define UPLOAD "."
#endif

#define LOGDATEI "hochladen.log"

#define TITEL u8"hochladen"
#define DANKE u8"Danke"
#define FEHLER u8"Fehler"
#define VERZEICHNISFEHLER u8"In das Verzeichnis konnte nicht gewechselt werden."
#define FORMULARFEHLER u8"Die Formulareigenschaften sind nicht richtig."
#define GRENZFEHLER u8"Kein 'boundary' gefunden."
#define MMAPFEHLER u8"Kein mmap m\u00F6glich."
#define KONFLIKTFEHLER u8"Eine Datei mit dem Namen existiert bereits. Bitte umbenennen."
#define ANLEGEFEHLER u8"Die Datei konnte nicht angelegt werden."
#define GROESSENFEHLER u8"Wegen \u00DCberf\u00FCllung geschlossen."
#define DATEIGROESSENFEHLER u8"Die Datei ist zu gro\u00DF."
#define SPEICHERFEHLER u8"Fehler beim Speichern."
#define DATEIFEHLER u8"Keine Datei gefunden."

#define SCHLIESSEN u8"\u00D7"
#define FEHLERMELDUNG u8"<span class='emoji'>\u2639</span> " FEHLER
#define ERFOLG u8"<span class='emoji'>\u263A</span> " DANKE

// faengt String s mit der Konstanten c an?
#define Praefix(s,c) (strncasecmp (s, "" c "", sizeof(c)-1) == 0)
#define Text(s)  write(STDOUT_FILENO, "" s "", sizeof(s)-1)

#define CGI_KOPF "Content-Type: text/html; charset=UTF-8\n\n"
#define HTML_ANFANG  "<!DOCTYPE html>\n<html>\n\n"

#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifndef O_NOFOLLOW
#define O_NOFOLLOW 0
#endif

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

static char Dateiname[256];
static const char *Position, *Ende;
static const char *Methode, *Typ, *Dateityp, *Grenze;
static char *Benutzer;
static size_t Grenzlaenge;

static int Verzeichniswechsel (void);
static int Formular (void);
static int uebertragen (void);
static size_t Inhaltszugang (void);
static size_t Zeilenlaenge (const char *);
static const char *Zeilensuche (const char *, const char *);
static const char *Folgezeile (const char *);
static const char *Datenanfang (const char *);
static int Mixed (const char **);
static char *Dateisuche (void);
static int Dateinamenermittlung (const char *);
static int speichern (int);
static void Logdatei (int);
static size_t Blocklaenge (const char *);
static const char *Block_ignorieren (const char *);
static int Fehler (const char *);
static int Fehlerstatus (const char *, const char *);
static int Erfolg (void);
static void Bedienelemente (void);
static void Ausgabe (const char *);
static char *Benutzername (void);


int
main (void)
{
  Methode = getenv ("REQUEST_METHOD");
  if (!Methode || !getenv ("GATEWAY_INTERFACE"))
    {
      Text ("Es muss von einem Webserver"
	    " als CGI-Programm aufgerufen werden.\n");
      return EXIT_FAILURE;
    }

  if (Verzeichniswechsel ())
    return Fehler (VERZEICHNISFEHLER);

  Benutzer = Benutzername ();

  if (strcmp (Methode, "POST") != 0)
    return Formular ();

  Typ = getenv ("CONTENT_TYPE");
  if (!Typ || strncmp (Typ, "multipart/form-data", 19) != 0)
    return Fehlerstatus ("415 Unsupported Media Type", FORMULARFEHLER);

  Grenze = strstr (Typ, "boundary=");
  if (!Grenze)
    return Fehlerstatus ("415 Unsupported Media Type", GRENZFEHLER);

  Grenze += sizeof ("boundary=") - 1;
  if (*Grenze == '"')
    ++Grenze;
  Grenzlaenge = strcspn (Grenze, "\"");

  return uebertragen ();
}


static void
Kopfdaten (void)
{
  Text ("<meta name='robots' content='noindex, nofollow'>\n"
	"<meta name='viewport' content='width=device-width, initial-scale=1'>\n"
	"<style>\n"
	"@media (min-width:1024px) { body{font-size:x-large} }\n"
	"@media (prefers-color-scheme:light) {\n"
	"html{background-color:#F4ECD8;color:#000}\n"
	"div#Inhalt{background-color:#CBA;color:#653;\n"
	"border-color:#653}}\n"
	"@media (prefers-color-scheme:dark) {\n"
	"html{background-color:#000;color:#FFF}\n"
	"div#Inhalt{background-color:#210;color:#CBA;\n"
	"border-color:#653}}\n"
	".emoji{font-family:emoji,serif}\n"
	"div#Inhalt{display:table;margin:10ex auto;padding:1em;\n"
	"border-style:solid;border-width:medium;border-radius:1ex}\n"
	"input{font-size:inherit;margin-bottom:1ex}\n"
	"h1,div.info{text-align:center}\n"
	"h1.Server{font-size:inherit;margin-bottom:4ex}\n"
	"div#schliessen{text-align:right}\n"
	"div#schliessen a{color:inherit;text-decoration:none}\n"
	"</style>\n");
}


static int
Formular (void)
{
  const char *Servername, *Adresse;

  /*
     Selbstverstaendlich kann man auch ein entsprechendes Formular
     in anderen Seiten einbetten.
   */

  Text (CGI_KOPF);

  if (!strcmp (Methode, "HEAD"))
    return EXIT_SUCCESS;

  Servername = getenv ("SERVER_NAME");
  Adresse = getenv ("REMOTE_ADDR");

  Text (HTML_ANFANG);
  Text ("<head>\n<title>" TITEL ": ");
  Ausgabe (Servername);
  Text ("</title>\n");
  Kopfdaten ();
  Text ("</head>\n\n<body>\n<div id='Inhalt'>\n"
	"<div id='schliessen'><a href='/'>" SCHLIESSEN "</a></div>\n\n"
	"<h1 class='Server'>");
  Ausgabe (Servername);
  Text ("</h1>\n\n"
	"<form method='post' enctype='multipart/form-data'>\n"
	"<div class='Formular'>\n"
	"<input name='Datei' type='file' required multiple>\n"
	"<br>\n<input type='submit'>\n</div></form>\n\n"
	"<div class='info'>");

  // Adresse anzeigen, falls nicht localhost
  if (strcmp (Adresse, "127.0.0.1") && strcmp (Adresse, "::1"))
    {
      Text ("\n<span class='emoji'>\U0001F516</span>");
      Ausgabe (Adresse);
    }

  if (Benutzer)
    {
      Text ("\n<span class='emoji'>\U0001F464</span>");
      Ausgabe (Benutzer);
    }

  Text ("\n</div>\n\n</div>\n</body></html>\n");

  return EXIT_SUCCESS;
}


static int
Erfolg (void)
{
  Text (CGI_KOPF);

  if (!strcmp (Methode, "HEAD"))
    return EXIT_SUCCESS;

  Text (HTML_ANFANG);
  Text ("<head>\n<title>" DANKE "</title>\n");
  Kopfdaten ();
  Text ("</head>\n\n<body>\n<div id='Inhalt'>\n"
	"<div id='schliessen'><a href='/'>" SCHLIESSEN "</a></div>\n\n"
	"<h1>" ERFOLG);

  if (Benutzer)
    {
      Text (", ");
      Ausgabe (Benutzer);
    }

  Text ("!</h1>\n</div>\n</body></html>\n");

  return EXIT_SUCCESS;
}


static int
Fehlerstatus (const char *Status, const char *Ursache)
{
  Text ("Status: ");
  Ausgabe (Status);
  Text ("\n");
  Text (CGI_KOPF);

  if (!strcmp (Methode, "HEAD"))
    return EXIT_FAILURE;

  Text (HTML_ANFANG);
  Text ("<head>\n<title>" FEHLER "</title>\n"
	"<meta name='viewport' content='width=device-width, initial-scale=1'>\n"
	"<style>\n");
  write (STDOUT_FILENO, FehlerStil_css, sizeof (FehlerStil_css) - 1);
  Text ("</style>\n</head>\n\n<body>\n<div id='Fehler'>\n");
  Bedienelemente ();
  Text ("<h1>" FEHLERMELDUNG "</h1>\n\n");

  if (Ursache)
    {
      Text ("<p>\n");
      Ausgabe (Ursache);
      Text ("\n</p>\n\n");
    }

  Text ("</div>\n</body></html>\n");

  return EXIT_FAILURE;
}


static int
Fehler (const char *Ursache)
{
  return Fehlerstatus ("500 Internal Server Error", Ursache);
}


static int
uebertragen (void)
{
  unsigned int Anzahl = 0;
  int log;

  if (!Inhaltszugang ())
    return Fehler (MMAPFEHLER);

#ifdef LOGDATEI
  log = open (LOGDATEI, O_WRONLY | O_APPEND | O_CREAT | O_NOFOLLOW, 0666);
#else
  log = -1;
#endif
  // Fehlerprobe kommt spaeter

  // umask ignorieren
  umask (0);

  // Zeitbegrenzung vermeiden
  Text ("X-CGI-Pragma: warte\n");

  while (Dateisuche ())
    {
      int d;

      d = open (Dateiname,
		O_WRONLY | O_CREAT | O_EXCL | O_BINARY | O_NOFOLLOW, 0664);

      if (d < 0)
	{
	  if (errno == EEXIST)
	    return Fehlerstatus ("409 Conflict", KONFLIKTFEHLER);
	  else
	    return Fehler (ANLEGEFEHLER);
	}

      if (speichern (d) < 0 || close (d) < 0)
	{
	  int e = errno;

	  unlink (Dateiname);

	  switch (e)
	    {
	    case ENOSPC:
	    case EDQUOT:
	      return Fehlerstatus ("413 Payload Too Large", GROESSENFEHLER);

	    case EFBIG:
	      return Fehlerstatus ("413 Payload Too Large",
				   DATEIGROESSENFEHLER);

	    default:
	      return Fehler (SPEICHERFEHLER);
	    }
	}

      ++Anzahl;
      Logdatei (log);
    }

  if (log >= 0)
    close (log);

  if (Anzahl == 0)
    return Fehlerstatus ("400 No file", DATEIFEHLER);

  return Erfolg ();
}


static int
Verzeichniswechsel (void)
{
  const char *Verzeichnis = getenv ("UPLOAD");

  if (!Verzeichnis || !*Verzeichnis)
    Verzeichnis = UPLOAD;

  return (access (Verzeichnis, R_OK | W_OK | X_OK) < 0
	  || chdir (Verzeichnis) < 0);
}


// zurueck-Knopf
static void
Bedienelemente (void)
{
  const char *zurueck;

  zurueck = getenv ("HTTP_REFERER");	// sic
  if (!zurueck || !*zurueck)
    zurueck = getenv ("SCRIPT_NAME");
  if (!zurueck || !*zurueck)
    zurueck = "/";

  Text ("<div id='schliessen'><a href='");
  Ausgabe (zurueck);
  Text ("'>" SCHLIESSEN "</a></div>\n\n");
}


static size_t
Blocklaenge (const char *a)
{
  size_t l;

  for (l = 0; l < 8192; ++l, ++a)
    if (!memcmp (a, "\r\n--", 4) && !memcmp (a + 4, Grenze, Grenzlaenge))
      break;

  return l;
}


static const char *
Block_ignorieren (const char *p)
{
  while (p < Ende && memcmp (p, Grenze, Grenzlaenge))
    ++p;

  return Folgezeile (p);
}


static void
Logdatei (int log)
{
#ifdef LOGDATEI
  size_t zl;
  time_t t;
  char *Adresse;
  char Zeit[40];
  struct iovec iov[10];

  if (log < 0)
    return;

  time (&t);
  zl = strftime (Zeit, sizeof (Zeit), "%Y-%m-%d %H:%M:%S", localtime (&t));

  Adresse = getenv ("REMOTE_ADDR");
  if (!Adresse || !zl)
    return;

  // Daten zusammenstellen, um sie in einem Abwasch schreiben zu koennen
  iov[0].iov_base = Zeit;
  iov[0].iov_len = zl;
  iov[1].iov_base = ", ";
  iov[1].iov_len = 2;
  iov[2].iov_base = Adresse;
  iov[2].iov_len = strlen (Adresse);
  iov[3].iov_base = ", ";
  iov[3].iov_len = 2;
  iov[4].iov_base = Benutzer ? Benutzer : "-";
  iov[4].iov_len = strlen (iov[4].iov_base);
  iov[5].iov_base = ", ";
  iov[5].iov_len = 2;
  iov[6].iov_base = (char *) Dateityp;
  iov[6].iov_len = strcspn (Dateityp, "\r\n");
  iov[7].iov_base = ", ";
  iov[7].iov_len = 2;
  iov[8].iov_base = Dateiname;
  iov[8].iov_len = strlen (Dateiname);
  iov[9].iov_base = "\n";
  iov[9].iov_len = 1;

  writev (log, iov, 10);

#endif
}


static int
speichern (int d)
{
  const char *p;
  size_t l;

  p = Position;
  while (0 != (l = Blocklaenge (p)))
    {
      ssize_t r;

      r = write (d, p, l);
      if (r < 0)
	return -1;

      p += r;
    }

  Position = p;

  return 0;
}


static size_t
Inhaltszugang (void)
{
  unsigned long int l = 0;
  void *e;
  const char *ls;

  ls = getenv ("CONTENT_LENGTH");
  if (ls)
    l = strtoul (ls, NULL, 10);

  if (l == 0 || l == ULONG_MAX)
    return 0;

  e = mmap (NULL, l, PROT_READ, MAP_SHARED | MAP_FILE, STDIN_FILENO, 0);
  if (e == MAP_FAILED)
    return 0;

  posix_madvise (e, l, POSIX_MADV_SEQUENTIAL);

  Position = e;
  Ende = Position + l;

  return l;
}


static int
Dateinamenermittlung (const char *Zeile)
{
  size_t l;
  const char *n;

  Dateiname[0] = '\0';
  n = Zeilensuche (Zeile, "filename=");
  if (!n)
    return -1;

  n += sizeof ("filename=") - 1;
  n += strspn (n, " \t");

  if (*n == '"')
    l = strcspn (++n, "\"\r\n");
  else
    l = strcspn (n, ";\r\n");

  if (!l)
    return -1;

  // Name zu lang? => kuerzen
  if (l >= sizeof (Dateiname))
    l = sizeof (Dateiname) - 1;

  memcpy (Dateiname, n, l);
  Dateiname[l] = '\0';

  return 0;
}


static char *
Dateisuche (void)
{
  const char *p;

  Dateiname[0] = '\0';
  Dateityp = "-";

  p = Position;

Blockanalyse:
  while (p && !Praefix (p, "Content-Disposition:"))
    p = Folgezeile (p);

  if (!p)
    return NULL;

  if (Dateinamenermittlung (p) < 0)
    {
      if (!Mixed (&p))
	goto Blockanalyse;

      if (p)
	p = Block_ignorieren (p);

      if (!p)
	return NULL;

      goto Blockanalyse;
    }

  // darf kein Pfadelement enthalten
  if (strchr (Dateiname, '/'))
    return NULL;

  Position = Datenanfang (p);

  return Dateiname;
}



static size_t
Zeilenlaenge (const char *p)
{
  size_t zl = 0;

  while (*p != '\r' && *p != '\n' && p < Ende)
    {
      ++p;
      ++zl;
    }

  return zl;
}


static const char *
Zeilensuche (const char *p, const char *Ausdruck)
{
  size_t l, zl, al;

  al = strlen (Ausdruck);
  zl = Zeilenlaenge (p);

  if (al > zl)
    return NULL;

  for (l = zl - al; l; --l)
    if (!strncasecmp (p + l, Ausdruck, al))
      return p + l;

  return NULL;
}


static const char *
Folgezeile (const char *p)
{
  if (!p || p >= Ende)
    return NULL;

  ++p;
  while (p < Ende && *p != '\n')
    ++p;

  if (p >= Ende)
    return NULL;

  return ++p;
}


// sucht Leerzeile und gibt die Folgezeile aus
static const char *
Datenanfang (const char *p)
{
  while (p && *p != '\r' && *p != '\n')
    {
      if (Praefix (p, "Content-Type: "))
	Dateityp = p + sizeof ("Content-Type: ") - 1;
      p = Folgezeile (p);
    }

  if (!p)
    return NULL;

  return Folgezeile (p);
}


// multipart/mixed
static int
Mixed (const char **pos)
{
  const char *p = *pos;

  while (p && *p != '\r' && *p != '\n')
    {
      if (Praefix (p, "Content-Type: multipart/mixed;"))
	{
	  const char *b = Zeilensuche (p, "boundary=");

	  if (b)
	    {
	      size_t l;

	      b += sizeof ("boundary=") - 1;
	      b += strspn (b, " \t");
	      if (*b == '"')
		l = strcspn (++b, "\"\r\n");
	      else
		l = strcspn (b, ";\r\n");

	      Grenze = b;
	      Grenzlaenge = l;

	      *pos = Block_ignorieren (p);
	      return 0;
	    }
	}

      p = Folgezeile (p);
    }

  *pos = Folgezeile (p);

  return -1;
}


// versuche Namen des Benutzers zu ermitteln, wahrscheinlich erfolglos
static char *
Benutzername (void)
{
  char *Benutzer;

  Benutzer = getenv ("REMOTE_USER");

  if (!Benutzer || !*Benutzer)
    Benutzer = getenv ("HTTP_FROM");

  if (!Benutzer || !*Benutzer)
    Benutzer = getenv ("REMOTE_IDENT");

  // keinen Leerstring akzeptieren
  if (Benutzer && !*Benutzer)
    Benutzer = NULL;

  return Benutzer;
}


static void
Ausgabe (const char *s)
{
  if (s && *s)
    write (STDOUT_FILENO, s, strlen (s));
}
