/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * CGI: Gopher-Zugang fuer Webserver (Gateway / Proxy)
 * Copyright (c) 2024-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
Formularfelder:
u: Gopher-URL
s: Suchtext (Typ 7)
a: Aktion
a=dl: Download
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <strings.h>
#include <netinet/in.h>
#include "akfnetz.h"
#include "gophertyp.h"
#include "version.h"
#include "GopherStil.h"

#define TITEL "AKF-Gopherweb"

//#define ZEICHENSATZ "ISO-8859-1"
#define ZEICHENSATZ "UTF-8"

//#define CACHE_CONTROL "Cache-Control: no-cache\n"
#define CACHE_CONTROL "Cache-Control: public, max-age=43200\n"
// 12 Stunden -- 60*60*12=43200

// Bitte nicht Google-Fonts einbinden!
// Die sammeln darueber IP-Adressen der Nutzer

#define TYP_URL 1000
#define HTML_ENDE "\n</body>\n</html>"
#define GOPHER_PORT 70
#define Praefix(s,c) (strncasecmp (s, "" c "", sizeof(c)-1) == 0)


// akfnetz_Gopher funktioniert hier nicht mit der Suche
// ausserdem brauchen wir keinen weiteren Proxy
#pragma GCC poison akfnetz_Gopher akfnetz_Gopher_URL

enum Protokolltyp
{ P_GOPHER, P_FINGER };

static void Gopherweb (FILE * gopher, const char *url);
static void Einleitung (const char *Titel);
static int Startseite (void);
static void Adressleiste (const char *url);
static int Fehler (const char *deutsch, const char *englisch,
		   const char *esperanto, const char *url);
static int Umleitung (const char *url);
static int Browseradressen (const char *url);
static void Gophermenue (FILE * gopher, const char *Selektor,
			 const char *url);
static void Menuezeile (char *);
static void Sucheintrag (const char *Name, const char *Server,
			 const char *Port, const char *Selektor);
static void Sucheingabe (const char *url);
static void Suchanfrage (FILE * gopher, const char *Selektor,
			 const char *url);
static void Direktausgabe (FILE * gopher, const char *Selektor,
			   const char *url);
static void Textausgabe (FILE * gopher, const char *Selektor,
			 const char *url);
static void Gopherlink (const char *Name, const char *Server,
			const char *Port, const char *Selektor, int Typ);
static void Dienstlink (const char *Name, const char *Server,
			const char *Port, const char *Selektor, int Typ);
static const char *Dateiname (const char *Pfad);
static unsigned int Symbol (int Typ);
static void Gopheranfrage (FILE * gopher, const char *Selektor);
static void Servername (const char *Server);
static void Serverport (const char *Port, int Standardport);
static bool Portverbot (const char *url);
static FILE *Gopherverbindung (const char *url);
static void Gopherausgabe (FILE * gopher);
static void Gopher_html_Ausgabe (FILE * gopher);
static inline void Code (const char *);
static inline void Text (const char *);
static inline bool Abschluss (const char *Zeile);
static void CGI_Pruefung (void);


int
main (void)
{
  const char *url;
  FILE *gopher;

  CGI_Pruefung ();

  // nur Methode GET
  akfnetz_Formulardaten (getenv ("QUERY_STRING"));

  url = akfnetz_Formularfeld ("u");
  if (!url || !*url)
    return Startseite ();
  // url ist bereits dekodiert!

  // URL-Schema ueberpruefen
  if (!Praefix (url, "gopher://") && !Praefix (url, "finger://"))
    return Browseradressen (url);

  // URL-Einleitung "gopher://" ohne Name?
  if (!*(url + 9))
    return Startseite ();

  gopher = Gopherverbindung (url);
  if (!gopher)
    return Fehler ("Keine Route zum Zielrechner",
		   "No route to host", "Gastiganto ne atingeblas", url);

  Gopherweb (gopher, url);
  fclose (gopher);

  return EXIT_SUCCESS;
}


static void
Gopherweb (FILE *gopher, const char *url)
{
  const char *Selektor, *Aktion;
  int Typ;
  enum Protokolltyp Protokoll;
  char Benutzer[256];		// nur fuer Finger

  *Benutzer = '\0';
  Protokoll = (Praefix (url, "gopher:")) ? P_GOPHER : P_FINGER;
  Typ = (Protokoll == P_GOPHER) ? TYP_MENUE : TYP_TEXT;

  if (Portverbot (url))
    {
      // bereits verbunden => Port ist offen
      Fehler ("Port verboten", "Port forbidden", "pordo malpermesita", url);
      return;
    }

  Aktion = akfnetz_Formularfeld ("a");
  Selektor = akfnetz_URL_Pfad (url);

  if (!Selektor)
    Selektor = "";
  else
    {
      while (*Selektor == '/')
	++Selektor;

      if (*Selektor && Protokoll == P_GOPHER)
	{
	  // Gopher: Das erste Zeichen ist der Typ
	  Typ = (int) (unsigned char) *Selektor;
	  ++Selektor;
	}
    }

  // bei Finger evtl. auch Benutzernamen pruefen
  if (Protokoll == P_FINGER && !*Selektor
      && akfnetz_URL_Zugang (Benutzer, sizeof (Benutzer), url))
    {
      // Passwort ignorieren
      char *p = strchr (Benutzer, ':');
      if (p)
	*p = '\0';

      Selektor = akfnetz_url_dekodieren (Benutzer);
    }

  // besondere Aktion?
  if (Aktion && !strcmp ("dl", Aktion) && !strchr ("2378T", Typ))
    {
      Direktausgabe (gopher, Selektor, "application/octet-stream");
      // das erzwingt den Download
      return;
    }

  switch (Typ)
    {
    case TYP_TEXT:
    case TYP_MIME:
      Textausgabe (gopher, Selektor, url);
      break;

    case TYP_MENUE:
      Gophermenue (gopher, Selektor, url);
      break;

    case TYP_SUCHE:
      Suchanfrage (gopher, Selektor, url);
      break;

    case TYP_BINHEX:
    case TYP_ARCHIV:
    case TYP_UUE:
      // Download erzwingen
      Direktausgabe (gopher, Selektor, "application/octet-stream");
      break;

    case TYP_BINAER:
    case TYP_BILD:
    case TYP_PNG:
    case TYP_BITMAP:
    case TYP_VIDEO:
    case TYP_TON:
    case TYP_AUDIO:
    case TYP_DOKU:
      // ermittle Typ anhand der Dateiendung
      Direktausgabe (gopher, Selektor, akfnetz_Medientyp (Selektor));
      break;

    case TYP_GIF:
      Direktausgabe (gopher, Selektor, "image/gif");
      break;

    case TYP_HTML:
      Direktausgabe (gopher, Selektor, "text/html");
      // URL-Adressen werden in Gophermenue() behandelt
      break;

    case TYP_RTF:
      Direktausgabe (gopher, Selektor, "application/rtf");
      break;

    case TYP_PDF:
      Direktausgabe (gopher, Selektor, "application/pdf");
      break;

    case TYP_XML:
      Direktausgabe (gopher, Selektor, "application/xml");
      break;

    case TYP_KALENDER:
      Direktausgabe (gopher, Selektor, "text/calendar");
      break;

      // 2/3/8/T/+/i werden im Menue unterstuetzt, nicht in URL

    case TYP_FEHLER:
    case TYP_CCSO:
    case TYP_TELNET:
    case TYP_TN3270:
    case TYP_ALT:
    case TYP_INFO:
    default:
      Fehler ("Typ nicht unterst&#xFC;tzt",
	      "Type not supported", "Tipo ne subtenatas", url);
      break;
    }
}


static void
Einleitung (const char *Titel)
{
  // CGI-Kopf
  Code ("Content-Type: text/html; charset=" ZEICHENSATZ "\n");
  Code (CACHE_CONTROL);
  Code ("X-Robots-Tag: noindex, nofollow\n");
  Code ("Referrer-Policy: no-referrer\n");
  akfnetz_cgi_Kopfende ();

  // HTML5
  Code ("<!DOCTYPE html>\n<html>\n\n<head>\n"
	"<meta charset='" ZEICHENSATZ "'>\n");

  Code ("<title>");
  Text (Titel);
  Code ("</title>\n");

  Code ("<meta name='generator' content='AKF-Gopherweb (AKFNetz "
	AKFNETZ_VERSION ")'>\n"
	"<meta name='robots' content='noindex, nofollow'>\n"
	"<meta name='referrer' content='no-referrer'>\n"
	"<meta name='viewport' content='width=device-width, initial-scale=1'>\n"
	"<style>\n");
  Code (GopherStil);
  Code ("</style>\n</head>\n\n<body>\n");
}


// Verarbeitet eine Menuezeile
// (Zeile wird veraendert)
static void
Menuezeile (char *Zeile)
{
  int Typ = (int) (unsigned char) *Zeile;

  char *p = Zeile + 1;
  char *Name = strsep (&p, "\t");
  char *Selektor = strsep (&p, "\t");
  char *Server = strsep (&p, "\t");
  char *Port = strsep (&p, "\t\r\n");

  // Wenn Port nicht NULL ist, ist das davor auch nicht NULL
  // Aber alles kann ein Leerstring sein

  if (!Port)
    return;

  // "URL:"-Selektor
  if (*Name && Selektor && !strncmp ("URL:", Selektor, 4))
    Typ = TYP_URL;		// der eigentliche Typ wird ignoriert

  // Der gesetzte Zeichensatz ist nicht unbedingt UTF-8!
  printf ("<i>&#x%X;</i> ", Symbol (Typ));

  switch (Typ)
    {
    case TYP_INFO:		// Info
      if (*Name)
	Text (Name);
      putchar ('\n');
      break;

    case TYP_TEXT:
    case TYP_MENUE:
    case TYP_BINHEX:
    case TYP_ARCHIV:
    case TYP_UUE:
    case TYP_BINAER:
    case TYP_PDF:
    case TYP_RTF:
    case TYP_AUDIO:
    case TYP_DOKU:
    case TYP_HTML:
    case TYP_XML:
    case TYP_BILD:
    case TYP_MIME:
    case TYP_GIF:
    case TYP_PNG:
    case TYP_BITMAP:
    case TYP_VIDEO:
    case TYP_TON:
    case TYP_KALENDER:
    case TYP_ALT:
      Gopherlink (Name, Server, Port, Selektor, Typ);
      break;

    case TYP_SUCHE:
      Sucheintrag (Name, Server, Port, Selektor);
      break;

    case TYP_CCSO:
    case TYP_TELNET:
    case TYP_TN3270:
      Dienstlink (Name, Server, Port, Selektor, Typ);
      break;

    case TYP_FEHLER:
      Code ("<span class='Fehlertyp'>");
      Text (Name);
      Code ("</span>\n");
      break;

    case TYP_URL:
      Code ("<a href='");
      Text (Selektor + 4);
      Code ("' target='_blank'>");
      Text (Name);
      Code ("</a>\n");
      break;

    default:
      Code ("<span class='unbekannt'>");
      Text (Name);
      Code ("</span>\n");
      break;
    }
}


static void
Gophermenue (FILE *gopher, const char *Selektor, const char *url)
{
  char Zeile[4096];

  Gopheranfrage (gopher, Selektor);
  Einleitung ((Selektor && *Selektor) ? Selektor : TITEL);
  Adressleiste (url);
  Code ("<main>\n<pre>\n");

  while (fgets (Zeile, sizeof (Zeile), gopher) && !Abschluss (Zeile))
    Menuezeile (Zeile);

  Code ("</pre>\n</main>\n" HTML_ENDE "\n");
}


static void
Sucheintrag (const char *Name, const char *Server, const char *Port,
	     const char *Selektor)
{
  Text (Name);
  Code ("</pre>\n");

  Code ("<form>\n");
  Code ("<input type='hidden' name='u' value='gopher://");
  Servername (Server);
  Serverport (Port, GOPHER_PORT);
  Code ("/7");

  akfnetz_urlkodiert (stdout, Selektor, ":/()");

  Code ("'>\n&#xA0;&#xA0;&#xA0;&#xA0;"
	"<input type='text' name='s'>\n"
	"<button><i>&#x1F50D;</i></button>\n</form>\n<pre>");
}


static void
Sucheingabe (const char *url)
{
  char Host[1024];

  akfnetz_URL_Hostname (Host, sizeof (Host), url);

  Einleitung ("Gopher-Suche");
  Code ("<h1>");
  Text (Host);
  Code ("</h1>\n");

  Code ("<nav>\n<form>\n<input type='hidden' name='u' value='");
  Text (url);
  Code ("'>\n<input type='text' name='s' size='80'>\n"
	"<button><i>&#x1F50D;</i></button>\n</form>\n</nav>\n"
	HTML_ENDE "\n");
}


static void
Suchanfrage (FILE *gopher, const char *Selektor, const char *url)
{
  bool Suchfeld = false;
  size_t el = 0;

  const char *Eingabe = akfnetz_Formularfeld ("s");

  // Wenn kein Selektor, dann nur Suchstring uebertragen
  // TODO nur Typ?
  if (Eingabe && (!Selektor || !*Selektor))
    {
      Gophermenue (gopher, Eingabe, url);
      return;
    }

  if (Eingabe)
    el = strlen (Eingabe);

  size_t pl = strlen (Selektor);
  char pf[pl + 1 + el + 1];
  memcpy (pf, Selektor, pl);
  pf[pl] = '\0';

  pl = 0;

  if (!Suchfeld && strchr (pf, '\t'))
    Suchfeld = true;

  // Eingabe nur beruecksichtigen, wenn noch kein Suchfeld vorhanden
  if (!Suchfeld && Eingabe)
    {
      pl = strlen (pf);
      pf[pl] = '\t';
      memcpy (pf + pl + 1, Eingabe, el);
      pf[pl + 1 + el] = '\0';
      Suchfeld = true;
    }

  if (Suchfeld)
    Gophermenue (gopher, pf, url);
  else
    Sucheingabe (url);
}


static void
Direktausgabe (FILE *gopher, const char *Selektor, const char *mime)
{
  const char *Disposition;

  // keine Dateien sind hier verboten
  if (!mime || !*mime)
    mime = "application/octet-stream";

  // Wir koennen hier nicht alles testen :-(
  // Archive sollten eh mit Typ '5' gekennzeichnet werden
  if (!strcmp (mime, "application/octet-stream")
      || !strcmp (mime, "application/zip")
      || !strcmp (mime, "application/gzip")
      || !strcmp (mime, "application/x-xz"))
    Disposition = "attachment";	// erzwingt herunterladen
  else
    Disposition = "inline";

  // CGI-Kopf
  Code ("Content-Type: ");
  Code (mime);
  putchar ('\n');
  Code (CACHE_CONTROL);
  Code ("X-Robots-Tag: noindex, nofollow\n");
  Code ("Referrer-Policy: no-referrer\n");

  // Dateinamen mitgeben
  printf ("Content-Disposition: %s; filename=\"%s\"\n",
	  Disposition, Dateiname (Selektor));

  akfnetz_cgi_Kopfende ();

  Gopheranfrage (gopher, Selektor);
  Gopherausgabe (gopher);
}


static void
Textausgabe (FILE *gopher, const char *Selektor, const char *url)
{
  Einleitung (*Selektor == '/' ? Selektor + 1 : Selektor);
  Adressleiste (url);

  Gopheranfrage (gopher, Selektor);

  Code ("<main>\n<pre>\n");
  Gopher_html_Ausgabe (gopher);
  Code ("</pre>\n</main>\n" HTML_ENDE "\n");
}


static void
Gopherlink (const char *Name, const char *Server, const char *Port,
	    const char *Selektor, int Typ)
{
  static int letzter_Typ = TYP_ALT;

  // alle Strings sind nicht NULL

  Code ("<a href='?u=gopher://");
  Servername (Server);
  Serverport (Port, GOPHER_PORT);

  // Typ '+' nicht in URL
  if (Typ == TYP_ALT)
    Typ = letzter_Typ;
  else
    letzter_Typ = Typ;

  // eventuell Pfad anhaengen
  if (Typ != TYP_MENUE || strcmp (Selektor, "/") != 0)
    {
      // Typ
      putchar ('/');

      // problematische Typ-Zeichen kodieren
      if (strchr (" :;<>&|\"\'`%!?@^$+,/\\={}", Typ))
	printf ("%%%02X", Typ);
      else
	putchar (Typ);

      akfnetz_urlkodiert (stdout, Selektor, ":/()");
    }

  Code ("'>");
  Text (Name);
  Code ("</a>\n");
}


// Link auf externen Dienst
// derzeit CSO, telnet und tn3270, kann aber erweitert werden
static void
Dienstlink (const char *Name, const char *Server, const char *Port,
	    const char *Selektor, int Typ)
{
  const char *Protokoll;
  int Standardport = -1;

  switch (Typ)
    {
    case TYP_CCSO:
      Protokoll = "cso";
      Standardport = 105;
      break;

    case TYP_TELNET:
      Protokoll = "telnet";
      Standardport = 23;
      break;

    case TYP_TN3270:
      Protokoll = "tn3270";
      Standardport = 23;
      break;

    default:
      Code ("<span class='unbekannt'>");
      Text (Name);
      Code ("</span>\n");
      return;
    }

  Code ("<a href='");
  Code (Protokoll);
  Code ("://");

  // Selektor kann den Anmeldenamen enthalten
  while (*Selektor == '/')
    ++Selektor;

  if (*Selektor && strcmp ("none", Selektor) != 0)
    {
      Text (Selektor);
      putchar ('@');
    }

  Servername (Server);
  Serverport (Port, Standardport);
  Code ("'>");
  Text (Name);
  Code ("</a>\n");
}


// gibt Servernamen in URL aus
static void
Servername (const char *Server)
{
  if (strchr (Server, ':'))	// IPv6-Adresse?
    {
      putchar ('[');
      Text (Server);
      putchar (']');
    }
  else
    Text (Server);
}


// Haengt evtl. Port an
static void
Serverport (const char *Port, int Standardport)
{
  if (!Port || !*Port)
    return;

  int Nr = atoi (Port);
  if (Nr > 0 && Nr != Standardport)
    printf (":%d", Nr);
}

// Dies ist ein offener Proxy und kann missbraucht werden
static bool
Portverbot (const char *url)
{
  static const in_port_t Verboten[] = {
    20, 21, 989, 990, 80, 443, 8080, 25, 2525, 465, 587, 143, 110, 995,
    143, 93
  };

  char Port[8];
  if (!akfnetz_URL_Port (Port, sizeof (Port), url))
    return false;		// Standardport ist in Ordnung

  int PortNr = atoi (Port);
  if (PortNr <= 0)
    return true;

  for (unsigned i = 0; i < (sizeof (Verboten) / sizeof (Verboten[0])); ++i)
    if (PortNr == Verboten[i])
      return true;

  return false;
}


static unsigned int
Symbol (int Typ)
{
  unsigned int sym;

  switch (Typ)
    {
    case TYP_TEXT:
      sym = 0x1F4C4;
      break;

    case TYP_MENUE:
      sym = 0x1F4C1;
      break;

    case TYP_CCSO:
      sym = 0x1F5C2;
      break;

    case TYP_FEHLER:
      sym = 0x26A0;
      break;

    case TYP_BINHEX:
    case TYP_ARCHIV:
    case TYP_UUE:
      sym = 0x1F4E6;		// Paket
      break;

    case TYP_SUCHE:
      sym = 0x1F50D;
      break;

    case TYP_TELNET:
    case TYP_TN3270:
      sym = 0x1F4BB;
      break;

    case TYP_BINAER:
      sym = 0x2699;		// Zahnrad
      break;

    case TYP_BILD:
    case TYP_GIF:
    case TYP_PNG:
    case TYP_BITMAP:
      sym = 0x1F5BC;
      break;

    case TYP_ALT:
      sym = 0x26D3;		// Doppel-Ketten
      break;

    case TYP_HTML:
      sym = 0x1F4D8;		// blaues Buch
      break;

    case TYP_DOKU:
    case TYP_PDF:
    case TYP_RTF:
    case TYP_XML:
      sym = 0x1F4D7;		// gruenes Buch
      break;

    case TYP_TON:
    case TYP_AUDIO:
      sym = 0x1F442;		// Ohr
      break;

    case TYP_VIDEO:
      sym = 0x1F3A6;
      break;

    case TYP_KALENDER:
      sym = 0x1F4C5;
      break;

    case TYP_MIME:
      sym = 0x1F4E7;		// E-Mail
      break;

    case TYP_INFO:
      sym = 0xA0;		// geschuetztes Leerzeichen
      break;

    case TYP_URL:
      sym = 0x1F517;
      break;

    default:			// unbekannt
      sym = 0x1F914;		// denkendes Gesicht
      break;
    }

  return sym;
}


// Anfrage an Gopher-Server stellen
static void
Gopheranfrage (FILE *gopher, const char *Selektor)
{
  // Leerzeile erlaubt
  if (Selektor)
    fputs (Selektor, gopher);

  fputs ("\r\n", gopher);
  fflush (gopher);
}


// Gibt Daten vom Server direkt aus
static void
Gopherausgabe (FILE *gopher)
{
  size_t r;
  char Puffer[2048];

  while ((r = fread (Puffer, 1, sizeof (Puffer), gopher)) != 0)
    fwrite (Puffer, 1, r, stdout);
}


// Gibt Daten vom Server HTML-kodiert aus
static void
Gopher_html_Ausgabe (FILE *gopher)
{
  char Zeile[2048];

  while (fgets (Zeile, sizeof (Zeile), gopher) && !Abschluss (Zeile))
    Text (Zeile);
}


static inline void
Code (const char *s)
{
  fputs (s, stdout);
}


static inline void
Text (const char *s)
{
  akfnetz_xml_Text (stdout, s);
}


static inline bool
Abschluss (const char *Zeile)
{
  return (Zeile[0] == '.' && Zeile[1] < 0x20);
}


// Name der Datei ohne Pfad
// Im Zweifel "gophermap"
static const char *
Dateiname (const char *Pfad)
{
  const char *Name;

  if (!Pfad || !*Pfad)
    return "gophermap";

  Name = strrchr (Pfad, '/');
  if (Name)
    ++Name;
  else
    Name = Pfad;

  if (!*Name)
    Name = "gophermap";

  return Name;
}


static int
Startseite (void)
{
  Einleitung (TITEL);
  Code ("<h1>" TITEL "</h1>\n\n");
  Adressleiste (NULL);

  Code ("<main>\n<div><i>&#x1F4E6;&#x1F4E5;</i> <a href='"
	AKFNETZ_HOMEPAGE "' target='_blank'>AKFNetz "
	AKFNETZ_VERSION "</a></div>\n<div>Copyright &#xA9; "
	AKFNETZ_JAHR " Andreas K. F&#xF6;rster</div>\n"
	"<div>License <a href='http://gnu.org/licenses/gpl.html'"
	" target='_blank'>GPLv3+: GNU General Public License "
	"version 3 or newer</a></div>\n</main>\n" HTML_ENDE "\n");

  return (EXIT_SUCCESS);
}


static void
Adressleiste (const char *url)
{
  Code ("<nav>\n<a id='Logo' href='" AKFNETZ_HOMEPAGE "' target='_blank'>\n"
	"<img alt='AKFNetz' title='AKFNetz' width='32' height='32' src=\n'"
	AKFNETZ_AKFLOGO "'\n></a>\n\n");

  Code ("<form><div>\n<label>URL:\n"
	"<input type='url' name='u' required"
	" placeholder='gopher://&#x2026; | finger://&#x2026;'");

  if (url && *url)
    {
      Code ("\n value='");
      Text (url);
      putchar ('\'');
    }

  // fuer Aktionen immer name='a' verwenden
  Code (">\n</label>\n"
	"<button><i>&#x2192;</i></button>\n"
	"<button name='a' value='dl'><i>&#x1F4E5;</i></button>\n");

  if (url && *url)
    {
      Code ("<a href='");
      Text (url);
      Code ("'>Link</a>\n");
    }

  Code ("</div></form>\n</nav>\n\n");
}


#define FEHLERFORMAT \
  "<i>&#x%X;</i> <span class='Fehlertyp' lang='%s'>%s</span>\n"

static int
Fehler (const char *deutsch, const char *englisch, const char *esperanto,
	const char *url)
{
  const unsigned int sym = Symbol (TYP_FEHLER);

  Code ("Status: 502 Bad Gateway\n");
  Einleitung (englisch);
  Adressleiste (url);
  Code ("<main>\n<pre>\n");

  // nicht Text() verwenden
  if (deutsch)
    printf (FEHLERFORMAT, sym, "de", deutsch);

  if (englisch)
    printf (FEHLERFORMAT, sym, "en", englisch);

  if (esperanto)
    printf (FEHLERFORMAT, sym, "eo", esperanto);

  Code ("</pre>\n</main>\n" HTML_ENDE "\n");

  return EXIT_FAILURE;
}


static int
Umleitung (const char *url)
{
  printf ("Location: %s\n\n", url);
  // Spezialfall in CGI (vgl. RFC 3875, 6.2.3)
  // Webserver erzeugt Kopf und Inhalt automatisch.
  // Es muss eine absolute URL sein.

  return EXIT_SUCCESS;
}


// Einige Adressen koennen vom Browser direkt verarbeitet werden
static int
Browseradressen (const char *url)
{
  if (Praefix (url, "https://")
      || Praefix (url, "http://")
      || Praefix (url, "file://") || Praefix (url, "mailto:"))
    return Umleitung (url);
  else
    return Fehler ("Keine unterst&#xFC;tzte URL",
		   "Unsupported URL", "Nesubtenata URL", url);
}


// Verbindung zum Gopher-Server herstellen
static FILE *
Gopherverbindung (const char *url)
{
  int v;

  v = akfnetz_URL_Verbindung (url, 0);
  if (v < 0)
    return NULL;

  return fdopen (v, "r+");
}


// befinden wir uns ueberhaupt in einer CGI-Umgebung?
static void
CGI_Pruefung (void)
{
  const char *s;

  s = getenv ("GATEWAY_INTERFACE");
  if (s && !strncmp (s, "CGI/", 4))
    return;			// es ist eine CGI-Umgebung

  // keine CGI-Umgebung
  fputs (TITEL " " AKFNETZ_VERSION "\n" AKFNETZ_COPYRIGHT
	 "\nLizenz: GPLv3 oder neuer\n" AKFNETZ_HOMEPAGE "\n", stderr);
  exit (EXIT_FAILURE);
}
