---------------------------------------------------------------------------
-- CGI                                                                   --
-- Copyright (c) 2019-2020 Andreas K. Foerster <akf@akfoerster.de>       --
--                                                                       --
-- This program is free software: you can redistribute it and/or modify  --
-- it under the terms of the GNU General Public License as published by  --
-- the Free Software Foundation, either version 3 of the License, or     --
-- (at your option) any later version.                                   --
--                                                                       --
-- This program is distributed in the hope that it will be useful,       --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of        --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         --
-- GNU General Public License for more details.                          --
--                                                                       --
-- You should have received a copy of the GNU General Public License     --
-- along with this program.  If not, see <http://www.gnu.org/licenses/>. --
---------------------------------------------------------------------------

-- SPDX-License-Identifier: GPL-3.0-or-later
pragma License (Gpl);
pragma Ada_2005;

with Ada.Environment_Variables;
with Ada.Text_IO;
with Ada.Strings.Fixed;
with Ada.Strings.Maps;
with Ada.Characters.Latin_1; use Ada.Characters.Latin_1;

package body CGI is

   Request_Method   : Method_Type   := Get;
   Form_Initialised : Boolean       := False;
   Form_Data        : access String := null;
   Data_Length      : Natural       := 0;

   Separators : constant Ada.Strings.Maps.Character_Set :=
     Ada.Strings.Maps.To_Set (Ampersand & Semicolon);

   HTML_Characters : constant Ada.Strings.Maps.Character_Set :=
     Ada.Strings.Maps.To_Set
       (Less_Than_Sign &
        Greater_Than_Sign &
        Ampersand &
        Apostrophe &
        Quotation);

   procedure Initialise_Form;
   function URL_Decode (Item : String) return String;

   ---------------
   -- HTML_Text --
   ---------------

   procedure HTML_Text (Item : String) is
      Position : Positive := Item'First;
      I        : Natural  := 0;
   begin
      loop
         I := Ada.Strings.Fixed.Index (Item, HTML_Characters, Position);
         exit when I = 0;

         Ada.Text_IO.Put (Item (Position .. I - 1));

         case Item (I) is
            when Less_Than_Sign =>
               Ada.Text_IO.Put ("&#60;");

            when Greater_Than_Sign =>
               Ada.Text_IO.Put ("&#62;");

            when Ampersand =>
               Ada.Text_IO.Put ("&#38;");

            when Apostrophe =>
               Ada.Text_IO.Put ("&#39;");

            when Quotation =>
               Ada.Text_IO.Put ("&#34;");

            when others =>  -- shouldn't happen
               Ada.Text_IO.Put ("&#xFFFD;");
         end case;

         Position := I + 1;
         exit when Position > Item'Last;
      end loop;

      Ada.Text_IO.Put (Item (Position .. Item'Last));
   end HTML_Text;

   function Key_Found (Key, S : String) return Boolean is
   begin
      return S'Length > Key'Length
        and then S (S'First + Key'Length) = Equals_Sign
        and then S (S'First .. S'First + Key'Length - 1) = Key;
      -- comparing strings is slow, so that comes last
   end Key_Found;

   -- returns last column of dataset
   function Dataset (First : Positive) return Natural is
      Last : Natural;
   begin
      Last := Ada.Strings.Fixed.Index (Form_Data.all, Separators, First);

      if Last = 0 then
         Last := Data_Length;
      else
         Last := Last - 1;
      end if;

      return Last;
   end Dataset;

   ----------------
   -- Key_Exists --
   ----------------

   function Key_Exists (Key : String) return Boolean is
      First : Positive := 1;
      Last  : Natural;
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      while First < Data_Length loop
         Last := Dataset (First);

         if Key_Found (Key, Form_Data (First .. Last)) then
            return True;
         end if;

         First := Last + 2;
      end loop;

      return False;
   end Key_Exists;

   -----------
   -- Value --
   -----------

   function Value (Key : String) return String is
      First : Positive := 1;
      Last  : Natural;
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      while First < Data_Length loop
         Last := Dataset (First);

         declare
            S : constant String := Form_Data (First .. Last);
         begin
            if Key_Found (Key, S) then
               return URL_Decode (S (S'First + Key'Length + 1 .. S'Last));
            end if;
         end;

         First := Last + 2;
      end loop;

      return "";
   end Value;

   function Value (Position : Positive) return String is
      First : Positive := 1;
      Last  : Natural;
      P     : Positive := 1;
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      while First < Data_Length loop
         Last := Dataset (First);

         if P = Position then
            declare
               S     : constant String  := Form_Data (First .. Last);
               Equal : constant Natural := Ada.Strings.Fixed.Index (S, "=");
            begin
               return URL_Decode (S (Equal + 1 .. S'Last));
            end;
         end if;

         First := Last + 2;
         P     := P + 1;
      end loop;

      return "";
   end Value;

   function Key (Position : Positive) return String is
      First : Positive := 1;
      Last  : Natural;
      P     : Positive := 1;
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      while First < Data_Length loop
         Last := Dataset (First);

         if P = Position then
            declare
               S     : constant String  := Form_Data (First .. Last);
               Equal : constant Natural := Ada.Strings.Fixed.Index (S, "=");
            begin
               if Equal > 0 then
                  return URL_Decode (S (S'First .. Equal - 1));
               else
                  return "";
               end if;
            end;
         end if;

         First := Last + 2;
         P     := P + 1;
      end loop;

      return "";
   end Key;

   --------------------
   -- Argument_Count --
   --------------------

   function Argument_Count return Natural is
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      if Data_Length = 0 then
         return 0;
      else
         return Ada.Strings.Fixed.Count (Form_Data.all, Separators) + 1;
      end if;
   end Argument_Count;

   ----------------
   -- Put_Header --
   ----------------

   procedure Put_Header (Header : String := Default_Header) is
   begin
      if not Header_Sent then
         Ada.Text_IO.Put (Header);
         Ada.Text_IO.New_Line (2);
         Header_Sent := True;
      end if;
   end Put_Header;

   ------------
   -- Method --
   ------------

   function Method return Method_Type is
   begin
      return Request_Method;
   end Method;

   -------------------------
   -- For_Every_Parameter --
   -------------------------

   procedure For_Every_Parameter is
      First    : Positive := 1;
      Last     : Natural;
      Position : Positive := 1;
      Quit     : Boolean  := False;
   begin
      if not Form_Initialised then
         Initialise_Form;
      end if;

      while First < Data_Length loop
         Last := Dataset (First);

         declare
            S     : constant String  := Form_Data (First .. Last);
            Equal : constant Natural := Ada.Strings.Fixed.Index (S, "=");
         begin
            if Equal > 0 then
               Action
                 (Key      => S (S'First .. Equal - 1),
                  Value    => URL_Decode (S (Equal + 1 .. S'Last)),
                  Position => Position,
                  Quit     => Quit);
               exit when Quit;
            end if;
         end;

         Position := Position + 1;
         First    := Last + 2;
      end loop;
   end For_Every_Parameter;

   function Metavariable (Name : String) return String is
   begin
      return Ada.Environment_Variables.Value (Name);

   exception
      when others =>
         return "";
   end Metavariable;

   function Metavariable_Exists
     (Name : String) return Boolean renames
     Ada.Environment_Variables.Exists;

   ----------------
   -- URL_Decode --
   ----------------

   function URL_Decode (Item : String) return String is
      Length : constant Natural :=
        Item'Length - (2 * Ada.Strings.Fixed.Count (Item, "%"));
      Result : String (1 .. Length);
      Hex    : String   := "16#7F#";
      IP     : Positive := Item'First;
      RP     : Positive := 1;
   begin
      if Item'Length = 0 then
         return "";
      end if;

      loop
         case Item (IP) is
            when Plus_Sign =>
               Result (RP) := Space;

            when Percent_Sign =>
               Hex (4 .. 5) := Item (IP + 1 .. IP + 2);
               Result (RP)  := Character'Val (Natural'Value (Hex));
               IP           := IP + 2;

            when others =>
               Result (RP) := Item (IP);
         end case;

         exit when RP = Length;

         RP := RP + 1;
         IP := IP + 1;
      end loop;

      return Result;
   end URL_Decode;

   procedure Post_Import is
      Length : Natural := 0;
   begin
      if Metavariable_Exists ("CONTENT_LENGTH")
        and then
          Metavariable ("CONTENT_TYPE") =
          "application/x-www-form-urlencoded"
      then
         Length := Natural'Value (Metavariable ("CONTENT_LENGTH"));
      end if;

      if Length > 0 then
         Form_Data := new String (1 .. Length);
         Ada.Text_IO.Get (Form_Data.all);
         Data_Length := Length;
      end if;
   end Post_Import;

   procedure Query_Import is
      S : constant String := Metavariable ("QUERY_STRING");
   begin
      if S'Length > 0 then
         Form_Data   := new String'(S);
         Data_Length := S'Length;
      end if;
   end Query_Import;

   procedure Initialise_Form is
   begin
      if Request_Method = Post then
         Post_Import;
      else -- GET or HEAD
         Query_Import;
      end if;

      Form_Initialised := True;
   end Initialise_Form;

begin
   if not Metavariable_Exists ("GATEWAY_INTERFACE") then
      raise Program_Error with "No CGI environment";
   end if;

   Request_Method := Method_Type'Value (Metavariable ("REQUEST_METHOD"));
end CGI;
