/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * CGI-Beispiel fuer akfnetz
 * Copyright (c) 2016-2019,2024 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "akfnetz.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TITEL "Beispielformular"

//#define AUSGABETYP "Content-Type: text/html; charset=UTF-8"
#define AUSGABETYP "Content-Type: application/xhtml+xml; charset=UTF-8"

#define STIL \
  "body{font-size:x-large;max-width:50em;\n" \
  "margin:2ex auto;padding:.5em;display:table}\n" \
  "blockquote{padding-left:1em;border-left:solid .7ex #ccc}"

#define HTML_ANFANG "<!DOCTYPE html>\n" \
  "<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='de'>\n" \
  "<head>\n<title>" TITEL "</title>\n" \
  "<style>\n" STIL "\n</style>\n</head>\n\n<body>\n"

#define HTML_ENDE "</body></html>"

// Textausgabe mit maskierten HTML-Sonderzeichen
#define Text(s) akfnetz_xml_Text(stdout, s)

static void Auswertung (void);
static void Verarbeitung (void);
static void Formular (void);


int
main (void)
{
  // wurde es ueberhaupt per CGI aufgerufen?
  if (!getenv ("GATEWAY_INTERFACE"))
    {
      fputs ("Das sollte von einem Webserver "
	     "per CGI aufgerufen werden.\n", stderr);
      return EXIT_FAILURE;
    }

  if (akfnetz_cgi_Formulardaten ())
    Auswertung ();
  else
    Formular ();

  // Der Exitcode ist nicht wirklich wichtig
  return EXIT_SUCCESS;
}


static void
Auswertung (void)
{
  char *Anrede, *Name, *Anliegen, *Vorrang;
  bool eilig;

  Anrede = akfnetz_Formularfeld ("Anrede");
  Name = akfnetz_Formularfeld ("Name");
  Anliegen = akfnetz_Formularfeld ("Anliegen");
  Vorrang = akfnetz_Formularfeld ("Vorrang");
  eilig = (Vorrang && !strcmp ("eilig", Vorrang));

  // CGI-Kopf
  puts (AUSGABETYP);
  akfnetz_cgi_Kopfende ();

  puts (HTML_ANFANG);

  /*
     Auch wenn im Formular Eintraege vorausgesetzt werden,
     sollte man doch immer davon ausgehen, dass sie ungesetzt sein koennten.
   */
  if (Name && *Name)
    {
      fputs ("\n<h1>", stdout);
      if (!Anrede || !*Anrede)
	fputs ("Hallo ", stdout);
      else if (!strcmp ("Frau", Anrede))
	fputs ("Sehr geehrte Frau ", stdout);
      else
	fputs ("Sehr geehrter Herr ", stdout);
      Text (Name);
      puts (",</h1>");
    }

  /*
     Niemals Daten aus Formularen direkt in (X)HTML einfuegen!
     Ein Angreifer koennte sonst Code einschleusen.
     Das nennt man Cross-Site-Scripting (XSS).

     Um das zu verhindern ist die Funktion akfnetz_xml_Text() da.
     Hier ueber das Makro Text().
   */

  puts ("\n<p>uns liegt nun folgendes Anliegen vor:</p>\n\n"
	"<blockquote><pre>");
  Text (Anliegen);
  puts ("\n</pre></blockquote>\n");

  if (eilig)
    puts ("<p>Wir werden uns umgehend darum k\u00FCmmern.</p>");
  else
    puts ("<p>Wir werden uns bei Gelegenheit darum k\u00FCmmern.</p>");

  printf ("<p><small>Dieses Schreiben erreichte uns "
	  "\u00FCber die IP-Adresse %s</small></p>\n",
	  getenv ("REMOTE_ADDR"));
  /*
     In einem lokalen Netzwerk innerhalb eines Unternehmens
     koennte man zB. die IP-Adresse einem bestimmten Rechner
     in einer bestimmten Abteilung zuordnen.
   */

  printf ("<p><a href='%s'>Weiteres Anliegen melden...</a></p>\n",
	  getenv ("SCRIPT_NAME"));

  Verarbeitung ();

  puts (HTML_ENDE);
}


static void
Verarbeitung (void)
{
  /*
     Dies soll nur demonstrieren, wie man sequenziell
     auf die Feldwerte zugreift.
     Man koennte es zB. auch per E-Mail verschicken,
     oder irgendwo speichern.
   */

  puts ("\n<hr/>\n<h2>Felder:</h2>");

  puts ("<dl>");

  for (int i = 0; akfnetz_Formularfelder[i].Feld; ++i)
    {
      fputs ("<dt>", stdout);
      Text (akfnetz_Formularfelder[i].Feld);
      fputs ("</dt>\n<dd>", stdout);
      Text (akfnetz_Formularfelder[i].Wert);
      puts ("</dd>");
    }

  puts ("</dl>");
}


static void
Formular (void)
{
  // CGI-Kopf
  puts (AUSGABETYP);
  akfnetz_cgi_Kopfende ();

  puts (HTML_ANFANG);

  puts ("<h1>" TITEL "</h1>");

  // Dieses Skript ruft sich selbst auf.
  printf ("<form action='%s' method='POST'>\n", getenv ("SCRIPT_NAME"));
  // GET wuerde auch funktionieren,
  // aber POST ist in den meisten Faellen besser.

  puts ("Anrede:\n"
	"<label><input name='Anrede' type='radio' value='' "
	"checked='checked'/>Formlos</label>\n"
	"<label><input name='Anrede' type='radio' "
	"value='Frau'/>Frau</label>\n"
	"<label><input name='Anrede' type='radio' "
	"value='Herr'/>Herr</label><br/>");

  fputs ("<label>Name:\n<input type='text' "
	 "name='Name' required='required'", stdout);
  // Falls der Name bekannt ist, koennen wir ihn schon eintragen.
  char *Name = getenv ("REMOTE_USER");
  if (Name && *Name)
    printf (" value='%s'", Name);
  puts ("/></label><br/>");

  puts ("<label>Anliegen:<br/>\n"
	"<textarea name='Anliegen' cols='50' rows='15' required='required'"
	" wrap='hard'></textarea></label><br/>");

  puts ("Einreichen:\n"
	"<button name='Vorrang' value='normal'>normal</button>\n"
	"<button name='Vorrang' value='eilig'>eilig</button>");
  puts ("</form>");

  puts (HTML_ENDE);
}
