---------------------------------------------------------------------------
-- CGI-Beispiel fuer akfnetz                                             --
-- Copyright (c) 2019-2020 Andreas K. Foerster <akf@akfoerster.de>       --
--                                                                       --
-- This program is free software: you can redistribute it and/or modify  --
-- it under the terms of the GNU General Public License as published by  --
-- the Free Software Foundation, either version 3 of the License, or     --
-- (at your option) any later version.                                   --
--                                                                       --
-- This program is distributed in the hope that it will be useful,       --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of        --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         --
-- GNU General Public License for more details.                          --
--                                                                       --
-- You should have received a copy of the GNU General Public License     --
-- along with this program.  If not, see <http://www.gnu.org/licenses/>. --
---------------------------------------------------------------------------

-- SPDX-License-Identifier: GPL-3.0-or-later
pragma License (Gpl);

with CGI;
with Ada.Text_IO; use Ada.Text_IO;
with Ada.Exceptions;

procedure Beispiel_Ada is

   use type CGI.Method_Type;

   Titel : constant String := "Beispiel: Ada";
   NL : Character renames ASCII.LF;
   Fehler : exception;

   procedure HTML_Anfang;
   procedure Formular;
   procedure Auswertung;
   procedure Verarbeitung;

   procedure HTML_Anfang is
   begin
      Put_Line
        ("<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='de'>" &
         NL &
         "<head>" &
         NL &
         "<title>" &
         Titel &
         "</title>" &
         NL &
         "<style>" &
         NL &
         "body { font-size:x-large; max-width: 50em;" &
         NL &
         "margin: 2ex auto; padding: .5em; display: table; }" &
         NL &
         "blockquote { padding-left: 1em; border-left: solid .7ex #ccc}" &
         NL &
         "</style>" &
         NL &
         "</head>" &
         NL &
         NL &
         "<body>");
   end HTML_Anfang;

   procedure Formular is
   begin
      Put_Line ("Content-Type: application/xhtml+xml; charset=UTF-8");
      New_Line;
      if CGI.Method = CGI.Head then
         return;
      end if;

      HTML_Anfang;
      Put ("<h1>");
      Put (Titel);
      Put ("</h1>");
      New_Line;

      Put ("<form method='post' action='");
      Put (CGI.Metavariable ("SCRIPT_NAME"));
      Put ("'>");
      New_Line;

      Put
        ("Anrede:" &
         NL &
         "<label><input name='Anrede' type='radio' value='formlos'" &
         " checked='checked'/>Formlos</label>" &
         NL &
         "<label><input name='Anrede' type='radio'" &
         " value='Frau'/>Frau</label>" &
         NL &
         "<label><input name='Anrede' type='radio'" &
         " value='Herr'/>Herr</label><br/>" &
         NL);

      Put ("<label>Name: <input type='text' name='Name' required='required'");

      -- Falls der Name bekannt ist, koennen wir ihn schon eintragen.
      if CGI.Metavariable_Exists ("REMOTE_USER") then
         Put (" value='");
         Put (CGI.Metavariable ("REMOTE_USER"));
         Put (''');
      end if;

      Put
        ("/></label><br/>" &
         NL &
         "<label>Anliegen:<br/>" &
         NL &
         "<textarea name='Anliegen' cols='50' rows='15' required='required'" &
         " wrap='hard'></textarea></label><br/>" &
         NL &
         "Einreichen:" &
         NL &
         "<button name='Vorrang' value='normal'>normal</button>" &
         NL &
         "<button name='Vorrang' value='eilig'>eilig</button>" &
         NL &
         "</form></body></html>");
      New_Line;
   end Formular;

   procedure Auswertung is
      Anrede   : constant String := CGI.Value ("Anrede");
      Name     : constant String := CGI.Value ("Name");
      Anliegen : constant String := CGI.Value ("Anliegen");
      Vorrang  : constant String := CGI.Value ("Vorrang");
   begin
      if Anliegen = "" then
         raise Fehler;
      end if;

      Put_Line ("Content-Type: application/xhtml+xml; charset=UTF-8");
      New_Line;
      if CGI.Method = CGI.Head then
         return;
      end if;

      HTML_Anfang;

      if Name /= "" then
         Put ("<h1>");
         if Anrede = "Frau" then
            Put ("Sehr geehrte Frau ");
         elsif Anrede = "Herr" then
            Put ("Sehr geehrter Herr ");
         else
            Put ("Hallo ");
         end if;

         CGI.HTML_Text (Name);

         Put_Line (",</h1>");
      end if;

      Put ("<p>uns liegt nun folgendes Anliegen vor:</p>");
      New_Line (2);
      Put ("<blockquote><pre>");
      CGI.HTML_Text (Anliegen);
      New_Line;
      Put_Line ("</pre></blockquote>");

      if "eilig" = Vorrang then
         Put_Line ("<p>Wir werden uns umgehend darum k&#252;mmern.</p>");
      else
         Put_Line
           ("<p>Wir werden uns bei Gelegenheit darum k&#252;mmern.</p>");
      end if;

      Put ("<p><small>Dieses Schreiben erreichte uns von ");

      if CGI.Metavariable_Exists ("REMOTE_HOST") then
         Put ("dem Arbeitsplatz ");
         Put (CGI.Metavariable ("REMOTE_HOST"));
      else
         Put ("der IP-Adresse ");
         Put (CGI.Metavariable ("REMOTE_ADDR"));
         -- dieser Wert muss gesetzt sein
      end if;

      Put ("</small></p>");
      New_Line;

      Put ("<p><a href='");
      Put (CGI.Metavariable ("SCRIPT_NAME"));
      Put ("'>Weiteres Anliegen melden...</a></p>");
      New_Line;

      Verarbeitung;

      Put_Line ("</body></html>");
   end Auswertung;

   procedure Verarbeitung is
      Eintraege : Natural := CGI.Argument_Count;
   begin
      if Eintraege = 0 then
         return;
      end if;

      New_Line (2);
      Put_Line ("<hr/>" & NL & "<h2>Felder:</h2>");
      New_Line;

      Put_Line ("<dl>");
      for I in 1 .. Eintraege loop
         Put ("<dt>");
         CGI.HTML_Text (CGI.Key (I));
         Put ("</dt><dd><pre>");
         CGI.HTML_Text (CGI.Value (I));
         Put ("</pre></dd>");
         New_Line;
      end loop;
      Put_Line ("</dl>");
   end Verarbeitung;

   procedure Fehlerseite
     (Meldung : String := "Es ist ein Fehler aufgetreten")
   is
   begin
      Put_Line ("Status: 500 Server Error");
      Put_Line ("Content-Type: application/xhtml+xml; charset=UTF-8");
      New_Line;
      if CGI.Method = CGI.Head then
         return;
      end if;

      HTML_Anfang;
      Put_Line ("<div style='color:#A00'>");
      Put_Line (Meldung);
      Put_Line ("</div>");
      Put_Line ("</body></html>");
   end Fehlerseite;

begin
   if CGI.Argument_Count = 0 then
      Formular;
   else
      Auswertung;
   end if;

exception
   when E : others =>
      Fehlerseite (Ada.Exceptions.Exception_Message (E));
      raise;

end Beispiel_Ada;
