/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Automatische Sprachauswahl anhand der Browsereinstellung
 * Copyright (c) 2018-2021,2023,2024 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// Das wurde doch erheblich komplexer, als erwartet!

#define _GNU_SOURCE

#include "akfnetz.h"
#include <glob.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <limits.h>

#define STANDARDAKZEPTANZ "en,de;q=0.5"
#define GZ_ERWEITERUNG ".gz~"

#ifdef __linux__
#define MIT_SPLICE
#endif

// nicht mehr als 1024 Bytes pro Dateiname
#if !defined(NAME_MAX) || NAME_MAX > 1024
#undef NAME_MAX
#define NAME_MAX 1024
#endif


#define Text(s)  write(STDOUT_FILENO, "" s "", sizeof(s)-1)
#define MIN(a,b)  ((a)<(b)?(a):(b))

static int Dateisuche (void);
static void Basisname (void);
static int oeffnen (void);
static void Inhalt (int, off_t);
static int Fehler (void);
static void Groesse (off_t);
static void Modifikationszeit (time_t);

static bool gzip, xhtml;
static char Sprachkennung[2];
static const char *Methode;
static char Dateiname[NAME_MAX + 1];
static size_t Dateinamenslaenge;


int
main (void)
{
  static struct stat st;

  Methode = getenv ("REQUEST_METHOD");

  // CGI-Umgebung pruefen
  if (!Methode || !getenv ("GATEWAY_INTERFACE"))
    {
      Text ("Sprachindex\nLizenz: GPLv3 oder neuer\n"
	    AKFNETZ_HOMEPAGE "\n\n");
      return EXIT_SUCCESS;
    }

  if (Dateisuche () < 0)
    return Fehler ();

  int Datei = oeffnen ();
  if (Datei < 0 || fstat (Datei, &st) < 0 || !S_ISREG (st.st_mode))
    return Fehler ();

  // Alles klar, sende CGI-Kopf
  Text ("Vary: Accept-Language,Accept-Encoding\nContent-Language: ");
  write (STDOUT_FILENO, Sprachkennung, 2);
  Text ("\nContent-Location: ");
  write (STDOUT_FILENO, Dateiname, Dateinamenslaenge);	// ohne GZ_ERWEITERUNG!
  Text ("\nContent-Type: ");

  if (xhtml)
    Text ("application/xhtml+xml");
  else
    Text ("text/html");

  Text ("; charset=UTF-8\n");

  if (gzip)
    Text ("Content-Encoding: gzip\n");

  Groesse (st.st_size);
  Modifikationszeit (st.st_mtime);
  Text ("\n");			// Kopfende

  if (strcmp (Methode, "HEAD"))
    Inhalt (Datei, st.st_size);

  close (Datei);

  return EXIT_SUCCESS;
}


// sucht nach Dateien und gleicht die Sprache ab
// Setzt Variablen: Dateiname, Dateinamenslaenge, Sprachkennung
// -1 bei Fehler
static int
Dateisuche (void)
{
  glob_t gl;
  char Sprachangebot[256];
  char Muster[NAME_MAX + 1];
  size_t len;			// Laenge von Sprachangebot

  Basisname ();

  xhtml = false;
  memcpy (Muster, Dateiname, Dateinamenslaenge);
  memcpy (Muster + Dateinamenslaenge, ".??", 3);
  memcpy (Muster + Dateinamenslaenge + 3, ".html", sizeof (".html"));

  if (glob (Muster, GLOB_NOSORT, NULL, &gl))
    {
      globfree (&gl);

      // nochmal mit .xhtml versuchen
      // entweder oder - keine kombinierte Liste
      memcpy (Muster + Dateinamenslaenge + 3, ".xhtml", sizeof (".xhtml"));
      if (glob (Muster, GLOB_NOSORT, NULL, &gl))
	{
	  globfree (&gl);
	  return -1;
	}

      xhtml = true;
    }

  // Sprachangebot zusammenstellen
  // zum Beispiel "en,de,eo,"
  *Sprachangebot = '\0';
  len = 0;
  for (size_t i = 0; i < gl.gl_pathc; ++i)
    {
      if (len >= sizeof (Sprachangebot) - 4)
	break;

      char *sp = strchr (gl.gl_pathv[i], '.');
      if (!sp)
	continue;

      memcpy (Sprachangebot + len, sp + 1, 2);
      len += 2;
      Sprachangebot[len++] = ',';
    }

  Sprachangebot[len] = '\0';

  char *akzeptiert = getenv ("HTTP_ACCEPT_LANGUAGE");

  // Der Client weiß nicht, was er will
  if (!akzeptiert || !*akzeptiert)
    akzeptiert = STANDARDAKZEPTANZ;

  // Das Sprachangebot mit den akzeptierten Sprachen abgleichen
  // Das Komma am Ende von Sprachangebot stoert nicht
  int sp = akfnetz_Sprachauswahl (akzeptiert, Sprachangebot);

  // keine Übereinstimmung? => Nochmal versuchen
  if (sp < 0 || (size_t) sp >= gl.gl_pathc)
    sp = akfnetz_Sprachauswahl (STANDARDAKZEPTANZ, Sprachangebot);

  if (sp < 0 || (size_t) sp >= gl.gl_pathc)
    sp = 0;			// erste gefundene Sprache

  // Sprachkennung setzen
  memcpy (Sprachkennung, &Sprachangebot[sp * 3], 2);

  // Dateiname setzen
  strcpy (Dateiname, gl.gl_pathv[sp]);
  Dateinamenslaenge = strlen (Dateiname);

  globfree (&gl);

  return 0;
}


static int
oeffnen (void)
{
  int d;
  const char *accept_encoding;

  gzip = false;

  // gzip-komprimiert senden?
  accept_encoding = getenv ("HTTP_ACCEPT_ENCODING");
  if (accept_encoding && strstr (accept_encoding, "gzip")
      && Dateinamenslaenge + sizeof (GZ_ERWEITERUNG) <= NAME_MAX)
    {
      memcpy (Dateiname + Dateinamenslaenge, GZ_ERWEITERUNG,
	      sizeof (GZ_ERWEITERUNG));
      gzip = true;
      // Dateinamenslaenge nicht anpassen!
    }

  d = open (Dateiname, O_RDONLY);

  if (d < 0 && gzip)
    {
      // nochmal ohne GZ_ERWEITERUNG versuchen
      Dateiname[Dateinamenslaenge] = '\0';
      gzip = false;

      d = open (Dateiname, O_RDONLY);
    }

  return d;
}


static void
Basisname (void)
{
  const char *script_name, *n;
  size_t l;

  n = NULL;
  script_name = getenv ("SCRIPT_NAME");

  if (script_name && *script_name)
    n = strrchr (script_name, '/');

  // kein Dateiname?
  if (!n || !n[1])
    n = "/index";

  ++n;

  // Basisnamen ermitteln
  l = strcspn (n, ".");
  l = MIN (l, NAME_MAX - 3 - sizeof (".xhtml"));
  // kein Funktionsaufruf bei MIN

  memcpy (Dateiname, n, l);
  Dateiname[l] = '\0';
  Dateinamenslaenge = l;
}


// mit Groessenangabe muss es nicht blockorientiert ausgegeben werden
static void
Groesse (off_t Wert)
{
  size_t l;
  char *p;
  char b[20];

  if (Wert < 0)
    return;

  // Wert in String verwandeln
  p = b + sizeof (b) - 1;
  *p = '\n';
  l = 1;

  do
    {
      ++l;
      --p;
      *p = (Wert % 10) ^ 0x30;
      Wert /= 10;
    }
  while (Wert);

  Text ("Content-Length: ");
  write (STDOUT_FILENO, p, l);
}


// Zeitangabe ist fuer Cache nuetzlich
static void
Modifikationszeit (time_t t)
{
  size_t s;
  char b[80];

  s = strftime (b, sizeof (b),
		"Last-Modified: %a, %d %b %Y %T GMT\n", gmtime (&t));

  if (s)
    write (STDOUT_FILENO, b, s);
}


#ifdef MIT_SPLICE

static void
Inhalt (int Datei, off_t Groesse)
{
  ssize_t n;

  while (Groesse > 0)
    {
      n = splice (Datei, NULL, STDOUT_FILENO, NULL,
		  MIN (Groesse, INT_MAX), SPLICE_F_MOVE);

      if (n <= 0)
	break;

      Groesse -= n;
    }
}

#else // nicht MIT_SPLICE

static void
Inhalt (int Datei, off_t Groesse)
{
  ssize_t n;
  char b[4096];

  while (Groesse > 0)
    {
      n = read (Datei, b, sizeof (b));

      if (n <= 0)
	break;

      write (STDOUT_FILENO, b, n);

      Groesse -= n;
    }
}

#endif // nicht MIT_SPLICE


static int
Fehler (void)
{
  Text ("Status: 500 Server Error\n"
	"Content-Type: text/plain; charset=UTF-8\n\n");

  if (strcmp (Methode, "HEAD"))
    {
      Text ("Serverfehler: Datei ");
      write (STDOUT_FILENO, Dateiname, Dateinamenslaenge);
      Text (".??.html nicht gefunden.\n\n");

      Text ("Server Error: File ");
      write (STDOUT_FILENO, Dateiname, Dateinamenslaenge);
      Text (".??.html not found.\n\n");

      Text ("Servila eraro: Dosiero ");
      write (STDOUT_FILENO, Dateiname, Dateinamenslaenge);
      Text (".??.html ne trovita.\n\n");
    }

  return EXIT_FAILURE;
}
