#!/bin/sh

# geschrieben von Andreas K. Förster <https://akfoerster.de>
# Lizenz: CC0
# https://creativecommons.org/publicdomain/zero/1.0/deed.de
# SPDX-License-Identifier: CC0-1.0

# erstmal prüfen, ob es als CGI aufgerufen wurde
if [ -z "$GATEWAY_INTERFACE" ]
then
  echo "Ey, das sollte nur über einen Webserver aufgerufen werden." >&2
  exit 1
fi

# Dann den CGI-Kopf ausgeben

echo "Status: 200 OK"
# Damit kann der HTTP-Statuscode festgelegt werden.
# Das ist vor allem bei Fehlermeldungen nötig.
# Wenn das Skript zum Beispiel etwas nicht findet, dann sollte es
# "Status: 404 Not Found" ausgeben.
# Bei Erfolg kann die Status-Angabe aber auch weg gelassen werden.

echo "Content-Type: text/plain; charset=utf-8"
# Es muss entweder „Content-Type“ und/oder „Location“ angegeben sein.
# Für HTML nimmt man dann „text/html“ statt „text/plain“.
# Aber ich wollte dieses Skript jetzt nicht zu kompliziert machen.
# Ein Skript könnte aber auch eine Grafik,
# oder gar Audio- oder Video-Daten ausgeben.

echo "Cache-Control: no-cache"
# Es können auch beliebige HTTP-Kopfdaten mitgeschickt werden.

echo
# Der Kopf muss immer durch eine Leerzeile abgeschlossen werden,
# auch wenn danach nichts mehr folgt.

test "$REQUEST_METHOD" = "HEAD" && exit
# Falls nur der Kopf angefordert wurde,
# braucht der Rest nicht ausgeführt werden


# Ansonsten folgt dann der eigentliche Inhalt:

date "+Am %d.%m.%Y, um %H Uhr %M (Ortszeit des Servers)," 2>&1
# Beim Aufruf externer Befehle möglichst immer die Fehlermeldungen
# auf die Standardausgabe oder auf /dev/null umlenken,
# sonst landen die sonstwo.

echo
echo "$SERVER_SOFTWARE sendet schöne Grüße von $SERVER_NAME."
echo

# Bei den meisten Variablen muss man damit rechnen,
# dass sie eventuell nicht gesetzt oder leer sind.

if [ -n "$REMOTE_USER" ]
then echo "Hallo $REMOTE_USER!"
else echo "Leider kenne ich deinem Namen nicht. ☹"
fi

if [ -n "$AUTH_TYPE" ]
then
  if [ -n "$REMOTE_USER" ]
  then echo "Du hast dich mittels $AUTH_TYPE angemeldet. ♥ ☺"
  else echo "Du bist zwar mittels $AUTH_TYPE angemeldet,"
       echo "trotzdem kenne ich deinen Namen nicht."
       # Das ist ungewöhnlich, aber möglich.
  fi
fi

# Wenn kein Name bekannt ist, erfinden wir halt mal einen...
# den können wir dann später immer verwenden.
Anrede=${REMOTE_USER:-"Netzbürger"}

# Variablen, die mit HTTP_ beginnen, sind Sachen,
# vom User-Agent (Browser, Suchmaschine, etc.) gesendet wurden.

# Es ist zwar äußerst unwahrscheinlich,
# dass der Browser eine E-Mail-Adresse sendet,
# aber es gibt einen formalen Standard für sowas.
[ -n "$HTTP_FROM" ] && echo "Deine E-Mail-Adresse lautet $HTTP_FROM."

echo
echo "Dein Computer hat die Adresse $REMOTE_ADDR."
# Die sollte immer bekannt sein.
# Man sollte sowohl mit IPv4-, als auch mit IPv6-Adressen rechnen.

# Das folgende wird aber eher selten vom Server unterstützt:
[ -n "$REMOTE_HOST" ] && echo "Der Rechner ist bekannt als $REMOTE_HOST."
[ -n "$REMOTE_IDENT" ] && echo "Du bist auch bekannt als $REMOTE_IDENT."
# Diese Angaben kann ein Skript aber evtl. selber anhand $REMOTE_ADDR ermitteln.

if [ -n "$HTTP_USER_AGENT" ]
then
  echo "Dein Browser hat die Kennungung:"
  echo "$HTTP_USER_AGENT"
  # Viele Browser sind hier sehr gesprächig und verworren. ☹
fi

# Do not track
if [ -z "$HTTP_DNT" ]
then
  echo "„Do not track“ ist nicht aktiviert."
elif [ "$HTTP_DNT" -ge 1 ]
then
  echo "„Do not track“ ist aktiviert."
elif [ "$HTTP_DNT" -eq 0 ]
then
  echo "Dein „Do not track“ besagt, du willst ausdrücklich verfolgt werden!?"
fi

if [ -n "$HTTP_ACCEPT_LANGUAGE" ]
then
echo
cat <<ENDE
Die Spracheinstellungen des Browsers sehen so aus: $HTTP_ACCEPT_LANGUAGE
Das könnte man für mehrsprachige Programme verwenden.
Wir sind schließlich im Welt-Weiten Web, und da sprechen nicht alle Deutsch.
Manche sprechen auch English, oder русский язык, oder 普通话, oder …
Na gut, das muss man vielleicht doch nicht alles unterstützen.
Aber man könnte.
ENDE
fi

echo
if [ -n "$HTTP_REFERER" ]
then
  echo "Diese Seite wurde über folgende Seite aufgerufen:"
  echo "$HTTP_REFERER"
else
  echo "Diese Seite scheint nicht über einen Link aufgerufen worden zu sein."
  echo "Aber das kann man nie mit Sicherheit ausschließen."
fi

echo
echo "Die Datenübertragung geschieht über $SERVER_PROTOCOL auf Port $SERVER_PORT."
echo "Das Programm $SCRIPT_NAME kommuniziert über $GATEWAY_INTERFACE."
echo
echo "Der Server läuft auf dem Betriebssystem: $(uname -sr)"

echo
echo "In der Adresse könnten hinter dem CGI-Skript noch weitere"
echo "Pfadangaben und ein Dateiname folgen."
if [ -z "$PATH_INFO" ]
then
  echo "Du hast da aber jetzt nichts angegeben. Mach das doch mal …"
else
  echo "Du hast da $PATH_INFO angegeben."
  if [ -n "$PATH_TRANSLATED" ]
  then
    echo "Auf dem Server wäre das wohl auf $PATH_TRANSLATED."
  fi

cat <<ENDE
Diese Angaben können nicht nur angezeigt,
sondern natürlich auch verwendet werden.
Sie können durchaus auch als rein abstrakte Namen interpretiert werden.
ENDE
fi

echo
cat <<ENDE
Man kann auch Parameter an das CGI-Skript übergeben,
indem man an die Adresse ein Fragezeichen und dann die Parameter
getrennt durch Pluszeichen angibt.
Zum Beispiel:
http://$SERVER_NAME:$SERVER_PORT$SCRIPT_NAME$PATH_INFO?eins+zwei+drei.
Es darf aber kein unkodiertes Gleichzeichen vorkommen.
Problematische Zeichen kodiert man mit % und dem hexadezimalen Zeichenwert.
Zum Beispiel %2B für +, %25 für %, %20 für Leerzeichen, %3D für =.
ENDE
echo

if [ $# -eq 0 ]
then
  echo "Du hast aber jetzt keine Parameter angegeben."
else
  echo "Du hast jetzt $# Parameter angegeben:"

  for i in "$@"
  do
    echo " • $i"
  done
fi

echo
cat <<ENDE
Das hinter dem Fragezeichen nennt man den Query-String.
Darüber werden auch Formulardaten mit der Methode GET übertragen.
Dann muss man den aber selber auseinanderpulen und dekodieren.

Formulare mit GET sollte man nur verwenden, wenn die Daten kurz sind,
nicht geheim, und nichts auf dem Server gespeichert werden soll.
ENDE

if [ -n "$QUERY_STRING" ]
then
  echo
  echo "Der Query-String sieht jetzt so aus:"
  echo "$QUERY_STRING"
fi

echo
echo "Meistens werden Formulare jedoch mit der (Methode) POST übermittelt."
if [ "$REQUEST_METHOD" = "POST" -a -n "$CONTENT_LENGTH" ]
then
  echo "In der Post waren $CONTENT_LENGTH Bytes."
  echo "Kodierung: $CONTENT_TYPE"
  dd ibs=1 count=$CONTENT_LENGTH 2>/dev/null
else
  echo "Aber jetzt ist keine Post angekommen."
fi

echo
echo
echo "Ich zeige dir mal alle aktuellen Umgebungsvariablen:"
printenv | sort | sed "/=/ s/^/ • /"
echo
echo "Achtung: Nicht alle Variablen gehören zur CGI-Spezifikation!"


# und Tschüss
echo
echo "Tschüss, $SERVER_SOFTWARE"

exit 0
