/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * CGI fuer AKF-Netz-Server (Kindprozess)
 * Copyright (c) 2015-2024 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CGI/1.1 (Common Gateway Interface)
 * RFC 3875
 *
 * Kindprozess. Kein Logbuch.
 */

#ifndef KEIN_CGI

#define _POSIX
#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 700

#include <stdio.h>
#include <stdlib.h>
#include <stdnoreturn.h>
#include <limits.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "akfnetz.h"
#include "akfwebserver.h"

#define CGI_PARAMETER 50
#define CGI_METAVARIABLEN 200
// Unbegrenzte Ressourcen waeren eine Einladung fuer Angreifer

#define GATEWAY_INTERFACE "CGI/1.1"

#ifndef SERVER_SOFTWARE
#define SERVER_SOFTWARE "server"	// neutraler Name
#endif

// fuer Textkonstanten kann man die Laenge vorab ermitteln
#define Feld(f,e)  Feldeintrag (f "=", sizeof(f), e, 0)
#define Feldlaenge(f,e,l)  Feldeintrag (f "=", sizeof(f), e, l)
#define Festfeld(e) Feldzuweisung ("" e "")

#define Grossbuchstabe(z) (('a'<=(z)&&(z)<='z')?(z)^0x20:(z))
#define Fehler()  _exit (127)

// Wir brauchen uns hier keine Gedanken um Speicherlecks zu machen.
// Der Prozess wird entweder durch execve ersetzt, oder beendet.
#if defined (__GNUC__) && __GNUC__ >= 10
#pragma GCC diagnostic ignored "-Wanalyzer-malloc-leak"
#endif

#pragma GCC poison  exit

static char **Variablen;
static size_t Variablenanzahl;

static int Variablenerstellung (void);
static bool Kommandozeile (char **);
static void Kanalumleitung (int);
static void Begrenzungen (void);


// fuehrt CGI-Programm aus, kehrt nur bei Fehlern zurueck
extern noreturn void
akfnetz_cgi_ausfuehren (int Ausgabe, const char *Befehl, size_t Befehllaenge)
{
  char Programm[Befehllaenge + 1];
  memcpy (Programm, Befehl, Befehllaenge);
  Programm[Befehllaenge] = '\0';

  if (Variablenerstellung () < 0)
    Fehler ();

  char *Parameter[CGI_PARAMETER + 2];
  Parameter[0] = Programm;
  Parameter[1] = NULL;

  if (!Kommandozeile (Parameter))
    Fehler ();

  Kanalumleitung (Ausgabe);
  Begrenzungen ();

  execve (Programm, Parameter, Variablen);
  Fehler ();
}


// leitet Standard-Kanaele um, und schliesst Ausgabe
static void
Kanalumleitung (int Ausgabe)
{
  // stdout koennte bereits geschlossen sein
  dup2 (Ausgabe, STDOUT_FILENO);
  close (Ausgabe);

  int nirgendwo = open ("/dev/null", O_RDWR | O_NOCTTY);

  if (nirgendwo <= STDERR_FILENO)
    Fehler ();

  dup2 (nirgendwo, STDERR_FILENO);

  if (http.Inhalt < 0)
    dup2 (nirgendwo, STDIN_FILENO);
  else
    {
      dup2 (http.Inhalt, STDIN_FILENO);
      close (http.Inhalt);
    }

  close (nirgendwo);
}


static void
Begrenzungen (void)
{
  struct rlimit l;

  // CPU-Zeit begrenzen. Dies ist nicht die Realzeit!
  // Als Schutz vor Endlosschleife.
  l.rlim_cur = 5 * 60;
  l.rlim_max = 10 * 60;
  setrlimit (RLIMIT_CPU, &l);

  // Coredumps sind bereits verhindert

  // Berechtigungen fuer neue Dateien im CGI-Programm
  // Diese koennen vom CGI-Programm ignoriert werden
  umask (Einstellung.cgi_umask);

  // Programm mit verringerter Prioritaet ausfuehren
  nice (5);
}


static void
Feldzuweisung (char *f)
{
  if (Variablenanzahl <= CGI_METAVARIABLEN)
    Variablen[Variablenanzahl++] = f;
}


// Feld muss mit '=' enden, Eintragslaenge darf 0 sein
static void
Feldeintrag (const char *Feld, size_t Feldlaenge,
	     const char *Eintrag, size_t Eintragslaenge)
{
  char *p;

  if (Eintrag && !Eintragslaenge)
    Eintragslaenge = strlen (Eintrag);

  p = malloc (Feldlaenge + Eintragslaenge + 1);
  if (!p)
    return;

  memcpy (p, Feld, Feldlaenge);
  p[Feldlaenge] = '\0';

  if (Eintrag)
    {
      memcpy (p + Feldlaenge, Eintrag, Eintragslaenge);
      p[Feldlaenge + Eintragslaenge] = '\0';
    }

  Feldzuweisung (p);
}


static void
Feldzahl (const char *Feld, uintmax_t Wert)
{
  char f[256];
  int l;

  l = snprintf (f, sizeof (f), "%s=%ju", Feld, Wert);

  if (l < 0 || (size_t) l >= sizeof (f))
    return;

  char *p = malloc (l + 1);
  if (!p)
    return;

  memcpy (p, f, l + 1);
  Feldzuweisung (p);
}


static void
Serverprotokoll (void)
{
  char f[80];
  int l;

  l = snprintf (f, sizeof (f), "SERVER_PROTOCOL=HTTP/%hu.%hu",
		http.Version, http.Unterversion);

  // ein Ueberlauf ist hier ausgeschlossen
  if (l < 0)
    return;

  char *p = malloc (l + 1);
  if (!p)
    return;

  memcpy (p, f, l + 1);

  Feldzuweisung (p);
}


// DOCUMENT_ROOT, SCRIPT_NAME, PATH_INFO, PATH_TRANSLATED
static void
Pfadinfo (void)
{
  char Pfad[PATH_MAX];
  char *p;

  // DOCUMENT_ROOT: Erweiterung, nicht in CGI-Spezifikation
  p = stpcpy (Pfad, Einstellung.Wurzel);
  if (http.Virtuellname)
    {
      p = stpcpy (p, "/");
      p = stpcpy (p, http.Virtuellname);
    }

  Feld ("DOCUMENT_ROOT", Pfad);

  if (!http.Extrapfad)
    Feldlaenge ("SCRIPT_NAME", http.Pfad, http.Pfadlaenge);
  else
    {
      size_t Extralaenge = strlen (http.Extrapfad);
      size_t Virtuellaenge =
	http.Virtuellname ? strlen (http.Virtuellname) : 0;

      Feldlaenge ("PATH_INFO", http.Extrapfad, Extralaenge);

      if (Einstellung.Wurzellaenge + 1 + Virtuellaenge + Extralaenge
	  < PATH_MAX)
	{
	  p = stpcpy (Pfad, Einstellung.Wurzel);
	  if (http.Virtuellname)
	    {
	      p = stpcpy (p, "/");
	      p = stpcpy (p, http.Virtuellname);
	    }
	  p = stpcpy (p, http.Extrapfad);

	  Feld ("PATH_TRANSLATED", Pfad);
	}

      // SCRIPT_NAME darf nicht den Extrapfad enthalten
      Feldlaenge ("SCRIPT_NAME", http.Pfad, http.Pfadlaenge - Extralaenge);
    }
}


// Sicherheitskritische oder ueberfluessige Angaben herausfiltern
static inline bool
Kopffilter (const char *z)
{
  return (!strchr (z, ':')
	  || Praefix (z, "Authorization:")
	  || Praefix (z, "Proxy-Authorization:")
	  || Praefix (z, "Proxy:")
	  || Praefix (z, "Content-Type:") || Praefix (z, "Content-Length:"));

  /*
     Ein Proxy-Kopf darf nicht zu HTTP_PROXY uebersetzt werden,
     da diese Variable von etlichen Klienten verwendet wird.
     Siehe <https://httpoxy.org/>.
   */
}


// HTTP-Angaben in Metavariablen umwandeln
static void
http_Angaben (void)
{
  char **Kopf = http.Kopf;

  // HTTP/0.9?
  if (!Kopf)
    return;

  char f[256];
  memcpy (f, "HTTP_", 5);

  for (int i = 0; Kopf[i]; ++i)
    {
      char *z = Kopf[i];

      // Nicht endlos Ressourcen verschwenden, sonst angreifbar
      if (Variablenanzahl > CGI_METAVARIABLEN)
        break;

      if (Kopffilter (z))
	continue;

      size_t fl;

      // Es ist sichergestellt, dass ':' vorkommt
      for (fl = 5; fl < sizeof (f) && *z != ':'; ++fl, ++z)
	f[fl] = (*z == '-') ? '_' : Grossbuchstabe (*z);

      f[fl++] = '=';
      Feldeintrag (f, fl, Textanfang (z + 1), 0);
    }
}


// Variablen aus der Konfiguration eintragen
static void
Konfigurationsvariablen (void)
{
  if (Einstellung.cgiMeta)
    {
      struct akfnetz_Liste *d;
      for (d = Einstellung.cgiMeta; d; d = d->weiter)
	Feldzuweisung ((char *) d->Eintrag);
    }
}


static int
Variablenerstellung (void)
{
  Variablen = malloc (sizeof (*Variablen) * (CGI_METAVARIABLEN + 1));
  if (!Variablen)
    return -1;

  Festfeld ("GATEWAY_INTERFACE=" GATEWAY_INTERFACE);

  if (Einstellung.Signatur)
    Festfeld ("SERVER_SOFTWARE=" SERVER_SOFTWARE);
  else
    Festfeld ("SERVER_SOFTWARE=server");

  Feld ("SERVER_NAME",
	http.Virtuellname ? http.Virtuellname : Einstellung.Servername);
  Serverprotokoll ();

  if (Einstellung.Serverport)
    Feldzahl ("SERVER_PORT", Einstellung.Serverport);
  else
    Festfeld ("SERVER_PORT=80");

  Feld ("REQUEST_METHOD", akfnetz_Methodenname[http.Methode]);
  Feld ("QUERY_STRING", http.Anfrage);

  if (*http.Klientenadresse)
    Feld ("REMOTE_ADDR", http.Klientenadresse);
  else				// leider muss das gesetzt sein
    Festfeld ("REMOTE_ADDR=127.0.0.1");

  if (http.Klientenname)
    Feld ("REMOTE_HOST", http.Klientenname);

  // REMOTE_IDENT wird nicht unterstuetzt

  Pfadinfo ();

  if (http.Namenslaenge)
    Feldlaenge ("REMOTE_USER", http.Name, http.Namenslaenge);
  // auch wenn nicht autorisiert

  if (http.autorisiert)
    Festfeld ("AUTH_TYPE=Basic");

  if (http.Methode == POST)
    {
      Feldzahl ("CONTENT_LENGTH", http.Inhaltslaenge);
      Feld ("CONTENT_TYPE", akfnetz_Kopfeintrag (http.Kopf, "Content-Type"));
    }

  /*
     Erweiterungen - Die folgenden sind kein Teil der CGI-Spezifikation,
     werden aber auch von einigen anderen Servern unterstuetzt.

     DOCUMENT_ROOT ist in Pfadinfo implementiert.
   */

  Feld ("REQUEST_URI", http.URI);
  if (http.Klientenport != 0)
    Feldzahl ("REMOTE_PORT", http.Klientenport);
  // REMOTE_PORT kann fuer ident-Anfragen genutzt werden

  if (http.Originalpfad)
    {
      Festfeld ("REDIRECT_STATUS=200");
      Feld ("REDIRECT_URL", http.Originalpfad);
      // diese Variablen werden unter anderem fuer PHP benoetigt
    }

  Konfigurationsvariablen ();
  http_Angaben ();

  // Abschluss kennzeichnen
  Variablen[Variablenanzahl] = NULL;

  return 0;
}


// sammelt Kommandozeilenparameter fuer CGI (IsIndex)
static bool
Kommandozeile (char **Parameter)
{
  if (!http.Anfrage || !*http.Anfrage || strchr (http.Anfrage, '='))
    return true;		// keine Parameter, aber okay

  /*
    Alle in einem Speicherbereich
    Das URL-Dekodieren mach die Ausgabe nur kleiner,
    so dass alles in dem Bereich bleiben kann.
  */
  char *as = strdup (http.Anfrage);
  if (!as)
    return false;

  size_t nr = 1;		// 0 fuer Programmname reserviert
  char *s, *sz = NULL;
  for (s = strtok_r (as, "+", &sz); s && nr <= CGI_PARAMETER;
       s = strtok_r (NULL, "+", &sz))
    Parameter[nr++] = akfnetz_url_dekodieren (s);

  Parameter[nr] = NULL;

  // weitere Parameter? -> Fehler
  return !s;
}

#endif
