/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef AKFWEBSERVER_H
#define AKFWEBSERVER_H

#include <stdio.h>
#include <stdbool.h>
#include <stdnoreturn.h>
#include <stdint.h>
#include <sys/types.h>
#include "Sprachen.h"

#pragma GCC poison  strtok gmtime localtime sprintf vsprintf

#define SERVER_SOFTWARE "akfwebserver"
#define PROTOKOLL "HTTP/1.1"
#define ZEITLIMIT 30	// Sekunden, 0 fuer kein Limit
#define LISTEN_BACKLOG 3
#define STRAFSEKUNDEN 3		// bei falschen Zugangsdaten

/*
Die STRAFSEKUNDEN sollen vor allem automatische
Einlogversuche behindern.
*/

#ifndef TEXTKODIERUNG
#ifdef SYSTEMKODIERUNG
#define TEXTKODIERUNG SYSTEMKODIERUNG
#else
#define TEXTKODIERUNG "UTF-8"
#endif
#endif

#define TEXT "text/plain; charset=" TEXTKODIERUNG
#define HTML "text/html"
#define XHTML "application/xhtml+xml; charset=UTF-8"
#define BIN "application/octet-stream"

#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define MAX(x,y) ((x) > (y) ? (x) : (y))

#ifdef KEIN_CGI
#define NUTZE_CGI  false
#else
#define NUTZE_CGI  true
#endif

#ifdef KEIN_INDEXER
#define NUTZE_INDEXER  false
#else
#define NUTZE_INDEXER  true
#endif

#ifndef Fehlermeldung
#define Fehlermeldung  strerror
#endif

#define akfnetz_Logbuch(...) \
  fprintf (Einstellung.Logbuch, __VA_ARGS__)

#define redsam(l)  (Einstellung.Loglevel >= (l))

// faengt String s mit der Konstanten c an?
#define Praefix(s,c) (strncasecmp (s, "" c "", sizeof(c)-1) == 0)

// nuetzliche Symbole
#define LEER           u8"\u00A0"	// geschuetztes Leerzeichen
#define PFEIL_HOCH     u8"\u2191"
#define PFEIL_RUNTER   u8"\u2193"
#define PFEIL_LINKS    u8"\u2190"
#define PFEIL_RECHTS   u8"\u2192"
#define GEDANKENSTRICH u8"\u2014"
#define SCHLIESSEN     u8"\u00D7"
#define AUFZAEHLUNG    u8"\u2022"	// Bullet
#define SMILEY         u8"<span class='emoji'>\u263A</span>"
#define TRAURIG        u8"<span class='emoji'>\u2639</span>"
#define WARNUNG        u8"<span class='emoji'>\u26A0</span>"
#define TOTENKOPF      u8"<span class='emoji'>\u2620</span>"

enum Zustand
{
  Fehlerfrei, Neuanforderung, Persistenzfehler,
  Suchfehler, Zugriffsfehler, Verzeichnisfehler, Unterstuetzungsfehler,
  Anfragefehler, Methodenfehler, Voraussetzungsfehler, Berechtigungsfehler,
  Bereichsfehler, Adressfehler, Erwartungsfehler, Versionsfehler,
  Betriebsfehler, Platzfehler, Serverfehler, Scheisse
};

/*
- Neuanforderung wird fuer lokale Umleitungen benutzt (CGI)
- Persistenzfehler, wenn die Laenge unbekannt oder falsch ist
  und die Verbindung geschlossen werden soll
- alles ueber Persistenzfehler erzeugt eine Fehlerseite
- Anfragefehler: unspezifizierter Fehler in der Anfrage (400)
- Zugriffsfehler, wenn etwas nicht erlaubt ist (403)
- Suchfehler, wenn etwas nicht gefunden wurde (404)
- Methodenfehler, wenn Methode bekannt, aber nicht erlaubt ist (405)
- Berechtigungsfehler, wenn Zugangsdaten angefragt werden sollen (401)
- Unterstuetzungsfehler, wenn etwas nicht unterstuetzt wird (501)
- Betriebsfehler im Wartungsmodus (503)
- Serverfehler: unspezifizierter Fehler am Server (500)
- Scheisse sagt man nicht; falls doch, ist das ein Serverfehler (500)
*/

struct Einstellungen
{
  char *Servername, *Tempverzeichnis, *Wurzel, *Schutzbereich;
  char *Onion_Location;
  struct akfnetz_Liste *cgiDatei, *cgiMeta;
  struct akfnetz_Liste *Zugang, *Kopfzeilen, *Erweiterungen;
  size_t Wurzellaenge, Schutzbereichslaenge;
  FILE *Logbuch;
  int Protokoll;
  int Loglevel;
  mode_t cgi_umask;
  unsigned short int Serverport, Port;
  bool offen, gzip, Datenschutz, cgi, beschreibbar, Schreibschutz;
  bool Hostname, virtuell, Trace, Signatur, Schutzbereichsblockierung;
  volatile bool Wartungsmodus;
  char Serveradresse[46];	// nur fuer IP-Adressen!
};


struct HTTP_Angaben
{
  char *Klientenname, *Virtuellname;
  char *Anfragezeile, *URI, *Pfad, *Originalpfad;
  char *Host, *Anfrage, *cgi, *Extrapfad;	// nur indirekte Zeiger
  char **Kopf;
  size_t Pfadlaenge, Namenslaenge, CGIlaenge;
  uintmax_t Inhaltslaenge, Antwortlaenge;
  off_t Dateigroesse;
  time_t Zeitpunkt;
  unsigned short int Version, Unterversion;
  unsigned short int Status;
  unsigned short int Klientenport;
  enum akfnetz_Anfragemethoden Methode;
  int Inhalt;
  bool komplett, autorisiert, ip_gesetzt, onion;
  char Name[80];
  char Klientenadresse[46];	// nur fuer IP-Adressen!
};


extern enum Sprache Systemsprache;	// nur unterstuetzte Sprache
extern struct HTTP_Angaben http;
extern struct Einstellungen Einstellung;


// wartet auf eingehende Verbindungen am Socket
noreturn void akfnetz_Verbindungsannahme (int);

int akfnetz_inetd (void);

// HTTP
void akfnetz_HTTP_Anfrage (FILE * ein, FILE * aus);
time_t http_Anfang (FILE *, int, const char *);
void http_Kopf (FILE *, const char *, const char *);
void http_ContentLength (FILE *, uintmax_t);
enum Zustand http_Dauerverbindung (FILE *);
void http_Methodenerlaubnis (FILE *);
void http_Ende (FILE *);

// Webfehler
void http_Webfehler (FILE *, enum Zustand);
void http_ueberlastet (int);
enum Zustand http_sende_Fehlerstil (FILE *);

// Logbuch
void Logbucheintrag (void);
void Logbuch_Verbindung (void);

// Datei
int Dateiinhalt (FILE *, int, off_t Anfang, off_t Datenmenge);

enum Zustand akfnetz_cgi (FILE *, const char *);
noreturn void akfnetz_cgi_ausfuehren (int, const char *, size_t);

#ifdef KEIN_CGI
#define akfnetz_cgi(a,b) (Zugriffsfehler)
#define akfnetz_Variable(x) (-1)
#endif


static inline char *
Textanfang (const char *s)
{
  while (*s && (*s == ' ' || *s == '\t'))
    ++s;

  return (char *) s;
}

#endif
