/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfwebserver
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <time.h>
#include <stdbool.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <getopt.h>

#include "akfnetz.h"
#include "akfwebserver.h"
#include "version.h"

#ifdef SERVER_SOFTWARE
#define PROGRAMMNAME SERVER_SOFTWARE
#else
#define PROGRAMMNAME "akfnetzserver"
#endif

struct Einstellungen Einstellung;
enum Sprache Systemsprache;

static char *Programmname, *Socketname;
static struct passwd *Nutzer;
static bool Daemon, inetd;

#define Kurzoptionen "hbgVBCD46lPsvQTiU:H:c:e:S:L:p:a:d:u:G:M:N:"

enum OPT
{
  OPT_OFFEN = 256, OPT_DATENSCHUTZ, OPT_VAR, OPT_WARTUNG, OPT_TMPDIR,
  OPT_HOSTNAME, OPT_SCHUTZB, OPT_ZUGANG, OPT_SERVERPORT, OPT_VIRTUELL,
  OPT_ONION
};

static const struct option Langoptionen[] = {
  {"Hilfe", no_argument, NULL, 'h'},
  {"help", no_argument, NULL, 'h'},
  {"Version", no_argument, NULL, 'V'},
  {"version", no_argument, NULL, 'V'},
  {"config", required_argument, NULL, 'c'},
  {"Konfiguration", required_argument, NULL, 'c'},
  {"Servername", required_argument, NULL, 'N'},
  {"servername", required_argument, NULL, 'N'},
  {"socket", required_argument, NULL, 'U'},
  {"Socket", required_argument, NULL, 'U'},
  {"unix-socket", required_argument, NULL, 'U'},
  {"IPv4", no_argument, NULL, '4'},
  {"ipv4", no_argument, NULL, '4'},
  {"IPv6", no_argument, NULL, '6'},
  {"ipv6", no_argument, NULL, '6'},
  {"localhost", no_argument, NULL, 'l'},
  {"Adresse", required_argument, NULL, 'a'},
  {"address", required_argument, NULL, 'a'},
  {"Port", required_argument, NULL, 'p'},
  {"port", required_argument, NULL, 'p'},
  {"Serverport", required_argument, NULL, OPT_SERVERPORT},
  {"serverport", required_argument, NULL, OPT_SERVERPORT},
  {"Verzeichnis", required_argument, NULL, 'd'},
  {"directory", required_argument, NULL, 'd'},
  {"still", no_argument, NULL, 's'},
  {"verbose", no_argument, NULL, 'v'},
  {"wortreich", no_argument, NULL, 'v'},
  {"log", required_argument, NULL, 'L'},
  {"gzip", no_argument, NULL, 'g'},
  {"Datenschutz", no_argument, NULL, OPT_DATENSCHUTZ},
  {"privacy", no_argument, NULL, OPT_DATENSCHUTZ},
  {"offen", no_argument, NULL, OPT_OFFEN},
  {"open", no_argument, NULL, OPT_OFFEN},
  {"internet", no_argument, NULL, OPT_OFFEN},
  {"Internet", no_argument, NULL, OPT_OFFEN},
  {"beschreibbar", no_argument, NULL, 'b'},
  {"writable", no_argument, NULL, 'b'},
  {"Schreibschutz", no_argument, NULL, 'P'},
  {"writeprotection", no_argument, NULL, 'P'},
  {"protected-zone", required_argument, NULL, OPT_SCHUTZB},
  {"Schutzbereich", required_argument, NULL, OPT_SCHUTZB},
  {"block-protected-zone", no_argument, NULL, 'B'},
  {"Schutzbereichsblockierung", no_argument, NULL, 'B'},
  {"Zugang", required_argument, NULL, OPT_ZUGANG},
  {"Account", required_argument, NULL, OPT_ZUGANG},
  {"account", required_argument, NULL, OPT_ZUGANG},
  {"Nutzer", required_argument, NULL, 'u'},
  {"cgi", no_argument, NULL, 'C'},
  {"cgi-Datei", required_argument, NULL, 'G'},
  {"cgi-file", required_argument, NULL, 'G'},
  {"cgi-umask", required_argument, NULL, 'M'},
  {"Erweiterung", required_argument, NULL, 'e'},
  {"extension", required_argument, NULL, 'e'},
  {"header", required_argument, NULL, 'H'},
  {"Kopf", required_argument, NULL, 'H'},
  {"var", required_argument, NULL, OPT_VAR},
  {"daemon", no_argument, NULL, 'D'},
  {"virtuell", no_argument, NULL, OPT_VIRTUELL},
  {"virtual", no_argument, NULL, OPT_VIRTUELL},
  {"servicemode", no_argument, NULL, OPT_WARTUNG},
  {"Wartungsmodus", no_argument, NULL, OPT_WARTUNG},
  {"trace", no_argument, NULL, 'T'},
  {"Sprache", required_argument, NULL, 'S'},
  {"language", required_argument, NULL, 'S'},
  {"tmpdir", required_argument, NULL, OPT_TMPDIR},
  {"hostname", no_argument, NULL, OPT_HOSTNAME},
  {"inetd", no_argument, NULL, 'i'},
  {"onion", required_argument, NULL, OPT_ONION},
  {"Onion", required_argument, NULL, OPT_ONION},
  {0, 0, 0, 0}
};


static void Initialisierung (char *);
static void Optionen (int, char **);
static bool Option (int, char *);
static void Einrichtung (void);
static void Logbuchoeffnung (const char *);
static void Signaleinrichtung (void);
static noreturn void Servermodus (void);


int
main (int argc, char **argv)
{
  Initialisierung (argv[0]);
  Optionen (argc, argv);
  Einrichtung ();

  if (inetd)
    return akfnetz_inetd ();
  else
    Servermodus ();

  return EXIT_FAILURE;
}


static void
Version (FILE *d)
{
  switch (Systemsprache)
    {
    case Deutsch:
      fputs (PROGRAMMNAME " " AKFNETZ_VERSION "\n" AKFNETZ_COPYRIGHT "\n"
	     "Lizenz GPLv3+: GNU GPL Version 3 oder neuer "
	     "<http://gnu.org/licenses/gpl.html>\n\n"
	     "Dies ist freie Software. Sie darf veraendert "
	     "und verteilt werden.\n"
	     "Dieses Programm wird ohne Gewaehrleistung geliefert,\n"
	     "soweit dies gesetzlich zulaessig ist.\n\n"
	     "erstellt: " __DATE__ "\n\n" "Homepage: <" AKFNETZ_HOMEPAGE
	     ">\n", d);
      break;

    case Englisch:
    default:
      fputs (PROGRAMMNAME " " AKFNETZ_VERSION "\n" AKFNETZ_COPYRIGHT "\n"
	     "License GPLv3+: GNU GPL version 3 or later "
	     "<http://gnu.org/licenses/gpl.html>\n\n"
	     "This is free software; you are free to change and redistribute it.\n"
	     "There is NO WARRANTY, to the extent permitted by law.\n\n"
	     "build: " __DATE__ "\n\n" "Homepage: <" AKFNETZ_HOMEPAGE ">\n",
	     d);
      break;
    }

  // Jahr-2038-Problem?
  if (sizeof (time_t) == 4 && (time_t) (-1) < 1)
    switch (Systemsprache)
      {
      case Deutsch:
	fputs ("\nAchtung: Diese Variante ist nur bis zum "
	       "Jahr 2038 verwendbar!\n", d);
	break;

      case Englisch:
      default:
	fputs ("\nAttention: This version is only usable up to "
	       "the year 2038!\n", d);
	break;
      }

  putc ('\n', d);
}


static noreturn void
Hilfe (void)
{
  Version (stdout);

  switch (Systemsprache)
    {
    case Deutsch:
      puts ("  -h, --help, --Hilfe\t\tDiese Hilfe\n"
	    "  -V, --version, --Version\tVersion\n"
	    "  -c, --Konfiguration=<Datei>\tKonfiguration aus Datei lesen\n"
	    "  -U, --Socket=<Dateiname>\tlokaler Socket\n"
	    "  -4, --IPv4\t\t\tIPv4\n"
	    "  -6, --IPv6\t\t\tIPv6 (Vorgabe)\n"
	    "  -p, --Port=<Nr>\t\tPortnummer\n"
	    "      --Serverport=<Nr>\t\tnominaler Port\n"
	    "  -l, --localhost\t\tnur auf localhost hoeren\n"
	    "  -a, --Adresse=<IP-Adresse>\tan bestimmte Adresse binden\n"
	    "      --Datenschutz\t\tkeine IP-Adressen speichern\n"
	    "  -d, --Verzeichnis=<Verz.>\tWurzelverzeichnis [.]\n"
	    "      --virtuell\t\tNamensbasiertes Virtual Hosting\n"
	    "  -N, --Servername=<Name>\tServername\n"
	    "  -L, --log=<Datei>\t\tLogbuch in Datei schreiben\n"
	    "  -s, --still\t\t\tnichts ausgeben\n"
	    "  -v, --wortreich\t\tausfuehrlichere Ausgabe\n"
	    "  -g, --gzip\t\t\tnach komprimierten Dateien suchen\n"
	    "  -G, --cgi-Datei=<Datei>\tDatei ist CGI-Programm\n"
	    "  -C, --cgi\t\t\t*.cgi als CGI-Programme ausfuehren\n"
	    "  -e, --Erweiterung=<E=Aktion>\tAktion fuer Erweiterung festlegen\n"
	    "  -M, --cgi-umask\t\tumask fuer CGI-Programme\n"
	    "  -D, --daemon\t\t\tals Daemon im Hintergrund laufen\n"
	    "  -b, --beschreibbar\t\thochladen & loeschen erlauben\n"
	    "  -P, --Schreibschutz\t\thochladen & loeschen nur mit Zugangsdaten\n"
	    "      --Schutzbereich=<Verz.>\tVerzeichnis schuetzen\n"
	    "  -B, --Schutzbereichsblockierung\n"
	    "      --Zugang=<Name:Kennung>\tschwache Zugangsbeschraenkung\n"
	    "      --offen\t\t\tfuer das gesamte Internet offen!\n"
	    "  -u, --Nutzer=<Name>\t\tNutzerkennung setzen (nur fuer root)\n"
	    "  -H, --Kopf=<Kopf>\t\tHTTP-Kopfzeilen hinzufuegen\n"
	    "      --var=<VAR=Wert>\t\tzusaetzliche Metavariablen\n"
	    "      --Wartungsmodus\t\tausser Betrieb\n"
	    "  -T, --trace\t\t\tMethode TRACE erlauben\n"
	    "      --hostname\t\tHostnamen ermitteln\n"
	    "  -S, --Sprache=<Sprache>\tlokale System-Sprache\n"
	    "      --tmpdir=<Verz.>\t\tVerzeichnis fuer temporaere Daten (POST)\n");
      break;

    case Englisch:
    default:
      puts ("  -h, --help, --Hilfe\t\tthis help\n"
	    "  -V, --version, --Version\tversion\n"
	    "  -c, --config=<file>\t\tread configuration from file\n"
	    "  -U, --Socket=<file name>\tlocal socket\n"
	    "  -4, --IPv4\t\t\tIPv4\n"
	    "  -6, --IPv6\t\t\tIPv6 (default)\n"
	    "  -p, --Port=<nr>\t\tport number\n"
	    "      --Serverport=<nr>\t\tnominal Port\n"
	    "  -l, --localhost\t\tjust listen on localhost\n"
	    "  -a, --address=<address>\tbind to a specific address\n"
	    "      --privacy\t\t\tdon't store IP addresses\n"
	    "  -d, --directory=<dir>\t\troot directory [.]\n"
	    "      --virtual\t\t\tName-based Virtual Hosting\n"
	    "  -N, --Servername=<Name>\tServername\n"
	    "  -L, --log=<file>\t\twrite log to a file\n"
	    "  -s, --still\t\t\tbe silent\n"
	    "  -v, --verbose\t\t\tbe more verbose\n"
	    "  -g, --gzip\t\t\tsearch for compressed files\n"
	    "  -G, --cgi-file=<file>\t\tfile is a CGI program\n"
	    "  -C, --cgi\t\t\texecute *.cgi as CGI programs\n"
	    "  -e, --extension=<E=action>\tdefine action for extension\n"
	    "  -M, --cgi-umask\t\tumask for CGI programs\n"
	    "  -D, --daemon\t\t\trun as daemon in the background\n"
	    "  -b, --writable\t\tallow uploads and deletions\n"
	    "  -P, --writeprotected\t\tuploads and deletions only with access control\n"
	    "      --protected-zone=<dir>\tprotect directory\n"
	    "  -B, --block-protected-zone\n"
	    "      --account=<name:password>\tweak access control\n"
	    "      --open\t\t\topen to the whole Internet\n"
	    "  -u, --Nutzer=<name>\t\tset user (just for root)\n"
	    "  -H, --header=<header>\t\tadd HTTP header\n"
	    "      --var=<VAR=value>\t\tadd metavariable\n"
	    "      --servicemode\t\tout of order\n"
	    "  -T, --trace\t\t\tallow method TRACE\n"
	    "      --hostname\t\tget the hostname\n"
	    "  -S, --language=<lang>\t\tlocal system language\n"
	    "      --tmpdir=<dir>\t\tdirectory for temporary data (POST)\n");
      break;
    }

  puts ("Homepage: <" AKFNETZ_HOMEPAGE ">\n");

  exit (EXIT_SUCCESS);
}


static noreturn void
Fehler (int Fehlernr, const char *Ursache)
{
  fputs (Programmname, stderr);

  if (Ursache && *Ursache)
    fprintf (stderr, ": %s", Ursache);

  if (Fehlernr)
    fprintf (stderr, ": %s (%d)",
	     (Systemsprache == Deutsch)
	     ? Fehlermeldung (Fehlernr) : strerror (Fehlernr), Fehlernr);

  fputc ('\n', stderr);
  exit (EXIT_FAILURE);
}


static char *
Arbeitsverzeichnis (void)
{
  char s[10240];

  if (!getcwd (s, sizeof (s)))
    return NULL;

  return strdup (s);
}


// erstellt base64-kodierte Liste
// (diese Liste wird nie freigegeben)
static void
Zugangsdatenimport (const char *s)
{
  // Wenn ':' am Anfang, dann ist der Rest bereits kodiert
  if (*s == ':')
    {
      if (akfnetz_Listeneintrag (&Einstellung.Zugang, s + 1, 0))
	Fehler (errno, NULL);
    }
  else if (!strchr (s, ':'))
    Fehler (EINVAL, s);
  else				// Kodieren
    {
      size_t sl = strlen (s);
      size_t pl = akfnetz_base64_Platz (sl);
      char Puffer[pl];

      akfnetz_base64_kodiert (Puffer, pl, s, sl, AKFNETZ_BASE64);
      if (akfnetz_Listeneintrag (&Einstellung.Zugang, Puffer, pl - 1))
	Fehler (errno, NULL);
    }
}


// entferne Freiraum am Ende des Strings
static void
bereinige_Ende (char *s)
{
  if (!s || !*s)
    return;

  size_t l = strlen (s) - 1;
  while (l && (s[l] == ' ' || s[l] == '\t'))
    --l;

  s[l + 1] = '\0';
}


// nur fuer Konfigurationsdatei
static char *
Argumentbearbeitung (char *a)
{
  if (!a)
    return NULL;

  a = Textanfang (a);

  if (*a == '"')
    {
      ++a;
      char *p = strchr (a, '"');
      if (!p)
	Fehler (EINVAL, a);
      *p = '\0';
    }
  else
    bereinige_Ende (a);

  return a;
}


static void
Konfigurationsdatei (const char *Name)
{
  char Zeile[10240];
  FILE *d;
  char *sz;

  d = fopen (Name, "r");
  if (!d)
    Fehler (errno, Name);

  while (fgets (Zeile, sizeof (Zeile), d))
    {
      char *op = strtok_r (Zeile, "= \t\r\n", &sz);

      // Leerzeilen oder Kommentare ignorieren
      if (!op || *op == '#')
	continue;

      // Kurzoptionen
      if (!op[1])
	{
	  char *p = strchr (Kurzoptionen, *op);
	  char *Argument = Argumentbearbeitung (strtok_r (NULL, "\r\n", &sz));

	  if (!p
	      || (p[1] == ':' && !Argument)
	      || (p[1] != ':' && Argument) || !Option (*op, Argument))
	    Fehler (EINVAL, op);

	  continue;
	}

      // Langoptionen
      int i;
      for (i = 0; Langoptionen[i].name; ++i)
	{
	  const struct option *o = &Langoptionen[i];
	  int Token = o->val;

	  if (!strcmp (op, o->name))
	    {
	      char *Argument =
		Argumentbearbeitung (strtok_r (NULL, "\r\n", &sz));

	      if ((!Argument && o->has_arg == required_argument)
		  || (Argument && o->has_arg == no_argument))
		Fehler (EINVAL, op);

	      if (o->flag)
		{
		  *o->flag = Token;
		  Token = 0;
		}

	      if (!Option (Token, Argument))
		Fehler (EINVAL, op);

	      break;
	    }
	}

      if (!Langoptionen[i].name)
	Fehler (EINVAL, op);
    }

  fclose (d);
}


// nur druckbares ASCII, ohne Steuerzeichen
static inline bool
nur_ascii (const void *s)
{
  const signed char *t = s;
  // nicht-ASCII-Zeichen werden durch die Umwandlung negativ

  while (*t)
    if (*t++ < 0x20)
      return false;

  return true;
}


// Nutzerkennung aendern (Nummer oder Name)
// es werden hier nur die Variablen gesetzt
static uid_t
Nutzerkennung (const char *Name)
{
  if (!Name || !*Name)
    return Nutzer->pw_uid;

  char *e;
  uid_t u = (uid_t) strtoul (Name, &e, 10);

  Nutzer = *e ? getpwnam (Name) : getpwuid (u);

  if (!Nutzer)
    Fehler (EINVAL, Name);

  return Nutzer->pw_uid;
}


static void
Systemsprachenerkennung (void)
{
  char *l;

  Systemsprache = Englisch;

  if (!(l = getenv ("LC_ALL")) && !(l = getenv ("LC_MESSAGES"))
      && !(l = getenv ("LANG")))
    l = "C";

  if (!strncmp (l, "de", 2))
    Systemsprache = Deutsch;
  else if (!strncmp (l, "eo", 2))
    Systemsprache = Esperanto;

  // Nur unterstuetzte Sprachen erkennen!
}


// core dumps verhindern - die koennten Zugangsdaten enthalten
static void
Kernabwurfvermeidung (void)
{
  struct rlimit Limits;

  getrlimit (RLIMIT_CORE, &Limits);
  Limits.rlim_cur = 0;
  setrlimit (RLIMIT_CORE, &Limits);
}


static void
Initialisierung (char *argv0)
{
  Kernabwurfvermeidung ();

  // fuer Fehlermeldungen
  setlocale (LC_MESSAGES, "");
  setlocale (LC_CTYPE, "");
  Systemsprachenerkennung ();

  Programmname = strrchr (argv0, '/');
  if (Programmname)
    ++Programmname;
  else
    Programmname = argv0;

  memset (&Einstellung, 0, sizeof (Einstellung));

  uid_t uid = getuid ();
  if (uid == 0)
    uid = geteuid ();

  errno = EPERM;
  Nutzer = getpwuid (uid);
  if (!Nutzer)
    Fehler (errno, NULL);

  Einstellung.Protokoll = PF_INET6;
  Einstellung.Logbuch = stdout;
  Einstellung.Loglevel = 1;
  Einstellung.cgi = false;
  Einstellung.cgi_umask = 0022;
  Einstellung.Port = 0;
  Einstellung.Serverport = 0;
  Einstellung.gzip = false;
  Einstellung.virtuell = false;
  Einstellung.Trace = false;
  Einstellung.Signatur = true;

  char *t = getenv ("TMPDIR");
  if (t && *t)
    Einstellung.Tempverzeichnis = strdup (t);
}


static unsigned short int
Portnummer (char *Argument)
{
  long int v;

  v = strtol (Argument, NULL, 10);
  if (0 >= v || v > 0xFFFF)
    Fehler (EDOM, Argument);

  return (unsigned short int) v;
}


static void
Servername (const char *Name)
{
  // Zeichen verbieten, die verwirren koennten
  // Internationale Domainnamen muessen als ACE-Strings angegeben werden
  if (!nur_ascii (Name) || strpbrk (Name, " /@:?#[]\"<>"))
    Fehler (EINVAL, Name);

  free (Einstellung.Servername);
  Einstellung.Servername = strdup (Name);
}


static void
Logbuchoeffnung (const char *Name)
{
  FILE *d;

  d = fopen (Name, "a");

  if (!d)
    {
      int Fehlernr = errno;

      fprintf (stderr, "%s: %s: %s (%d)\n", Programmname, Name,
	       Fehlermeldung (Fehlernr), Fehlernr);

      return;
    }

  // Zeilenpufferung / schliessen bei Ausfuehrung anderer Programme
  setvbuf (d, NULL, _IOLBF, BUFSIZ);
  fcntl (fileno (d), F_SETFD, FD_CLOEXEC);

  Einstellung.Logbuch = d;
}


static int
Variablenkonflikt (const char *s)
{
  static const char *const var[] = {
    "HTTP_", "REMOTE_", "CONTENT_", "REDIRECT_",
    "SERVER_NAME=", "SERVER_PORT=", "SERVER_SOFTWARE=", "SERVER_PROTOCOL=",
    "GATEWAY_INTERFACE=", "SCRIPT_NAME=", "REQUEST_METHOD=", "REQUEST_URI=",
    "QUERY_STRING=", "AUTH_TYPE=", "PATH_INFO=", "PATH_TRANSLATED=",
    "CONTENT_LENGTH=", "CONTENT_TYPE=", "DOCUMENT_ROOT=", NULL
  };

  if (!s || !strchr (s, '='))
    return -1;

  // Das ist nicht effizient, aber auch nicht zeitkritisch
  const char *b;
  for (int i = 0; (b = var[i]); ++i)
    if (!strncmp (s, b, strlen (b)))
      return -1;

  return 0;
}


static bool
Option (int Kurzoption, char *Argument)
{
  size_t l;
  char *e;

  // umgehe Problem mit statischer Analyse
  if (!Argument)
    Argument = "";

  switch (Kurzoption)
    {
    case 'h':			// help
      Hilfe ();
      break;

    case 'V':			// version
      Version (stdout);
      exit (EXIT_SUCCESS);
      break;

    case 'c':			// config, Konfiguration
      Konfigurationsdatei (Argument);
      break;

    case 'N':			// Servername, servername
      Servername (Argument);
      break;

    case 'U':			// socket, Socket
      if (*Argument != '/')
	Fehler (EINVAL, Argument);

      Einstellung.Protokoll = PF_UNIX;
      free (Socketname);
      Socketname = strdup (Argument);
      break;

    case '4':			// IPv4
      Einstellung.Protokoll = PF_INET;
      break;

    case '6':			// IPv6
      Einstellung.Protokoll = PF_INET6;
      break;

    case 'l':			// localhost
      strcpy (Einstellung.Serveradresse, "127.0.0.1");
      break;

    case 'a':			// Adresse, address
      l = strlen (Argument) + 1;
      if (l > sizeof (Einstellung.Serveradresse))
	Fehler (EFAULT, Argument);

      memcpy (Einstellung.Serveradresse, Argument, l);
      break;

    case 'p':			// Port, port
      Einstellung.Port = Portnummer (Argument);
      break;

    case OPT_SERVERPORT:	// Serverport, serverport
      Einstellung.Serverport = Portnummer (Argument);
      break;

    case 'd':			// Verzeichnis, directory
      // access benutzt reale Identitaet, chdir die effektive
      if (access (Argument, X_OK) < 0 || chdir (Argument) < 0)
	Fehler (errno, Argument);
      break;

    case 'C':			// cgi
      if (!NUTZE_CGI)
	Fehler (ENOSYS, "CGI");

      Einstellung.cgi = true;
      break;

    case 'G':			// cgi-Datei
      if (!NUTZE_CGI)
	Fehler (ENOSYS, "CGI");

      if (akfnetz_Listeneintrag (&Einstellung.cgiDatei, Argument, 0))
	Fehler (errno, NULL);
      break;

    case 'M':			// cgi-umask
      if (!NUTZE_CGI)
	Fehler (ENOSYS, "CGI");

      Einstellung.cgi_umask = (mode_t) strtoul (Argument, &e, 8);

      if (*e || Einstellung.cgi_umask > 0777)
	Fehler (EDOM, Argument);
      break;

    case 'e':			// Erweiterung, extension
      if (!NUTZE_CGI)
	Fehler (ENOSYS, "CGI");

      if (*Argument == '=' || *Argument == '.' || !strchr (Argument, '='))
	Fehler (EINVAL, Argument);

      if (akfnetz_Listeneintrag (&Einstellung.Erweiterungen, Argument, 0))
	Fehler (errno, NULL);
      break;

    case 's':			// still
      Einstellung.Loglevel = 0;
      break;

    case 'v':			// wortreich, verbose
      Einstellung.Loglevel = 2;
      break;

    case 'g':			// gzip
      Einstellung.gzip = true;
      break;

    case OPT_DATENSCHUTZ:
      Einstellung.Datenschutz = true;
      break;

    case 'L':			// log
      // es kann nur eine Logdatei geben
      if (fileno (Einstellung.Logbuch) > STDERR_FILENO)
	Fehler (EINVAL, Argument);

      Logbuchoeffnung (Argument);
      break;

    case OPT_ZUGANG:		// Zugang, account
      if (!nur_ascii (Argument))
	Fehler (EINVAL, Argument);
      Zugangsdatenimport (Argument);
      // kodierte Daten werden ungeprueft uebernommen
      break;

    case OPT_SCHUTZB:
      if (Einstellung.Schutzbereich || *Argument != '/')
	Fehler (EINVAL, Argument);

      l = strlen (Argument);
      Einstellung.Schutzbereich = malloc (l + 1);
      if (!Einstellung.Schutzbereich)
	Fehler (ENOMEM, NULL);
      memcpy (Einstellung.Schutzbereich, Argument, l + 1);
      Einstellung.Schutzbereichslaenge = l;
      break;

    case 'B':
      Einstellung.Schutzbereichsblockierung = true;
      break;

    case 'u':			// Nutzer
      // nur root als reale uid darf die uid aendern
      if (getuid () != 0)
	Fehler (EPERM, Argument);

      Nutzerkennung (Argument);
      break;

    case 'D':			// daemon
      Daemon = true;
      break;

    case OPT_VIRTUELL:		// virtuell, virtual
      Einstellung.virtuell = true;
      break;

    case 'b':			// beschreibbar
      Einstellung.beschreibbar = true;
      break;

    case 'P':			// Schreibschutz, writeprotection
      Einstellung.Schreibschutz = true;
      Einstellung.beschreibbar = true;
      break;

    case 'H':			// Header, Kopf
      if (!strchr (Argument, ':'))
	Fehler (EINVAL, Argument);

      if (akfnetz_Listeneintrag (&Einstellung.Kopfzeilen, Argument, 0))
	Fehler (errno, NULL);
      break;

    case OPT_VAR:		// var
      if (!NUTZE_CGI)
	Fehler (ENOSYS, "CGI");

      if (Variablenkonflikt (Argument)
	  || akfnetz_Listeneintrag (&Einstellung.cgiMeta, Argument, 0))
	Fehler (EINVAL, Argument);
      break;

    case OPT_OFFEN:		// offen, open
      Einstellung.offen = true;
      break;

    case OPT_WARTUNG:		// Wartungsmodus, servicemode
      Einstellung.Wartungsmodus = true;
      Einstellung.Trace = true;
      break;

    case 'T':			// trace
      Einstellung.Trace = true;
      break;

    case OPT_HOSTNAME:		// hostname
      Einstellung.Hostname = true;
      break;

    case 'i':			// inetd
      inetd = true;
      // stderr harmlos machen
      {
	int n = open ("/dev/null", O_WRONLY | O_NOCTTY);
	if (n >= 0)
	  {
	    dup2 (n, STDERR_FILENO);
	    close (n);
	  }
      }
      break;

    case 'S':			// Sprache, language
      if (!strcasecmp ("en", Argument))
	Systemsprache = Englisch;
      else if (!strcasecmp ("de", Argument))
	Systemsprache = Deutsch;
      else if (!strcasecmp ("eo", Argument))
	Systemsprache = Esperanto;
      else
	Fehler (EINVAL, Argument);
      break;

    case OPT_TMPDIR:
      free (Einstellung.Tempverzeichnis);
      Einstellung.Tempverzeichnis = strdup (Argument);
      break;

    case OPT_ONION:		// onion
      if ((!Praefix (Argument, "http://") && !Praefix (Argument, "https://"))
	  || strcasecmp (".onion", Argument + strlen (Argument) - 6))
	Fehler (EINVAL, Argument);

      free (Einstellung.Onion_Location);
      Einstellung.Onion_Location = strdup (Argument);
      break;

    case 'Q':
      Einstellung.Signatur = false;
      break;

    case 0:
      break;

    case '?':
    case ':':
    default:
      return false;
    }

  return true;
}


static void
Optionen (int argc, char **argv)
{
  optind = 1;
  opterr = 0;
  int opt;
  while ((opt = getopt_long (argc, argv, Kurzoptionen,
			     Langoptionen, NULL)) != -1)
    {
      if (!Option (opt, optarg))
	Fehler (EINVAL, argv[optind - 1]);
    }

  if (optind < argc)
    Fehler (EINVAL, argv[optind]);

  if (Einstellung.Schreibschutz && !Einstellung.Zugang)
    Fehler (EPERM, Deutsch == Systemsprache
	    ? "keine Zugangsdaten angegeben"
	    : Esperanto == Systemsprache
	    ? "neniuj akreditajhoj difinitaj" : "no credentials defined");
}


static void
Einrichtung (void)
{
  // 0 ist der Superuser - der darf zu viel
  // Die eigentliche Rechteaufgabe erfolgt aber erst spaeter
  if (Nutzer->pw_uid == 0 && Nutzerkennung (getenv ("LOGNAME")) == 0
      && Nutzerkennung (getenv ("SUDO_UID")) == 0
      && Nutzerkennung (getenv ("SU_FROM")) == 0
      && Nutzerkennung ("nobody") == 0)
    Fehler (EPERM, Deutsch == Systemsprache
	    ? "mit root-Rechten laufen"
	    : Esperanto == Systemsprache
	    ? "kuri kun chefuzanto (root) privilegioj"
	    : "run with root privileges");

  if (Einstellung.Datenschutz)
    Einstellung.Hostname = false;

  if (Einstellung.Onion_Location && Einstellung.virtuell)
    Fehler (EPERM, "Onion");

  if (Einstellung.Schreibschutz && Einstellung.Schutzbereich)
    Fehler (EPERM, Einstellung.Schutzbereich);

  if (!Einstellung.Servername)
    {
      char *h;

      h = getenv ("HOSTNAME");
      if (!h)
	h = getenv ("HOST");

      if (h && *h)
	Einstellung.Servername = h;
      else
	{
	  char Hostname[1024];
	  if (!gethostname (Hostname, sizeof (Hostname)))
	    Einstellung.Servername = strdup (Hostname);
	}
    }

  Einstellung.Wurzel = Arbeitsverzeichnis ();
  if (!Einstellung.Wurzel)
    Fehler (ENAMETOOLONG, Deutsch == Systemsprache
	    ? "Arbeitsverzeichnis"
	    : Esperanto == Systemsprache
	    ? "labordosierujo" : "working directory");

  Einstellung.Wurzellaenge = strlen (Einstellung.Wurzel);

  // an IPv4-Adresse binden? -> auf IPv4 einschraenken
  if (PF_INET6 == Einstellung.Protokoll && *Einstellung.Serveradresse
      && !strchr (Einstellung.Serveradresse, ':'))
    Einstellung.Protokoll = PF_INET;

  if (!Einstellung.Port && !inetd && Einstellung.Protokoll != PF_UNIX)
    Einstellung.Port = (geteuid () == 0) ? 80u : 8080u;

  if (!Einstellung.Serverport)
    Einstellung.Serverport = Einstellung.Port;

  Signaleinrichtung ();

  // eventuell Loglevel erhoehen, wenn Logbuch in Datei geschrieben wird
  if (Einstellung.Loglevel <= 0
      && fileno (Einstellung.Logbuch) > STDERR_FILENO)
    Einstellung.Loglevel = 1;
}


static void
Signalverarbeitung (int signum)
{
  switch (signum)
    {
    case SIGUSR1:
    case SIGUSR2:
      Einstellung.Wartungsmodus = (signum == SIGUSR1);
      break;

    case SIGCHLD:
      {
	int e = errno;
	while (waitpid (-1, NULL, WNOHANG) > 0);
	errno = e;
      }
      break;
    }
}


static void
Signaleinrichtung (void)
{
  struct sigaction sa;

  // Signal zum Abholen der Kinder etablieren
  // Das holt auch die CGI-Prozesse ab
  sa.sa_handler = &Signalverarbeitung;
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART;
  sigaction (SIGCHLD, &sa, NULL);
  sigaction (SIGUSR1, &sa, NULL);
  sigaction (SIGUSR2, &sa, NULL);
}


// wird nur bei Bedarf eingerichtet
static void
Abbruchsignal (int signum)
{
  if (Socketname)
    unlink (Socketname);

  signal (signum, SIG_DFL);
  raise (signum);
}


// wird nur bei Bedarf eingerichtet
static void
Aufraeumauftrag (void)
{
  struct sigaction sa;

  sa.sa_handler = &Abbruchsignal;
  sa.sa_flags = 0;
  sigfillset (&sa.sa_mask);

  sigaction (SIGINT, &sa, NULL);
  sigaction (SIGTERM, &sa, NULL);
  sigaction (SIGHUP, &sa, NULL);
  sigaction (SIGALRM, &sa, NULL);
}


static int
Gruppenzuweisung (const char *Name, gid_t Gruppe)
{
  size_t Anzahl = 0;
  gid_t Liste[256];

  Liste[Anzahl++] = Gruppe;

  setgrent ();

  while (Anzahl < 256)
    {
      struct group *g;

      g = getgrent ();
      if (!g)
	break;

      if (g->gr_gid == Gruppe)
	continue;

      // Namensliste durchsuchen
      for (char **nl = g->gr_mem; *nl; ++nl)
	if (!strcmp (*nl, Name))
	  {
	    Liste[Anzahl++] = g->gr_gid;
	    break;
	  }
    }

  endgrent ();

  return setgroups (Anzahl, Liste);
}


// eventuell root-Rechte ab jetzt aufgeben
static void
Rechteaufgabe (void)
{
  uid_t euid;

  euid = geteuid ();

  // unprivilegiert per setuid, aber dann als root aufgerufen?
  // => erst Privilegien erlangen, um sie dann richtig abweisen zu koennen.
  if (euid != 0 && getuid () == 0)
    {
      if (seteuid (0) < 0)
	Fehler (errno, "seteuid");
      euid = geteuid ();
    }

  if (euid != 0)
    return;

  // die uid zum Schluss, sonst hat man nicht mehr die noetigen Rechte
  if (Nutzer->pw_uid == 0
      || setgid (Nutzer->pw_gid) < 0
      || Gruppenzuweisung (Nutzer->pw_name, Nutzer->pw_gid) < 0
      || setuid (Nutzer->pw_uid) < 0 || (euid = geteuid ()) == 0)
    Fehler (errno,
	    Deutsch == Systemsprache ? "Rechte setzen"
	    : Esperanto == Systemsprache
	    ? "shanghanta uzanton" : "changing user");

  // Hiernach sollte das Zurueckerlangen von root-Rechten unmoeglich sein.

  if (euid != Nutzer->pw_uid)
    {
      errno = EPERM;
      Nutzer = getpwuid (euid);
      if (!Nutzer)
	Fehler (errno, "getpwuid");
    }
}


static void
Statusinformationen (FILE *d)
{
  Version (d);

  if (*Einstellung.Serveradresse)
    fprintf (d, Deutsch == Systemsprache ? "Bind.:\t%s" : "Bound:\t%s",
	     Einstellung.Serveradresse);
  else if (Einstellung.offen)
    fputs (Deutsch == Systemsprache ? "OFFEN!" : "OPEN!", d);
  else
    fputs (Deutsch == Systemsprache ? "gefiltert" : "filtered", d);

  fputc ('\n', d);

  fputs ("Proto:\t", d);
  switch (Einstellung.Protokoll)
    {
    case PF_INET6:
      fputs ("IPv6\n", d);
      break;

    case PF_INET:
      fputs ("IPv4\n", d);
      break;

    case PF_UNIX:
      fprintf (d, "Socket: %s\n", Socketname);
      break;
    }

  fprintf (d, "PID:\t%ld\n", (long) getpid ());
  fputs (Deutsch == Systemsprache ? "Nutzer" :
	 Esperanto == Systemsprache ? "Uzanto" : "User", d);
  fprintf (d, ":\t%s (%d)\n\n", Nutzer->pw_name, (int) Nutzer->pw_uid);

  if (Einstellung.Servername)
    fprintf (d, Esperanto == Systemsprache
	     ? "Servilo:%s\n" : "Server:\t%s\n", Einstellung.Servername);

  if (Einstellung.Protokoll == PF_INET || Einstellung.Protokoll == PF_INET6)
    {
      fprintf (d, Esperanto == Systemsprache
	       ? "Pordo:\t%u\n" : "Port:\t%u\n",
	       (unsigned int) Einstellung.Port);

      fputs ("\nURL:\thttp://", d);
      if (*Einstellung.Serveradresse)
	{
	  if (strchr (Einstellung.Serveradresse, ':'))
	    fprintf (d, "[%s]", Einstellung.Serveradresse);
	  else
	    fputs (Einstellung.Serveradresse, d);
	}
      else
	fputs (Einstellung.Servername, d);

      if (Einstellung.Port != 80)
	fprintf (d, ":%u", (unsigned int) Einstellung.Port);

      fputs ("\n\n", d);
    }
}


static noreturn void
Servermodus (void)
{
  int Lauscher;
  int Wert;

  Lauscher = socket (Einstellung.Protokoll, SOCK_STREAM, 0);

  // Kein IPv6 => IPv4 versuchen
  if (Lauscher < 0 && PF_INET6 == Einstellung.Protokoll
      && (errno == EPFNOSUPPORT || errno == EAFNOSUPPORT))
    {
      Einstellung.Protokoll = PF_INET;
      Lauscher = socket (Einstellung.Protokoll, SOCK_STREAM, 0);
    }

  if (Lauscher < 0)
    Fehler (errno, "socket");

  fcntl (Lauscher, F_SETFD, FD_CLOEXEC);

  Wert = true;
  setsockopt (Lauscher, SOL_SOCKET, SO_REUSEADDR, &Wert, sizeof (Wert));

  // Namen zuweisen
  switch (Einstellung.Protokoll)
    {
    case PF_INET6:
      if (akfnetz_IPv6_Bindung (Lauscher,
				Einstellung.Serveradresse,
				Einstellung.Port) < 0)
	Fehler (errno, Einstellung.Serveradresse);
      break;

    case PF_INET:
      if (akfnetz_IPv4_Bindung (Lauscher,
				Einstellung.Serveradresse,
				Einstellung.Port) < 0)
	Fehler (errno, Einstellung.Serveradresse);
      break;

    case PF_UNIX:
      // Socket darf nicht mit root-Rechten angelegt werden!
      Rechteaufgabe ();

      unlink (Socketname);
      if (akfnetz_Lokalbindung (Lauscher, Socketname) < 0)
	Fehler (errno, Socketname);

      // Socketname bei Abbruch loeschen
      Aufraeumauftrag ();
      break;

    default:
      Fehler (EPFNOSUPPORT, NULL);
    }

  Rechteaufgabe ();

  if (Einstellung.Protokoll != PF_INET6 && Einstellung.Protokoll != PF_INET)
    {
      *Einstellung.Serveradresse = '\0';
      Einstellung.Port = 0;	// reservierter Port
    }

  if (listen (Lauscher, LISTEN_BACKLOG) < 0)
    Fehler (errno, "listen");

  if (Daemon)
    {
      if (akfnetz_daemonisieren () < 0)
	Fehler (errno, "daemon");
      if (fileno (Einstellung.Logbuch) <= STDERR_FILENO)
	Einstellung.Loglevel = 0;
    }
  else if (redsam (1))
    Statusinformationen (stderr);

  // kehrt nie zurueck
  akfnetz_Verbindungsannahme (Lauscher);
  abort ();
}
