/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _POSIX_C_SOURCE 200112L
#define _XOPEN_SOURCE 500

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stdnoreturn.h>
#include <errno.h>
#include <locale.h>
#include <getopt.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>

#include "akfnetz.h"
#include "httpclient.h"
#include "version.h"

#define Programmname "akfweb-dl"
#define KOPFMAXIMUM 256		// Maximale Anzahl der Kopfzeilen +1

#define OPT_SOCKET 1000
#define OPT_GZIP 1001

static void Optionen (struct akfnetz_Clientdaten *, int, char **);
static inline bool Verzeichnisadresse (const char *);
static noreturn void Fehler (int, const char *);


int
main (int argc, char **argv)
{
  struct akfnetz_Clientdaten vb;

  setlocale (LC_MESSAGES, "");
  setlocale (LC_CTYPE, "");

  akfnetz_Clientinitialisierung (&vb);
  Optionen (&vb, argc, argv);

  // Schreibfehler zum Server sollen nicht gleich das Programm abbrechen
  signal (SIGPIPE, SIG_IGN);

  int r = akfnetz_Netzaufruf (&vb, argv + optind, argc - optind - 1);
  if (r < 0)
    Fehler (errno, NULL);

  if (r >= 300)
    return EXIT_FAILURE;

  return EXIT_SUCCESS;
}


static inline void
Version (FILE * d)
{
  if (deutsch)
    fputs (Programmname " (Deutsch) (akfnetz " AKFNETZ_VERSION ")\n"
	   AKFNETZ_COPYRIGHT "\nLizenz GPLv3+: "
	   "GNU GPL Version 3 oder neuer "
	   "<http://gnu.org/licenses/gpl.html>\n\n"
	   "Dies ist freie Software. "
	   "Sie darf veraendert und verteilt werden.\n"
	   "Dieses Programm wird ohne Gewaehrleistung geliefert,\n"
	   "soweit dies gesetzlich zulaessig ist.\n\n"
	   "erstellt: " __DATE__ "\n", d);
  else
    fputs (Programmname " (English) (akfnetz " AKFNETZ_VERSION ")\n"
	   AKFNETZ_COPYRIGHT "\nLicense GPLv3+: "
	   "GNU GPL version 3 or later "
	   "<http://gnu.org/licenses/gpl.html>\n\n"
	   "This is free software; you are free to change "
	   "and redistribute it.\n"
	   "There is NO WARRANTY, to the extent permitted by law.\n\n"
	   "build: " __DATE__ "\n", d);
}


static noreturn void
Hilfe (void)
{
  Version (stdout);
  if (deutsch)
    puts ("Verwendung: " Programmname
	  " [Optionen] [Datei ...] <URL>\n\n"
	  "  -h, --help, --Hilfe\t\tDiese Hilfe\n"
	  "  -V, --version, --Version\tVersion\n"
	  "  -c, --stdout\t\t\tAusgabe ueber Standardausgabe\n"
	  "  -o, --output=<Name>\n"
	  "      --Ausgabe=<Name>\n"
	  "      --Datei=<Name>\t\tDateiname fuer Ausgabe\n"
	  "      --gzip\t\t\takzeptiere komprimierte Dateien\n"
	  "  -a, --aktualisiere=<Name>\t"
	  "aktualisiere die Datei, falls noetig\n"
	  "  -S\t\t\t\tneue Dateien sind nur fuer den Benutzer nutzbar\n"
	  "  -u, --upload, --cp, --senden\tDateien auf den Server senden,"
	  " falls moeglich\n"
	  "  -L, --delete\n"
	  "      --entfernen\t\tAdresse unzugaenglich machen,"
	  " falls moeglich\n"
	  "  -i, --info\t\t\tNur Infos ueber die Adresse einholen\n"
	  "  -I, --head\t\t\tHTTP-Kopf ungefiltert anzeigen\n"
	  "  -O, --options\n"
	  "      --Optionen\t\tOptionen fuer die Adresse anzeigen\n"
	  "  -T, --trace\t\t\tGibt Trace auf Standardausgabe aus\n"
	  "  -l, --links, --Links\t\textrahiehre Links aus (X)HTML\n"
	  "  -s, --still\t\t\tNur das Noetigste ausgeben\n"
	  "  -U, --nicht-umleiten\t\tUmleitungen als Fehler behandeln\n"
	  "  -M, --nicht-interaktiv\t"
	  "nicht interaktiv nach Zugangsdaten fragen\n"
	  "  -N, --nohup\t\t\tSIGHUP ignorieren\n"
	  "  -Z, --Zugang=<Name:Kennung>\t"
	  "Zugangsdaten mitschicken (Basic)\n"
	  "  -P, --proxy=<URL>\n"
	  "      --Proxy=<URL>\t\tProxy verwenden\n"
	  "  -t, --tor\t\t\tTor verwenden\n"
	  "  -r, --referrer=<URL>\t\tVerweisende URL\n"
	  "  -A, --user-agent=<Name>\tUser-Agent angeben\n"
	  "  -m, --mail=<E-Mail-Adresse>\tKontakt-Adresse mitschicken\n"
	  "      --socket=<Pfad>\t\tVerbindung per lokalem Socket\n"
	  "\nUmgebungsvariablen:\n"
	  " - http_proxy\n - AKFUSERAGENT\n - AKFAGENTMAIL\n"
	  " - HTTPSOCKET\n");
  else
    puts ("Usage: " Programmname
	  " [Options] [file [...]] <URL>\n\n"
	  "  -h, --help, --Hilfe\t\tthis help\n"
	  "  -V, --version, --Version\tversion\n"
	  "  -c, --stdout\t\t\tuse standard output\n"
	  "  -o, --output=<name>\n"
	  "      --Datei=<name>\t\tfile name for output\n"
	  "      --gzip\t\t\taccept compressed files\n"
	  "  -a, --aktualisiere=<Name>\t"
	  "update the file, if necessary\n"
	  "  -S\t\t\t\tnew files are just usable for the user\n"
	  "  -u, --upload, --cp, --senden\tsend files to the server,"
	  " if supported\n"
	  "  -L, --delete\n"
	  "      --entfernen\t\tremove the address, if supported\n"
	  "  -i, --info\t\t\tget infos about the address\n"
	  "  -I, --head\t\t\tshow the whole HTTP head\n"
	  "  -O, --options\n"
	  "      --Optionen\t\tshow options for the address\n"
	  "  -T, --trace\t\t\ttrace to standard output\n"
	  "  -l, --links, --Links\t\textract links from (X)HTML\n"
	  "  -s, --still\t\t\tbe silent\n"
	  "  -U, --nicht-umleiten\t\ttreat redirects as errors\n"
	  "  -M, --nicht-interaktiv\t"
	  "don't ask for credentials interactively\n"
	  "  -N, --nohup\t\t\tignore SIGHUP\n"
	  "  -Z, --Zugang=<Name:Kennung>\t"
	  "send credetials (Basic)\n"
	  "  -P, --proxy=<URL>\n"
	  "      --Proxy=<URL>\t\tuse the proxy\n"
	  "  -t, --tor\t\t\tuse Tor\n"
	  "  -r, --referrer=<URL>\t\treferrer URL\n"
	  "  -A, --user-agent=<Name>\tUser-Agent\n"
	  "  -m, --mail=<e-mail address>\tsend contact address\n"
	  "      --socket=<Path>\t\tconnect via local socket\n"
	  "\nEnvironment variables:\n"
	  " - http_proxy\n - AKFUSERAGENT\n - AKFAGENTMAIL\n"
	  " - HTTPSOCKET\n");

  puts ("Homepage: <" AKFNETZ_HOMEPAGE ">\n");

  exit (EXIT_SUCCESS);
}


static noreturn void
Fehler (int Fehlernr, const char *Ursache)
{
  fputs (Programmname, stderr);

  if (Ursache && *Ursache)
    fprintf (stderr, ": %s", Ursache);

  if (Fehlernr)
    fprintf (stderr, ": %s (%d)", Fehlermeldung (Fehlernr), Fehlernr);

  fputs ("\a\n", stderr);

  exit (EXIT_FAILURE);
}


static void
Optionen (struct akfnetz_Clientdaten *vb, int argc, char **argv)
{
  static const struct option Langoption[] = {
    {"Hilfe", no_argument, NULL, 'h'},
    {"help", no_argument, NULL, 'h'},
    {"Version", no_argument, NULL, 'V'},
    {"version", no_argument, NULL, 'V'},
    {"Zugang", required_argument, NULL, 'Z'},
    {"nicht-interaktiv", no_argument, NULL, 'M'},
    {"nicht-umleiten", no_argument, NULL, 'U'},
    {"info", no_argument, NULL, 'i'},
    {"head", no_argument, NULL, 'I'},
    {"still", no_argument, NULL, 's'},
    {"stdout", no_argument, NULL, 'c'},
    {"output", required_argument, NULL, 'o'},
    {"Ausgabe", required_argument, NULL, 'o'},
    {"Datei", required_argument, NULL, 'o'},
    {"aktualisiere", required_argument, NULL, 'a'},
    {"proxy", required_argument, NULL, 'P'},
    {"Proxy", required_argument, NULL, 'P'},
    {"referrer", required_argument, NULL, 'r'},
    {"referer", required_argument, NULL, 'r'},	// bekannter Schreibfehler
    {"user-agent", required_argument, NULL, 'A'},
    {"mail", required_argument, NULL, 'm'},
    {"trace", no_argument, NULL, 'T'},
    {"Optionen", no_argument, NULL, 'O'},
    {"options", no_argument, NULL, 'O'},
    {"entfernen", no_argument, NULL, 'L'},
    {"loeschen", no_argument, NULL, 'L'},
    {"delete", no_argument, NULL, 'L'},
    {"senden", no_argument, NULL, 'u'},
    {"upload", no_argument, NULL, 'u'},
    {"cp", no_argument, NULL, 'u'},
    {"Links", no_argument, NULL, 'l'},
    {"links", no_argument, NULL, 'l'},
    {"secure", no_argument, NULL, 'S'},
    {"sicher", no_argument, NULL, 'S'},
    {"socket", required_argument, NULL, OPT_SOCKET},
    {"Socket", required_argument, NULL, OPT_SOCKET},
    {"unix-socket", required_argument, NULL, OPT_SOCKET},
    {"IPv4", no_argument, NULL, '4'},
    {"ipv4", no_argument, NULL, '4'},
    {"IPv6", no_argument, NULL, '6'},
    {"ipv6", no_argument, NULL, '6'},
    {"gzip", no_argument, NULL, OPT_GZIP},
    {"nohup", no_argument, NULL, 'N'},
    {"tor", no_argument, NULL, 't'},
    {0, 0, 0, 0}
  };

  opterr = 0;
  int opt;
  while ((opt = getopt_long (argc, argv, "46hVcstiIlLMNSTOUuP:o:Z:A:m:r:a:",
			     Langoption, NULL)) != -1)
    {
      switch (opt)
	{
	case 'h':		// --help
	  Hilfe ();
	  break;

	case 'V':		// --version
	  Version (stdout);
	  exit (EXIT_SUCCESS);
	  break;

	case '4':
	  vb->Adressfamilie = AF_INET;
	  break;

	case '6':
	  vb->Adressfamilie = AF_INET6;
	  break;

	case 'c':		// --stdout
	  vb->Ausgabename = "-";
	  break;

	case 'Z':		// --Zugang
	  free (vb->Zugang);
	  vb->Zugang = strdup (optarg);
	  vb->nicht_interaktiv = true;
	  break;

	case 'M':		// --nicht-interaktiv
	  vb->nicht_interaktiv = true;
	  break;

	case 'U':		// --nicht-umleiten
	  vb->nicht_umleiten = true;
	  break;

	case 'i':		// --info
	  vb->Methode = HEAD;
	  vb->nicht_umleiten = true;
	  break;

	case 'I':		// --head
	  vb->Methode = HEAD;
	  vb->nicht_umleiten = true;
	  vb->ungefiltert = true;
	  break;

	case 's':		// --still
	  vb->still = true;
	  break;

	case 't':		// --tor
	  vb->HTTP_Proxy = "socks4a://127.0.0.1:9050/";
	  break;

	case 'o':		// --Datei, --output
	  // Argumente bleiben erhalten
	  vb->Ausgabename = optarg;
	  break;

	case 'a':		// --aktualisiere
	  vb->Ausgabename = optarg;
	  vb->aktualisiere = true;
	  break;

	case 'P':		// --Proxy, --proxy
	  vb->HTTP_Proxy = optarg;
	  if (vb->HTTP_Proxy && !*vb->HTTP_Proxy)
	    vb->HTTP_Proxy = NULL;
	  break;

	case 'r':		// --referrer, --referer
	  vb->Referrer = optarg;
	  break;

	case 'A':		// --user-agent
	  vb->Agent = optarg;
	  break;

	case 'm':		// --mail
	  vb->Mail = optarg;
	  break;

	case 'O':		// --Optionen, --options
	  vb->Methode = OPTIONS;
	  vb->ungefiltert = true;
	  break;

	case 'T':		// --trace
	  vb->Methode = TRACE;
	  vb->Ausgabename = "-";
	  vb->nicht_interaktiv = true;
	  vb->still = true;
	  break;

	case 'L':		// --entfernen, --loeschen, --delete
	  vb->Methode = DELETE;
	  vb->Ausgabename = "-";
	  break;

	case 'u':		// --senden, --upload
	  vb->Methode = PUT;
	  vb->Ausgabename = "-";
	  break;

	case 'l':		// --links, --Links
	  vb->Links = true;
	  vb->Ausgabename = "-";
	  break;

	case 'S':		// --sicher, --secure
	  umask (0077);
	  break;

	case OPT_SOCKET:	// --socket, --Socket
	  vb->Socketpfad = optarg;
	  break;

	case OPT_GZIP:		// --gzip
	  vb->gzip = true;
	  break;

	case 'N':		// --nohup
	  vb->still = true;
	  vb->nicht_interaktiv = true;
	  signal (SIGHUP, SIG_IGN);
	  break;

	case '?':
	case ':':
	default:
	  Fehler (EINVAL, argv[optind - 1]);
	  break;
	}
    }

  // eine URL muss angegeben sein
  if (argc < optind + 1)
    Fehler (EDESTADDRREQ, NULL);

  // letztes Argument ist die Adresse
  if (akfnetz_neue_Adresse (vb, argv[argc - 1]) < 0)
    Fehler (EINVAL, argv[argc - 1]);

  if (PUT == vb->Methode)
    {
      if (optind == argc - 1)
	Fehler (0, deutsch ?
		"Es muss mindestens ein Dateiname angegeben werden." :
		"There must be at least one filename.");

      if (argc - optind > 2 && !Verzeichnisadresse (vb->Adresse))
	Fehler (0, deutsch ?
		"Bei mehreren Dateien muss die URL mit einem \"/\" enden." :
		"For more than one file the URL must end on \"/\",");
    }

  // sind Zugangsdaten in der URL?
  char Z[1024];
  if (akfnetz_URL_Zugang (Z, sizeof (Z), vb->Adresse))
    {
      free (vb->Zugang);
      vb->Zugang = strdup (akfnetz_url_dekodieren (Z));
      vb->nicht_interaktiv = true;
    }
}


// endet die Adresse mit einem Schraegstrich?
static inline bool
Verzeichnisadresse (const char *Adresse)
{
  return (Adresse[strlen (Adresse) - 1] == '/');
}
