/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfscraper - Webscraper
 * Copyright (c) 2019-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <locale.h>
#include <errno.h>
#include "akfnetz.h"
#include "version.h"

#ifndef Fehlermeldung
#define Fehlermeldung  strerror
#endif

#define OPTIONEN "ghVleufnL:o:c:"

static const struct option Langoption[] = {
  {"Hilfe", no_argument, NULL, 'h'},
  {"help", no_argument, NULL, 'h'},
  {"Version", no_argument, NULL, 'V'},
  {"version", no_argument, NULL, 'V'},
  {"charset", required_argument, NULL, 'c'},
  {"Zeichensatz", required_argument, NULL, 'c'},
  {"output", required_argument, NULL, 'o'},
  {"Ausgabe", required_argument, NULL, 'o'},
  {"Datei", required_argument, NULL, 'o'},
  {"Links", no_argument, NULL, 'l'},
  {"links", no_argument, NULL, 'l'},
  {"extended", no_argument, NULL, 'e'},
  {"erweitert", no_argument, NULL, 'e'},
  {"force", no_argument, NULL, 'f'},
  {"forciere", no_argument, NULL, 'f'},
  {0, 0, 0, 0}
};


static int
Hilfe (const char *n)
{
  printf ("%s [Optionen] [HTML-Dateien]\n\n", n);
  puts ("Optionen\n"
	"  -u\t\tEingabe ist UTF-8 kodiert\n"
	"  -c ZS\t\tEingabe-Zeichensatz\n"
	"  -l\t\tExterne Links als Referenzen\n"
	"  -L #\t\tignoriere # Links am Anfang\n"
	"  -g\t\tGrafik-Links\n"
	"  -n\t\tkeine automatischen Zeilenumbrueche\n"
	"  -e\t\terweitert: SGR-Steuersequenzen verwenden\n"
	"  -o Datei\tAusgabe in Datei schreiben\n"
	"  -f\t\tAusgabe forcieren\n");

  puts ("Homepage: <" AKFNETZ_HOMEPAGE ">\n");

  return EXIT_SUCCESS;
}


static int
Version (void)
{
  puts ("akfscraper (akfnetz " AKFNETZ_VERSION ")\n\n"
	AKFNETZ_COPYRIGHT "\n"
	"Lizenz GPLv3+: GNU GPL Version 3 oder neuer "
	"<http://gnu.org/licenses/gpl.html>\n\n"
	"Dies ist freie Software. Sie darf veraendert und verteilt werden.\n"
	"Dieses Programm wird ohne Gewaehrleistung geliefert,\n"
	"soweit dies gesetzlich zulaessig ist.\n");

  return EXIT_SUCCESS;
}


static int
Dateibearbeitung (const char *Name, FILE * aus,
		  int Linkignoranz, int Schalter)
{
  FILE *ein;

  if (!strcmp (Name, "-"))
    {
      akfnetz_scrape (stdin, aus, Linkignoranz, Schalter);
      return 0;
    }

  ein = fopen (Name, "r");

  if (!ein)
    {
      perror (Name);
      return -1;
    }

  akfnetz_scrape (ein, aus, Linkignoranz, Schalter);

  fclose (ein);

  return 0;
}


int
main (int argc, char *argv[])
{
  int opt;
  int Linkignoranz, Schalter;
  bool forcieren;
  char *Ausgabename;
  FILE *Ausgabe;

  Linkignoranz = 0;
  Schalter = 0;
  forcieren = false;
  Ausgabe = stdout;
  Ausgabename = NULL;

  setlocale (LC_ALL, "");

  while ((opt = getopt_long (argc, argv, OPTIONEN, Langoption, NULL)) != -1)
    {
      switch (opt)
	{
	case 'h':
	  return Hilfe (argv[0]);

	case 'V':
	  return Version ();

	case 'u':
	  Schalter |= AKFNETZ_SC_UTF8;
	  break;

	case 'c':
	  if (!strcasecmp (optarg, "UTF-8") || !strcasecmp (optarg, "UTF8"))
	    Schalter |= AKFNETZ_SC_UTF8;
	  break;

	case 'e':
	  Schalter |= AKFNETZ_SC_TERMINAL;
	  break;

	case 'l':
	  Schalter |= AKFNETZ_SC_LINKS;
	  break;

	case 'L':
	  Linkignoranz = (int) strtol (optarg, NULL, 10);

	  if (1 > Linkignoranz || Linkignoranz > 10000)
	    {
	      fprintf (stderr, "%s: -L: %s\n", argv[0], Fehlermeldung (EDOM));
	      return EXIT_FAILURE;
	    }

	  Schalter |= AKFNETZ_SC_LINKS;
	  break;

	case 'n':
	  Schalter |= AKFNETZ_SC_LANGZEILEN;
	  break;

	case 'g':
	  Schalter |= AKFNETZ_SC_GRAFIK;
	  break;

	case 'f':
	  forcieren = true;
	  break;

	case 'o':
	  if (strcmp (optarg, "-"))
	    Ausgabename = optarg;
	  break;

	case '?':
	  return EXIT_FAILURE;
	}
    }

  if (Ausgabename)
    {
      Ausgabe = fopen (Ausgabename, forcieren ? "w" : "wx");
      if (!Ausgabe)
	{
	  perror (Ausgabename);
	  return EXIT_FAILURE;
	}
    }

  int Status = EXIT_SUCCESS;

  if (optind >= argc)
    akfnetz_scrape (stdin, Ausgabe, Linkignoranz, Schalter);
  else
    while (optind < argc)
      {
	if (Dateibearbeitung (argv[optind++], Ausgabe,
			      Linkignoranz, Schalter) < 0)
	  Status = EXIT_FAILURE;
      }

  fclose (Ausgabe);
  return Status;
}
