/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _XOPEN_SOURCE 600
#define __STDC_CONSTANT_MACROS

#include "akfnetz.h"
#include "Entitaet.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <time.h>
#include <stdint.h>
#include <unistd.h>
#include <limits.h>

#define ANTWORTSPALTE "40"	// +1

#define CSI           "\033["
#define LOESCHEN      CSI "2J" CSI "H"
#define UEBERSCHRIFT  CSI "1;4m"
#define ANTWORT       "\r" CSI ANTWORTSPALTE "C"
#define NORMAL        CSI "m"
#define OKAY          CSI "32m"
#define FEHLER        CSI "31;7m"
#define WARNUNG       CSI "1;4m"

static char Puffer[1024];

extern int akfnetz_Medientypfehler (char *, size_t);
static inline void teste (const char *);
static inline void erwarte (const char *);
static inline void Okay (void);
static void Fehler (const char *);
static void Warnung (const char *);

int
main (void)
{
  uintmax_t Zahl;
  intmax_t Ganzzahl;
  struct tm Zeit;
  int i;

  puts (LOESCHEN UEBERSCHRIFT "AKFNetz-Check" NORMAL "\n");

  /*******************************************************************/

  teste ("time_t");
  printf (ANTWORT "%zu Bit\n", sizeof (time_t) * CHAR_BIT);
  teste ("size_t");
  printf (ANTWORT "%zu Bit\n", sizeof (size_t) * CHAR_BIT);
  teste ("off_t");
  printf (ANTWORT "%zu Bit\n", sizeof (off_t) * CHAR_BIT);
  teste ("uintmax_t");
  printf (ANTWORT "%zu Bit\n", sizeof (uintmax_t) * CHAR_BIT);

  akfnetz_Datengroessenstring (Puffer, sizeof (Puffer), 0);
  printf ("%s" ANTWORT "kleinste Datenmenge\n", Puffer);

  akfnetz_Datengroessenstring (Puffer, sizeof (Puffer), UINTMAX_MAX);
  printf ("%s" ANTWORT "groesste Datenmenge (theoretisch)\n\n", Puffer);

  /*******************************************************************/

  teste ("ASCII");
  if ('a' == 0x81 && 'A' == 0xC1 && '0' == 0xF0)
    Fehler ("EBCDIC? - Igitt!");
  if (' ' != 0x20 || '0' != 0x30 || 'A' != 0x41 || 'z' != 0x7A || '~' != 0x7E)
    Fehler (NULL);
  Okay ();

  /*******************************************************************/

  teste ("Zahlstring");
  Zahl = UINTMAX_C (42);
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 10, 1);
  erwarte ("42");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 10, 4);
  erwarte ("0042");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 16, 4);
  erwarte ("002A");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 2, 8);
  erwarte ("00101010");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 2, 1);
  erwarte ("101010");

  Zahl = UINTMAX_C (0);
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 10, 0);
  erwarte ("0");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 10, 1);
  erwarte ("0");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 16, 1);
  erwarte ("0");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 2, 1);
  erwarte ("0");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 2, 8);
  erwarte ("00000000");

  Zahl = UINTMAX_C (0xFFFFFFFFFFFFFFFF);
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 10, 1);
  erwarte ("18446744073709551615");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 16, 1);
  erwarte ("FFFFFFFFFFFFFFFF");
  akfnetz_Zahlstring (Puffer, sizeof (Puffer), Zahl, 2, 1);
  erwarte ("11111111111111111111111111111111"
	   "11111111111111111111111111111111");
  Okay ();


  /*******************************************************************/

  teste ("Ganzzahlstring");
  Ganzzahl = INTMAX_C (42);
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 1);
  erwarte ("42");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 4);
  erwarte ("0042");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 16, 4);
  erwarte ("002A");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 8);
  erwarte ("00101010");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 1);
  erwarte ("101010");

  Ganzzahl = INTMAX_C (0);
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 0);
  erwarte ("0");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 1);
  erwarte ("0");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 16, 1);
  erwarte ("0");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 1);
  erwarte ("0");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 8);
  erwarte ("00000000");

  Ganzzahl = INTMAX_C (-42);
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 1);
  erwarte ("-42");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 4);
  erwarte ("-042");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 16, 4);
  erwarte ("-02A");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 8);
  erwarte ("-0101010");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 1);
  erwarte ("-101010");

  // kleinste Zahl bei 64-Bit (System koennte aber mehr Bit unterstuetzen)
  Ganzzahl = INTMAX_C (-9223372036854775807) - 1;
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 10, 1);
  erwarte ("-9223372036854775808");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 16, 1);
  erwarte ("-8000000000000000");
  akfnetz_Ganzzahlstring (Puffer, sizeof (Puffer), Ganzzahl, 2, 1);
  erwarte ("-10000000000000000000000000000000"
	   "00000000000000000000000000000000");
  Okay ();


  /*******************************************************************/

  teste ("MIME-Types");
  i = akfnetz_Medientypfehler (Puffer, sizeof (Puffer));
  if (i < 0)
    Fehler (Puffer);
  printf (" (%i)", i);
  if (strcmp (akfnetz_Medientyp ("/tmp/Kalender-01.01.2017.icalendar"),
	      "text/calendar") != 0)
    Fehler (".icalendar");
  if (strcmp (akfnetz_Medientyp ("/tmp/Kalender-01.01.2017.icalendar.Z"),
	      "application/x-compress") != 0)
    Fehler (".Z");
  Okay ();


  /*******************************************************************/

  teste ("Enitaeten");
  printf (" (%i)", ENTITAETSANZAHL);
  if (!Entitaetsname[ENTITAETSANZAHL - 1].Name)
    Fehler ("Anzahl");
  for (size_t i = 1; i < ENTITAETSANZAHL; ++i)
    {
      if (strcmp (Entitaetsname[i - 1].Name, Entitaetsname[i].Name) >= 0)
	Fehler (Entitaetsname[i].Name);
    }
  Okay ();


  /*******************************************************************/

  teste ("strncpy, strncmp, strncasecmp");
  Puffer[8] = '!';
  strncpy (Puffer, "ABCDEFGHIJKLMNOPQRSTUVWXYZ", 8);
  if (Puffer[7] != 'H' || Puffer[8] != '!'
      || strncmp (Puffer, "ABCDEFGH###", 8)
      || strncasecmp (Puffer, "aBcDeFgH###", 8))
    Fehler (NULL);
  Okay ();


  /*******************************************************************/

  teste ("Base64 kodieren");	// Tests aus RFC 4648
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "", 0, AKFNETZ_BASE64);
  erwarte ("");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "f", 1, AKFNETZ_BASE64);
  erwarte ("Zg==");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "fo", 2, AKFNETZ_BASE64);
  erwarte ("Zm8=");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "foo", 3, AKFNETZ_BASE64);
  erwarte ("Zm9v");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "foob", 4, AKFNETZ_BASE64);
  erwarte ("Zm9vYg==");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "fooba", 5,
			  AKFNETZ_BASE64);
  erwarte ("Zm9vYmE=");
  akfnetz_base64_kodiert (Puffer, sizeof (Puffer), "foobar", 6,
			  AKFNETZ_BASE64);
  erwarte ("Zm9vYmFy");
  Okay ();


  /*******************************************************************/

  teste ("Base64 dekodieren");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "");
  erwarte ("");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zg==");
  erwarte ("f");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zm8=");
  erwarte ("fo");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zm9v");
  erwarte ("foo");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zm9vYg==");
  erwarte ("foob");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zm9vYmE=");
  erwarte ("fooba");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), "Zm9vYmFy");
  erwarte ("foobar");
  akfnetz_base64_dekodieren (Puffer, sizeof (Puffer),
			     " \tZ\n  m 9  v\t  Y  \ng\n\t=\n=######");
  erwarte ("foob");
  Okay ();


  /*******************************************************************/

  teste ("URL dekodieren");
  strcpy (Puffer, "Test+Test++Test%25%2BTest%20%E2%98%ba");
  akfnetz_url_dekodieren (Puffer);
  erwarte ("Test Test  Test%+Test \u263A");
  Okay ();


  /*******************************************************************/

  teste ("Zeitfunktionen");
  akfnetz_analysiere_http_Zeit (&Zeit, "  \tWed, 09 Nov 1994 08:49:37 GMT");
  akfnetz_http_Zeitstring (Puffer, sizeof (Puffer), &Zeit);
  erwarte ("Wed, 09 Nov 1994 08:49:37 GMT");

  akfnetz_analysiere_http_Zeit (&Zeit, "Wednesday, 09-Nov-94 08:49:37 GMT");
  akfnetz_http_Zeitstring (Puffer, sizeof (Puffer), &Zeit);
  erwarte ("Wed, 09 Nov 1994 08:49:37 GMT");

  akfnetz_analysiere_http_Zeit (&Zeit, "  \tWed Nov  9 08:49:37 1994");
  akfnetz_http_Zeitstring (Puffer, sizeof (Puffer), &Zeit);
  erwarte ("Wed, 09 Nov 1994 08:49:37 GMT");

  Okay ();

  /*******************************************************************/

  putchar ('\n');
  if (sizeof (off_t) <= sizeof (int_least32_t))
    Warnung ("Probleme bei sehr grossen Dateien. (ca. ab 2GB)");

  if (sizeof (time_t) == 4 && (time_t) (-1) < 1)
    Warnung ("Jahr-2038-Problem entdeckt.");

  return EXIT_SUCCESS;
}


static inline void
teste (const char *t)
{
  fputs (t, stdout);
}


static inline void
erwarte (const char *t)
{
  if (strcmp (Puffer, t))
    {
      printf (ANTWORT FEHLER "FEHLER!\a\n" NORMAL
	      "erwartet: %s\nerhalten: %s\n", t, Puffer);
      exit (EXIT_FAILURE);
    }

  memset (Puffer, '#', sizeof (Puffer));
}

static inline void
Okay (void)
{
  puts (ANTWORT OKAY "Okay" NORMAL);
}


static void
Warnung (const char *Problem)
{
  printf (WARNUNG "Warnung:" NORMAL " %s\n", Problem);
}


static void
Fehler (const char *Problem)
{
  if (Problem)
    printf (ANTWORT FEHLER "FEHLER:" NORMAL " %s\a\n", Problem);
  else
    puts (ANTWORT FEHLER "FEHLER!\a" NORMAL);

  exit (EXIT_FAILURE);
}
