/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfnetzpass
 * Copyright (c) 2016-2018,2020 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "akfnetz.h"
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <locale.h>

static int Hilfe (const char *);
static int Fehler (int, const char *);
static inline bool nur_ascii (const void *);

#define Text(fd, s)  write((fd), "" s "", sizeof(s)-1)

int
main (int argc, char **argv)
{
  int Datei;
  ssize_t l, l1, l2;
  bool inakzeptabel;
  char z[1024], kodiert[1024];

  Datei = STDOUT_FILENO;
  setlocale (LC_ALL, "");

  if (argc > 2 ||
      (argc == 2 && (!strcmp (argv[1], "-h") || !strcmp (argv[1], "--help")
		     || !strcmp (argv[1], "--Hilfe")
		     || !strcmp (argv[1], "--version"))))
    return Hilfe (argv[0]);

  if (argc == 2)
    {
      Datei = open (argv[1], O_WRONLY | O_APPEND | O_CREAT,
		    S_IRUSR | S_IWUSR);

      if (Datei < 0)
	return Fehler (errno, argv[1]);
    }

  do
    {
      Text (STDERR_FILENO, "Name: ");
      l1 = read (STDIN_FILENO, z, sizeof (z));
      if (l1 <= 1)
	return EXIT_FAILURE;

      z[l1 - 1] = '\0';

      inakzeptabel = !nur_ascii (z) || strchr (z, ':');

      if (inakzeptabel)
	Fehler (EILSEQ, NULL);
    }
  while (inakzeptabel);

  z[l1 - 1] = ':';

  do
    {
      Text (STDERR_FILENO, "Password: ");
      l2 = read (STDIN_FILENO, z + l1, sizeof (z) - l1);
      l = l1 + l2 - 1;
      z[l] = '\0';

      if (l2 < (10 + 1))
	{
	  inakzeptabel = true;
	  Fehler (EBADMSG, "zu kurz");
	}
      else
	{
	  inakzeptabel = !nur_ascii (z + l1);

	  if (inakzeptabel)
	    Fehler (EILSEQ, NULL);
	}
    }
  while (inakzeptabel);

  if (!akfnetz_base64_kodiert (kodiert, sizeof (kodiert), z, l,
			       AKFNETZ_BASE64))
    return Fehler (EMSGSIZE, NULL);

  Text (Datei, ":");
  write (Datei, kodiert, strlen (kodiert));
  Text (Datei, "\n");
  close (Datei);

  return EXIT_SUCCESS;
}


static int
Hilfe (const char *prg)
{
  write (STDOUT_FILENO, prg, strlen (prg));
  Text (STDOUT_FILENO, " [file]\n\n"
	"Appends obfuscated account data to file.\n"
	"Haengt Zugangsdaten verschleiert an file an.\n");

  return EXIT_SUCCESS;
}


static int
Fehler (int e, const char *Ursache)
{
  const char *m;

  if (Ursache && *Ursache)
    {
      write (STDERR_FILENO, Ursache, strlen (Ursache));
      Text (STDERR_FILENO, ": ");
    }

  m = strerror (e);
  write (STDERR_FILENO, m, strlen (m));
  Text (STDERR_FILENO, "\a\n");

  return EXIT_FAILURE;
}


static inline bool
nur_ascii (const void *s)
{
  const signed char *t = s;

  while (*t)
    if (*t++ < 0x20)
      return false;

  return true;
}
