/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfnetz - Netzwerk Bibliothek
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Hinweis:
 * Die AGPL ist mit der GPL kompatibel. Siehe jeweils Abschnitt 13.
 * Falls man diese Funktionen fuer CGI-Programme oder Server verwenden will,
 * darf man diese unter der AGPL lizenzieren.
 * Die AGPL wird hiermit ausdruecklich empfohlen.
 */

#ifndef AKFNETZ_H
#define AKFNETZ_H

#include <stdio.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <uchar.h>
#include <time.h>

#define AKFNETZ_HOMEPAGE "https://akfoerster.de/p/akfnetz/"

// TODO: besser sortieren

// Fuer CGI-Programme

// gibt Leerzeile aus; bei einer HEAD-Anfrage wird der Prozess beendet
void akfnetz_cgi_Kopfende (void);

// Gibt Text aus mit kodierten HTML/XML-Sonderzeichen <&>'"
void akfnetz_xml_Zeichen (FILE *, char);
void akfnetz_xml_Text (FILE *, const char *);
#define akfnetz_html_Text(d, s) akfnetz_xml_Text(d, s)

/***************
 ** Formulare **
 ***************/

// Erstmal Daten mit einer dieser Funktionen importieren
// erwartetes Format: application/x-www-form-urlencoded
bool akfnetz_cgi_Formulardaten (void);
void akfnetz_Formulardaten (const char *);
void akfnetz_Formulardaten_einlesen (size_t);

// Formularfeld-Wert mit bestimmten Namen erhalten
char *akfnetz_Formularfeld (const char *);

/*
akfnetz_Formularfelder[0].Feld ist das erste Feld.
Der letzte Eintrag ist NULL.
*/
struct akfnetz_Feld { char *Feld, *Wert; };
extern struct akfnetz_Feld *akfnetz_Formularfelder;

void akfnetz_Formularfreigabe (void);
// Die explizite Formularfreigabe ist meist unnoetig

// wachsende Textvariablen
// VERALTET: Nicht mehr benutzen! Das soll in Zukunft entfernt werden.
struct akfnetz_Text
{
  char *Inhalt;
  size_t Laenge, reserviert;
};

struct akfnetz_Text *
akfnetz_Texterstellung (struct akfnetz_Text *, size_t Anfangsgroesse);
void akfnetz_Textfreigabe (struct akfnetz_Text *);
void akfnetz_hinzufuegen (struct akfnetz_Text *, const char *, size_t);
void akfnetz_Textbyte (struct akfnetz_Text *, int);
int akfnetz_Textreservierung (struct akfnetz_Text *, size_t);
void akfnetz_xml_Textvariable (struct akfnetz_Text *, const char *);
#define akfnetz_html_Textvariable(t, s) akfnetz_xml_Textvariable(t, s)

// nur verwenden, falls Variable laengerfristig gebraucht wird
#define akfnetz_Textoptimierung(t) \
  akfnetz_Textreservierung(t, (t)->Laenge + 1)


// Ausgabe
// 0 bei Erfolg oder -1 bei Fehler, errno gesetzt
int akfnetz_Ausgabe (int, const void *, size_t);

// gibt terminierten C-String aus
int akfnetz_ausgeben (int, const char *);

// Eingabe - wartet bis alle Daten zusammen sind
// 0 bei Erfolg oder -1 bei Fehler, errno gesetzt
// ohne String-Terminator!
int akfnetz_Eingabe (int, void *, size_t);

// Gibt String-Konstante aus
#define akfnetz_str(fd, s)  akfnetz_Ausgabe((fd), "" s "", sizeof(s)-1)

#define akfnetz_Zeilenende(fd)  akfnetz_Ausgabe((fd), "\r\n", 2)

// Gibt natuerliche Zahl aus
// Basis: dezimal: 10, hexadezimal: 16, binaer: 2, usw.
void akfnetz_Zahl (int, uintmax_t Wert, int Basis, size_t Mindeststellen);
size_t akfnetz_Zahlstring (char *Puffer, size_t Groesse,
                          uintmax_t Wert, int Basis, size_t Mindeststellen);

// das Gleiche mit Unterstuetzung fuer negative Zahlen
void akfnetz_Ganzzahl (int, intmax_t Wert, int Basis, size_t Mindeststellen);
size_t akfnetz_Ganzzahlstring (char *Puffer, size_t Groesse,
                              intmax_t Wert, int Basis,
                              size_t Mindeststellen);

// liest in Speicher ein (muss mit free() freigegeben werden)
// wird mit binaerer 0 abgeschlossen
// gibt bei Fehler 0 zurueck
extern size_t akfnetz_einlesen (int, char **);

// kopiert Daten voll gepuffert
extern int akfnetz_kopiere (int Eingabe, int Ausgabe);

// Gibt Datengroesse in menschenlesbarer Einheit aus (B,kB,MB,GB,TB,PB,EB)
void akfnetz_Datengroesse (FILE *, uintmax_t);
size_t akfnetz_Datengroessenstring (char *, size_t, uintmax_t);

/*
Extrahiert Text aus XHTML/HTML.
Die Ausgabe ist immer UTF-8
*/
#define AKFNETZ_SC_UTF8 1          // erzwinge UTF-8 Eingabe
#define AKFNETZ_SC_TERMINAL 2      // SGR-Steuersequenzen einfuegen (ECMA-48)
#define AKFNETZ_SC_LINKS 4         // Links als Referenzen
#define AKFNETZ_SC_GRAFIK 8        // Grafik-Links
#define AKFNETZ_SC_LANGZEILEN 16   // keine automatischen Zeilenumbrueche
int akfnetz_scrape (FILE *Eingabe, FILE *Ausgabe,
                    int Linkignoranz, int Schalter);

// Konvertiert einzelnes Zeichen nach UTF-8
// Puffer muss mindestens 5 Byte Platz haben.
size_t akfnetz_UTF8 (char *, char32_t);

// Gibt Unicode-Zeichen fuer Codepage 1252 aus
char32_t akfnetz_cp1252 (int);

/*
Gibt Unicode-Zeichen fuer HTML-Entitaet aus.
Wenn Name mit '&' beginnt, wird dieses uebersprungen.
Gelesen wird bis zum Semikolon oder anderem Trennzeichen.
Es kann eine benannte, oder nummerische Entitaet sein.
Bei unbekannten Namen oder ungueltigen Zeichen wird 0xFFFD ausgegeben.

Unterstuetzt nur Entitaetsnamen, die mit einem einzelnen Unicode-Zeichen
dargestellt werden koennen.
HTML5 vollstaendig unterstuetzen zu wollen, grenzt an Wahnsinn!
*/
char32_t akfnetz_Entitaet (const char *Name);

/*********
 ** URL **
 *********/

// Gibt Hostnamen und evtl. Port aus
// zB. localhost, localhost:8080, [::1]:8080
char *akfnetz_URL_Host (char *, size_t, const char *URL);

// Gibt nur den Hostnamen oder die IP-Adresse aus
char *akfnetz_URL_Hostname (char *, size_t, const char *URL);

// Gibt Port aus, falls angegeben, sonst NULL
char *akfnetz_URL_Port (char *, size_t, const char *URL);

// Standardport fuer einige Protokolle
// unabhaengig von evtl. angegebenen Port
extern char *akfnetz_Standardport (const char *URL);

/*
Gibt Zeiger auf Pfad aus (evtl. mit Query-String und Fragment)
Ist der Pfad leer und ist kein Query-String oder Fragment vorhanden,
wird NULL ausgegeben.
*/
const char *akfnetz_URL_Pfad (const char *URL);

// Gibt evtl. Zugangsdaten aus URL zurueck, oder NULL.
// Das sollte nochmal mit akfnetz_url_dekodieren nachbearbeitet werden.
char *akfnetz_URL_Zugang (char *, size_t, const char *URL);


// Gibt String URL-kodiert aus
void akfnetz_urlkodiert (FILE *, const char *, const char *erlaubt);
char *akfnetz_urlkodiert_puf (const char *s, const char *erlaubt);
void akfnetz_urlkodiert_Text (struct akfnetz_Text *, const char *);

// dekodiert URL-Kodierung an Ort und Stelle
char *akfnetz_url_dekodieren (char *);


// Ruft Funktion mit Links (Querverweisen) aus dem (X)HTML-Body auf.
// Es wird nur nach Links gesucht, nicht nach Adressen eingebetteter Elemente.
void akfnetz_Links (const char *, void (*) (const char *, size_t, void *d),
                    void *d);

/*
 * Sprachauswahl
 * Nur fuer Hauptsprachen, keine Sprachvarianten.
 *
 * Akzeptanz ist der Wert "Accept-Language" des Clients.
 * untertuetzt ist eine Komma-separierte Liste unterstuetzter Sprachen.
 *
 * Das Ergebnis ist die Nummer der Sprache, angefangen bei 0.
 * Bei keiner Uebereinstimmung wird -1 zurueck gegeben.
 *
 * Beispiel:
 * l = akfnetz_Sprachauswahl (getenv ("HTTP_ACCEPT_LANGUAGE"), "en,de,eo");
 * if (l < 0) l = 0;
 */
int akfnetz_Sprachauswahl (const char *Akzeptanz, const char *unterstuetzt);


/*
 * HTTP-Vereinbarungen
 * Akzeptanz ist einer der HTTP "Accept-*"-Kopzeilen.
 * Angebot ist eine kommaseparierte Liste mit Angeboten.
 *
 * Bei Erfolg hat Angebot nur noch den besten Wert
 * und wird als Ergebnis zurueckgegeben.
 * Bei keiner Uebereinstimmung wird NULL zurueckgegeben und Angebot ist leer.
*/
char *akfnetz_Vereinbarung (const char *Akzeptanz, char *Angebot);


// HTTP-Zeitangabe ausgeben (GMT)
void akfnetz_http_Zeitangabe (FILE *, const struct tm *);
#define akfnetz_http_Zeitstring(p,l,z)  \
  strftime ((p), (l), "%a, %d %b %Y %T GMT", (z))

/*
 * analysiert HTTP-Zeitangabe
 * Leerzeichen am Anfang werden ignoriert, am Ende darf beliebiger Text folgen
 * tm_yday bleibt ungesetzt
 * Gibt bei Fehler NULL zurueck
 */
struct tm *akfnetz_analysiere_http_Zeit (struct tm *, const char *);

// Vergleicht Zeitangaben ohne Beruecksichtigung der Zeitzone
// Ergebnis ist kleiner 0, gleich 0, oder groesser 0
int akfnetz_Zeitenvergleich (const struct tm *, const struct tm *);

// deutschsprachige Wochentagsnamen und Monatsnamen
extern const char *const akfnetz_Wochentag[7];
extern const char *const akfnetz_Monat[12];

// Gibt die verstrichene Zeit seit der angegebenen Startzeit aus
// Rueckgabewert: Zeit in Sekunden, oder 0
extern long unsigned int akfnetz_Dauer (FILE *, time_t, bool deutsch);


// verwandelt Internet-Adresse in Text, wie inet_ntop,
// aber gemappte IPv4-Adresse wird als reine IPv4-Adresse ausgegeben
char *akfnetz_Adressentext (int af, const void *a, char *Ziel, size_t);

// verwandelt Adresse von Text in interne Darstellung, wie inet_pton,
// aber eine IPv4-Adresse wird evtl. nach IPv6 gemappt
int akfnetz_Adresszuweisung (int af, const char *a, void *Ziel);

// gibt Adresse aus Socketadresse aus, NULL wenn weder IPv4 noch IPv6
void *akfnetz_Socketadresse (const void *);

// gibt Port des Sockets aus, oder -1
int akfnetz_Socketport (int);


/************
 ** GOPHER **
 ************/

/*
 * stellt Verbindung zu Gopher- oder Finger-Server her
 * und stellt die Anfrage
 * Verbindung laeuft eventuell ueber Proxy, wenn nicht NULL oder ""
 * gibt Socket zurueck, von dem die Antwort gelesen werden kann
 * -1 bei Fehler und errno gesetzt
 * Unterstuetzt Gopher- und Finger-URLs
 */
int akfnetz_Gopher_URL (const char *url, const char *Proxy);

// gibt Selektor mit CRLF an Gopher-Verbindung
int akfnetz_Gopher_Selektor (int Verbindung, const char *Selektor);


/**********
 ** HTTP **
 **********/

enum akfnetz_Anfragemethoden
{
  akfnetz_unbekannte_Methode,
  GET, HEAD, POST, OPTIONS, TRACE, PUT, DELETE, PATCH, CONNECT,
  PROPFIND, PROPPATCH, MKCOL, COPY, MOVE, LOCK, UNLOCK,
  SEARCH, MERGE, CHECKIN, CHECKOUT, UNCHECKOUT, REPORT, MKACTIVITY,
  MKCALENDAR
};

extern const char *const akfnetz_Methodenname[];
enum akfnetz_Anfragemethoden akfnetz_Methodentyp (const char *);


// liest Kopfzeilen
char **akfnetz_Kopflesen (FILE *);
void akfnetz_Kopffreigabe (char **);
char *akfnetz_Kopfeintrag (char **, const char *);
intmax_t akfnetz_Kopfzahl (char **, const char *);
// INTMAX_MIN, wenn nicht angegeben

// sucht Token in Kopfeintrag
const char *akfnetz_Tokensuche (char **, const char *Eintrag, const char *Token);

/*
 * gibt "Name: Basic ..." aus.
 * Als Zugang wird entweder "Name:Kennung" im Klartext erwartet,
 * oder gleich die Base64-kodierte Kennung.
 */
void akfnetz_Authorization_Basic (FILE *, const char *Name, const char *Zugang);


/*
 * MIME-Typen
 * Erwartet Dateinamen, evtl. mit Pfad oder URL.
 * Gibt MIME-Typ anhand des Namens oder der Erweiterung aus,
 * oder einen Leerstring fuer verbotene Dateien.
 */
const char *akfnetz_Medientyp (const char *);


/*
Gibt HTML-Darstellung des Verzeichnisses aus
die Anfrage (query) ist mit Apache kompatibel
Sprachakzeptanz gemaess "Accept-Language"
*/
void
akfnetz_html_Verzeichnis (FILE *,
			  const char *Verzeichnis,
			  const char *Titel,
			  bool zurueck,
			  const char *Anfrage,
			  const char *Sprachakzeptanz,
			  const char *Nutzer);


// Gibt schlichte HTML-Seite mit Link aus (geeignet fuer Umleitungen)
extern void akfnetz_html_Linkseite (FILE *, const char *URL);


// Funktionen fuer gestueckelten (chunked) Inhalt.
// Dahinter kann noch ein Trailer folgen, zumindest aber eine Leerzeile

// Inhalt ungestueckelt weiterleiten.
uintmax_t akfnetz_gestueckelt_Filter (FILE *ein, FILE *aus);

// Inhalt in den Speicher einlesen.
struct akfnetz_Text *akfnetz_gestueckelt_Text (FILE *, struct akfnetz_Text *);

/*
Inhalt ueber Funktion ausgeben.
Die Funktion f sollte im Fehlerfall einen Wert ungleich 0 zurueckgeben.
Der Wert d wird unveraendert durchgeleitet
Wenn f NULL ist, wird der Inhalt einfach ignoriert.
Gibt die Groesse zurueck, oder 0 bei Fehlern.
*/
uintmax_t akfnetz_gestueckelt (FILE *,
			       int (*f) (void *, size_t, void *d),
			       void *d);

/*
Mit Base64 koennen Binaerdaten in reines ASCII konvertiert werden.
Es wird manchmal auch verwendet, um Informationen zu "verschleiern".
Zum Auffuellen werden evtl. noch Gleichzeichen (=) angehaengt.
Die Datenmenge vergroessert sich um etwa 33%.
Kodieren ist effizienter als dekodieren.
*/
#define AKFNETZ_BASE64 \
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"

// besser fuer URLs und Dateinamen geeignet
#define AKFNETZ_BASE64_URL \
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"

char *akfnetz_base64_kodiert (char *, size_t, const void *, size_t,
                              const char *Zeichensatz);

// wieviel Platz fuer base64-kodierte Variable benoetigt wird
#define akfnetz_base64_Platz(l)  ((4 * (((l) + 2) / 3)) + 1)

/*
dekodiert Base64 oder Base64url
Leerzeichen, Steuerzeichen und nicht-ASCII-Zeichen werden ignoriert,
Falsche ASCII-Zeichen fuehren zum Abbruch mit Laenge 0.
Der Ausgabepuffer darf hier mit dem Eingabepuffer identisch sein.
*/
size_t akfnetz_base64_dekodieren (void *, size_t, const char *);

// wie akfnetz_base64_dekodieren, bricht aber bei bestimmten Zeichen ab
size_t akfnetz_base64_Teilbereich (void *, size_t, const char *, int Abbruch);


// Versucht eine Verbindung zu einem Internet-Server herzustellen
// Gibt Dateinummer zurueck, oder -1 bei Fehler
// Port muss nummerisch sein, zB. "80"
// af ist AF_INET, AF_INET6, oder AF_UNSPEC
int akfnetz_Serververbindung (const char *Host, const char *Port, int af);

// stellt Verbindung zu dem Server in der URL her
// af ist AF_INET, AF_INET6, oder AF_UNSPEC(=0)
int akfnetz_URL_Verbindung (const char *URL, int af);

// Stellt Verbindung zu lokalem Unix-Socket her
int akfnetz_Lokalverbindung (const char *Pfad);

// testet, ob die Socketadresse ein privates Netz ist
// IPv4/IPv6/UNIX, sonst false
bool akfnetz_privates_Netz (const void *);


/***********
 ** Proxy **
 ***********/

/*
SOCKS ist ein Protokoll fuer Proxies.
Man baut dabei eine Verbindung zu dem Proxy-Server auf, und sagt dem,
ueber folgende Funktionen, mit welchem Rechner man verbunden werden will -
wie bei einer Vermittlung.
Bei Erfolg verwendet man einfach dieselbe Verbindung weiter, um mit dem
Zielrechner zu sprechen. Es wird aber keine direkte Verbindung mit dem Ziel
aufgebaut, alles wird ueber den Proxy vermittelt.
Bei Fehlern geben die Funktionen -1 zurueck und setzen errno.
Die Verbindung kann dann nicht einfach weiterverwendet werden, sondern muss
erstmal geschlossen werden!

Id, Name und Passwort duerfen NULL sein, falls sie nicht benoetigt werden.
*/

// Verbindung indirekt ueber proxy_url (SOCKS4, SOCKS4a, SOCKS5)
// Wenn proxy_url NULL oder "" ist, dann Direktverbindung
int akfnetz_Proxy_Verbindung  (const char *proxy_url, const char *url);
int akfnetz_Proxy_Verbindung_Host (const char *proxy_url,
                                   const char *Host, const char *Port);


// ermittle IP Adresse (.onion verboten)
void *akfnetz_IP_Adresse (void *Adresse, const char *Name, const char *Port,
     int af);

// Stellt Verbindung ueber SOCKS4 her (nur IPv4)
int akfnetz_socks4_Verbindung (int proxy, const char *Id,
                               const void *sockaddr);

// Stellt Verbindung ueber SOCKS4a her, Adresse wird am Proxy ermittelt
int akfnetz_socks4a_Verbindung (int proxy, const char *Id,
                      const char *Host, int Port);

// Stellt Verbindung ueber SOCKS5 her (IPv4 oder IPv6)
int akfnetz_socks5_Verbindung (int proxy,
                               const char *Name, const char *Passwort,
                               const void *sockaddr);
int akfnetz_socks5h_Verbindung (int proxy,
                                const char *Name, const char *Passwort,
                                const char *Host, int Port);


// Server: bindet Socket an Port und evtl. Adresse (bind(2))
// wenn Adresse NULL oder "" ist, lauscht er auf allen Adressen
// gibt bei Fehler -1 zurueck und setzt errno
int akfnetz_IPv6_Bindung (int, const char *Adresse, int Port);
int akfnetz_IPv4_Bindung (int, const char *Adresse, int Port);
int akfnetz_Lokalbindung (int, const char *Pfad);


// erzeugt eine Temporaerdatei ohne Dateinamen, oder mit mkstemp
// wenn Verzeichnis NULL oder "" ist, wird "/tmp" verwendet
int akfnetz_Temporaerdatei (const char *Verzeichnis);

// Macht aus dem Prozess einen Daemon
// Standard-Ein-/Ausgaben werden auf /dev/null umgeleitet
int akfnetz_daemonisieren (void);

// Ist das System deutschsprachig?
// (setlocale muss nicht gesetzt sein)
bool akfnetz_deutschsprachig (void);

// rein deutschsprachiger Ersatz fuer strerror
char *akfnetz_Systemfehler (int);

// deutsche Kurzmeldung zu einem HTTP-Statuscode
// 9XX fuer interne Meldungen
// Leerstring bei unbekannter Kategorie (erste Ziffer)
char *akfnetz_Statusmeldung (int);


/************
 ** Listen **
 ************/

struct akfnetz_Liste
{
  struct akfnetz_Liste *weiter;
  const char *Eintrag;
  size_t Laenge;
};

// fuegt Kopie von Wert am Anfang der Liste hinzu
// Wenn Laenge == 0 ist, wird strlen (Wert) verwendet
int akfnetz_Listeneintrag (struct akfnetz_Liste **Liste,
		       const char *Wert, size_t Laenge);

void akfnetz_Listenfreigabe (struct akfnetz_Liste **Liste);


// AKF-Logo fuer CGI-Programme
// width="32" height="32"
// siehe auch RFC 2397 -- The "data" URL scheme
#define AKFNETZ_AKFLOGO "data:image/png;base64,\
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgAgMAAAAOFJJnAAAACVBMVEUAAAAAAP/w\
2xHg3qDHAAAAAXRSTlMAQObYZgAAAHhJREFUGNOVzCESxDAIBdAfkSP0Pj1CRIpY\
vwLug2g9IjlloXRWVOxMI5I34X+wlV7bsm74h/u8xEenHFMDzKQOUTbaffQ1MRoO\
MR7Xj47IBJiFtINset06ltz3Q0N9YEV5AMBdQtaao2YEGQIy1C+UnES85YsS1wkl\
BUu6moPy1wAAAABJRU5ErkJggg=="

#endif
