/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfgopherserver
 * Copyright (c) 2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* siehe RFC 1436 */

/*
 * Umgebungsvariablen:
 * HOSTNAME | HOST
 *
 * Dateien:
 * gophertag
 * gophermap, *.gophermap
 */

#define _GNU_SOURCE
#define _DEFAULT_SOURCE
// fuer FNM_CASEFOLD

#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdbool.h>
#include <stdnoreturn.h>
#include <getopt.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <dirent.h>
#include <fnmatch.h>
#include <pwd.h>
#include <grp.h>
#include "akfnetz.h"
#include "gophertyp.h"
#include "version.h"


#define SERVER_NAME "akfgopherserver"
#define PROGRAMMNAME SERVER_NAME

#define ZEITLIMIT 30		// halbe Minute
#define ERSATZZEICHEN '#'
#define TYP_UNBENUTZBAR '\025'	// NAK
#define FNM_OPTIONEN  FNM_CASEFOLD
#define ELTERNVERZEICHNIS ".."	// Anzeigename
#define DUMMYHOST "host.invalid"	// siehe RFC 6761, 6.4.
#define LISTEN_BACKLOG 3

#define Kurzoptionen "hV46tlDiN:p:a:d:u:"

enum OPT
{
  OPT_OFFEN = 256, OPT_SERVERPORT
};

static const struct option Langoptionen[] = {
  {"Hilfe", no_argument, NULL, 'h'},
  {"help", no_argument, NULL, 'h'},
  {"Version", no_argument, NULL, 'V'},
  {"version", no_argument, NULL, 'V'},
  {"Servername", required_argument, NULL, 'N'},
  {"servername", required_argument, NULL, 'N'},
  {"IPv4", no_argument, NULL, '4'},
  {"ipv4", no_argument, NULL, '4'},
  {"IPv6", no_argument, NULL, '6'},
  {"ipv6", no_argument, NULL, '6'},
  {"localhost", no_argument, NULL, 'l'},
  {"Adresse", required_argument, NULL, 'a'},
  {"address", required_argument, NULL, 'a'},
  {"Port", required_argument, NULL, 'p'},
  {"port", required_argument, NULL, 'p'},
  {"Serverport", required_argument, NULL, OPT_SERVERPORT},
  {"serverport", required_argument, NULL, OPT_SERVERPORT},
  {"Verzeichnis", required_argument, NULL, 'd'},
  {"directory", required_argument, NULL, 'd'},
  {"offen", no_argument, NULL, OPT_OFFEN},
  {"open", no_argument, NULL, OPT_OFFEN},
  {"internet", no_argument, NULL, OPT_OFFEN},
  {"Internet", no_argument, NULL, OPT_OFFEN},
  {"daemon", no_argument, NULL, 'D'},
  {"inetd", no_argument, NULL, 'i'},
  {"Nutzer", required_argument, NULL, 'u'},
  {"text", no_argument, NULL, 't'},
  {0, 0, 0, 0}
};


#ifndef Fehlermeldung
#define Fehlermeldung  strerror
#endif

#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX 255
#endif

static noreturn void Hilfe (void);
static void Version (FILE *);
static void Optionen (int, char **);
static void Initialisiere (void);
static int gopher_inetd (void);
static void Verarbeitung (FILE * aus, const char *Pfad, const char *Suchtext);
static void Verzeichnis (FILE * aus, const char *Pfad);
static int Dateifilter (const struct dirent *);
static void Dateiauflistung (FILE * aus, const char *Pfad);
static bool Verzeichnisname (char *tag, size_t Laenge, const char *Pfad);
static void Elternverzeichnis (FILE * aus, const char *Pfad,
			       const char *Name);
static char *Dateipfad (char *Selektor, size_t Selektorlaenge,
			const char *Pfad, const char *Name);
static void Textausgabe (FILE * aus, const char *Datei);
static void Textualausgabe (FILE * aus, const char *Datei);
static void Binaerausgabe (FILE * aus, const char *Datei);
static char *Zeilenlesen (FILE * ein, char *Zeile, size_t Laenge);
static char *lies_Zeile (FILE * ein);
static char *bereinigter_Pfad (char *Pfad);
static const char *Dateiname (const char *Pfad);
static void HTML_Umleitung (FILE * aus, const char *url);
static char *Filter (char *Zeile);
static void Verbindungseinstellungen (int);
static noreturn void Fehler (int Fehlernr, const char *Ursache);
static void Gopher_Anfrage (FILE * ein, FILE * aus);
static bool Gophermap_Datei (FILE * aus, const char *Pfad, const char *Datei);
static bool Gophermap (FILE * aus, const char *Pfad, FILE * Datei);
static inline void Gopherfehler (FILE * aus);
static inline void Gopherabschluss (FILE * aus);
static inline void Gophermenuezeile (FILE * aus, int Typ,
				     const char *Name, const char *Selektor,
				     const char *Server, int Port);

static noreturn void Servermodus (void);
static noreturn void Verbindungsannahme (int Lauscher);
static int Verbindungsverarbeitung (int Verbindung);
static void Rechteaufgabe (void);
static void Nutzerinitialisierung (void);
static uid_t Nutzerkennung (const char *Name);
static void Statusinformationen (FILE *);
static void Signalverarbeitung (int signum);


static int Kategorisierung (const char *Pfad, const char *Datei);
static inline bool Textdatei (const char *Datei);
static inline bool Binaerdatei (const char *Datei);
static inline bool HTMLdatei (const char *Datei);
static inline bool XMLdatei (const char *Datei);
static inline bool Dokumentdatei (const char *Datei);
static inline bool GIFdatei (const char *Datei);
static inline bool Bilddatei (const char *Datei);
static inline bool Archivdatei (const char *Datei);
static inline bool Audiodatei (const char *Datei);
static inline bool Videodatei (const char *Datei);
static inline bool UUEdatei (const char *Datei);
static inline bool BinHexdatei (const char *Datei);
static inline bool MIMEdatei (const char *Datei);
static inline bool Kalenderdatei (const char *Datei);
static bool verbotene_Datei (const char *Datei);


// Variablen
static char *Programmname, *Hostname, *Bindungsadresse;
static struct passwd *Nutzer;
static int Hostport, Serverport, Protokoll;
static char Standardtyp;
static bool deutsch, offen, inetd, Daemon;


int
main (int argc, char **argv)
{
  Programmname = argv[0];
  setlocale (LC_ALL, "");

  Optionen (argc, argv);
  Initialisiere ();

  if (inetd)
    return gopher_inetd ();

  Servermodus ();
  return EXIT_FAILURE;
}


static noreturn void
Hilfe (void)
{
  puts (Programmname);
  putchar ('\n');

  if (deutsch)
    {
      puts ("Optionen");
      puts ("  -h, --help, --Hilfe\t\tDiese Hilfe\n"
	    "  -V, --version, --Version\tVersion\n"
	    "  -t, --text\t\t\tStandardtyp ist Text (0)\n"
	    "  -N, --Servername=<Name>\tServername\n"
	    "  -p, --Port=<Nr>\t\tPortnummer\n"
	    "  -4, --IPv4\t\t\tIPv4\n"
	    "  -6, --IPv6\t\t\tIPv6 (Vorgabe)\n"
	    "  -l, --localhost\t\tnur auf localhost hoeren\n"
	    "  -a, --Adresse=<IP-Adresse>\tan bestimmte Adresse binden\n"
	    "  -d, --Verzeichnis=<Verz.>\tWurzelverzeichnis [.]\n"
	    "  -D, --daemon\t\t\tals Daemon im Hintergrund laufen\n"
	    "      --offen\t\t\tfuer das gesamte Internet offen!\n"
	    "  -u, --Nutzer\t\t\tBenutzer festlegen (nur root)\n"
	    "  -i, --inetd\t\t\tMit inetd(8) verwenden\n");
    }
  else				// nicht deutsch
    {
      puts ("Options");
      puts ("  -h, --help, --Hilfe\t\tthis help\n"
	    "  -V, --version, --Version\tversion\n"
	    "  -t, --text\t\t\tdefault type is text (0)\n"
	    "  -N, --Servername=<Name>\tservername\n"
	    "  -p, --Port=<Nr>\t\tport number\n"
	    "  -4, --IPv4\t\t\tIPv4\n"
	    "  -6, --IPv6\t\t\tIPv6 (default)\n"
	    "  -l, --localhost\t\tjust listen on localhost\n"
	    "  -a, --address=<address>\tbind to a specific address\n"
	    "  -d, --directory=<dir>\t\troot directory [.]\n"
	    "  -D, --daemon\t\t\trun as daemon in the background\n"
	    "      --open\t\t\topen to the whole Internet\n"
	    "  -u, --Nutzer=<name>\t\tset user (just for root)\n"
	    "  -i, --inetd\t\t\tuse with inetd(8)\n");
    }

  putchar ('\n');
  puts ("Homepage: <" AKFNETZ_HOMEPAGE ">\n");

  exit (EXIT_SUCCESS);
}


static void
Version (FILE *d)
{
  fputs (PROGRAMMNAME " " AKFNETZ_VERSION "\n" AKFNETZ_COPYRIGHT "\n", d);

  if (deutsch)
    fputs ("Lizenz GPLv3+: GNU GPL Version 3 oder neuer "
	   "<http://gnu.org/licenses/gpl.html>\n\n"
	   "Dies ist freie Software. Sie darf veraendert "
	   "und verteilt werden.\n"
	   "Dieses Programm wird ohne Gewaehrleistung geliefert,\n"
	   "soweit dies gesetzlich zulaessig ist.\n\n" "erstellt: "
	   __DATE__ "\n", d);
  else
    fputs ("License GPLv3+: GNU GPL version 3 or later "
	   "<http://gnu.org/licenses/gpl.html>\n\n"
	   "This is free software; you are free to change and redistribute it.\n"
	   "There is NO WARRANTY, to the extent permitted by law.\n\n"
	   "build: " __DATE__ "\n", d);

  fputs ("\nHomepage: <" AKFNETZ_HOMEPAGE ">\n", d);
}


static void
Gopher_Anfrage (FILE *ein, FILE *aus)
{
  char *Anfrage = lies_Zeile (ein);
  if (!Anfrage)
    return;

  // Anfrage auseinandernehmen (wird veraendert)
  char *p = Anfrage;
  char *Pfad = strsep (&p, "\t");
  char *Suchtext = strsep (&p, "\t");
  // char *GPlus = strsep (&p, "\t");

  Verarbeitung (aus, bereinigter_Pfad (Pfad), Suchtext);
  free (Anfrage);
}


static void
Verarbeitung (FILE *aus, const char *Pfad, const char *Suchtext)
{
  if (!Pfad)
    {
      Gopherfehler (aus);
      return;
    }
  else if (!strncmp (Pfad, "URL:", 4) && Pfad[4] > 0x20)
    {
      HTML_Umleitung (aus, Pfad + 4);
      return;
    }

  // Hier brauchen nur die Typen von Kategorisierung() auftauchen
  // Es kommen keine Typen von aussen
  switch (Kategorisierung (Pfad, Dateiname (Pfad)))
    {
    case TYP_TEXT:
      Textausgabe (aus, Pfad);
      // Zeilen mit CR/LF ohne Steuerzeichen ausgegeben
      // mit Abschlusszeile
      break;

    case TYP_MENUE:
      Verzeichnis (aus, Pfad);
      break;

    case TYP_BINHEX:
    case TYP_UUE:
    case TYP_HTML:
    case TYP_XML:
    case TYP_MIME:
    case TYP_KALENDER:
      Textualausgabe (aus, Pfad);
      // weitgehend unveraenderte Ausgabe, mit Abschlusszeile
      break;

    case TYP_BINAER:
    case TYP_ARCHIV:
    case TYP_BILD:
    case TYP_GIF:
    case TYP_AUDIO:
    case TYP_VIDEO:
    case TYP_DOKU:
      Binaerausgabe (aus, Pfad);
      // unveraenderte Ausgabe, ohne Abschlusszeile
      break;

    case TYP_UNBENUTZBAR:
    default:
      Gopherfehler (aus);
      break;
    }
}


static void
Verzeichnis (FILE *aus, const char *Pfad)
{
  // gophermap verarbeiten
  if (!Gophermap_Datei (aus, Pfad, "gophermap"))
    Dateiauflistung (aus, Pfad);
}


// Verzeichniseintraege vorfiltern
static int
Dateifilter (const struct dirent *e)
{
  const char *n = e->d_name;
  return (*n != '.' && !verbotene_Datei (n)
	  && strcmp (n, "robots.txt") && strcasecmp (n, "caps.txt"));
}


// erstellt automatische Verzeichnisse
static void
Dateiauflistung (FILE *aus, const char *Pfad)
{
  int Anzahl;
  struct dirent **e;

  Anzahl = scandir ((Pfad && *Pfad) ? Pfad : ".", &e, Dateifilter, alphasort);
  if (Anzahl < 0)
    {
      Gopherfehler (aus);
      return;
    }

  // Eintrag fuer Elternverzeichnis
  if (Pfad && *Pfad)
    Elternverzeichnis (aus, Pfad, ELTERNVERZEICHNIS);

  // Verzeichnis auflisten
  for (int i = 0; i < Anzahl; ++i)
    {
      int Typ;
      char *Name, Selektor[4096], Anzeige[256];

      Name = e[i]->d_name;

      // *.gophermap
      if (!fnmatch ("*.gophermap", Name, 0))
	{
	  Gophermap_Datei (aus, Pfad, Name);
	  continue;
	}

      Dateipfad (Selektor, sizeof (Selektor), Pfad, Name);
      Typ = Kategorisierung (Selektor, Name);
      if (Typ == TYP_UNBENUTZBAR)
	continue;

      if (Typ == TYP_MENUE)
	{
	  // gophertag - anderer Name fuer das Verzeichnis
	  if (!Verzeichnisname (Anzeige, sizeof (Anzeige), Selektor))
	    strncpy (Anzeige, Name, sizeof (Anzeige));
	}
      else
	strncpy (Anzeige, Name, sizeof (Anzeige));

      Anzeige[sizeof (Anzeige) - 1] = '\0';

      // Dateiendungen .text, .txt, .md und .markdown verbergen
      // und Unterstriche in Leerzeichen verwandeln
      if (Typ == TYP_TEXT)
	{
	  char *p = strrchr (Anzeige, '.');

	  if (p && (!strcasecmp (p, ".txt")
		    || !strcasecmp (p, ".text")
		    || !strcasecmp (p, ".md")
		    || !strcasecmp (p, ".markdown")))
	    {
	      *p = '\0';	// Endung abschneiden

	      // Unterstriche zu Leerzeichen umwandeln
	      for (p = Anzeige; *p; ++p)
		if (*p == '_')
		  *p = ' ';
	    }
	}

      Gophermenuezeile (aus, Typ, Filter (Anzeige), Selektor,
			Hostname, Serverport);
    }

  Gopherabschluss (aus);

  // aufraeumen
  for (int i = 0; i < Anzahl; ++i)
    free (e[i]);
  free (e);
}


// gophertag lesen
static bool
Verzeichnisname (char *gophertag, size_t Laenge, const char *Pfad)
{
  FILE *d;
  char Name[4096];

  *gophertag = '\0';

  d = fopen (Dateipfad (Name, sizeof (Name), Pfad, "gophertag"), "r");
  if (!d)
    return false;

  Zeilenlesen (d, gophertag, Laenge);
  fclose (d);

  return true;
}


static void
Elternverzeichnis (FILE *aus, const char *Pfad, const char *Name)
{
  size_t l = strlen (Pfad) + 1;
  char *p;
  char Selektor[l];

  memcpy (Selektor, Pfad, l);
  p = strrchr (Selektor, '/');

  if (p)
    *p = '\0';
  else
    *Selektor = '\0';		// Hauptverzeichnis

  Gophermenuezeile (aus, TYP_MENUE, Name, Selektor, Hostname, Serverport);
}


// Fuegt Pfad/Name zusammen, Pfad darf NULL sein
static char *
Dateipfad (char *Selektor, size_t Selektorlaenge,
	   const char *Pfad, const char *Name)
{
  if (Pfad && *Pfad)
    {
      if (snprintf (Selektor, Selektorlaenge, "%s/%s", Pfad, Name)
	  >= Selektorlaenge)
	return NULL;
    }
  else
    {
      strncpy (Selektor, Name, Selektorlaenge - 1);
      Selektor[Selektorlaenge - 1] = '\0';
    }

  return Selektor;
}


// true = fertig, false = weiter auflisten
static bool
Gophermap_Datei (FILE *aus, const char *Pfad, const char *Datei)
{
  FILE *d;

  if (!Pfad || !*Pfad)
    d = fopen (Datei, "r");
  else
    {
      size_t Namenslaenge = strlen (Pfad) + 1 + strlen (Datei) + 1;
      char Name[Namenslaenge];

      d = fopen (Dateipfad (Name, Namenslaenge, Pfad, Datei), "r");
    }

  if (!d)
    return false;

  bool e = Gophermap (aus, Pfad, d);
  fclose (d);

  return e;
}


// true = fertig, false = weiter auflisten
static bool
Gophermap (FILE *aus, const char *Pfad, FILE *Datei)
{
  char z[4096];

  while (Zeilenlesen (Datei, z, sizeof (z)))
    {
      if (!*z)			// Leerzeile
	{
	  Gophermenuezeile (aus, TYP_INFO, "", "", DUMMYHOST, 1);
	  continue;
	}

      if (*z == '#')		// Kommentar
	continue;

      if (*z == '!')		// Titel
	{
	  Gophermenuezeile (aus, TYP_INFO, Filter (z + 1), "TITLE",
			    DUMMYHOST, 1);
	  continue;
	}

      // Kein TAB
      if (!strchr (z, '\t'))
	{
	  if (*z == '.' && !z[1])
	    return true;	// fertig
	  else if (*z == '*' && !z[1])
	    return false;	// weitere Eintraege
	  else if (*z == '=')
	    {
	      Gophermap_Datei (aus, Pfad, z + 1);
	      continue;
	    }
	  else			// Info
	    Gophermenuezeile (aus, TYP_INFO, Filter (z), "", DUMMYHOST, 1);
	}
      else			// mit TAB
	{
	  int Typ = (int) *z;

	  char *p = z + 1;
	  char *Name = strsep (&p, "\t");
	  char *Selektor = strsep (&p, "\t");
	  char *Server = strsep (&p, "\t");
	  char *Port = strsep (&p, "\t");

	  Filter (Name);

	  if (Server && *Server)	// Normalfall
	    {
	      int PortNr = 70;
	      if (Port && *Port)
		PortNr = atoi (Port);
	      Gophermenuezeile (aus, Typ, Name, Selektor, Server, PortNr);
	    }
	  else			// lokal
	    {
	      if (!Selektor || !*Selektor)
		Selektor = Name;

	      if (Typ == TYP_MENUE && !strcmp (Selektor, ".."))
		Elternverzeichnis (aus, Pfad, Name);
	      else if (strchr (Selektor, '/'))	// Pfad
		Gophermenuezeile (aus, Typ, Name, Selektor,
				  Hostname, Serverport);
	      else		// nur Dateiname
		{
		  char s[4096];
		  if (Dateipfad (s, sizeof (s), Pfad, Selektor))
		    Gophermenuezeile (aus, Typ, Name, s,
				      Hostname, Serverport);
		}
	    }
	}
    }

  return true;			// fertig
}


// gibt Binaer-Dateien unveraendert aus
static void
Binaerausgabe (FILE *aus, const char *Datei)
{
  FILE *ein;
  size_t l;
  unsigned char Puffer[4096];

  ein = fopen (Datei, "rb");
  if (!ein)
    {
      Gopherfehler (aus);
      return;
    }

  while ((l = fread (Puffer, 1, sizeof (Puffer), ein)) != 0)
    if (fwrite (Puffer, 1, l, aus) < l)
      break;

  fclose (ein);
  // keine Abschluss-Zeile!
}


// gibt reine Textdateien aus,
// Zeilenende CR/LF, auch wenn es nur LF war
// Steuerzeichen groesstenteils rausgefiltert
static void
Textausgabe (FILE *aus, const char *Datei)
{
  FILE *ein;
  char Zeile[4096];

  // Zeilen auf 80 Bytes einschraenken funktioniert nicht,
  // wegen UTF-8

  ein = fopen (Datei, "r");
  if (!ein)
    {
      Gopherfehler (aus);
      return;
    }

  while (Zeilenlesen (ein, Zeile, sizeof (Zeile)))
    {
      // Punkt als Abschlusszeichen interpretieren
      if (!strcmp (Zeile, "."))
	break;

      fputs (Filter (Zeile), aus);
      fwrite ("\r\n", 2, 1, aus);
    }

  fclose (ein);
  Gopherabschluss (aus);
}


// Fuer Textformate, die nicht direkt gelesen werden sollen
static void
Textualausgabe (FILE *aus, const char *Datei)
{
  Binaerausgabe (aus, Datei);
  fwrite ("\r\n.\r\n", 5, 1, aus);
}


static inline void
Gophermenuezeile (FILE *aus, int Typ, const char *Name,
		  const char *Selektor, const char *Server, int Port)
{
  fprintf (aus, "%c%s\t%s\t%s\t%d\r\n", Typ, Name, Selektor, Server, Port);
}


// bei Textdateien aller Art einschliesslich Menues
static inline void
Gopherabschluss (FILE *aus)
{
  fwrite (".\r\n", 3, 1, aus);
}


static inline void
Gopherfehler (FILE *aus)
{
  Gophermenuezeile (aus, TYP_FEHLER, "Fehler / Error / Eraro", "",
		    DUMMYHOST, 1);
  Gopherabschluss (aus);
}


/*
 * Datei-Typen
 */

// ausfuehrbar, wenn irgendein Bit fuer Ausfuehrbarkeit gesetzt ist
static inline bool
ausfuehrbar (mode_t Modus)
{
  return ((Modus & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0);
}


// gibt Typ zurueck
static int
Kategorisierung (const char *Pfad, const char *Datei)
{
  int Typ;
  struct stat status;

  if (!Pfad || !*Pfad)
    Pfad = ".";

  // Die Reihenfolge ist nicht beliebig

  if (verbotene_Datei (Datei))
    Typ = TYP_UNBENUTZBAR;
  else if (stat (Pfad, &status) < 0)	// stat folgt symbolischen Links
    Typ = TYP_UNBENUTZBAR;
  else if (S_ISDIR (status.st_mode))
    Typ = TYP_MENUE;
  else if (!S_ISREG (status.st_mode))
    Typ = TYP_UNBENUTZBAR;	// Geraete, Sockets, Pipes
  else if (ausfuehrbar (status.st_mode))
    Typ = TYP_BINAER;
  else if (Textdatei (Datei))
    Typ = TYP_TEXT;
  else if (Binaerdatei (Datei))
    Typ = TYP_BINAER;
  else if (GIFdatei (Datei))
    Typ = TYP_GIF;
  else if (Bilddatei (Datei))
    Typ = TYP_BILD;
  else if (HTMLdatei (Datei))
    Typ = TYP_HTML;
  else if (XMLdatei (Datei))
    Typ = TYP_XML;
  else if (Dokumentdatei (Datei))
    Typ = TYP_DOKU;
  else if (Archivdatei (Datei))
    Typ = TYP_ARCHIV;
  else if (Videodatei (Datei))
    Typ = TYP_VIDEO;
  else if (Audiodatei (Datei))
    Typ = TYP_AUDIO;
  else if (UUEdatei (Datei))
    Typ = TYP_UUE;
  else if (BinHexdatei (Datei))
    Typ = TYP_BINHEX;
  else if (MIMEdatei (Datei))
    Typ = TYP_MIME;
  else if (Kalenderdatei (Datei))
    Typ = TYP_KALENDER;
  else				// unbekannt
    Typ = (int) Standardtyp;

  return Typ;
}


static bool
verbotene_Datei (const char *Datei)
{
  // nicht nur versteckt, sondern auch nicht abrufbar
  return (!strcmp ("gophertag", Datei)
	  || !strcmp ("gophermap", Datei)
	  || !strcmp ("lost+found", Datei)
	  || !fnmatch ("*~", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.bak", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tmp", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.temp", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.old", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cgi", Datei, FNM_OPTIONEN)
	  || !strcasecmp ("core", Datei)
	  || !fnmatch ("*.core", Datei, FNM_OPTIONEN)
	  || !fnmatch ("core.*", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.part", Datei, FNM_OPTIONEN));

  // *.gophermap wird gebraucht
}


static inline bool
Textdatei (const char *Datei)
{
  // nicht ausfuehrbare Dateien ohne Punkt seien Textdateien
  return (!strchr (Datei, '.')
	  || !fnmatch ("*.txt", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.text", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.md", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.markdown", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.asc", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ascii", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.inf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.NFO", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.info", Datei, FNM_OPTIONEN)
	  || !strcasecmp ("FILE_ID.DIZ", Datei)
	  || !fnmatch ("*.lsm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cnf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.conf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.config", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ini", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.log", Datei, FNM_OPTIONEN)
	  || !fnmatch ("README.*", Datei, FNM_OPTIONEN)
	  // Programmcode (unvollstaendig)
	  || !fnmatch ("*.sh", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.csh", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.bat", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cmd", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.[chpsdfm]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.c[cs]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cxx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.h[hp]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.hxx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.h++", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.d[di]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.[bp]as", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.p[lmpy]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.asm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.sx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.awk", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tcl", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tk", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.wat", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ad[abcs]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cbl", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.go", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.lua", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.java", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rs", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.scm", Datei, FNM_OPTIONEN));

  // FILE_ID.DIZ: DIZ = "Description In ZIP"
  // .NFO Dateien sind meist in Codepage 437
}


static inline bool
Binaerdatei (const char *Datei)
{
  return (!fnmatch ("*.bin", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.[oa]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.obj", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.so", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.exe", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.dll", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.com", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.bat", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.sys", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ttf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.woff", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.woff2", Datei, FNM_OPTIONEN));
}


static inline bool
HTMLdatei (const char *Datei)
{
  return (!fnmatch ("*.html", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.htm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xhtml", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xht", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.shtml", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.sht", Datei, FNM_OPTIONEN));
}


static inline bool
XMLdatei (const char *Datei)
{
  return (!fnmatch ("*.xml", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rss", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.atom", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rdf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xsl", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xslt", Datei, FNM_OPTIONEN));
}


static inline bool
Dokumentdatei (const char *Datei)
{
  return (!fnmatch ("*.od[tspmf]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.abw", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.pdf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rtf", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ps", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.dvi", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.djv", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.djvu", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cb[ztr]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.doc", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.docx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xls", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xlsx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ppt", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.pptx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.epub", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.azw", Datei, FNM_OPTIONEN));
}


static inline bool
GIFdatei (const char *Datei)
{
  return (!fnmatch ("*.gif", Datei, FNM_OPTIONEN));
}


static inline bool
Bilddatei (const char *Datei)
{
  return (!fnmatch ("*.png", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.jp[g2fx]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.jpeg", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.pcx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.bmp", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.svg", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.svgz", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tif", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tiff", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ico", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.x[pb]m", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.dcm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.dicom", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.webp", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.avif", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.avifz", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.jxl", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.hei[cf]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.wm[fz]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.exr", Datei, FNM_OPTIONEN));

  // kein *.gif
}

static inline bool
Archivdatei (const char *Datei)
{
  return (!fnmatch ("*.zip", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tar", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cpio", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.tgz", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.gz", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.z", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.bz2", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.xz", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rar", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.sea", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.7z", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.lha", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.lz[hs]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.cab", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.iso", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.img", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.image", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.deb", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.rpm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.dmg", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.apk", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.jar", Datei, FNM_OPTIONEN));
}


static inline bool
Audiodatei (const char *Datei)
{
  return (!fnmatch ("*.wav", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.au", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.snd", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.aif[fc]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.aif", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.wma", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mka", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mp[321]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mpga", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mpega", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.m4[ab]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.flac", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.wv", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.og[ga]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.opus", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.spx", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.weba", Datei, FNM_OPTIONEN));
}


static inline bool
Videodatei (const char *Datei)
{
  return (!fnmatch ("*.mov", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mng", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.avi", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.flv", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mp[ge]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mpeg", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mp4", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.m4v", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.mkv", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.wmv", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.og[vx]", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.webm", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.3g[p2]", Datei, FNM_OPTIONEN));
}


static inline bool
UUEdatei (const char *Datei)
{
  return (!fnmatch ("*.uue", Datei, FNM_OPTIONEN));
}


static inline bool
BinHexdatei (const char *Datei)
{
  return (!fnmatch ("*.h[ecq]x", Datei, FNM_OPTIONEN));
}


static inline bool
MIMEdatei (const char *Datei)
{
  return (!fnmatch ("*.mbox", Datei, FNM_OPTIONEN));
}


static inline bool
Kalenderdatei (const char *Datei)
{
  // RFC 5545 - iCalendar
  return (!fnmatch ("*.[iv]cs", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ical", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.icalendar", Datei, FNM_OPTIONEN)
	  || !fnmatch ("*.ifb", Datei, FNM_OPTIONEN));
}


// liest Zeile ohne Zeilenendezeichen
static char *
Zeilenlesen (FILE *ein, char *Zeile, size_t Laenge)
{
  if (!fgets (Zeile, Laenge, ein))
    return NULL;

  // Ende bereinigen, aber Leerzeichen am Ende belassen
  ssize_t pos = ((ssize_t) strlen (Zeile)) - 1;

  while (pos >= 0 && (Zeile[pos] == '\n' || Zeile[pos] == '\r'))
    Zeile[pos--] = '\0';

  return Zeile;
}


static char *
lies_Zeile (FILE *ein)
{
  char Puffer[10240];

  if (Zeilenlesen (ein, Puffer, sizeof (Puffer)))
    return strdup (Puffer);

  return NULL;
}


static char *
bereinigter_Pfad (char *Pfad)
{
  if (!Pfad)
    return NULL;

  // URLs unveraendert uebernehmen
  if (!strncmp (Pfad, "URL:", 4))
    return Pfad;

  // Schraegstriche am Anfang ignorieren
  // immer relativer Pfad (sicherheitsrelevant!)
  while (*Pfad == '/')
    ++Pfad;

  // kein Schraegstrich am Ende des Pfades
  size_t l = strlen (Pfad) - 1;
  if (Pfad[l] == '/')
    Pfad[l] = '\0';

  // Path-Traversal verhindern
  if (!strncmp (Pfad, "..", 2) || strstr (Pfad, "/../"))
    return NULL;

  return Pfad;
}


// Dateinamen in Pfad finden
static const char *
Dateiname (const char *Pfad)
{
  const char *Name;

  Name = strrchr (Pfad, '/');
  if (Name)
    ++Name;
  else
    Name = Pfad;

  return Name;
}


// Steuerzeichen rausfiltern
static char *
Filter (char *Zeile)
{
  for (size_t i = 0; Zeile[i]; ++i)
    {
      unsigned char z = (unsigned char) Zeile[i];
      if (z < 0x20 && z != '\t')
	Zeile[i] = ERSATZZEICHEN;
    }

  // nur C0-Steuerzeichen, TAB und DEL erlaubt
  // C1-Steuerzeichen sind problematisch, wegen UTF-8

  return Zeile;
}


// HTML-Code fuer Clients, die keine URL-Eintraege kennen, aber HTML
static void
HTML_Umleitung (FILE *aus, const char *url)
{
  fprintf (aus,
	   "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\r\n"
	   "<HTML>\r\n<HEAD>\r\n"
	   "<TITLE>Redirect</TITLE>\r\n"
	   "<META HTTP-EQUIV=\"refresh\" CONTENT=\"10; URL=%s\">\r\n"
	   "</HEAD>\r\n\r\n<BODY>\r\n<DIV ALIGN=CENTER>\r\n"
	   "<P>\r\nYou are following a link from gopher to a web site."
	   " You will be automatically taken to the web site"
	   " shortly.\r\n</P>\r\n"
	   "<P>\r\nIf you do not get sent there, please click"
	   " the following link to go to the web site.\r\n</P>\r\n"
	   "<P>\r\nThe URL linked is:\r\n</P>\r\n"
	   "<P>\r\n<BIG><A HREF=\"%s\">%s</A></BIG>\r\n</P>\r\n"
	   "<P>\r\nThanks for using gopher!\r\n</P>\r\n"
	   "<HR>\r\n<ADDRESS>\r\n" SERVER_NAME " " AKFNETZ_VERSION
	   "\r\n</ADDRESS>\r\n</DIV>\r\n</BODY>\r\n</HTML>\r\n",
	   url, url, url);

  /*
     Die Spezifikation besagt:
     "It must adhere to the W3C HTML 3.2 standard."

     Es sollte auch mit sehr alten Webbrowsern funktionieren.
     Die sind fuer das heutige Web zwar ungeeignet, aber die
     unterstuetzten noch Gopher.
   */
}


static int
gopher_inetd (void)
{
  // root ist verboten!
  if (!geteuid ())
    return EXIT_FAILURE;

  // Ein- und Ausgabe sollten sich auf den selben Socket beziehen
  Verbindungseinstellungen (STDIN_FILENO);

  Gopher_Anfrage (stdin, stdout);

  return EXIT_SUCCESS;
}


// setzt Zeitlimit fuer Verbindung
static void
Verbindungseinstellungen (int v)
{
  struct timeval z;

  // Auszeit setzen
  z.tv_sec = ZEITLIMIT;
  z.tv_usec = 0;

  // Fehler ignorieren (v muss nicht unbedingt ein Socket sein)
  setsockopt (v, SOL_SOCKET, SO_RCVTIMEO, &z, sizeof (z));
  setsockopt (v, SOL_SOCKET, SO_SNDTIMEO, &z, sizeof (z));

#ifdef TCP_CORK
  // moeglichst keine unvollstaendigen Frames senden
  int i = true;
  setsockopt (v, IPPROTO_TCP, TCP_CORK, &i, sizeof (i));
#endif
}


static noreturn void
Fehler (int Fehlernr, const char *Ursache)
{
  fputs (Programmname, stderr);

  if (Ursache && *Ursache)
    fprintf (stderr, ": %s", Ursache);

  if (Fehlernr)
    fprintf (stderr, ": %s (%d)",
	     deutsch ? Fehlermeldung (Fehlernr) : strerror (Fehlernr),
	     Fehlernr);

  fputc ('\n', stderr);

  exit (EXIT_FAILURE);
}


static void
Statusinformationen (FILE *d)
{
  Version (d);
  fputc ('\n', d);

  if (Bindungsadresse)
    fprintf (d, deutsch ? "Bind.:\t%s" : "Bound:\t%s", Bindungsadresse);
  else if (offen)
    fputs (deutsch ? "OFFEN!" : "OPEN!", d);
  else
    fputs (deutsch ? "gefiltert" : "filtered", d);

  fputc ('\n', d);

  fputs ("Proto:\t", d);
  switch (Protokoll)
    {
    case PF_INET6:
      fputs ("IPv6", d);
      break;

    case PF_INET:
      fputs ("IPv4", d);
      break;
    }

  fputc ('\n', d);

  fprintf (d, "PID:\t%ld\n", (long) getpid ());
  fputs (deutsch ? "Nutzer" : "User", d);
  fprintf (d, ":\t%s (%d)\n\n", Nutzer->pw_name, (int) Nutzer->pw_uid);

  fprintf (d, "Server:\t%s\n", Hostname);

  if (Protokoll == PF_INET || Protokoll == PF_INET6)
    {
      fprintf (d, "Port:\t%d\n", Hostport);
      fprintf (d, "\nURL:\tgopher://%s", Hostname);

      if (Hostport != 70)
	fprintf (d, ":%d", Hostport);

      fputs ("\n\n", d);
    }
}


/*
 * Servermodus
 */

// TODO
static noreturn void
Servermodus (void)
{
  int Lauscher;
  int Wert;

  Lauscher = socket (Protokoll, SOCK_STREAM, 0);

  // Kein IPv6 => IPv4 versuchen
  if (Lauscher < 0 && PF_INET6 == Protokoll
      && (errno == EPFNOSUPPORT || errno == EAFNOSUPPORT))
    Lauscher = socket ((Protokoll = PF_INET), SOCK_STREAM, 0);

  if (Lauscher < 0)
    Fehler (errno, "socket");

  fcntl (Lauscher, F_SETFD, FD_CLOEXEC);

  Wert = true;
  setsockopt (Lauscher, SOL_SOCKET, SO_REUSEADDR, &Wert, sizeof (Wert));

  // Namen zuweisen
  switch (Protokoll)
    {
    case PF_INET6:
      if (akfnetz_IPv6_Bindung (Lauscher, Bindungsadresse, Hostport) < 0)
	Fehler (errno, Bindungsadresse);
      break;

    case PF_INET:
      if (akfnetz_IPv4_Bindung (Lauscher, Bindungsadresse, Hostport) < 0)
	Fehler (errno, Bindungsadresse);
      break;

    default:
      Fehler (EPFNOSUPPORT, NULL);
    }

  Rechteaufgabe ();

  if (listen (Lauscher, LISTEN_BACKLOG) < 0)
    Fehler (errno, "listen");

  if (Daemon)
    {
      if (akfnetz_daemonisieren () < 0)
	Fehler (errno, "daemon");
    }
  else
    Statusinformationen (stderr);

  // kehrt nie zurueck
  Verbindungsannahme (Lauscher);
  abort ();
}


static noreturn void
Verbindungsannahme (int Lauscher)
{
  struct sockaddr_storage Klient;
  socklen_t Klientengroesse;
  int Verbindung;
  pid_t Prozess;

  while (true)
    {
      do
	{
	  Klientengroesse = sizeof (Klient);
	  Verbindung = accept (Lauscher,
			       (struct sockaddr *) &Klient, &Klientengroesse);
	}
      while (Verbindung < 0);

      // Zugriffe von nicht privaten Adressen abweisen
      if (!offen && !akfnetz_privates_Netz (&Klient))
	{
	  close (Verbindung);
	  continue;
	}

      fcntl (Verbindung, F_SETFD, FD_CLOEXEC);

      // Kindprozess erstellen
      Prozess = fork ();

      switch (Prozess)
	{
	case 0:		// Kindprozess
	  close (Lauscher);
	  // Klientenadresse (&Klient, Klientengroesse);  TODO
	  _exit (Verbindungsverarbeitung (Verbindung));
	  break;

	case -1:		// Fehler
	  // TODO
	  break;
	}

      close (Verbindung);
    }				// Schleife

  // wird nie erreicht
  abort ();
}


// Kindprozess
static int
Verbindungsverarbeitung (int Verbindung)
{
  Verbindungseinstellungen (Verbindung);

  // zwei Dateistroeme fuer unabhaengige Puffer
  FILE *ein, *aus;
  aus = NULL;
  ein = fdopen (Verbindung, "rb");
  if (!ein)
    {
      close (Verbindung);
      return 127;
    }

  int n = dup (Verbindung);
  fcntl (n, F_SETFD, FD_CLOEXEC);
  aus = fdopen (n, "wb");
  if (!aus)
    {
      fclose (ein);
      close (n);
      return 127;
    }

  Gopher_Anfrage (ein, aus);
  fclose (aus);
  fclose (ein);

  return EXIT_SUCCESS;
}


/*
 * Benutzer-Rechte
 */

static int
Gruppenzuweisung (const char *Name, gid_t Gruppe)
{
  size_t Anzahl = 0;
  gid_t Liste[256];

  Liste[Anzahl++] = Gruppe;

  setgrent ();

  while (Anzahl < 256)
    {
      struct group *g;

      g = getgrent ();
      if (!g)
	break;

      if (g->gr_gid == Gruppe)
	continue;

      // Namensliste durchsuchen
      for (char **nl = g->gr_mem; *nl; ++nl)
	if (!strcmp (*nl, Name))
	  {
	    Liste[Anzahl++] = g->gr_gid;
	    break;
	  }
    }

  endgrent ();

  return setgroups (Anzahl, Liste);
}


// eventuell root-Rechte ab jetzt aufgeben
static void
Rechteaufgabe (void)
{
  uid_t euid;

  euid = geteuid ();

  // unprivilegiert per setuid, aber dann als root aufgerufen?
  // => erst Privilegien erlangen, um sie dann richtig abweisen zu koennen.
  if (euid != 0 && getuid () == 0)
    {
      if (seteuid (0) < 0)
	Fehler (errno, "seteuid");
      euid = geteuid ();
    }

  if (euid != 0)
    return;

  // die uid zum Schluss, sonst hat man nicht mehr die noetigen Rechte
  if (Nutzer->pw_uid == 0
      || setgid (Nutzer->pw_gid) < 0
      || Gruppenzuweisung (Nutzer->pw_name, Nutzer->pw_gid) < 0
      || setuid (Nutzer->pw_uid) < 0 || (euid = geteuid ()) == 0)
    Fehler (errno, deutsch ? "Rechte setzen" : "setting privileges");

  // Hiernach sollte das Zurueckerlangen von root-Rechten unmoeglich sein.

  if (euid != Nutzer->pw_uid)
    {
      errno = EPERM;
      Nutzer = getpwuid (euid);
      if (!Nutzer)
	Fehler (errno, "getpwuid");
    }
}


static void
Nutzerinitialisierung (void)
{
  uid_t uid = getuid ();
  if (uid == 0)
    uid = geteuid ();

  errno = EPERM;
  Nutzer = getpwuid (uid);
  if (!Nutzer)
    Fehler (errno, NULL);

  // 0 ist der Superuser - der darf zu viel
  // Die eigentliche Rechteaufgabe erfolgt aber erst spaeter
  if (Nutzer->pw_uid == 0 && Nutzerkennung (getenv ("LOGNAME")) == 0
      && Nutzerkennung (getenv ("SUDO_UID")) == 0
      && Nutzerkennung (getenv ("SU_FROM")) == 0
      && Nutzerkennung ("nobody") == 0)
    Fehler (EPERM, "root");
}


// Nutzerkennung aendern (Nummer oder Name)
// es werden hier nur die Variablen gesetzt
static uid_t
Nutzerkennung (const char *Name)
{
  if (!Name || !*Name)
    return Nutzer->pw_uid;

  char *e;
  uid_t u = (uid_t) strtoul (Name, &e, 10);

  Nutzer = *e ? getpwnam (Name) : getpwuid (u);

  if (!Nutzer)
    Fehler (EINVAL, Name);

  return Nutzer->pw_uid;
}


static void
Signalverarbeitung (int signum)
{
  if (signum == SIGCHLD)
    {
      int e = errno;
      while (waitpid (-1, NULL, WNOHANG) > 0);
      errno = e;
    }
}


/*
 * Initialisierung
 * nach Einlesen der Optionen
 */

static void
Initialisiere (void)
{
  if (!Hostname)
    {
      // Hostname ermitteln
      char *h = getenv ("HOSTNAME");
      if (!h)
	h = getenv ("HOST");

      if (h && *h)
	Hostname = strdup (h);
      else
	{
	  char Name[HOST_NAME_MAX + 1];
	  if (!gethostname (Name, sizeof (Name)))
	    Hostname = strdup (Name);
	  else
	    Hostname = strdup ("localhost");	// letzter Ausweg
	}
    }

  if (!Hostport)
    {
      if (inetd)
	{
	  int Port = akfnetz_Socketport (STDIN_FILENO);
	  Hostport = (Port < 0) ? 70 : Port;
	}
      else			// Server-Modus
	Hostport = (geteuid () == 0) ? 70 : 7070;
    }

  if (!Serverport)
    Serverport = Hostport;

  // an IPv4-Adresse binden? -> auf IPv4 einschraenken
  if (PF_INET6 == Protokoll && Bindungsadresse
      && !strchr (Bindungsadresse, ':'))
    Protokoll = PF_INET;

  // Zombies vermeiden
  struct sigaction sa;
  sa.sa_handler = &Signalverarbeitung;
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART;
  if (sigaction (SIGCHLD, &sa, NULL) < 0)
    Fehler (errno, "sigaction");
}


// gibt Portnummer als Zahl zurueck, mit Bereichsueberpruefung
static int
Portnummer (const char *Argument)
{
  long int v;

  v = strtol (Argument, NULL, 10);
  if (0 >= v || v > 0xFFFF)
    Fehler (EDOM, Argument);

  return v;
}


static void
Optionen (int argc, char **argv)
{
  int opt;

  Hostname = Bindungsadresse = NULL;
  Protokoll = PF_INET6;
  offen = inetd = Daemon = false;
  Standardtyp = TYP_BINAER;
  deutsch = akfnetz_deutschsprachig ();
  Nutzerinitialisierung ();

  optind = 1;
  opterr = 0;

  while ((opt = getopt_long (argc, argv, Kurzoptionen,
			     Langoptionen, NULL)) != -1)
    {
      switch (opt)
	{
	case 'h':
	  Hilfe ();
	  break;

	case 'V':
	  Version (stdout);
	  exit (EXIT_SUCCESS);
	  break;

	case 'N':
	  if (Hostname)
	    free (Hostname);
	  Hostname = strdup (optarg);
	  break;

	case 'p':
	  Hostport = Portnummer (optarg);
	  break;

	case OPT_SERVERPORT:
	  Serverport = Portnummer (optarg);
	  break;

	case 'd':
	  if (chdir (optarg) < 0)
	    Fehler (errno, optarg);
	  break;

	case 't':		// text
	  Standardtyp = TYP_TEXT;
	  break;

	case 'i':
	  inetd = true;
	  break;

	case '4':		// IPv4
	  Protokoll = PF_INET;
	  break;

	case '6':		// IPv6
	  Protokoll = PF_INET6;
	  break;

	case 'D':		// daemon
	  Daemon = true;
	  break;

	case 'a':		// Adresse, address
	  if (Bindungsadresse)
	    free (Bindungsadresse);
	  Bindungsadresse = strdup (optarg);
	  break;

	case 'l':		// localhost
	  Protokoll = PF_INET;
	  if (Bindungsadresse)
	    free (Bindungsadresse);
	  Bindungsadresse = strdup ("127.0.0.1");
	  if (Hostname)
	    free (Hostname);
	  Hostname = strdup ("localhost");
	  break;

	case 'u':		// Nutzer
	  // nur root als reale uid darf die uid aendern
	  if (getuid () != 0)
	    Fehler (EPERM, optarg);

	  Nutzerkennung (optarg);
	  break;

	case OPT_OFFEN:
	  offen = true;
	  break;

	case '?':
	  Fehler (EINVAL, argv[optind - 1]);
	  break;
	}
    }
}
