/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Simpler Gopher Client
 * Copyright (c) 2024-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <locale.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "akfnetz.h"
#include "version.h"

#define Fehlertext(s)  write(STDERR_FILENO, "" s "", sizeof(s)-1)

#ifndef Fehlermeldung
#define Fehlermeldung  strerror
#endif

static int Hilfe (const char *PrgName);
static char *var (const char *);
static const char *Proxyerkennung (void);
static inline void Fehlerstring (const char *);
static inline int Fehlernummer (const char *Name, int e);

int
main (int argc, char **argv)
{
  setlocale (LC_ALL, "");

  // nur genau ein Parameter erlaubt, keine Option
  // Das deckt auch --help und --version ab
  if (argc < 2)
    return Fehlernummer (argv[0], EDESTADDRREQ);
  else if (argc != 2 || argv[1][0] == '-')
    return Hilfe (argv[0]);

  int v = akfnetz_Gopher_URL (argv[1], Proxyerkennung ());
  if (v < 0)
    return Fehlernummer (argv[0], errno);

  if (akfnetz_kopiere (v, STDOUT_FILENO) < 0)
    return Fehlernummer (argv[0], errno);

  close (v);
  return EXIT_SUCCESS;
}


static int
Hilfe (const char *PrgName)
{
  Fehlertext ("akfgopher-dl (akfnetz " AKFNETZ_VERSION ")\n"
	      AKFNETZ_COPYRIGHT "\n"
	      "Lizenz GPLv3+: GNU GPL Version 3 oder neuer"
	      " <http://gnu.org/licenses/gpl.html>\n\n");

  Fehlerstring (PrgName);
  Fehlertext (" gopher://gopher.example.net/0/info | less\n");
  Fehlerstring (PrgName);
  Fehlertext (" gopher://gopher.example.net/1/ | gophermenu | less\n");
  Fehlerstring (PrgName);
  Fehlertext (" gopher://gopher.example.net/7/v2/vs%09Suchanfrage"
	      " | gophermenu | less\n\n");

  Fehlertext ("Umgebungsvariablen:\n - GOPHER_PROXY\n - ALL_PROXY\n");

  return EXIT_FAILURE;
}


static char *
var (const char *n)
{
  char *e = getenv (n);

  if (e && !*e)
    e = NULL;

  return e;
}


static const char *
Proxyerkennung (void)
{
  const char *proxy;

  if (!(proxy = var ("GOPHER_PROXY")) && !(proxy = var ("gopher_proxy"))
      && !(proxy = var ("ALL_PROXY")))
    proxy = var ("all_proxy");

  return proxy;
}


static inline void
Fehlerstring (const char *s)
{
  write (STDERR_FILENO, s, strlen (s));
}


static inline int
Fehlernummer (const char *Programmname, int e)
{
  Fehlerstring (Programmname);
  Fehlertext (": ");
  Fehlerstring (Fehlermeldung (e));
  Fehlertext ("\n");

  return EXIT_FAILURE;
}
