/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfgb - Gopher-Browser
 * Copyright (c) 2024-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * keine harten Abhaengigkeiten,
 * kein stdio.h
 */

#include <locale.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdbool.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <limits.h>
#include <fnmatch.h>
#include "akfnetz.h"
#include "gophertyp.h"
#include "akfgb.h"
#include "version.h"

#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX 255
#endif

//#define schwarzweiss

// Wieviele Zeilen der Anzeige reserviert sind
#define RESERVIERT 2

#define TYP_URL 0x7F
#define TYP_KAPUTT 0x15		// NAK

#define Text(s)       write (STDOUT_FILENO, "" s, sizeof(s)-1)
#define Steuerung(s)  Text(s)
// Fuer Variablen: Ausgabe(), Ausgabe_l()  oder Zahl()

#define Text_rechts(z, s) \
  do{Anzeige_Position(z,Anzeige.Breite+2-sizeof(s));Text(s);}while(0)
// +2 wegen Terminator und weil Koordinaten bei 1 beginnen

// Steuersequenzen
#define ESC  "\033"		// Escape
#define CSI  "\033["		// Control Sequence Introducer

#define Glocke             "\a"
#define Zeilenanfang       "\r"
#define Zeile_runter       ESC "D"	// IND
#define Zeile_hoch         ESC "M"	// RI
#define Anzeige_leeren     CSI "2J"	// ED
#define Zeile_leeren       CSI "2K"	// EL
#define Rest_leeren        CSI "K"	// EL
#define Heim               CSI "H"	// CUP
#define Textanfang         CSI "2H"
#define normal             CSI "m"	// SGR
#define Tastendarstellung  CSI "0;7m"
#define Gesamtrollbereich  CSI "r"	// DECSTBM
#define Cursor_an          CSI "?25h"	// SM
#define Cursor_aus         CSI "?25l"	// RM
#define Umbruch_an         CSI "?7h"
#define Umbruch_aus        CSI "?7l"
#define AltBuf_an          CSI "?47h"	// xterm (kein Rollbalken)
#define AltBuf_aus         CSI "?47l"
// siehe auch: Anzeige_Position, Anzeige_vorwaerts, Anzeige_Rollbereich

#ifdef schwarzweiss
#define Typdarstellung     CSI "1m"
#define Kommentare         CSI "1m"
#define Rollbalken         CSI "7m"
#define Statuszeile        CSI "0;7m"
#define Aktion             CSI "0;7m"
#define Warnung            CSI "0;7;1;5m"
#else // farbig
#define Typdarstellung     CSI "33m"
#define Kommentare         CSI "36m"
#define Rollbalken         CSI "0;30;46m"
#define Statuszeile        CSI "0;37;44m"
#define Aktion             CSI "0;37;42m"
#define Warnung            CSI "0;1;33;41m"
#endif

#define Tastentext(t)  Tastendarstellung " " #t " " normal

// Tastenkombination Buchstabe mit Ctrl | Strg
#define T_CTRL(b)  ((b) & 0x1F)

// Tasten
#define T_LEERTASTE 32
#define T_ESC 27		// CTRL [
#define T_TAB 9			// CTRL I
#define T_ENTER 13		// CTRL M
#define T_BACKSPACE 127		// CTRL ?

// interne Tastennummern
// muss ueber 255 sein, ansonsten beliebig
#define T_HOCH        1000
#define T_RUNTER      1001
#define T_LINKS       1002
#define T_RECHTS      1003
#define T_BILD_HOCH   1004
#define T_BILD_RUNTER 1005
#define T_POS1        1006
#define T_ENDE        1007
#define T_F1          2001
#define T_F2          2002
#define T_F3          2003
#define T_F4          2004
#define T_F5          2005


#ifndef ENODATA
// passt nicht wirklich, aber...
#define ENODATA ENOENT
#endif

#ifndef FNM_CASEFOLD
#define FNM_CASEFOLD 0
#endif

#ifndef TELNET
#define TELNET NULL
#endif

#ifndef TN3270
#define TN3270 NULL
#endif

#ifndef PH
#define PH NULL
#endif

#ifndef BILDBETRACHTER
#define BILDBETRACHTER NULL
#endif

#ifndef VIDEOABSPIELER
#define VIDEOABSPIELER NULL
#endif

#ifndef AUDIOABSPIELER
#define AUDIOABSPIELER NULL
#endif

#ifndef URL_BEHANDLUNG
#define URL_BEHANDLUNG NULL
#endif

typedef unsigned int Koordinate;

struct Inhalte
{
  Koordinate Versatz, aktiv;
  Koordinate Zeilenanzahl, Zeilenmaximum;
  char **Zeile;
  char *url;
  char *Puffer;
};

struct Anzeigedaten
{
  Koordinate Hoehe, Breite;
  bool aktualisieren;
};

static const char *Programmname, *Startadresse, *Proxy;
static struct termios Grundeinstellungen;
static volatile struct Anzeigedaten Anzeige;
static struct Inhalte Menue;
static bool deutsch;

#define HISTORIE_ANZAHL 20
static char *Historie[HISTORIE_ANZAHL];
// FIFO


static void Initialisierung (void);
static void URL_ausfuehren (int Typ, const char *url);
static void ausfuehren (const char *Zeile);
static void Normalmodus (void);
static void Rohmodus (void);
static void Anzeigegroesse (void);
static void Anzeige_Position (Koordinate Zeile, Koordinate Spalte);
static void Anzeige_vorwaerts (Koordinate Nr);
static void Anzeige_Rollbereich (Koordinate oben, Koordinate unten);
static int Binaerdatei (const char *url);
static int Befehlsaufruf (const char *url, const char *Befehl);
static int speichern (const char *Zeile);
static int URL_speichern (int Typ, const char *url);
static bool Speichernachfrage (int Typ, const char *Name, const char *url);
static void Speichererfolg (const char *Name);
static int Dienstprogramm (int Typ, const char *url);
static int Adressaufruf (const char *url);
static int Menue_einlesen (const char *url);
static int Inhalt_einlesen (struct Inhalte *Inhalt, const char *url);
static int Inhalt_uebernehmen (struct Inhalte *Inhalt,
			       const char *Text, size_t Laenge);
static void Inhalt_aufteilen (struct Inhalte *Inhalt);
static void Inhalt_bereinigen (struct Inhalte *Inhalt);
static void Inhalt_freigeben (struct Inhalte *Inhalt);
static char *Eingabeaufforderung (const char *Aufforderung);
static int suchen (const char *url);
static int URL_Eingabe (void);
static void Kopfzeile (const char *url);
static void Fusszeile (void);
static void warte (time_t Sekunden);
static void Statusmeldung (const char *Meldung);
static inline void Infostatus (int);
static int Fehlerstatus (int);
static void Menue_anzeigen (void);
static void Menue_Zeile_anzeigen (Koordinate);
static void Menue_Auswahl (void);
static void Menue_hoch (void);
static char *Plus (void);
static void Rollbalken_aktualisieren (Koordinate alt);
static void runterblaettern (void);
static void hochblaettern (void);
static char *URL_erstellen (const char *Zeile);
static bool externe_Adresse (const char *Zeile);
static int Gophertyp (const char *url);
static const char *Typname (int Typ);
static int Hilfe (void);
static int Version (void);
static void Onlinehilfe (void);
static char *var (const char *);
static const char *Proxyerkennung (void);
static inline int Fehlernummer (int);
static inline void Ausgabe_l (const void *Meldung, size_t Laenge);
static inline void Ausgabe (const void *Meldung);
static void Zahl (unsigned long int);
static void Wertanzeige (const char *);
static bool Tastendruck (void);
static int Taste (void);
static int Funktionstaste (void);
static inline void Tastenpuffer_leeren (void);
static inline void sende_Taste (char c);
static void Textbetrachter (const char *url);
static void Inhaltsbetrachter (struct Inhalte *Inhalt);
static void Text_anzeigen (struct Inhalte *);
static void Einruecken (void);
static void Historie_neu (const char *url);
static void Historie_zurueck (void);
static int Historie_anzeigen (void);
static char *Fehlerbericht (int);


int
main (int argc, char **argv)
{
  setlocale (LC_ALL, "");
  Programmname = argv[0];
  Proxy = Proxyerkennung ();
  deutsch = akfnetz_deutschsprachig ();

  if (argc >= 2)
    {
      if (!strcmp (argv[1], "--version"))
	return Version ();
      else if (*argv[1] == '-' || argc > 2)
	return Hilfe ();
      else
	Startadresse = argv[1];
    }

  if (!Startadresse)
    Startadresse = var ("GOPHER_HOME");

  if (Startadresse
      && strncasecmp ("gopher://", Startadresse, 9)
      && strncasecmp ("finger://", Startadresse, 9)
      && strncasecmp ("telnet://", Startadresse, 9)
      && strncasecmp ("tn3270://", Startadresse, 9)
      && strncasecmp ("cso://", Startadresse, 6))
    return Fehlernummer (EPROTONOSUPPORT);

  Initialisierung ();

  if (!Startadresse)
    {
      Steuerung (Textanfang);
      Version ();
      Infostatus (EDESTADDRREQ);
      // Startadresse auf Heap, wird aber nie freigegeben
      Startadresse = Eingabeaufforderung ("URL: ");
      if (!Startadresse)
	{
	  Steuerung (Anzeige_leeren Heim);
	  return EXIT_SUCCESS;
	}
    }

  register int Typ = Gophertyp (Startadresse);
  if (Typ != TYP_MENUE && Typ != TYP_SUCHE)	// kein Menue
    {
      URL_ausfuehren (Typ, Startadresse);
      Steuerung (Anzeige_leeren Heim);
      return EXIT_SUCCESS;
    }

  // Menue
  if (Menue_einlesen (Startadresse) < 0)
    {
      Fehlerstatus (errno);
      Tastenpuffer_leeren ();
      Steuerung (Anzeige_leeren Heim);
      return EXIT_FAILURE;
    }

  Historie_neu (Startadresse);

  while (true)
    Menue_Auswahl ();

  return EXIT_SUCCESS;
}


static int
Version (void)
{
  Text ("akfgb (akfnetz " AKFNETZ_VERSION ")\n" AKFNETZ_COPYRIGHT "\n");

  if (deutsch)
    Text ("Lizenz GPLv3+: GNU GPL Version 3 oder neuer");
  else
    Text ("License GPLv3+: GNU GPL version 3 or newer");

  Text (" <http://gnu.org/licenses/gpl.html>\n\n");

  return EXIT_SUCCESS;
}


static int
Hilfe (void)
{
  Version ();
  if (deutsch)
    Text ("Verwendung: akfgb [URL]\n\nUmgebungsvariablen:\n");
  else
    Text ("Usage: akfgb [URL]\n\nEnvironment:\n");

  Text (" - GOPHER_HOME");
  Wertanzeige (getenv ("GOPHER_HOME"));
  Text ("\n - GOPHER_PROXY | ALL_PROXY");
  Wertanzeige (Proxy);

  Text ("\n\nURL:\t");
  Ausgabe (URL_BEHANDLUNG ? URL_BEHANDLUNG : "-");
  Text ("\nTelnet:\t");
  Ausgabe (TELNET ? TELNET : "-");
  Text ("\nTN3270:\t");
  Ausgabe (TN3270 ? TN3270 : "-");
  Text ("\nPH:\t");
  Ausgabe (PH ? PH : "-");
  if (deutsch)
    Text ("\nBild:\t");
  else
    Text ("\nImage:\t");
  Ausgabe (BILDBETRACHTER ? BILDBETRACHTER : "-");
  Text ("\nVideo:\t");
  Ausgabe (VIDEOABSPIELER ? VIDEOABSPIELER : "-");
  Text ("\nAudio:\t");
  Ausgabe (AUDIOABSPIELER ? AUDIOABSPIELER : "-");
  Text ("\n");
  // siehe auch: Onlinehilfe ()

  return EXIT_SUCCESS;
}


static void
URL_ausfuehren (int Typ, const char *url)
{
  if (!url)
    return;

  switch (Typ)
    {
    case TYP_TEXT:
    case TYP_BINHEX:
    case TYP_UUE:
    case TYP_HTML:		// roh anzeigen besser als nichts tun
    case TYP_XML:
    case TYP_MIME:
    case TYP_KALENDER:
      Textbetrachter (url);
      break;

    case TYP_MENUE:
      if (Menue_einlesen (url) < 0)
	Fehlerstatus (errno);
      Historie_neu (url);
      break;

    case TYP_SUCHE:
      if (suchen (url) < 0)
	Fehlerstatus (errno);
      break;

    case TYP_BILD:
    case TYP_GIF:
    case TYP_BITMAP:
    case TYP_PNG:
      if (Befehlsaufruf (url, BILDBETRACHTER) < 0)
	Fehlerstatus (errno);
      break;

    case TYP_AUDIO:
    case TYP_TON:
      if (Befehlsaufruf (url, AUDIOABSPIELER) < 0)
	Fehlerstatus (errno);
      break;

    case TYP_VIDEO:
      if (Befehlsaufruf (url, VIDEOABSPIELER) < 0)
	Fehlerstatus (errno);
      break;

    case TYP_CCSO:
    case TYP_TELNET:
    case TYP_TN3270:
      if (Dienstprogramm (Typ, url) < 0)
	Fehlerstatus (errno);
      break;

    case TYP_INFO:
    case TYP_FEHLER:
      // nichts machen
      break;

    case TYP_BINAER:
      // Binaerdatei kann evtl. Medien-Dateien wiedergeben
      if (Binaerdatei (url) < 0)
	URL_speichern (Typ, url);
      break;

    case TYP_ARCHIV:
    case TYP_DOKU:
    case TYP_RTF:
    case TYP_PDF:
      URL_speichern (Typ, url);
      break;

    case TYP_KAPUTT:
      Kopfzeile (url);
      Fehlerstatus (EPROTONOSUPPORT);
      Kopfzeile (Menue.url);
      break;

      // was er wirklich nicht kann
    case TYP_ALT:
    default:
      Kopfzeile (url);
      Fehlerstatus (ENOSYS);
      Kopfzeile (Menue.url);
      break;
    }
}


// fuehrt Menuezeile aus
static void
ausfuehren (const char *Zeile)
{
  char *url = URL_erstellen (Zeile);
  if (!url)
    return;			// still ignorieren

  // gopher kann hier nur kleingeschrieben sein
  // finger kann ueber "URL:" eingefuegt worden sein
  if ((*url == 'g' && !strncmp ("gopher:", url, 7))
      || !strncasecmp ("finger:", url, 7))
    URL_ausfuehren (*Zeile, url);
  else				// kein Gopher/Finger
    {
      Kopfzeile (url);
      if (Adressaufruf (url) < 0)
	Fehlerstatus (errno);
      Kopfzeile (Menue.url);
    }

  free (url);
}


static void
Kopfzeile (const char *url)
{
  Steuerung (Heim Statuszeile Zeile_leeren);

  if (url)
    {
      if (Anzeige.Breite >= 80)
	Text ("URL: ");

      Ausgabe (url);
    }

  Steuerung (normal);
}


static void
Fusszeile (void)
{
  Anzeige_Position (Anzeige.Hoehe, 1);
  Steuerung (Statuszeile Zeile_leeren);

  Text (" akfgb (AKFNetz)");

  if (deutsch)
    Text_rechts (Anzeige.Hoehe, "Hilfe: ? ");
  else
    Text_rechts (Anzeige.Hoehe, "Help: ? ");

  Steuerung (normal);
}


static void
Statusmeldung (const char *Meldung)
{
  Anzeige_Position (Anzeige.Hoehe, 1);
  Steuerung (Aktion Zeile_leeren);
  if (Meldung)
    Ausgabe (Meldung);
  Steuerung (normal);
}


static inline void
Infostatus (int info)
{
  Statusmeldung (Fehlerbericht (info));
}


static int
Fehlerstatus (int Fehler)
{
  Anzeige_Position (Anzeige.Hoehe, 1);
  Steuerung (Warnung Zeile_leeren);
  Ausgabe (Fehlerbericht (Fehler));
  Steuerung (normal Glocke);

  warte (MELDUNGSZEIT);
  Fusszeile ();

  return -1;
}


static char *
Fehlerbericht (int e)
{
#ifdef Fehlermeldung
  return (deutsch ? Fehlermeldung (e) : strerror (e));
#else
  return (strerror (e));
#endif
}



static void
Menue_anzeigen (void)
{
  Steuerung (normal Anzeige_leeren);
  Kopfzeile (Menue.url);
  Fusszeile ();

  Steuerung (Textanfang);
  Anzeige_Rollbereich (2, Anzeige.Hoehe - 1);

  Koordinate Maximum = Menue.Zeilenanzahl;
  if (Maximum > Anzeige.Hoehe - RESERVIERT)
    Maximum = Anzeige.Hoehe - RESERVIERT;

  if (Menue.aktiv >= Maximum + Menue.Versatz)
    Menue.aktiv = Maximum + Menue.Versatz - 1;

  for (Koordinate i = 0; i < Maximum; ++i)
    {
      Koordinate Nr = i + Menue.Versatz;
      if (Nr >= Menue.Zeilenanzahl)
	break;

      Text ("\n");
      Menue_Zeile_anzeigen (Nr);
    }

  Anzeige.aktualisieren = false;
}


// Zeigt Menue_Zeile
// Cursor muss schon an richtiger Position sein
static void
Menue_Zeile_anzeigen (Koordinate Nr)
{
  if (Menue.aktiv == Nr)
    Steuerung (Rollbalken);

  Steuerung (Zeile_leeren);
  Einruecken ();

  char *Zeile = Menue.Zeile[Nr];
  int Typ = (int) *Zeile;

  if (externe_Adresse (Zeile))
    Typ = TYP_URL;

  // Es gibt wirklich grausamen Datenmuell da draussen
  if (Typ > 0x20 && Typ <= TYP_URL)
    {
      if (Anzeige.Breite >= 80)
	{
	  Steuerung (CSI "4D" Typdarstellung);
	  Ausgabe_l (Typname (Typ), 4);
	  Text (" ");

	  if (Menue.aktiv == Nr)
	    Steuerung (Rollbalken);
	  else
	    Steuerung (normal);
	}

      char *Name = Zeile + 1;
      Ausgabe_l (Name, strcspn (Name, "\t"));
    }

  if (Menue.aktiv == Nr)
    Steuerung (normal);
}


static void
Menue_hoch (void)
{
  if (!Menue.url)
    return;

  size_t l = strlen (Menue.url) + 1;
  char url[l];
  memcpy (url, Menue.url, l);

  // Pfad zeigt auf url
  // bewusst aenderbar machen
  char *Pfad = (char *) akfnetz_URL_Pfad (url);
  if (!Pfad)
    return;

  size_t p = strlen (Pfad);

  // nur noch Typ?
  if (p <= 3)
    return;

  --p;

  // endet mit Schraegstrich?
  if (Pfad[p] == '/')
    --p;

  while (p && Pfad[p] != '/')
    --p;

  // Aenderung an Pfad aendert auch url
  Pfad[p + 1] = '\0';

  Menue_einlesen (url);
}


// Typ '+'? - veraendern
// (veraendert Menue.Zeile)
static char *
Plus (void)
{
  char *Zeile = Menue.Zeile[Menue.aktiv];

  if (Menue.aktiv > 0 && TYP_ALT == *Zeile)
    {
      Koordinate letzte = Menue.aktiv - 1;
      while (letzte > 0 && *Menue.Zeile[letzte] == TYP_ALT)
	--letzte;
      *Zeile = *Menue.Zeile[letzte];
    }

  return Zeile;
}


static void
Menue_Auswahl (void)
{
  char *url = NULL;
  Koordinate alt;

  if (Anzeige.aktualisieren)
    Menue_anzeigen ();

  switch (Taste ())
    {
    case T_ESC:
    case 'q':
    case 'Q':
      //case T_CTRL ('C'):
      Steuerung (Anzeige_leeren Heim);
      exit (EXIT_SUCCESS);
      break;

    case T_F1:
    case '?':
      Onlinehilfe ();
      break;

    case 'h':
    case T_POS1:
      Menue.Versatz = Menue.aktiv = 0;
      Anzeige.aktualisieren = true;
      break;

    case T_ENDE:
      if (Menue.Zeilenanzahl > Anzeige.Hoehe - RESERVIERT)
	Menue.Versatz = Menue.Zeilenanzahl - (Anzeige.Hoehe - RESERVIERT);

      if (Menue.Zeilenanzahl)
	Menue.aktiv = Menue.Zeilenanzahl - 1;
      Anzeige.aktualisieren = true;
      break;

    case 'j':			// runter
    case T_RUNTER:
      if (!Menue.Zeilenanzahl)
	break;

      alt = Menue.aktiv;
      if (Menue.aktiv < Menue.Zeilenanzahl - 1)
	++Menue.aktiv;

      if (Menue.aktiv >= Menue.Versatz + (Anzeige.Hoehe - RESERVIERT))
	{
	  Steuerung (Zeile_runter);
	  ++Menue.Versatz;
	}

      Rollbalken_aktualisieren (alt);
      break;

    case 'k':			// rauf
    case T_HOCH:
      if (!Menue.Zeilenanzahl)
	break;

      alt = Menue.aktiv;
      if (Menue.aktiv > 0)
	--Menue.aktiv;

      if (Menue.aktiv < Menue.Versatz)
	{
	  Steuerung (Textanfang Zeile_hoch);
	  --Menue.Versatz;
	}

      Rollbalken_aktualisieren (alt);
      break;

    case T_BILD_RUNTER:
    case T_LEERTASTE:
      if (Menue.Zeilenanzahl)
	runterblaettern ();
      Menue.aktiv = Menue.Versatz;
      Anzeige.aktualisieren = true;
      break;

    case T_BILD_HOCH:
    case 'b':
      if (Menue.Zeilenanzahl)
	hochblaettern ();
      Menue.aktiv = Menue.Versatz;
      Anzeige.aktualisieren = true;
      break;

    case 'o':
      URL_Eingabe ();
      break;

    case T_F2:
    case 'm':
      Menue_einlesen (Startadresse);
      Historie_neu (Startadresse);
      break;

    case T_ENTER:
    case T_RECHTS:
      if (Menue.Zeilenanzahl)
	ausfuehren (Plus ());
      break;

    case T_LINKS:
    case T_BACKSPACE:
    case 'u':
      Historie_zurueck ();
      break;

    case 't':			// als Text anzeigen
      if (!strchr ("i2378T", *Menue.Zeile[Menue.aktiv]))
	{
	  url = URL_erstellen (Plus ());
	  URL_ausfuehren (TYP_TEXT, url);
	  free (url);
	  url = NULL;
	}
      break;

    case 'a':
      url = URL_erstellen (Plus ());
      if (url)
	{
	  Kopfzeile (url);
	  free (url);
	  url = NULL;
	  warte (MELDUNGSZEIT);
	  Kopfzeile (Menue.url);
	}
      break;

    case T_F4:
    case T_CTRL ('S'):
      if (Menue.Zeilenanzahl)
	speichern (Menue.Zeile[Menue.aktiv]);
      break;

    case 'w':			// TODO ????
      Menue_hoch ();
      break;

    case T_CTRL ('L'):
    case T_CTRL ('R'):
      Anzeige.aktualisieren = true;
      break;

    case T_CTRL ('H'):
      Historie_anzeigen ();
      Anzeige.aktualisieren = true;
      break;
    }
}


// nur fuer Menue
static void
runterblaettern (void)
{
  Menue.Versatz += (Anzeige.Hoehe - RESERVIERT);
  if (Menue.Versatz >= Menue.Zeilenanzahl)
    Menue.Versatz = Menue.Zeilenanzahl - 1;
}


// nur fuer Menue
static void
hochblaettern (void)
{
  if (Menue.Versatz > Anzeige.Hoehe - RESERVIERT)
    Menue.Versatz -= (Anzeige.Hoehe - RESERVIERT);
  else
    Menue.Versatz = 0;
}


static void
Rollbalken_aktualisieren (Koordinate alt)
{
  // +2: Zeilen werden ab 1 gezaehlt, Kopfzeile ueberspringen
  Anzeige_Position (alt - Menue.Versatz + 2, 1);
  Menue_Zeile_anzeigen (alt);
  Anzeige_Position (Menue.aktiv - Menue.Versatz + 2, 1);
  Menue_Zeile_anzeigen (Menue.aktiv);
}


static void
Onlinehilfe (void)
{
  static const char Text_de[] =
    "akfgb (akfnetz " AKFNETZ_VERSION ")\n"
    AKFNETZ_COPYRIGHT "\n"
    "Lizenz GPLv3+: GNU GPL Version 3 oder neuer"
    " <http://gnu.org/licenses/gpl.html>\n"
    "\n"
    "F1, ?                   diese Hilfe\n"
    "Esc, q, Q               beenden\n"
    "h, Pos1                 erste Zeile\n"
    "Ende                    letzte Zeile\n"
    "j, runter\n"
    "k, hoch                 Navigation / Men\u00FCpunkt ausw\u00E4hlen\n"
    "Enter, rechts           Men\u00FCpunkt starten\n"
    "u, links                Men\u00FC zur\u00FCck\n"
    "Leertaste, Bild runter  n\u00E4chste Seite\n"
    "b, Bild hoch            Seite zur\u00FCck\n"
    "w                       \u00FCbergeordnetes Men\u00FC\n"
    "F2, m                   Startadresse\n"
    "t                       als Text anzeigen\n"
    "a                       Adresse von Men\u00FCpunkt anzeigen\n"
    "o                       neue URL eingeben\n"
    "F4, ^S                  speichern\n"
    "^L, ^R                  Anzeige auffrischen\n";

  static const char Text_en[] =
    "akfgb (akfnetz " AKFNETZ_VERSION ")\n"
    AKFNETZ_COPYRIGHT "\n"
    "License GPLv3+: GNU GPL version 3 or newer"
    " <http://gnu.org/licenses/gpl.html>\n"
    "\n"
    "F1, ?                   this help\n"
    "Esc, q, Q               quit\n"
    "h, Pos1                 first line\n"
    "End                     last line\n"
    "j, down\n"
    "k, up                   navigation / select menu entry\n"
    "Enter, right            start menu entry\n"
    "u, left                 back menu\n"
    "Space, Page down        next page\n"
    "b, Page up              previous page\n"
    "w                       menu up\n"
    "F2, m                   startaddress\n"
    "t                       show as text\n"
    "a                       show address\n"
    "o                       enter new URL\n"
    "F4, ^S                  save\n"
    "^L, ^R                  refresh display\n";

  struct Inhalte Inhalt;
  int r;

  memset (&Inhalt, 0, sizeof (Inhalt));

  if (deutsch)
    r = Inhalt_uebernehmen (&Inhalt, Text_de, sizeof (Text_de));
  else
    r = Inhalt_uebernehmen (&Inhalt, Text_en, sizeof (Text_en));

  if (r < 0)
    {
      Fehlerstatus (errno);
      return;
    }

  Inhalt.url = NULL;
  Inhaltsbetrachter (&Inhalt);
  Inhalt_freigeben (&Inhalt);
}


static void
Textbetrachter (const char *url)
{
  struct Inhalte Inhalt;

  memset (&Inhalt, 0, sizeof (Inhalt));

  if (Inhalt_einlesen (&Inhalt, url) < 0)
    {
      Fehlerstatus (errno);
      return;
    }

  Inhalt_bereinigen (&Inhalt);
  Inhaltsbetrachter (&Inhalt);
  Inhalt_freigeben (&Inhalt);
}


static void
Inhaltsbetrachter (struct Inhalte *Inhalt)
{
  bool fortsetzen = true;

  Anzeige.aktualisieren = true;

  while (fortsetzen)
    {
      if (Anzeige.aktualisieren)
	Text_anzeigen (Inhalt);

      switch (Taste ())
	{
	case 'Q':
	  //case T_CTRL ('C'):
	  Steuerung (Gesamtrollbereich Anzeige_leeren Heim);
	  exit (EXIT_SUCCESS);
	  break;

	case T_F1:
	case '?':
	  if (Inhalt->url)
	    Onlinehilfe ();
	  break;

	case 'h':
	case T_POS1:
	  if (Inhalt->Versatz > 0)
	    {
	      Inhalt->Versatz = 0;
	      Anzeige.aktualisieren = true;
	    }
	  break;

	case T_ENDE:
	  if (Inhalt->Zeilenanzahl > Anzeige.Hoehe - RESERVIERT)
	    {
	      Inhalt->Versatz =
		Inhalt->Zeilenanzahl - (Anzeige.Hoehe - RESERVIERT);
	      Anzeige.aktualisieren = true;
	    }
	  break;

	case 'j':		// runter
	case T_RUNTER:
	  if (Inhalt->Zeilenanzahl > Anzeige.Hoehe - RESERVIERT
	      && Inhalt->Versatz < Inhalt->Zeilenanzahl
	      - (Anzeige.Hoehe - RESERVIERT))
	    {
	      ++Inhalt->Versatz;
	      Anzeige_Position (Anzeige.Hoehe - 1, 1);
	      Steuerung (Zeile_runter);
	      Einruecken ();
	      Ausgabe (Inhalt->Zeile
		       [Inhalt->Versatz + Anzeige.Hoehe - 2 - 1]);
	    }
	  break;

	case 'k':		// rauf
	case T_HOCH:
	  if (Inhalt->Versatz > 0)
	    {
	      Steuerung (Textanfang Zeile_hoch);
	      --Inhalt->Versatz;
	      Einruecken ();
	      Ausgabe (Inhalt->Zeile[Inhalt->Versatz]);
	    }
	  break;

	case T_BILD_RUNTER:
	case T_LEERTASTE:
	  Inhalt->Versatz += (Anzeige.Hoehe - RESERVIERT);
	  if (Inhalt->Versatz > Inhalt->Zeilenanzahl
	      - (Anzeige.Hoehe - RESERVIERT))
	    Inhalt->Versatz =
	      Inhalt->Zeilenanzahl - (Anzeige.Hoehe - RESERVIERT);
	  Anzeige.aktualisieren = true;
	  break;

	case T_BILD_HOCH:
	case 'b':
	  if (Inhalt->Versatz > Anzeige.Hoehe - RESERVIERT)
	    Inhalt->Versatz -= (Anzeige.Hoehe - RESERVIERT);
	  else
	    Inhalt->Versatz = 0;

	  Anzeige.aktualisieren = true;
	  break;

	case T_LINKS:		// zum Menue zurueckkehren
	case T_BACKSPACE:
	case T_ESC:
	case 'u':
	case 'q':
	  Steuerung (Gesamtrollbereich Anzeige_leeren Heim);
	  Anzeige.aktualisieren = true;
	  fortsetzen = false;
	  break;

	case 'm':		// zum Startmenue zurueckkehren
	case T_F2:
	  if (Inhalt->url && Menue.Zeilenanzahl > 0)
	    {
	      Steuerung (Gesamtrollbereich Anzeige_leeren Heim);
	      Menue_einlesen (Startadresse);
	      Historie_neu (Startadresse);
	      fortsetzen = false;
	    }
	  break;

	case 'o':		// URL wechseln
	  if (Inhalt->url && Menue.Zeilenanzahl > 0)
	    {
	      Steuerung (Gesamtrollbereich Anzeige_leeren Heim);
	      URL_Eingabe ();
	      fortsetzen = false;
	    }
	  break;

	case T_F4:		// speichern
	case T_CTRL ('S'):
	  // der genaue Typ ist hier nicht ganz so wichtig
	  if (Inhalt->url)
	    URL_speichern (TYP_TEXT, Inhalt->url);
	  break;

	case T_CTRL ('L'):
	case T_CTRL ('R'):
	  Anzeige.aktualisieren = true;
	  break;
	}
    }
}


// zeigt aktuelle Textseite
static void
Text_anzeigen (struct Inhalte *Inhalt)
{
  Steuerung (normal Anzeige_leeren);
  Kopfzeile (Inhalt->url);
  Text ("\n");
  Anzeige_Rollbereich (2, Anzeige.Hoehe - 1);

  Koordinate Maximum = Inhalt->Zeilenanzahl;
  if (Maximum > Anzeige.Hoehe - RESERVIERT)
    Maximum = Anzeige.Hoehe - RESERVIERT;

  for (Koordinate i = 0; i < Maximum; ++i)
    {
      Koordinate Nr = i + Inhalt->Versatz;
      if (Nr >= Inhalt->Zeilenanzahl)
	break;

      Text ("\n");
      Einruecken ();
      Ausgabe (Inhalt->Zeile[Nr]);
    }

  Fusszeile ();
  Anzeige.aktualisieren = false;
}


// alle C0-Steuerzeichen ausser TAB ersetzen
// (C1-Steuerzeichen waeren schwierig, wegen UTF-8)
static void
Inhalt_bereinigen (struct Inhalte *Inhalt)
{
  for (Koordinate zl = 0; zl < Inhalt->Zeilenanzahl; ++zl)
    {
      unsigned char *s = (unsigned char *) Inhalt->Zeile[zl];

      for (; *s; ++s)
	{
	  if (*s < 0x20 && *s != '\t')
	    *s = ERSATZZEICHEN;
	}
    }
}


static void
Einruecken (void)
{
  // Texte sollten auf maximal 80 Spalten getrimmt sein
  if (Anzeige.Breite > 80)
    Anzeige_vorwaerts ((Anzeige.Breite - 80) / 2);
}


// ist es ein UTF-8 Folgebyte?
static inline bool
Folgebyte (unsigned char c)
{
  return ((c & 0xC0) == 0x80);
}


static char *
Eingabeaufforderung (const char *Aufforderung)
{
  size_t l, Aufforderungslaenge;
  int c;
  unsigned char s[4096];

  *s = '\0';
  l = 0;
  Aufforderungslaenge = strlen (Aufforderung);

  Steuerung (Heim Aktion Zeile_leeren Cursor_an);
  Ausgabe_l (Aufforderung, Aufforderungslaenge);

  while ((c = Taste ()) != T_ENTER)
    {
      if (T_BACKSPACE == c)
	{
	  // erstmal UTF-8 Folgebytes entfernen
	  while (l && Folgebyte (s[l - 1]))
	    --l;

	  if (l)
	    --l;
	}
      else if (T_CTRL ('U') == c)
	l = 0;
      else if (T_LEERTASTE <= c && c <= 255 && l < (sizeof (s) - 1))
	s[l++] = (unsigned char) c;

      Steuerung (Zeilenanfang);
      Ausgabe_l (Aufforderung, Aufforderungslaenge);
      Ausgabe_l (s, l);
      Steuerung (Rest_leeren);
    }

  Steuerung (Cursor_aus normal);
  Anzeige.aktualisieren = true;

  // Steuerzeichen und Leerzeichen am Ende entfernen
  while (l && s[l - 1] <= T_LEERTASTE)
    --l;

  if (!l)
    return NULL;

  s[l++] = '\0';

  char *e = malloc (l);
  if (e)
    memcpy (e, s, l);

  return e;
}


static int
suchen (const char *url)
{
  char *Eingabe = Eingabeaufforderung ("[?]: ");
  if (!Eingabe)
    return 0;			// kein Fehler

  char *Anfrage = akfnetz_urlkodiert_puf (Eingabe, ":/()");
  free (Eingabe);

  char *neu = malloc (strlen (url) + 3 + strlen (Anfrage) + 1);
  if (!neu)
    {
      free (Anfrage);
      return -1;
    }

  // die Anfrage muss nicht unbedingt kodiert werden

  char *p = neu;
  p = stpcpy (p, url);
  p = stpcpy (p, "%09");
  p = stpcpy (p, Anfrage);

  free (Anfrage);

  // Suchanfrage gibt Menue aus
  int e = Menue_einlesen (neu);

  if (e >= 0)
    Historie_neu (neu);

  free (neu);

  return e;
}


static int
URL_Eingabe (void)
{
  char *url = Eingabeaufforderung ("URL: ");

  if (!url)
    return -1;

  URL_ausfuehren (Gophertyp (url), url);
  free (url);

  return 0;
}


static int
speichern (const char *Zeile)
{
  int Typ = (int) *Zeile;
  if (Typ < 0x20 || Typ >= 0x7F)
    return Fehlerstatus (EPERM);
  else if (Typ == TYP_INFO || Typ == TYP_FEHLER)
    return -1;			// still ignorieren

  char *url = URL_erstellen (Zeile);
  if (!url)
    return -1;			// still ignorieren

  int r = URL_speichern (Typ, url);
  free (url);

  return r;
}


static int
URL_speichern (int Typ, const char *url)
{
  if (!url || Typ == TYP_INFO || Typ == TYP_FEHLER)
    return -1;			// still ignorieren

  const char *Pfad = akfnetz_URL_Pfad (url);
  if (!Pfad)
    return Fehlerstatus (EPERM);

  // Gopher-URL: Typ ueberspringen
  if (strlen (Pfad) >= 2)
    Pfad += 2;

  // nicht speicherbare Typen
  if (strchr ("278T", Typ) || !strncmp ("URL:", Pfad, 4))
    return Fehlerstatus (EFAULT);

  // Name ermitteln
  const char *n = strrchr (Pfad, '/');

  if (n)
    ++n;
  else
    n = Pfad;

  // endet es mit einem Schraegstrich?
  if (!*n)
    n = "gophermap";

  char Name[NAME_MAX + 1];
  strncpy (Name, n, NAME_MAX);
  Name[NAME_MAX] = '\0';
  akfnetz_url_dekodieren (Name);

  if (!Speichernachfrage (Typ, Name, url))
    return -1;

  // Verbindung
  int Verbindung = akfnetz_Gopher_URL (url, Proxy);
  if (Verbindung < 0)
    return Fehlerstatus (errno);

  int Datei = open (Name, O_WRONLY | O_CREAT | O_EXCL, 0666);
  if (Datei < 0)
    {
      int e = errno;
      close (Verbindung);
      Fehlerstatus (e);
      Tastenpuffer_leeren ();
      return -1;
    }

  Infostatus (EINPROGRESS);

  if (akfnetz_kopiere (Verbindung, Datei) < 0)
    {
      int Fehler = errno;
      close (Verbindung);
      close (Datei);
      Fehlerstatus (Fehler);
      Tastenpuffer_leeren ();
      return -1;
    }

  close (Verbindung);		// unkritisch

  if (close (Datei) < 0)
    {
      Fehlerstatus (errno);
      Tastenpuffer_leeren ();
      return -1;
    }

  Speichererfolg (Name);

  return 0;
}


static bool
Speichernachfrage (int Typ, const char *Name, const char *url)
{
  int t;
  char Verzeichnis[4096];

  Anzeige.aktualisieren = true;

  Steuerung (normal Anzeige_leeren);
  Kopfzeile (url);
  Fusszeile ();

  Steuerung (normal);
  Anzeige_Position ((Anzeige.Hoehe / 2) - 3, 1);

  Einruecken ();
  if (getcwd (Verzeichnis, sizeof (Verzeichnis)))
    Ausgabe (Verzeichnis);

  Text ("\n\n");
  Einruecken ();
  Steuerung (Typdarstellung);
  Ausgabe (Typname (Typ));
  Steuerung (normal);
  Text (": ");
  Ausgabe (Name);
  Text ("\n\n");

  Einruecken ();
  if (deutsch)
    Text ("Speichern? - Ja: " Tastentext (Enter) ", Nein: " Tastentext (Esc));
  else
    Text ("Save? - Yes: " Tastentext (Enter) ", No: " Tastentext (Esc));

  Tastenpuffer_leeren ();

  do
    t = Taste ();
  while (t != T_ENTER && t != T_ESC);

  Steuerung (normal Anzeige_leeren);
  Kopfzeile (url);
  Fusszeile ();

  return (t == T_ENTER);
}


// zeigt Dateinamen nach Speicherung
static void
Speichererfolg (const char *Name)
{
  char Zeile[4096], *p;

  p = stpcpy (Zeile, Name);
  p = stpcpy (p, ": ");
  p = stpcpy (p, deutsch ? "gespeichert" : "saved");

  Statusmeldung (Zeile);
  warte (MELDUNGSZEIT);
  Fusszeile ();
  Tastenpuffer_leeren ();
}


static int
Inhalt_einlesen (struct Inhalte *Inhalt, const char *url)
{
  char *Puffer;

  Kopfzeile (url);
  Infostatus (EINPROGRESS);

  // kann Gopher oder Finger lesen
  int Verbindung = akfnetz_Gopher_URL (url, Proxy);
  if (Verbindung < 0)
    return -1;

  Inhalt_freigeben (Inhalt);
  Inhalt->url = strdup (url);

  // Puffer einlesen - alles komplett
  if (!akfnetz_einlesen (Verbindung, &Puffer))
    {
      close (Verbindung);
      errno = ENODATA;
      return -1;
    }

  close (Verbindung);

  Inhalt->Puffer = Puffer;
  Inhalt_aufteilen (Inhalt);

  return 0;
}


static int
Inhalt_uebernehmen (struct Inhalte *Inhalt, const char *Text, size_t Laenge)
{
  char *Puffer;

  Inhalt_freigeben (Inhalt);

  Puffer = malloc (Laenge);
  if (!Puffer)
    return -1;

  memcpy (Puffer, Text, Laenge);

  Inhalt->Puffer = Puffer;
  Inhalt_aufteilen (Inhalt);

  return 0;
}


// teilt Puffer in Zeilen auf
// Der Puffer wird veraendert
static void
Inhalt_aufteilen (struct Inhalte *Inhalt)
{
  // weder strtok noch strsep sind geeignet
  char *Anfang = Inhalt->Puffer;
  char *p = Anfang;

  while (p)
    {
      // mehr Speicher anfordern?
      if (Inhalt->Zeilenanzahl >= Inhalt->Zeilenmaximum)
	{
	  int num = Inhalt->Zeilenmaximum + 1024;
	  void *neu = realloc (Inhalt->Zeile, num * sizeof (*Inhalt->Zeile));
	  if (!neu)
	    break;

	  Inhalt->Zeile = neu;
	  Inhalt->Zeilenmaximum = num;
	}

      Inhalt->Zeile[Inhalt->Zeilenanzahl++] = Anfang;

      // Textabschluss?
      if (Anfang[0] == '.' && strchr ("\r\n", Anfang[1]))
	break;

      p = strchr (Anfang, '\n');
      if (!p)
	break;

      if (*(p - 1) == '\r')
	*(p - 1) = '\0';

      *p++ = '\0';
      Anfang = p;
    }

  --Inhalt->Zeilenanzahl;

  Anzeige.aktualisieren = true;
}


static void
Inhalt_freigeben (struct Inhalte *Inhalt)
{
  free (Inhalt->Zeile);
  free (Inhalt->Puffer);
  free (Inhalt->url);

  Inhalt->Zeilenanzahl = Inhalt->Zeilenmaximum = 0;
  Inhalt->Versatz = Inhalt->aktiv = 0;

  Inhalt->Zeile = NULL;
  Inhalt->Puffer = NULL;
  Inhalt->url = NULL;
}


// schiebt url an den Anfang der Historie
static void
Historie_neu (const char *url)
{
  if (Historie[HISTORIE_ANZAHL - 1])
    free (Historie[HISTORIE_ANZAHL - 1]);

  memmove (&Historie[1], &Historie[0],
	   (HISTORIE_ANZAHL - 1) * sizeof (Historie[0]));

  Historie[0] = strdup (url);
}


// nimmt aktuelles Menue aus Historie heraus
// und laedt das naechste Menue (FIFO)
static void
Historie_zurueck (void)
{
  // nie ganz leer werden lassen
  if (!Historie[1])
    return;

  memmove (&Historie[0], &Historie[1],
	   (HISTORIE_ANZAHL - 1) * sizeof (Historie[0]));

  Historie[HISTORIE_ANZAHL - 1] = NULL;

  Menue_einlesen (Historie[0]);
}


static int
Historie_anzeigen (void)
{
  struct Inhalte Inhalt;

  Inhalt.Zeile = (char **) Historie;
  Inhalt.Zeilenanzahl = Inhalt.Zeilenmaximum = HISTORIE_ANZAHL;
  Inhalt.Versatz = Inhalt.aktiv = 0;
  Inhalt.url = NULL;
  Inhalt.Puffer = NULL;

  Inhaltsbetrachter (&Inhalt);
  Inhalt.Zeile = NULL;

  // nicht Inhalt_freigeben()!

  return 0;
}


// liest Inhalt und bereinigt Menue
static int
Menue_einlesen (const char *url)
{
  Anzeige.aktualisieren = true;

  if (Inhalt_einlesen (&Menue, url) < 0)
    return -1;

  if (Menue.Zeilenanzahl == 0)
    return 0;

  int n = Menue.Zeilenanzahl - 1;

  // bereinigt am Ende Leerzeilen
  while (n >= 0 && !*Menue.Zeile[n])
    --n;

  // n kann -1 sein
  Menue.Zeilenanzahl = n + 1;

  if (Menue.Zeilenanzahl == 0)
    {
      errno = ENODATA;
      return -1;
    }

  // es gibt unglaublichen Datenmuell da draussen!
  Inhalt_bereinigen (&Menue);

  return 0;
}


// URL aus Menuezeile
// muss mit free(3) freigegeben werden
static char *
URL_erstellen (const char *Zeile)
{
  char Typ = *Zeile;

  if (Typ == TYP_INFO || Typ == TYP_FEHLER)
    return NULL;

  size_t l = strlen (Zeile) + 1;
  char z[l];
  memcpy (z, Zeile, l);

  char *p = z;

  (void) strsep (&p, "\t");
  char *Selektor = strsep (&p, "\t");
  char *Server = strsep (&p, "\t");
  char *Port = strsep (&p, "\t");

  // Wenn Port gesetzt ist, sind auch die davor gesetzt
  // aber evtl. leer
  if (!Port)
    return NULL;

  // externer Link?
  if (!strncmp (Selektor, "URL:", 4))
    return strdup (Selektor + 4);

  if (strlen (Server) > HOST_NAME_MAX)
    return NULL;

  char url[8192];

  p = url;
  p = stpcpy (p, "gopher://");
  if (strchr (Server, ':'))	// IPv6-Adresse?
    {
      *p++ = '[';
      p = stpcpy (p, Server);
      *p++ = ']';
    }
  else
    p = stpcpy (p, Server);

  if (strcmp (Port, "70"))
    {
      *p++ = ':';
      p = stpcpy (p, Port);
    }
  *p++ = '/';
  *p++ = Typ;

  char *Pfad = akfnetz_urlkodiert_puf (Selektor, ":/()");
  if (strlen (Pfad) >= (sizeof (url) - (ptrdiff_t) (p - url)))
    {
      free (Pfad);
      return NULL;
    }

  p = stpcpy (p, Pfad);
  free (Pfad);

  // Zwischen Typ und Selektor steht kein '/'.
  // Ein '/' nach dem Typ ist Teil des Selektors.

  return strdup (url);
}


// gibt den Typ der URL zurueck
static int
Gophertyp (const char *url)
{
  int Typ = TYP_KAPUTT;

  if (!strncasecmp ("gopher://", url, 9))
    {
      const char *Pfad = akfnetz_URL_Pfad (url);
      if (!Pfad || *Pfad != '/' || !Pfad[1])
	Typ = TYP_MENUE;	// Standardtyp
      else
	Typ = (int) (unsigned char) Pfad[1];
    }
  else if (!strncasecmp ("finger://", url, 9))
    Typ = TYP_TEXT;
  else if (!strncasecmp ("telnet://", url, 9))
    Typ = TYP_TELNET;
  else if (!strncasecmp ("tn3270://", url, 9))
    Typ = TYP_TN3270;
  else if (!strncasecmp ("cso://", url, 6))
    Typ = TYP_CCSO;

  return Typ;
}


// handelt es sich bei der Zeile um eine externe Adresse?
static bool
externe_Adresse (const char *Zeile)
{
  if (Zeile)
    {
      const char *p = strchr (Zeile, '\t');
      if (p && !strncmp (p + 1, "URL:", 4))
	return true;
    }

  return false;
}


// Anzeigegroesse bei SIGWINCH automatisch anpassen
static void
sig_anpassen (int sig)
{
  if (SIGWINCH == sig)
    Anzeigegroesse ();
}


static void
sig_beenden (int sig)
{
  Normalmodus ();

  signal (sig, SIG_DFL);
  raise (sig);
}


static void
Signaleinrichtung (void)
{
  struct sigaction sa;

  // Probleme mit unvollstaendig gelesenem Filter ignorieren
  signal (SIGPIPE, SIG_IGN);
  // kann nicht vom Terminal angehalten werden (^Z)
  signal (SIGTSTP, SIG_IGN);

  // Signale zur Beendigung
  sa.sa_handler = sig_beenden;
  sigfillset (&sa.sa_mask);
  sa.sa_flags = SA_RESETHAND;
  sigaction (SIGINT, &sa, NULL);
  sigaction (SIGQUIT, &sa, NULL);
  sigaction (SIGHUP, &sa, NULL);
  sigaction (SIGTERM, &sa, NULL);
  sigaction (SIGSEGV, &sa, NULL);
  sigaction (SIGFPE, &sa, NULL);
  sigaction (SIGBUS, &sa, NULL);
  sigaction (SIGABRT, &sa, NULL);
  sigaction (SIGILL, &sa, NULL);
  sigaction (SIGXFSZ, &sa, NULL);
  sigaction (SIGXCPU, &sa, NULL);

  // Signal SIGWINCH
  // Anzeigegroesse aktuell halten
  sa.sa_handler = sig_anpassen;
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = 0;		// nicht SA_RESTART wegen Taste()
  sigaction (SIGWINCH, &sa, NULL);
}


static void
Initialisierung (void)
{
  // Ist die Ausgabe ueberhaupt ein Terminal?
  if (!isatty (STDOUT_FILENO))
    exit (Fehlernummer (ENOTTY));

  // Eingabe-Einstellungen speichern
  if (tcgetattr (STDIN_FILENO, &Grundeinstellungen) < 0)
    exit (Fehlernummer (errno));

  // erstmal vorlaeufige Werte
  Anzeige.Breite = 80;
  Anzeige.Hoehe = 24;
  Anzeige.aktualisieren = true;

  Anzeigegroesse ();
  memset (&Menue, 0, sizeof (Menue));
  memset (&Historie, 0, sizeof (Historie));


  // Dateien werden im Arbeitsverzeichnis gespeichert
  // Verzeichnis wechseln
  chdir (getenv ("HOME"));
  chdir ("Downloads");
  // wenn "Downloads" nicht existiert, bleibt man zumindest
  // schonmal im Heimatverzeichnis

  Steuerung (Anzeige_leeren Heim);
  Rohmodus ();
  Steuerung (AltBuf_an normal Anzeige_leeren Heim);

  atexit (Normalmodus);
  Signaleinrichtung ();
}


static char *
var (const char *n)
{
  char *e = getenv (n);

  if (e && !*e)
    e = NULL;

  return e;
}


static const char *
Proxyerkennung (void)
{
  const char *proxy;

  if (!(proxy = var ("GOPHER_PROXY")) && !(proxy = var ("gopher_proxy"))
      && !(proxy = var ("ALL_PROXY")))
    proxy = var ("all_proxy");

  return proxy;
}


static inline void
Ausgabe_l (const void *Meldung, size_t Laenge)
{
  write (STDOUT_FILENO, Meldung, Laenge);
}


static inline void
Ausgabe (const void *Meldung)
{
  write (STDOUT_FILENO, Meldung, strlen (Meldung));
}


// gibt Wert in spitzen Klammern aus (Hilfe)
static void
Wertanzeige (const char *Wert)
{
  if (Wert && *Wert)
    {
      Text ("  <");
      Ausgabe_l (Wert, strlen (Wert));
      Text (">");
    }
}


// gibt Zahl aus
static void
Zahl (unsigned long int Wert)
{
  size_t Laenge = 0;
  char *p, Puffer[40];

  // eins hinter Puffer
  p = &Puffer[sizeof (Puffer)];

  do
    {
      --p;
      *p = (Wert % 10) ^ 0x30;
      Wert /= 10;
      ++Laenge;
    }
  while (Wert);

  Ausgabe_l (p, Laenge);
}


// nur im Normalmodus verwenden, sonst Fehlerstatus
static inline int
Fehlernummer (int e)
{
  char *Meldung = Fehlerbericht (e);

  write (STDERR_FILENO, Programmname, strlen (Programmname));
  write (STDERR_FILENO, ": ", 2);
  write (STDERR_FILENO, Meldung, strlen (Meldung));
  write (STDERR_FILENO, "\n", 1);

  return EXIT_FAILURE;
}


// Ausgabe 4 Zeichen
static const char *
Typname (int Typ)
{
  const char *n;

  switch (Typ)
    {
    case TYP_TEXT:
      n = "TEXT";
      break;

    case TYP_MENUE:
      n = "   >";
      break;

    case TYP_CCSO:
      n = "CCSO";
      break;

    case TYP_FEHLER:
      n = "!!!!";
      break;

    case TYP_BINHEX:
      n = " HQX";
      break;

    case TYP_ARCHIV:
      n = "ARCH";
      break;

    case TYP_UUE:
      n = " UUE";
      break;

    case TYP_SUCHE:
      n = "[?]:";
      break;

    case TYP_TELNET:
      n = "TELN";
      break;

    case TYP_BINAER:
      n = " BIN";
      break;

    case TYP_ALT:
      n = " ALT";
      break;

    case TYP_GIF:
      n = " GIF";
      break;

    case TYP_BITMAP:
    case TYP_PNG:
    case TYP_BILD:
      n = deutsch ? "BILD" : "PICT";
      break;

    case TYP_TN3270:
      n = "3270";
      break;

    case TYP_VIDEO:
      n = " VID";
      break;

    case TYP_TON:
    case TYP_AUDIO:
      n = " AUD";
      break;

    case TYP_DOKU:
      n = "DOKU";
      break;

    case TYP_HTML:
      n = "HTML";
      break;

    case TYP_RTF:
      n = " RTF";
      break;

    case TYP_PDF:
      n = " PDF";
      break;

    case TYP_XML:
      n = " XML";
      break;

    case TYP_MIME:
      n = "MIME";
      break;

    case TYP_KALENDER:
      n = " CAL";
      break;

    case TYP_INFO:
      n = "    ";
      break;

    case TYP_URL:
      n = " URL";
      break;

    default:
      n = "????";
      break;
    }

  return n;
}


/*****************
 * Terminal-Zeug *
 *****************/

// stellt Normalmodus wieder her und schaltet den alternativen Puffer aus
// loescht den Bildschirm
// wird auch bei Programmende aufgerufen
static void
Normalmodus (void)
{
  Steuerung (Gesamtrollbereich Umbruch_an Cursor_an AltBuf_aus
	     normal Anzeige_leeren Heim);

  tcsetattr (STDIN_FILENO, TCSANOW, &Grundeinstellungen);
}


// rohen Eingabe-Modus setzen (ohne alternativen Puffer)
static void
Rohmodus (void)
{
  struct termios n;

  // alles bei Normalmodus rueckgaengig machen!
  Steuerung (Gesamtrollbereich Umbruch_aus Cursor_aus);

  memcpy (&n, &Grundeinstellungen, sizeof (n));

  // nicht cfmakeraw verwenden - dies ist nicht ganz roh
  n.c_iflag &=
    ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
  n.c_lflag &= ~(ICANON | ECHO | ECHONL | IEXTEN);
  n.c_lflag |= ISIG;		// Abbruch zulassen
  n.c_oflag |= OPOST;		// Ausgabe nicht roh
  n.c_cflag &= ~(CSIZE | PARENB);
  n.c_cflag |= CS8;
  n.c_cc[VERASE] = T_BACKSPACE;
  n.c_cc[VSUSP] = _POSIX_VDISABLE;
  n.c_cc[VMIN] = 1;
  n.c_cc[VTIME] = 0;

  tcsetattr (STDIN_FILENO, TCSAFLUSH, &n);
}



// aktualisiert Anzeigegroesse
// wird am Anfang und bei dem Signal SIGWINCH aufgerufen
static void
Anzeigegroesse (void)
{
  struct winsize Groesse;

  if (ioctl (STDOUT_FILENO, TIOCGWINSZ, &Groesse) >= 0)
    {
      Anzeige.Hoehe = (Koordinate) Groesse.ws_row;
      Anzeige.Breite = (Koordinate) Groesse.ws_col;
      Anzeige.aktualisieren = true;
    }
}


// Cursor-Position (oben links = 1,1)
// siehe auch: Heim, Textanfang
static void
Anzeige_Position (Koordinate Zeile, Koordinate Spalte)
{
  if (Zeile <= 1 && Spalte <= 1)
    {
      // kuerzere Sequenz
      Steuerung (Heim);
      return;
    }

  Steuerung (CSI);
  Zahl (Zeile);

  if (Spalte > 1)
    {
      Steuerung (";");
      Zahl (Spalte);
    }

  Steuerung ("H");		// CUP
}


// Cursor vorwaerts
static void
Anzeige_vorwaerts (Koordinate Nr)
{
  Steuerung (CSI);
  Zahl (Nr);
  Steuerung ("C");		// CUF
}


static void
Anzeige_Rollbereich (Koordinate oben, Koordinate unten)
{
  Steuerung (CSI);
  Zahl (oben);
  Steuerung (";");
  Zahl (unten);
  Steuerung ("r");		// DECSTBM
}


// wartet Sekunden ab, oder bis Tastendruck
// Tastendruck bleibt im Puffer
static void
warte (time_t Sekunden)
{
  fd_set Eingaben;
  struct timeval Zeitlimit;

  Tastenpuffer_leeren ();

  FD_ZERO (&Eingaben);
  FD_SET (STDIN_FILENO, &Eingaben);

  Zeitlimit.tv_sec = Sekunden;
  Zeitlimit.tv_usec = 0;

  select (STDIN_FILENO + 1, &Eingaben, NULL, NULL, &Zeitlimit);
}


// prueft, ob etwas im Tastenpuffer ist,
// ohne es zu entfernen
static bool
Tastendruck (void)
{
  fd_set Eingaben;
  struct timeval Zeitlimit;

  FD_ZERO (&Eingaben);
  FD_SET (STDIN_FILENO, &Eingaben);
  Zeitlimit.tv_sec = 0;
  Zeitlimit.tv_usec = 1000;	// 1 Millisekunde

  return (select (STDIN_FILENO + 1, &Eingaben, NULL, NULL, &Zeitlimit) != 0);
}


static int
Funktionstaste (void)
{
  int c;
  size_t pos;
  char Puffer[20];

  // unterschiedliche Terminals haben unterschiedliche Tastenkodierungen

  pos = 0;
  c = Taste ();
  if (c != '[' && c != 'O')
    return -1;

  Puffer[pos++] = c;

  do
    {
      c = Taste ();
      Puffer[pos++] = c;
    }
  while (c < '@' && pos < sizeof (Puffer));

  switch (c)
    {
    case 'A':
      return T_HOCH;
    case 'B':
      return T_RUNTER;
    case 'C':
      return T_RECHTS;
    case 'D':
      return T_LINKS;
    case 'F':
      return T_ENDE;
    case 'H':
      return T_POS1;
    case 'P':
      return T_F1;
    case 'Q':
      return T_F2;
    case 'R':
      return T_F3;
    case 'S':
      return T_F4;

    case '~':
      switch (atoi (Puffer + 1))
	{
	case 1:
	  return T_POS1;
	case 4:
	  return T_ENDE;
	case 5:
	  return T_BILD_HOCH;
	case 6:
	  return T_BILD_RUNTER;
	case 7:
	  return T_POS1;
	case 8:
	  return T_ENDE;
	case 11:
	  return T_F1;
	case 12:
	  return T_F2;
	case 13:
	  return T_F3;
	case 14:
	  return T_F4;
	case 15:
	  return T_F5;
	}
      break;

    case '[':			// Linux-Textkonsole ist seltsam
      switch (Taste ())
	{
	case 'A':
	  return T_F1;
	case 'B':
	  return T_F2;
	case 'C':
	  return T_F3;
	case 'D':
	  return T_F4;
	case 'E':
	  return T_F5;
	}
      break;
    }

  return -1;
}


// wartet auf Tastendruck und gibt Taste zurueck
// nur fuer den Rohmodus
static int
Taste (void)
{
  unsigned char c;
  ssize_t l;

  c = 0;
  l = read (STDIN_FILENO, &c, sizeof (c));

  // Verbindung verloren?
  // Oder EINTR wegen SIGWINCH?
  if (l <= 0)
    return -1;

  if (c == T_ESC && Tastendruck ())
    return Funktionstaste ();

  return c;
}


static inline void
Tastenpuffer_leeren (void)
{
  tcflush (STDIN_FILENO, TCIFLUSH);
}


// nur mit Vorsicht anzuwenden!
static inline void
sende_Taste (char c)
{
  ioctl (STDIN_FILENO, TIOCSTI, &c);
}



/*********************
 * externe Programme *
 *********************/

static inline bool
Audiodatei (const char *Pfad)
{
  return (!fnmatch ("*.wav", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.au", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.snd", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.aif[fc]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.aif", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.wma", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mka", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mp[321]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mpga", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mpega", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.m4[ab]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.flac", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.wv", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.og[ga]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.opus", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.spx", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.weba", Pfad, FNM_CASEFOLD));
}


static inline bool
Videodatei (const char *Pfad)
{
  return (!fnmatch ("*.mov", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mng", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.avi", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.flv", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mp[ge]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mpeg", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mp4", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.m4v", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.mkv", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.wmv", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.og[vx]", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.webm", Pfad, FNM_CASEFOLD)
	  || !fnmatch ("*.3g[p2]", Pfad, FNM_CASEFOLD));
}


// Typ '9'
static int
Binaerdatei (const char *url)
{
  const char *Pfad = akfnetz_URL_Pfad (url);

  Kopfzeile (url);

  if (Audiodatei (Pfad))
    return Befehlsaufruf (url, AUDIOABSPIELER);
  else if (Videodatei (Pfad))
    return Befehlsaufruf (url, VIDEOABSPIELER);

  errno = ENOMSG;
  return -1;
}


// Login anzeigen fuer Typ '8' oder 'T'
static void
Gopher_Login (const char *url)
{
  char Puffer[1024];

  if (!strncmp ("gopher://", url, 9))
    {
      // Pfad kann Login enthalten
      const char *Selektor = akfnetz_URL_Pfad (url);
      if (!Selektor)
	Selektor = "";

      // Gopher-URL: Typ ueberspringen
      if (strlen (Selektor) >= 2)
	Selektor += 2;

      while (*Selektor == '/')
	++Selektor;

      if (*Selektor)
	Ausgabe (Selektor);
    }
  else if (akfnetz_URL_Zugang (Puffer, sizeof (Puffer), url))
    {
      // eventuell Passwort entfernen
      char *p = strchr (Puffer, ':');
      if (p)
	*p = '\0';

      Ausgabe (akfnetz_url_dekodieren (Puffer));
    }
}


// stellt Befehlszeile fuer telnet oder tn3270 zusammen
static int
Telnetdienst (char *Befehlszeile, const char *Befehl,
	      const char *Server, const char *Port)
{
  if (!Befehl || !*Befehl)
    {
      errno = ENOSYS;
      return -1;
    }

  char *p = Befehlszeile;
  p = stpcpy (p, "exec ");
  p = stpcpy (p, Befehl);
  *p++ = ' ';
  p = stpcpy (p, Server);

  if (*Port && strcmp (Port, "23") != 0)
    {
      *p++ = ' ';
      p = stpcpy (p, Port);
    }

  return 0;
}


// stellt Befehlszeile fuer ph zusammen
static int
PHdienst (char *Befehlszeile, const char *Befehl,
	  const char *Server, const char *Port)
{
  if (!Befehl || !*Befehl)
    {
      errno = ENOSYS;
      return -1;
    }

  char *p = Befehlszeile;
  p = stpcpy (p, "exec ");
  p = stpcpy (p, Befehl);
  p = stpcpy (p, " -s ");
  p = stpcpy (p, Server);

  if (*Port && strcmp (Port, "105") != 0)
    {
      p = stpcpy (p, " -p ");
      p = stpcpy (p, Port);
    }

  return 0;
}


// fuehrt ph oder telnet oder tn3270 aus
static int
Dienstprogramm (int Typ, const char *url)
{
  char Port[8], Server[HOST_NAME_MAX + 1];
  char Befehlszeile[1024];

  akfnetz_URL_Hostname (Server, sizeof (Server), url);
  if (!akfnetz_URL_Port (Port, sizeof (Port), url))
    *Port = '\0';

  errno = ENOSYS;

  // Befehlszeile zusammensetzen
  switch (Typ)
    {
    case TYP_CCSO:
      if (PHdienst (Befehlszeile, PH, Server, Port) < 0)
	return -1;
      break;

    case TYP_TELNET:
      if (Telnetdienst (Befehlszeile, TELNET, Server, Port) < 0)
	return -1;
      break;

    case TYP_TN3270:
      if (Telnetdienst (Befehlszeile, TN3270, Server, Port) < 0)
	return -1;
      break;

    default:			// sollte nicht vorkommen
      errno = ENOSYS;
      return -1;
    }

  Normalmodus ();
  Steuerung (Kommentare);
  Ausgabe (Befehlszeile);
  Text ("\n");

  if (Typ == TYP_TELNET || Typ == TYP_TN3270)
    {
      Text ("Login: ");
      Gopher_Login (url);
      Text ("\n");
    }

  Steuerung (normal);

  // ausfuehren
  int e = system (Befehlszeile);

  Rohmodus ();

  // Bei Fehlern des Befehles auf Taste warten
  if (e >= 0 && WIFEXITED (e) && WEXITSTATUS (e) != EXIT_SUCCESS)
    warte (MELDUNGSZEIT);

  Tastenpuffer_leeren ();
  Steuerung (normal Anzeige_leeren AltBuf_an normal Anzeige_leeren Heim);
  // Anzeige_leeren fuer beide Darstellungen
  Anzeige.aktualisieren = true;

  return e;
}


// URL aufrufen
static int
Adressaufruf (const char *url)
{
  char *Befehl = URL_BEHANDLUNG;

  if (!Befehl || !*Befehl)
    {
      errno = EADDRNOTAVAIL;
      return -1;
    }

  char Befehlszeile[5 + strlen (Befehl) + 3 + strlen (url) + 1];

  char *p = Befehlszeile;
  p = stpcpy (p, "exec ");
  p = stpcpy (p, Befehl);
  p = stpcpy (p, " '");
  p = stpcpy (p, url);
  p = stpcpy (p, "'");

  Normalmodus ();
  Kopfzeile (url);
  Steuerung (Textanfang);

  int e = system (Befehlszeile);

  Rohmodus ();

  // Bei Fehlern des Befehles auf Taste warten
  if (e >= 0 && WIFEXITED (e) && WEXITSTATUS (e) != EXIT_SUCCESS)
    warte (MELDUNGSZEIT);

  Tastenpuffer_leeren ();
  Steuerung (normal Anzeige_leeren AltBuf_an);
  Anzeige.aktualisieren = true;

  return 0;
}


// ruft Befehl ueber Shell auf und sendet Daten an dessen Standardeingabe
// gibt Wartestatus oder -1 zurueck
static int
Befehlsaufruf (const char *url, const char *Befehl)
{
  int e = -1;

  if (!Befehl || !*Befehl)
    {
      errno = ENOSYS;
      return -1;
    }

  // erstmal schauen, ob es das ueberhaupt gibt
  int Verbindung = akfnetz_Gopher_URL (url, Proxy);
  if (Verbindung < 0)
    return -1;

  Normalmodus ();
  Kopfzeile (url);
  Fusszeile ();

  // Befehl anzeigen
  Steuerung (Textanfang Kommentare);
  Ausgabe (Befehl);
  Steuerung (normal);
  Text ("\n");

  int Rohr[2];
  if (pipe (Rohr) < 0)
    {
      close (Verbindung);
      goto raus;
    }

  pid_t pid = fork ();
  switch (pid)
    {
    case -1:			// Fehler
      close (Verbindung);
      close (Rohr[0]);
      close (Rohr[1]);
      break;


    case 0:			// Kindprozess
      close (Verbindung);
      close (Rohr[1]);

      if (dup2 (Rohr[0], STDIN_FILENO) >= 0)
	{
	  close (Rohr[0]);
	  signal (SIGINT, SIG_DFL);

	  char Befehlszeile[1024];
	  strcpy (Befehlszeile, "exec ");
	  strcpy (Befehlszeile + 5, Befehl);
	  execl ("/bin/sh", "sh", "-c", Befehlszeile, (char *) NULL);
	}

      _exit (127);
      break;


    default:			// Elternprozess
      close (Rohr[0]);

      akfnetz_kopiere (Verbindung, Rohr[1]);

      close (Verbindung);
      close (Rohr[1]);

      if (waitpid (pid, &e, 0) < 0)
	e = -1;
      break;
    }

raus:
  Rohmodus ();

  // Bei Fehlern des Befehles auf Taste warten
  if (e >= 0 && WIFEXITED (e) && WEXITSTATUS (e) != EXIT_SUCCESS)
    warte (MELDUNGSZEIT);

  Steuerung (normal Anzeige_leeren Heim);	// Normalmodus leeren
  Steuerung (AltBuf_an normal Anzeige_leeren Heim);

  Tastenpuffer_leeren ();
  Anzeige.aktualisieren = true;

  return e;
}
