/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * akfbase64
 * Copyright (c) 2021,2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _POSIX_C_SOURCE 200112L

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <stdbool.h>
#include <string.h>
#include <libgen.h>
#include <errno.h>
#include <locale.h>
#include "akfnetz.h"
#include "version.h"

#define BLOCK 19
#define ZUFALLSGENERATOR "/dev/urandom"

#ifndef Fehlermeldung
#define Fehlermeldung  strerror
#endif

static const char *prgname;

static int kodieren (FILE *, FILE *, bool, bool, const char *);
static int dekodieren (FILE *, FILE *);
static int Hilfe (void);
static int Version (void);
static int Fehler (int);
static unsigned int Dateimodus (FILE *);


int
main (int argc, char **argv)
{
  bool dekodiere, Zeilenumbruch, url, uuformat, Zufall;
  char *Zeichensatz, *Eingabedatei, *Ausgabedatei;

  prgname = argv[0];
  setlocale (LC_ALL, "");

  Zeichensatz = AKFNETZ_BASE64;
  Eingabedatei = Ausgabedatei = NULL;
  dekodiere = url = Zufall = uuformat = false;
  Zeilenumbruch = true;

  int a;
  for (a = 1; a < argc; ++a)
    {
      const char *arg = argv[a];

      if (*arg != '-')
	break;

      if ((!strcmp (arg, "-h") || !strcmp (arg, "--help")
	   || !strcmp (arg, "--Hilfe")))
	return Hilfe ();
      else if (!strcmp (arg, "--version"))
	return Version ();
      else if (!strcmp (arg, "-u"))
	{
	  url = true;
	  Zeichensatz = AKFNETZ_BASE64_URL;
	}
      else if (!strcmp (arg, "-n"))
	Zeilenumbruch = false;
      else if (!strcmp (arg, "-m"))
	uuformat = true;
      else if (!strcmp (arg, "-d"))
	dekodiere = true;
      else if (!strcmp (arg, "-z"))
	Zufall = true;
      else
	return Fehler (EINVAL);
    }

  if (a < argc)
    {
      Eingabedatei = argv[a++];

      if (a < argc)
	Ausgabedatei = argv[a++];
    }

  // Bedingungen fuer Zufall
  if (Zufall && (Eingabedatei || dekodiere || uuformat))
    return Fehler (EINVAL);

  // Bedingungen fuer uudecode
  if (uuformat && (!Eingabedatei || url || !Zeilenumbruch || dekodiere))
    return Fehler (EINVAL);

  if (Zufall)
    Eingabedatei = ZUFALLSGENERATOR;

  FILE *ein = stdin;
  if (Eingabedatei && !(ein = fopen (Eingabedatei, "rb")))
    return Fehler (errno);

  FILE *aus = stdout;
  if (Ausgabedatei && !(aus = fopen (Ausgabedatei, "wb")))
    return Fehler (errno);

  int e;
  if (dekodiere)
    e = dekodieren (ein, aus);
  else
    {
      if (uuformat)
	fprintf (aus, "begin-base64 %o %s\n",
		 Dateimodus (ein), basename (Eingabedatei));

      e = kodieren (ein, aus, Zeilenumbruch, Zufall, Zeichensatz);

      if (uuformat)
	fputs ("====\n", aus);
    }

  if (fclose (aus) < 0 || fclose (ein) < 0 || e < 0)
    return Fehler (errno);

  return EXIT_SUCCESS;
}


static int
Hilfe (void)
{
  printf ("Base64 encode or decode\n\n"
	  "%s [options] [infile] [outfile]\n"
	  "%s -z [-u]\n\n"
	  "options:\n"
	  "  -u\tuse base64url for encoding\n"
	  "  -n\tno linebreak\n"
	  "  -m\tcompatible to uudecode encoding\n"
	  "  -d\tdecode\n"
	  "  -z\toutputs random characters (" ZUFALLSGENERATOR ")\n",
	  prgname, prgname);

  return EXIT_SUCCESS;
}


static int
Version (void)
{
  puts ("akfbase64 (akfnetz " AKFNETZ_VERSION ")\n"
	AKFNETZ_COPYRIGHT "\n\n"
	"License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.\n"
	"This is free software: you are free to change and redistribute it.\n"
	"There is NO WARRANTY, to the extent permitted by law.");

  return EXIT_SUCCESS;
}


static int
Fehler (int e)
{
  fprintf (stderr, "%s: %s\n", prgname, Fehlermeldung (e));
  return EXIT_FAILURE;
}


static unsigned int
Dateimodus (FILE *d)
{
  struct stat st;

  if (fstat (fileno (d), &st) < 0)
    return (0644);

  // keine suid oder sgid Bits ausgeben
  return (st.st_mode & 0777);
}


static int
kodieren (FILE *ein, FILE *aus, bool Zeilenumbruch, bool Zufall,
	  const char *Zeichensatz)
{
  size_t l;
  char Eingabe[3 * BLOCK];

  while ((l = fread (Eingabe, 1, sizeof (Eingabe), ein)))
    {
      char Ausgabe[4 * BLOCK + 1];

      if (akfnetz_base64_kodiert (Ausgabe, sizeof (Ausgabe),
				  Eingabe, l, Zeichensatz))
	{
	  if (fputs (Ausgabe, aus) == EOF
	      || (Zeilenumbruch && putc_unlocked ('\n', aus) == EOF))
	    return -1;

	  if (Zufall)
	    break;
	}
    }

  return ferror (ein) ? -1 : 0;
}


static int
dekodieren (FILE *ein, FILE *aus)
{
  int c;
  size_t l;
  char Puffer[256 * 4 + 1];

  l = 0;

  do
    {
      c = getc_unlocked (ein);

      if (32 < c && c < 127)
	Puffer[l++] = (char) c;

      if (l == (sizeof (Puffer) - 1) || c == EOF)
	{
	  size_t Laenge;

	  Puffer[l] = '\0';
	  l = 0;

	  Laenge =
	    akfnetz_base64_dekodieren (Puffer, sizeof (Puffer), Puffer);

	  if (fwrite (Puffer, 1, Laenge, aus) != Laenge)
	    return -1;
	}
    }
  while (c != EOF);

  return ferror (ein) ? -1 : 0;
}
