/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015,2016,2023 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "akfnetz.h"
#include <time.h>
#include <string.h>
#include <stdlib.h>

static const char Wochentagsabk[7][4] =
  { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

static const char Monatsabk[12][4] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
  "Nov", "Dec"
};

static int
Monatsnummer (const char *s)
{
  for (int i = 0; i < 12; ++i)
    if (!memcmp (s, Monatsabk[i], 3))
      return i;

  return -1;
}


static int
Wochentagsnummer (const char *s)
{
  for (int i = 0; i < 7; ++i)
    if (!memcmp (s, Wochentagsabk[i], 3))
      return i;

  return -1;
}


// bevorzugtes Format - alles an festen Positionen
static void
analysiere_rfc1123_Zeit (struct tm *z, const char *Zeit)
{
  z->tm_wday = Wochentagsnummer (Zeit);
  z->tm_mday = strtol (Zeit + 5, NULL, 10);
  z->tm_mon = Monatsnummer (Zeit + 8);
  z->tm_year = strtol (Zeit + 12, NULL, 10) - 1900;
  z->tm_hour = strtol (Zeit + 17, NULL, 10);
  z->tm_min = strtol (Zeit + 20, NULL, 10);
  z->tm_sec = strtol (Zeit + 23, NULL, 10);
  z->tm_isdst = 0;		// GMT
}


static void
analysiere_rfc1036_Zeit (struct tm *z, const char *Zeit)
{
  char *p;

  p = strchr (Zeit, ' ');
  if (!p)
    p = (char *) Zeit + 5;	// geraten

  z->tm_wday = Wochentagsnummer (Zeit);
  z->tm_mday = strtol (p, &p, 10);
  z->tm_mon = Monatsnummer (p + 1);
  z->tm_year = strtol (p + 5, &p, 10);	// zweistellig
  z->tm_hour = strtol (p + 1, &p, 10);
  z->tm_min = strtol (p + 1, &p, 10);
  z->tm_sec = strtol (p + 1, NULL, 10);
  z->tm_isdst = 0;		// GMT
}


static void
analysiere_asctime_Zeit (struct tm *z, const char *Zeit)
{
  char *p;

  z->tm_wday = Wochentagsnummer (Zeit);
  z->tm_mon = Monatsnummer (Zeit + 4);

  z->tm_mday = strtol (Zeit + 8, &p, 10);
  z->tm_hour = strtol (p + 1, &p, 10);
  z->tm_min = strtol (p + 1, &p, 10);
  z->tm_sec = strtol (p + 1, &p, 10);
  z->tm_year = strtol (p + 1, NULL, 10) - 1900;
  z->tm_isdst = 0;		// GMT
}


struct tm *
akfnetz_analysiere_http_Zeit (struct tm *z, const char *Zeit)
{
  // Auf jeden Fall alles loeschen, einschliesslich undokumentierter Felder
  memset (z, 0, sizeof (*z));

  if (!Zeit)
    return NULL;

  while (*Zeit && (*Zeit == ' ' || *Zeit == '\t'))
    ++Zeit;

  if (strlen (Zeit) >= 24)
    {
      if (Zeit[3] == ',')
	analysiere_rfc1123_Zeit (z, Zeit);
      else if (Zeit[3] == ' ')
	analysiere_asctime_Zeit (z, Zeit);
      else
	analysiere_rfc1036_Zeit (z, Zeit);
    }

  // Plausibilitaetspruefung
  if (z->tm_wday < 0 || z->tm_mon < 0 || z->tm_mday <= 0)
    {
      memset (z, 0, sizeof (*z));
      return NULL;
    }

  return z;
}
