/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright © 2015-2025 Andreas K. Förster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _POSIX
#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 700

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "akfnetz.h"
#include "akfwebserver.h"
#include "Sprachen.h"
#include "FehlerStil.h"

#define Text(fd, s)  write((fd), "" s, sizeof(s)-1)
#define gross(c) ((((c) & 0xE0) == 0x60) ? (c) & 0x5F : (c))

static void
Fehlerseite (FILE *, const char *[], const char *[], const char *);
static void Meldung (FILE *, enum Sprache, const char *[], const char *[]);
static void Autorisationsanfrage (FILE *);


extern void
http_Webfehler (FILE *aus, enum Zustand Art)
{
  const char *Fehler[SPRACHANZAHL], *Grund[SPRACHANZAHL], *Ursache;

  // Initialisieren
  memset (Fehler, 0, sizeof (Fehler));
  memset (Grund, 0, sizeof (Grund));
  Ursache = NULL;

  /*
     Grund und Ursache brauchen nicht angegeben werden.
     Grund ist ein Teilsatz, eingeleitet mit "Tut mir leid, aber "
   */

// *INDENT-OFF*

  switch (Art)
    {
    case Anfragefehler: // 400
      Fehler[Deutsch] = u8"Anfragefehler";
      Grund[Deutsch] = u8"die Anfrage war fehlerhaft.";
      Fehler[Esperanto] = u8"Malĝusta peto";
      Fehler[Englisch] = u8"Bad Request";
      Grund[Englisch] = u8"the request could not be understood.";
      Fehler[Russisch] = u8"плохой запрос";
      http_Anfang (aus, 400, "Bad Request");
      break;


    case Berechtigungsfehler: // 401
      Fehler[Deutsch] = u8"Nicht berechtigt";
      Grund[Deutsch] =
        u8"für den Zugang braucht man die richtigen Zugangsdaten.";
      Fehler[Esperanto] = u8"Eraro en aŭtentokontrolo";
      Grund[Esperanto] =
        u8"vi ne havas permeson eniri ĉi tien. Vi bezonas pasvorton.";
      Fehler[Englisch] = u8"Authorization Required";
      Grund[Englisch] = u8"you need the correct credentials to gain access.";
      Fehler[Russisch] = u8"не авторизован";
      Grund[Russisch] =
        u8"для доступа к нему требуются правильные учетные данные.";
      http_Anfang (aus, 401, "Authorization Required");
      Autorisationsanfrage (aus);
      break;


    case Zugriffsfehler: // 403
      Fehler[Deutsch] = u8"Gesperrt";
      Grund[Deutsch] = u8"der Zugang ist gesperrt.";
      Fehler[Esperanto] = u8"Malpermesita";
      Grund[Esperanto] = u8"tio ne estas alirebla.";
      Fehler[Englisch] = u8"Forbidden";
      Grund[Englisch] = u8"the access is denied.";
      Fehler[Russisch] = u8"запрещено";
      Grund[Russisch] = u8"доступ закрыт.";
      Ursache = http.Pfad;
      http_Anfang (aus, 403, "Forbidden");
      break;


    case Verzeichnisfehler: // 403 nochmal
      Fehler[Deutsch] = u8"Das ist ein Verzeichnis";
      Grund[Deutsch] = u8"die Adresse muss mit einem Schrägstrich enden.";
      Fehler[Esperanto] = u8"Ĉi tio estas dosierujo";
      Grund[Esperanto] = u8"la adreso devas finiĝi per oblikvo (/).";
      Fehler[Englisch] = u8"That is a directory";
      Grund[Englisch] = u8"the address has to end with a slash.";
      Fehler[Russisch] = u8"Это каталог";
      Grund[Russisch] = u8"адрес должен заканчиваться косой чертой.";
      Ursache = http.Pfad;
      http_Anfang (aus, 403, "That is a directory");
      break;


    case Suchfehler: // 404
      Fehler[Deutsch] = u8"Nicht gefunden";
      Grund[Deutsch] = u8"das gibt es hier nicht.";
      Fehler[Esperanto] = u8"Ne trovita";
      Grund[Esperanto] = u8"tio ne ekzistas ĉi tie.";
      Fehler[Englisch] = u8"Not Found";
      Grund[Englisch] = u8"that doesn't exist here.";
      Fehler[Russisch] = u8"не найдено";
      Grund[Russisch] = u8"здесь такого нет.";
      Ursache = http.Pfad;
      http_Anfang (aus, 404, "Not Found");
      break;


    case Methodenfehler: // 405, nur bei bekannten Methoden
      Fehler[Deutsch] = u8"Methode nicht erlaubt";
      Fehler[Esperanto] = u8"Metodo ne permesita";
      Fehler[Englisch] = u8"Method Not Allowed";
      Fehler[Russisch] = u8"метод не поддерживается";

      if (GET <= http.Methode && http.Methode <= MKCALENDAR)
	Ursache = akfnetz_Methodenname[http.Methode];

      http_Anfang (aus, 405, "Method Not Allowed");
      http_Methodenerlaubnis (aus);
      break;


    case Voraussetzungsfehler: // 412
      http_Anfang (aus, 412, "Precondition Failed");
      http_Kopf (aus, "Content-Length", "0");
      http_Ende (aus);		// kein Inhalt
      return; // kein Inhalt!


    case Platzfehler: // 413
      Fehler[Deutsch] = u8"Inhalt ist zu groß";
      Grund[Deutsch] = u8"das ist zu viel für mich.";
      Fehler[Esperanto] = u8"Enhavo tro granda";
      Grund[Esperanto] = u8"tio estas tro granda por mi.";
      Fehler[Englisch] = u8"Content too large";
      Grund[Englisch] = u8"that's too large for me.";
      Fehler[Russisch] = u8"полезная нагрузка слишком велика";
      Grund[Russisch] = u8"это слишком много для меня.";
      http_Anfang (aus, 413, "Payload Too Large");
      break;


    case Adressfehler: // 414
      Fehler[Deutsch] = u8"URI zu lang";
      Grund[Deutsch] = u8"die Adresse ist zu lang.";
      Fehler[Esperanto] = u8"Retadreso tro longa";
      Fehler[Englisch] = u8"URI too long";
      Grund[Englisch] = u8"the address is too long.";
      Fehler[Russisch] = u8"URI слишком длинный";
      Grund[Russisch] = u8"адрес слишком длинный.";
      http_Anfang (aus, 414, "URI Too Long");
      break;


    case Bereichsfehler: // 416
      Fehler[Deutsch] = u8"Bereichsfehler";
      Grund[Deutsch] = u8"der angeforderte Bereich ist nicht verfügbar.";
      Fehler[Esperanto] = u8"amplekso ne kontentigebla"; // ???
      Fehler[Englisch] = u8"Range Not Satisfiable";
      Grund[Englisch] = u8"the request was out of range.";
      Fehler[Russisch] = u8"диапазон не достижим";
      Grund[Russisch] = u8"запрошенный диапазон недоступен.";
      http_Anfang (aus, 416, "Range Not Satisfiable");

      if (http.Dateigroesse > 0)
	{
	  char s[80];
	  snprintf (s, sizeof (s), "bytes */%jd",
		    (intmax_t) http.Dateigroesse);
	  http_Kopf (aus, "Content-Range", s);
	}
      break;


    case Erwartungsfehler: // 417
      Fehler[Deutsch] = u8"Erwartungsfehler";
      Grund[Deutsch] = u8"die Erwartung wird nicht unterstützt.";
      Fehler[Esperanto] = u8"Atendo malsukcesis";
      Fehler[Englisch] = u8"Expectation Failed";
      Grund[Englisch] = u8"the expectation can not be fullfilled.";
      Fehler[Russisch] = u8"ожидание не оправдалось";
      http_Anfang (aus, 417, "Expectation Failed");
      break;


    case Serverfehler: // 500, auch Fehler von Webanwendungen
    case Scheisse:
    default:
      Fehler[Deutsch] = u8"Serverfehler";
      Grund[Deutsch] = u8"es liegt irgendein Fehler auf dem Server vor.";
      Fehler[Esperanto] = u8"Servila eraro";
      Grund[Esperanto] = u8"ĉi tio estas ia eraro en la servilo.";
      Fehler[Englisch] = u8"Server Error";
      Grund[Englisch] = u8"there is something wrong on the server.";
      Fehler[Russisch] = u8"внутренняя ошибка сервера";
      Grund[Russisch] = u8"на сервере произошла какая-то ошибка.";
      http_Anfang (aus, 500, "Server Error");
      break;


    case Unterstuetzungsfehler: // 501
      Fehler[Deutsch] = u8"nicht unterstützt";
      Grund[Deutsch] = u8"diese Anfrage wird nicht unterstützt.";
      Fehler[Esperanto] = u8"ne realiĝis";
      Grund[Esperanto] = u8"tiu peto ne estas subtenata.";
      Fehler[Englisch] = u8"Not Implemented";
      Grund[Englisch] = u8"that is not supported.";
      Fehler[Russisch] = u8"не реализовано";
      Grund[Russisch] = u8"этот запрос не поддерживается.";
      http_Anfang (aus, 501, "Not Implemented");
      http_Methodenerlaubnis (aus);
      break;


    case Betriebsfehler: // 503
      Fehler[Deutsch] = u8"Außer Betrieb";
      Grund[Deutsch] = u8"dieses Angebot ist zur Zeit nicht verfügbar.";
      Fehler[Esperanto] = u8"Servo ne havebla";
      Grund[Esperanto] = u8"tiu servo dumtempe ne estas havebla.";
      Fehler[Englisch] = u8"Under Construction";
      Grund[Englisch] = u8"this service is currently not available.";
      Fehler[Russisch] = u8"сервис недоступен";
      Grund[Russisch] = u8"в настоящее время эта услуга недоступна.";
      http_Anfang (aus, 503, "Service Unavailable");
      // temporärer Zustand
      break;


    case Versionsfehler: // 505
      Fehler[Deutsch] = u8"Falsche HTTP-Version";
      Fehler[Esperanto] = u8"HTTP-eldono ne subtenata";
      Fehler[Englisch] = u8"HTTP Version Not Supported";
      Fehler[Russisch] = u8"версия HTTP не поддерживается";
      http_Anfang (aus, 505, "HTTP Version Not Supported");
      break;
    }
// *INDENT-ON*

  Fehlerseite (aus, Fehler, Grund, Ursache);
}


#define HTML_EINLEITUNG "\
<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"\r\n\
 \"http://www.w3.org/TR/html4/strict.dtd\">\r\n\r\n\
<html>\r\n"

static void
Fehlerseite (FILE *aus, const char *Fehlermeldung[], const char *Grund[],
	     const char *Ursache)
{
  FILE *t;
  char *Inhalt;
  size_t Inhaltsgroesse;

  t = open_memstream (&Inhalt, &Inhaltsgroesse);
  if (!t)
    _exit (1);

  fprintf (t, HTML_EINLEITUNG "<head>\r\n"
	   "<meta http-equiv='Content-Type'"
	   " content='text/html; charset=UTF-8'>\r\n"
	   "<title>%03hu: %s</title>\r\n"
	   "<meta name='viewport'"
	   " content='width=device-width, initial-scale=1'>\r\n"
	   "<style type='text/css'>\r\n",
	   http.Status, Fehlermeldung[Englisch]);
  fputs (FehlerStil_css, t);
  // Stylesheet nicht velinken; evtl. nicht erreichbar
  fputs ("</style>\r\n</head>\r\n\r\n<body>\r\n<div id='Fehler'>\r\n", t);

  char *zurueck = akfnetz_Kopfeintrag (http.Kopf, "Referer");	// sic
  if (zurueck)
    fprintf (t, "<div id='schliessen'><a href='%s'>" SCHLIESSEN
	     "</a></div>\r\n", zurueck);

  fprintf (t, "<h1>" TRAURIG " %03hu</h1>\r\n\r\n", http.Status);

  if (Ursache)
    fprintf (t, "<div class='Ursache'>%s</div>\r\n\r\n", Ursache);

  // hier Ausgabe-Reihenfolge festlegen
  Meldung (t, Deutsch, Fehlermeldung, Grund);
  Meldung (t, Englisch, Fehlermeldung, Grund);
  Meldung (t, Russisch, Fehlermeldung, Grund);
  Meldung (t, Esperanto, Fehlermeldung, Grund);

  if (Einstellung.Signatur && !http.onion)
    fputs ("<div id='Signatur'>\r\n<a href='"
	   AKFNETZ_HOMEPAGE "'>akfnetz</a>\r\n</div>\r\n", t);

  fputs ("</div>\r\n</body></html>\r\n", t);
  fclose (t);

  http_Kopf (aus, "Content-Type", "text/html; charset=UTF-8");
  http_ContentLength (aus, Inhaltsgroesse);
  http_Dauerverbindung (aus);
  http_Ende (aus);

  if (http.Methode != HEAD)
    fwrite (Inhalt, 1, Inhaltsgroesse, aus);

  free (Inhalt);
}


static void
Meldung (FILE *t, enum Sprache Sprache,
	 const char *Fehlermeldung[], const char *Grund[])
{
  const char *Sprachkennung, *Fehlerwort, *Fehlerbezeichnung;
  const char *Grundeinleitung;

  Fehlerbezeichnung = Fehlermeldung[Sprache];
  if (!Fehlerbezeichnung)
    Fehlerbezeichnung = "???";

  switch (Sprache)
    {
    // *INDENT-OFF*
    case Englisch:
      Sprachkennung = "en";
      Fehlerwort = u8"Error";
      Grundeinleitung = u8"Sorry, but";
      break;
    case Esperanto:
      Sprachkennung = "eo";
      Fehlerwort = u8"Eraro";
      Grundeinleitung = u8"Mi bedaŭras, sed";
      break;
    case Deutsch:
      Sprachkennung = "de";
      Fehlerwort = u8"Fehler";
      Grundeinleitung = u8"Tut mir leid, aber";
      break;
    case Russisch:
      Sprachkennung = "ru";
      Fehlerwort = u8"Ошибка";
      Grundeinleitung = u8"Мне очень жаль, но";
      break;
    default:
      Sprachkennung = "??";
      Fehlerwort = u8"Error";
      Grundeinleitung = NULL;
      break;
    // *INDENT-ON*
    }

  fprintf (t, "<div class='Meldung' lang='%.2s'>\r\n"
	   "<small class='lang'>%c%c</small>\r\n"
	   "<h2>%s: %s</h2>\r\n", Sprachkennung,
	   gross (Sprachkennung[0]), gross (Sprachkennung[1]),
	   Fehlerwort, Fehlerbezeichnung);

  if (Grundeinleitung && Grund[Sprache])
    fprintf (t, "<p>%s %s</p>\r\n", Grundeinleitung, Grund[Sprache]);

  fputs ("</div>\r\n\r\n", t);
}


/*
Bei Überlastung kann man nur etwas sehr einfaches machen.
Das wird eventuell aus dem Hauptprozess aufgerufen.
Die Meldung wird direkt ungepuffert in einem Stück ausgegeben
und danach die Verbindung geschlossen.
*/
extern void
http_ueberlastet (int v)
{
  // *INDENT-OFF*
  Text (v, PROTOKOLL " 503 Service Unavailable\r\n"
	"Connection: close\r\n"
	"Content-Type: text/html; charset=UTF-8\r\n\r\n"
	HTML_EINLEITUNG
	u8"<head>\r\n"
	"<meta http-equiv='Content-Type'"
	" content='text/html; charset=UTF-8'>\r\n"
	"<title>Service Unavailable</title>\r\n"
	"</head>\r\n<body>\r\n"
	"<div lang='en'>\r\n"
	"<h1>Service not available due to overload.</h1>\r\n"
	"<p>Please try again later.</p>\r\n"
	"</div>\r\n<hr/>\r\n"
	"<div lang='de'>\r\n"
	"<h1>Dienst wegen Überlastung nicht verfügbar.</h1>\r\n"
	"<p>Bitte später noch einmal versuchen.</p>\r\n"
	"</div>\r\n<hr/>\r\n"
	"<div lang='ru'>\r\n"
	"<h1>Услуга недоступна из-за перегрузки.</h1>\r\n"
	"<p>Пожалуйста, попробуйте еще раз позже.</p>\r\n"
	"</div>\r\n<hr/>\r\n"
	"<div lang='eo'>\r\n"
	"<h1>Servo ne havebla pro troŝarĝo.</h1>\r\n"
	"<p>Bonvolu provi denove poste.</p>\r\n"
	"</div>\r\n<hr/>\r\n"
	"</body></html>\r\n");
  // *INDENT-ON*
}


enum Zustand
http_sende_Fehlerstil (FILE *aus)
{
  http_Anfang (aus, 200, "OK");
  http_Kopf (aus, "Content-Type", "text/css");
  http_Kopf (aus, "Cache-Control", "max-age=86400");
  http_ContentLength (aus, sizeof (FehlerStil_css) - 1);
  enum Zustand f = http_Dauerverbindung (aus);
  http_Ende (aus);

  if (http.Methode != HEAD)
    fwrite (FehlerStil_css, 1, sizeof (FehlerStil_css) - 1, aus);

  return f;
}


static void
Autorisationsanfrage (FILE *aus)
{
  if (http.Version != 0)
    {
      char *s = Einstellung.Schutzbereich;

      if (!s)
	s = "/";

      fprintf (aus, "WWW-Authenticate: Basic realm=\"%s\"\r\n", s);

      if (redsam (2))
	akfnetz_Logbuch ("> WWW-Authenticate: Basic realm=\"%s\"\n", s);
    }
}
