/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2023-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef VERZEICHNIS_H
#define VERZEICHNIS_H

#include "Sprachen.h"

#define H_PFEIL_HOCH    u8"\u2191"
#define H_PFEIL_RUNTER  u8"\u2193"
#define H_PFEIL_LINKS   u8"\u2190"
#define H_PFEIL_RECHTS  u8"\u2192"
#define H_VERZEICHNIS   u8"\U0001F4C2"
#define SPRACHTRENNER   u8"\u00B7 "

// nicht mehr als 255 Bytes pro Dateiname
// laengere Dateinamen werden ignoriert
#if !defined(NAME_MAX) || NAME_MAX > 255
#undef NAME_MAX
#define NAME_MAX 255
#endif

// Reihenfolge
#define AUFSTEIGEND  1
#define ABSTEIGEND  -1

#define Grossbuchstabe(z) (('a'<=(z)&&(z)<='z')?(z)^0x20:(z))


struct Dateiinfo
{
  off_t Groesse;
  time_t Datum;
  mode_t Modus;
  char Name[NAME_MAX + 1];
};


struct Meldungen
{
  const char *Sprachname, *Verzeichnis, *vorheriges, *Nutzer,
    *Name, *Groesse, *Datum, *Datumsformat;
  const char Wochentagsabk[7][5], Sprachkennung[3];
};


/*
 * Das Datumsformat bitte nicht von der Locale abhaengig machen!
 * Hier sollte die Browsereinstellung entscheidend sein,
 * nicht die Systemeinstellung des Servers!
 *
 * Falls Wochentage nicht mit maximal 4 Bytes gespeichert werden
 * koennen, nicht definieren.
 */

// *INDENT-OFF*

static const struct Meldungen m[SPRACHANZAHL] = {
    [Englisch] = {
      .Sprachname = u8"English",
      .Verzeichnis = u8"directory",
      .vorheriges = u8"previous directory",
      .Nutzer = u8"User",
      .Name = u8"Name",
      .Groesse = u8"Size",
      .Datum = u8"Date",
      .Datumsformat = u8"%Y-%m-%d, %H:%M",  // ISO-8601 international
      .Wochentagsabk =
        {u8"Sun", u8"Mon", u8"Tue", u8"Wed", u8"Thu", u8"Fri", u8"Sat"},
      .Sprachkennung = "en"
    },

    [Esperanto] = {
      .Sprachname = u8"Esperanto",
      .Verzeichnis = u8"dosierujo",
      .vorheriges = u8"Supren al dosierujo en pli alta nivelo",
      .Nutzer = u8"Uzanto",
      .Name = u8"Nomo",
      .Groesse = u8"Grando",
      .Datum = u8"Dato",
      .Datumsformat = u8"%Y-%m-%d, %H:%M",  // ISO-8601 international
      .Wochentagsabk =
        {u8"Di", u8"Lu", u8"Ma", u8"Me", u8"Ĵa", u8"Ve", u8"Sa"},
      .Sprachkennung = "eo"
    },

    [Deutsch] = {
      .Sprachname = u8"Deutsch",
      .Verzeichnis = u8"Verzeichnis",
      .vorheriges = u8"vorheriges Verzeichnis",
      .Nutzer = u8"Benutzer",
      .Name = u8"Name",
      .Groesse = u8"Größe",
      .Datum = u8"Datum",
      .Datumsformat = u8"%d.%m.%Y, %k\u00A0Uhr\u00A0%M",
      .Wochentagsabk =
        {u8"So", u8"Mo", u8"Di", u8"Mi", u8"Do", u8"Fr", u8"Sa"},
      .Sprachkennung = "de"
    },

    [Russisch] = {
      .Sprachname = u8"Русский",
      .Verzeichnis = u8"каталог",
      .vorheriges = u8"родительский каталог",
      .Nutzer = u8"Пользователь",
      .Name = u8"Название",
      .Groesse = u8"Размер",
      .Datum = u8"Дата",
      .Datumsformat = u8"%d.%m.%Y, %H:%M",
      .Wochentagsabk =
        {u8"Вс", u8"Пн", u8"Вт", u8"Ср", u8"Чт", u8"Пт", u8"Сб"},
      .Sprachkennung = "ru"
    },

    [Franzoesisch] = {
      .Sprachname = u8"Français",
      .Verzeichnis = u8"dossier",
      .vorheriges = u8"dossier précédent",
      .Nutzer = u8"Utilisateur",
      .Name = u8"Nom",
      .Groesse = u8"Taille",
      .Datum = u8"Date",
      .Datumsformat = u8"%d/%m/%Y, %H:%M",
      .Wochentagsabk =
        {u8"Dim", u8"Lun", u8"Mar", u8"Mer", u8"Jeu", u8"Ven", u8"Sam"},
      .Sprachkennung = "fr"
    },

    [Italienisch] = {
      .Sprachname = u8"Italiano",
      .Verzeichnis = u8"directory",
      .vorheriges = u8"directory principale",
      .Nutzer = u8"Utente",
      .Name = u8"Nome",
      .Groesse = u8"Dimensioni",
      .Datum = u8"Data",
      .Datumsformat = u8"%d/%m/%Y, %H:%M",
      .Wochentagsabk =
        {u8"Dom", u8"Lun", u8"Mar", u8"Mer", u8"Gio", u8"Ven", u8"Sab"},
      .Sprachkennung = "it"
    },

    [Spanisch] = {
      .Sprachname = u8"Español",
      .Verzeichnis = u8"directorio",
      .vorheriges = u8"directorio principal",
      .Nutzer = u8"Usuario",
      .Name = u8"Nombre",
      .Groesse = u8"Tamaño",
      .Datum = u8"Fecha",
      .Datumsformat = u8"%d/%m/%Y, %H:%M",
      .Wochentagsabk =
        {u8"Do", u8"Lu", u8"Ma", u8"Mi", u8"Ju", u8"Vi", u8"Sa"},
      .Sprachkennung = "es"
    },

    [Tuerkisch] = {
      .Sprachname = u8"Türkçe",
      .Verzeichnis = u8"dizin",
      .vorheriges = u8"üst dizin",
      .Nutzer = u8"Kullanıcı",
      .Name = u8"Ad",
      .Groesse = u8"Boyut",
      .Datum = u8"Tarihi",
      .Datumsformat = u8"%d-%m-%Y, %H:%M",
      .Wochentagsabk = {},
      .Sprachkennung = "tr"
    },

    [Chinesisch] = {
      .Sprachname = u8"中文",
      .Verzeichnis = u8"目录",
      .vorheriges = u8"上级目录",
      .Nutzer = u8"用户",
      .Name = u8"名称",
      .Groesse = u8"大小",
      .Datum = u8"日期",
      .Datumsformat = u8"%Y\u5E74%m\u6708%d\u65E5 %H\u65F6%M\u5206",
      .Wochentagsabk = {u8"\u65E5", u8"\u4E00", u8"\u4E8C", u8"\u4E09",
                        u8"\u56DB", u8"\u4E94", u8"\u516D"},
      .Sprachkennung = "zh"
    }
};

#endif
