/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
Das Format der Anfrage wurde kompatibel zu Apache gewaehlt.

C= legt das Sortierfeld fest (column)
N: Name, S: Groesse (Size), M: Modifikationszeit

O= legt die Reihenfolge fest
A: aufsteigend (Ascending), D: absteigend (Descending)

Vorgabe: C=N;O=A

L= legt die Sprache fest

Nicht unterstuetzte Angaben werden ignoriert.
*/

#define _POSIX
#define _POSIX_C_SOURCE 200112L
#define _XOPEN_SOURCE 600

#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include "akfnetz.h"
#include "Verzeichnis.h"
#include "VerzStil.h"
#include "Sprachen.h"

static int_fast8_t Reihenfolge;

static inline void Kopf (FILE *, const char *, enum Sprache);
static inline void Anfang (FILE *, const char *, bool,
			   const char *, enum Sprache, const char *Nutzer);
static void Sprachauswahl (FILE *, enum Sprache, int, int);
static inline void Tabellenkopf (FILE *, enum Sprache, int, int);
static void Kopfeintrag (FILE *, int, const char *, int, int, enum Sprache);
static inline void Tabellenende (FILE *);
static inline void Fuss (FILE *);
static void Eintrag (FILE *, const struct Dateiinfo *,
		     const char *, enum Sprache);
static inline void Datum (FILE *, time_t, enum Sprache);
static int Namenssortierung (const void *, const void *);
static int Groessensortierung (const void *, const void *);
static int Datumsortierung (const void *, const void *);
static inline bool versteckt (const char *, size_t, size_t);
static long Listenerstellung (struct Dateiinfo **, const char *,
			      const char *);
static void Anfrageanhang (FILE *, const char *);
static inline int Anfrageauswertung (const char *);
static inline enum Sprache Sprachvereinbarung (const char *, const char *);
static inline bool ist_Verzeichnis (const struct Dateiinfo *);
static inline bool einzelnes_Verzeichnis (const struct Dateiinfo *,
					  const struct Dateiinfo *);

// Das Makro KEIN_INDEXER bezieht sich nur auf den Server
// Das ist hier also irrelevant

extern void
akfnetz_html_Verzeichnis (FILE *t,
			  const char *Verz, const char *Titel,
			  bool zurueck, const char *Anfrage,
			  const char *Sprachakzeptanz, const char *Nutzer)
{
  int Feld = Anfrageauswertung (Anfrage);

  // Liste erstellen
  struct Dateiinfo *Liste = NULL;
  long Anzahl = Listenerstellung (&Liste, Verz, Nutzer);
  // im Fehlerfall wird ein leeres Verzeichnis angezeigt

  int (*Vergleich) (const void *, const void *);
  switch (Feld)
    {
    case 'S':			// Size
      Vergleich = &Groessensortierung;
      break;

    case 'M':			// Modification time
      Vergleich = &Datumsortierung;
      break;

    case 'N':			// Name
    default:
      Vergleich = &Namenssortierung;
      break;
    }

  // Sortieren
  if (Anzahl > 0)
    qsort (Liste, Anzahl, sizeof (*Liste), Vergleich);

  enum Sprache Sprache = Sprachvereinbarung (Sprachakzeptanz, Anfrage);

  Kopf (t, Titel, Sprache);
  Sprachauswahl (t, Sprache, Feld, Reihenfolge);
  Anfang (t, Titel, zurueck, Anfrage, Sprache, Nutzer);

  if (Anzahl > 0)
    Tabellenkopf (t, Sprache, Feld, Reihenfolge);

  for (long i = 0; i < Anzahl; ++i)
    Eintrag (t, Liste + i, Anfrage, Sprache);

  free (Liste);

  if (Anzahl > 0)
    Tabellenende (t);

  Fuss (t);
}


#ifdef LATIN1

// Latin1 als Entitaeten ausgeben
static void
Namensanzeige (FILE *t, const char *s)
{
  while (*s)
    {
      unsigned char c = (unsigned char) *s;

      if (c >= 0x80 || strchr ("<>&", c))
	fprintf (t, "&#%d;", (int) c);
      else if (c > 0x1F)
	fputc (*s, t);

      ++s;
    }
}

#else // nicht LATIN1
#define Namensanzeige(t, Name)  akfnetz_xml_Text (t, Name)
#endif


static inline void
Kopf (FILE *t, const char *Titel, enum Sprache Sprache)
{
  fputs ("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"\r\n"
	 "  \"http://www.w3.org/TR/html4/strict.dtd\">\r\n\r\n", t);

  fprintf (t, "<html lang='%s'>\r\n\r\n<head>\r\n<title>%s</title>\r\n",
	   m[Sprache].Sprachkennung, Titel);

  // Das robots-Tag ist konventioneller gestaltet,
  // damit es nicht uebersehen wird.

  fputs ("<meta http-equiv='Content-Type'"
	 " content='text/html; charset=UTF-8'>\r\n"
	 "<meta name=\"robots\" content=\"noindex,nofollow\">\r\n"
	 "<meta name='viewport'"
	 " content='width=device-width, initial-scale=1'>\r\n"
	 "<style type='text/css'>\r\n", t);
  fputs (VerzStil_css, t);
  fputs ("</style>\r\n</head>\r\n\r\n<body>\r\n", t);
}


static inline void
Anfang (FILE *t, const char *Titel, bool zurueck,
	const char *Anfrage, enum Sprache Sprache, const char *Nutzer)
{
  if (Titel && *Titel)
    fprintf (t, "<h1>%s</h1>\r\n\r\n", Titel);

  if (Nutzer && *Nutzer)
    fprintf (t, "<p class='Nutzer'>%s: %s</p>\r\n\r\n",
	     m[Sprache].Nutzer, Nutzer);

  if (zurueck)
    {
      fputs ("<p class='nav'><a href='../", t);
      Anfrageanhang (t, Anfrage);
      fprintf (t, "'>" H_PFEIL_HOCH " %s</a></p>\r\n\r\n",
	       m[Sprache].vorheriges);
    }
}


static void
Sprachauswahl (FILE *t, enum Sprache Sprache, int Feld, int Reihenfolge)
{
  fputs ("<form action='.'>\r\n<div class='nav lang'>\r\n", t);

  // nicht Standardsortierung?
  if (Feld != 'N' || Reihenfolge == ABSTEIGEND)
    {
      fprintf (t, "<input type='hidden' name='C' value='%c'>\r\n", Feld);
      fprintf (t, "<input type='hidden' name='O' value='%c'>\r\n",
	       (Reihenfolge == ABSTEIGEND) ? 'D' : 'A');
    }

  fputs ("<select name='L'>\r\n", t);

  for (enum Sprache i = 0; i < SPRACHANZAHL; ++i)
    {
      const struct Meldungen *s = &m[i];

      fprintf (t, "<option value='%s' lang='%s'%s>%s</option>\r\n",
	       s->Sprachkennung, s->Sprachkennung,
	       (i == Sprache) ? " selected" : "", s->Sprachname);
    }

  fputs ("</select>\r\n<button type='submit'>" H_PFEIL_RECHTS "</button>\r\n"
	 "</div>\r\n</form>\r\n\r\n", t);
}


static inline void
Tabellenkopf (FILE *t, enum Sprache Sprache, int Feld, int Reihenfolge)
{
  fputs ("<table>\r\n<thead><tr>\r\n", t);
  Kopfeintrag (t, 'N', m[Sprache].Name, Feld, Reihenfolge, Sprache);
  Kopfeintrag (t, 'S', m[Sprache].Groesse, Feld, Reihenfolge, Sprache);
  Kopfeintrag (t, 'M', m[Sprache].Datum, Feld, Reihenfolge, Sprache);
  fputs ("</tr></thead>\r\n\r\n<tbody>", t);
}


static void
Kopfeintrag (FILE *t, int Spalte, const char *Name,
	     int Feld, int Reihenfolge, enum Sprache Sprache)
{
  int Reihenfolgezeichen;

  // die Buchstaben repraesentieren die die gegenteilige Anordnung
  if (Feld == Spalte && AUFSTEIGEND == Reihenfolge)
    Reihenfolgezeichen = 'D';
  else
    Reihenfolgezeichen = 'A';

  fprintf (t, "<th class='%c'><a href='?C=%c;O=%c;L=%s'>%s</a>",
	   Spalte, Spalte, Reihenfolgezeichen, m[Sprache].Sprachkennung,
	   Name);

  // die Pfeile repraesentieren die aktuelle Anordnung
  if (Feld == Spalte)
    {
      if (AUFSTEIGEND == Reihenfolge)
	fputs (" " H_PFEIL_RUNTER, t);
      else
	fputs (" " H_PFEIL_HOCH, t);
    }

  fputs ("</th>\r\n", t);
}


static inline void
Tabellenende (FILE *t)
{
  fputs ("\r\n</tbody></table>\r\n\r\n", t);
}


static inline void
Fuss (FILE *t)
{
  fputs ("</body></html>\r\n", t);
}


static void
Eintrag (FILE *t, const struct Dateiinfo *e,
	 const char *Anfrage, enum Sprache Sprache)
{
  bool Verz = ist_Verzeichnis (e);

  fputs ("<tr>\r\n<td class='N'><a href=\r\n'", t);
  akfnetz_urlkodiert (t, e->Name, "()");

  if (Verz)
    {
      fputc ('/', t);
      Anfrageanhang (t, Anfrage);
    }

  fputs ("'\r\n>", t);

  Namensanzeige (t, e->Name);

  if (Verz)
    fputc ('/', t);

  fputs ("</a></td>\r\n<td class='S'", t);

  if (Verz)
    {
      fprintf (t, ">" H_VERZEICHNIS " %s", m[Sprache].Verzeichnis);
    }
  else
    {
      char Groesse[10];
      akfnetz_Datengroessenstring (Groesse, sizeof (Groesse), e->Groesse);
      fprintf (t, " title='%ju Byte'>%s", (uintmax_t) e->Groesse, Groesse);
    }

  fputs ("</td>\r\n<td class='M'>", t);
  Datum (t, e->Datum, Sprache);
  fputs ("</td>\r\n</tr>", t);
}


static inline void
Datum (FILE *t, time_t d, enum Sprache Sprache)
{
  const char *Wochentag;
  struct tm z;
  char Zeit[80];

  gmtime_r (&d, &z);
  // localtime_r ist langsam,
  // ausserdem sollte die Zeitone des Servers irrelevant sein

  Wochentag = m[Sprache].Wochentagsabk[z.tm_wday];
  if (*Wochentag)
    fprintf (t, "%s, ", Wochentag);

  strftime (Zeit, sizeof (Zeit), m[Sprache].Datumsformat, &z);
  fputs (Zeit, t);
}


// ist es ein Verzeichniseintrag?
static inline bool
ist_Verzeichnis (const struct Dateiinfo *e)
{
  return ((e->Modus & S_IFMT) == S_IFDIR);
}


// ist eines ein Verzeichnis und das andere nicht?
static inline bool
einzelnes_Verzeichnis (const struct Dateiinfo *a, const struct Dateiinfo *b)
{
  // auf einzelnes Bit konzentrieren
  // Sockets und Blockgeraete sind bereits weggefiltert
  return (((a->Modus ^ b->Modus) & S_IFDIR) == S_IFDIR);
}


static int
Namenssortierung (const void *p1, const void *p2)
{
  const struct Dateiinfo *a = p1, *b = p2;

  if (einzelnes_Verzeichnis (a, b))
    return (ist_Verzeichnis (a) ? -1 : 1);

  return (strcasecmp (a->Name, b->Name) * Reihenfolge);
}


static int
Groessensortierung (const void *p1, const void *p2)
{
  const struct Dateiinfo *a = p1, *b = p2;

  if (einzelnes_Verzeichnis (a, b))
    return (ist_Verzeichnis (a) ? -1 : 1);

  if (a->Groesse == b->Groesse)
    return strcasecmp (a->Name, b->Name);

  // die Differenz koennte zu gross fuer int sein
  return ((a->Groesse > b->Groesse) ? Reihenfolge : -Reihenfolge);
}


static int
Datumsortierung (const void *p1, const void *p2)
{
  const struct Dateiinfo *a = p1, *b = p2;

  if (einzelnes_Verzeichnis (a, b))
    return (ist_Verzeichnis (a) ? -1 : 1);

  if (a->Datum == b->Datum)
    return strcasecmp (a->Name, b->Name);

  return ((a->Datum > b->Datum) ? Reihenfolge : -Reihenfolge);
}


// keine versteckten Dateien und keine Backups, CSS, JS
static inline bool
versteckt (const char *n, size_t l, size_t vl)
{
  return (l > NAME_MAX || n[0] == '.' || n[l - 1] == '~'
	  || (vl == 1
	      && (!strcmp (n, "cgi-bin")
		  || !strcmp (n, "favicon.ico")
		  || !strcmp (n, "robots.txt")))
	  || !strcmp (n, "lost+found")
	  || !strcmp (n, "gophermap")
	  || !strcmp (n, "gophertag")
	  || (l >= 4 && (!strcasecmp (".bak", n + l - 4)
			 || !strcasecmp (".css", n + l - 4)
			 || !strcasecmp (".js", n + l - 3))));
}


static long
Listenerstellung (struct Dateiinfo **Liste, const char *Verz,
		  const char *Nutzer)
{
  if (!Verz || !*Verz)
    Verz = ".";

  DIR *d = opendir (Verz);
  if (!d)
    return 0;

  size_t vl = strlen (Verz);
  if (Verz[vl - 1] == '/')
    --vl;

  char Pfad[vl + NAME_MAX + 2];
  memcpy (Pfad, Verz, vl);
  Pfad[vl] = '/';
  char *Name = Pfad + vl + 1;
  *Name = '\0';
  // der Name wird spaeter ausgewechselt

  long Anzahl = 0, Kapazitaet = 0;
  struct dirent *Eintrag;
  *Liste = NULL;

  while ((Eintrag = readdir (d)))
    {
      if (Anzahl >= Kapazitaet)
	{
	  Kapazitaet += 100;
	  void *nl = realloc (*Liste, Kapazitaet * sizeof (**Liste));

	  // im Fehlerfall ist zumindest das Vorhandene noch in Ordnung
	  if (!nl)
	    break;

	  *Liste = nl;
	}

      const char *n = Eintrag->d_name;
      size_t l = strlen (n);

      if (versteckt (n, l, vl))
	continue;

      // Heimatverzeichnis nur fuer richtigen Nutzer
      if (*n == '~' && (!Nutzer || strcmp (n + 1, Nutzer) != 0))
	continue;

      memcpy (Name, n, l + 1);	// veraendert Variable Pfad

      // nur regulaere Dateien und Verzeichnisse
      struct stat status;
      if (!stat (Pfad, &status)
	  && (S_ISREG (status.st_mode) || S_ISDIR (status.st_mode)))
	{
	  struct Dateiinfo *e = *Liste + Anzahl;

	  memcpy (e->Name, n, l + 1);
	  e->Groesse = S_ISDIR (status.st_mode) ? 0 : status.st_size;
	  e->Datum = status.st_mtime;
	  e->Modus = status.st_mode;
	  ++Anzahl;
	}
    }

  closedir (d);

  return Anzahl;
}


static void
Anfrageanhang (FILE *t, const char *Anfrage)
{
  if (!Anfrage || !*Anfrage)
    return;

  fputc ('?', t);

  // ohne '&' einfache Ausgabe
  if (!strchr (Anfrage, '&'))
    fputs (Anfrage, t);
  else
    {
      // '&' durch ';' ersetzen
      for (const char *p = Anfrage; *p; ++p)
	fputc ((*p != '&') ? *p : ';', t);
    }
}


static inline int
Anfrageauswertung (const char *Anfrage)
{
  int Feld = 'N';

  Reihenfolge = AUFSTEIGEND;

  if (Anfrage && *Anfrage)
    {
      // erstmal in Grossbuchstaben umwandeln
      size_t l = strlen (Anfrage);
      char a[l + 1];
      for (size_t i = 0; i <= l; ++i)
	a[i] = Grossbuchstabe (Anfrage[i]);

      char *c = strstr (a, "C=");
      if (c)
	Feld = (int) c[2];

      if (strstr (a, "O=D"))
	Reihenfolge = ABSTEIGEND;
    }

  return Feld;
}


static enum Sprache
Sprachvereinbarung (const char *Akzeptanz, const char *Anfrage)
{
  // erstmal Sprache in Anfrage suchen
  if (Anfrage && *Anfrage)
    {
      const char *c;

      c = strstr (Anfrage, "L=");
      if (!c)
	c = strstr (Anfrage, "l=");

      if (c)
	{
	  c += 2;
	  for (enum Sprache i = 0; i < SPRACHANZAHL; ++i)
	    if (!strncasecmp (c, m[i].Sprachkennung, 2))
	      return i;
	}
    }

  // dann Browsereinstellung untersuchen
  enum Sprache s;
  s = (enum Sprache) akfnetz_Sprachauswahl (Akzeptanz, SPRACHANGEBOT);
  if (s == unverstaendlich)
    s = Englisch;

  return s;
}
