/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015,2016,2018,2019,2024 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _WIN32

#define _POSIX_C_SOURCE 200112L

#include "akfnetz.h"
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>

static inline int
Fehler (void)
{
  errno = EHOSTUNREACH;
  return -1;
}


int
akfnetz_Serververbindung (const char *Hostname, const char *Port, int af)
{
  struct addrinfo Hinweise, *Info;

  // keine DNS-Anfragen fuer Onion-Adressen!
  if (!Hostname || !Port
      || !strcasecmp (".onion", Hostname + strlen (Hostname) - 6))
    return Fehler ();

  Info = NULL;
  memset (&Hinweise, 0, sizeof (Hinweise));
  Hinweise.ai_family = af;
  Hinweise.ai_socktype = SOCK_STREAM;
  Hinweise.ai_protocol = 0;
  Hinweise.ai_flags = AI_NUMERICSERV | AI_ADDRCONFIG;
  // Port muss nummerisch sein

  int r = getaddrinfo (Hostname, Port, &Hinweise, &Info);
  if (r != 0)
    return Fehler ();

  int sock = -1;
  struct addrinfo *p;
  for (p = Info; p; p = p->ai_next)
    {
      int r;
      struct timeval Auszeit;

      sock = socket (p->ai_family, p->ai_socktype, p->ai_protocol);
      if (sock < 0)
	continue;

      Auszeit.tv_sec = 7;
      Auszeit.tv_usec = 0;

      setsockopt (sock, SOL_SOCKET, SO_RCVTIMEO, &Auszeit, sizeof (Auszeit));
      setsockopt (sock, SOL_SOCKET, SO_SNDTIMEO, &Auszeit, sizeof (Auszeit));

      do
	r = connect (sock, p->ai_addr, p->ai_addrlen);
      while (r < 0 && errno == EINTR);

      if (r >= 0)
	break;			// Erfolg

      do
	r = close (sock);
      while (r < 0 && errno == EINTR);
    }

  freeaddrinfo (Info);

  // keine Verbindung, Socket geschlossen
  if (!p)
    return Fehler ();

  return sock;
}

#endif
