/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2016,2018,2022,2023 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "akfnetz.h"
#include <stdlib.h>
#include <string.h>

#define MINDESTKAPAZITAET 8

static int Kapazitaetserhoehung (struct akfnetz_Text *, size_t);


extern struct akfnetz_Text *
akfnetz_Texterstellung (struct akfnetz_Text *Text, size_t Anfangskapazitaet)
{
  if (!Text)
    return NULL;

  Text->Laenge = Text->reserviert = 0;
  Text->Inhalt = NULL;

  if (Anfangskapazitaet)
    {
      if (Anfangskapazitaet < MINDESTKAPAZITAET)
	Anfangskapazitaet = MINDESTKAPAZITAET;
      char *Inhalt = malloc (Anfangskapazitaet);
      if (!Inhalt)
	return NULL;
      *Inhalt = '\0';
      Text->Inhalt = Inhalt;
      Text->reserviert = Anfangskapazitaet;
    }

  return Text;
}


extern void
akfnetz_Textfreigabe (struct akfnetz_Text *Text)
{
  if (!Text)
    return;

  free (Text->Inhalt);
  Text->Inhalt = NULL;
  Text->Laenge = Text->reserviert = 0;
}


extern void
akfnetz_hinzufuegen (struct akfnetz_Text *Text, const char *neu, size_t l)
{
  if (!Text || !Text->reserviert || !neu || !*neu)
    return;

  if (l == 0)
    l = strlen (neu);

  size_t Laenge = Text->Laenge;
  size_t neue_Laenge = Laenge + l;

  // es muss immer ein Byte mehr reserviert sein fuer den String-Terminator
  if (neue_Laenge >= Text->reserviert
      && Kapazitaetserhoehung (Text, neue_Laenge) < 0)
    return;

  memcpy (Text->Inhalt + Laenge, neu, l);
  Text->Inhalt[neue_Laenge] = '\0';
  Text->Laenge = neue_Laenge;
}


extern void
akfnetz_Textbyte (struct akfnetz_Text *Text, int ch)
{
  char c = (char) ch;
  akfnetz_hinzufuegen (Text, &c, 1);
}


extern int
akfnetz_Textreservierung (struct akfnetz_Text *Text, size_t Laenge)
{
  if (!Text || Laenge < MINDESTKAPAZITAET || Laenge == Text->reserviert)
    return 0;

  void *neu = realloc (Text->Inhalt, Laenge);
  if (!neu)
    {
      akfnetz_Textfreigabe (Text);
      return -1;
    }

  Text->Inhalt = neu;
  Text->reserviert = Laenge;

  // eventuell Text abschneiden
  if (Text->Laenge >= Laenge)
    {
      size_t l = Laenge - 1;
      Text->Laenge = l;
      Text->Inhalt[l] = '\0';
    }

  return 0;
}


static int
Kapazitaetserhoehung (struct akfnetz_Text *Text, size_t neue_Laenge)
{
  size_t nr = Text->reserviert;

  if (nr < MINDESTKAPAZITAET)
    nr = MINDESTKAPAZITAET;

  while (neue_Laenge >= nr)
    nr *= 2;

  return akfnetz_Textreservierung (Text, nr);
}
