/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2019,2023-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Die Compiler-Warnungen, wenn Endungen zu lang sind,
 * koennen ignoriert werden. Die Funktion prueft dann nur die
 * ersten 8 Zeichen der Endung.
 */

#define _POSIX_C_SOURCE 200112L
#include "akfnetz.h"

#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <stdbool.h>


#ifndef TEXTKODIERUNG
#ifdef SYSTEMKODIERUNG
#define TEXTKODIERUNG SYSTEMKODIERUNG
#else
#define TEXTKODIERUNG "UTF-8"
#endif
#endif

// allgemeine Medientypen:
#define TEXT "text/plain; charset=" TEXTKODIERUNG
#define BIN "application/octet-stream"
#define SONSTIGES BIN
#define VERBOTEN ""		// den Inhalt nicht direkt ausliefern

#define MAX_ENDUNG 8

#define Kleinbuchstabe(z) (('A'<=(z)&&(z)<='Z')?(z)^0x20:(z))


struct Medieneintrag
{
  char Endung[MAX_ENDUNG];
  const char *Typ;
};

static const struct Medieneintrag Medientypen[] = {
#include "mime-types.h"
};

static const size_t Groesse = sizeof (Medientypen[0]);
static const size_t Anzahl = sizeof (Medientypen) / sizeof (Medientypen[0]);

// vergleicht Dateiendungen
// bei voller Laenge ist es in Ordnung, wenn kein 0-Byte folgt.
static int
Vergleich (const void *s1, const void *s2)
{
  const struct Medieneintrag *e1 = s1, *e2 = s2;

  return memcmp (e1->Endung, e2->Endung, MAX_ENDUNG);
}


static const char *
Endungssuche (const char *Endung)
{
  struct Medieneintrag Suche, *e;

  memset (&Suche, 0, sizeof (Suche));

  for (int i = 0; *Endung && i < MAX_ENDUNG; ++i, ++Endung)
    Suche.Endung[i] = Kleinbuchstabe (*Endung);

  e = bsearch (&Suche, Medientypen, Anzahl, Groesse, &Vergleich);

  if (!e)
    return SONSTIGES;

  return e->Typ;
}


extern const char *
akfnetz_Medientyp (const char *Pfad)
{
  const char *Name, *Endung;

  Name = strrchr (Pfad, '/');
  Name = Name ? Name + 1 : Pfad;
  // Punkt am Anfang evtl. ueberspringen
  Endung = strrchr ((*Name == '.') ? Name + 1 : Name, '.');

  if (Endung)
    return Endungssuche (Endung + 1);

  // bekannter Dateiname ohne Endung?
  if (!strncasecmp (Name, "README", 6) || !strncasecmp (Name, "LIESMICH", 8)
      || !strncmp (Name, "COPYING", 7) || !strcasecmp (Name, "LICENSE")
      || !strcasecmp (Name, "LICENSES") || !strcasecmp (Name, "NEWS")
      || !strcasecmp (Name, "TODO") || !strcasecmp (Name, "Makefile")
      || !strcasecmp (Name, "ChangeLog") || !strcasecmp (Name, "INSTALL")
      || !strcasecmp (Name, "md5") || !strcasecmp (Name, "md5sum")
      || !strcasecmp (Name, "sha1") || !strcasecmp (Name, "sha1sum"))
    return TEXT;

  if (!strcasecmp (Name, "core") || !strcasecmp (Name, "lost+found"))
    return VERBOTEN;

  return SONSTIGES;
}
