/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2019 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stdint.h>
#include <inttypes.h>		// strtoimax
#include <errno.h>

#include "akfnetz.h"

#define KOPFMAXIMUM 256


extern char **
akfnetz_Kopflesen (FILE * d)
{
  char **Kopf;
  char Zeile[8192];
  size_t Anzahl;

  if (!d)
    return NULL;

  Kopf = malloc ((KOPFMAXIMUM + 1) * sizeof (*Kopf));
  if (!Kopf)
    return NULL;

  Anzahl = 0;

  do
    {
      if (!fgets (Zeile, sizeof (Zeile), d))
	{
	  // Datenende vor Leerzeile
	  Kopf[Anzahl] = NULL;
	  akfnetz_Kopffreigabe (Kopf);
	  return NULL;
	}

      size_t Laenge = strcspn (Zeile, "\r\n");
      Zeile[Laenge] = '\0';

      // Fortsetzungszeile?
      if (Anzahl && (*Zeile == ' ' || *Zeile == '\t'))
	{
	  char *p1 = Kopf[Anzahl - 1];
	  char *p2 = Zeile;
	  while (*p2 == ' ' || *p2 == '\t')
	    ++p2;
	  size_t l1 = strlen (p1);
	  size_t l2 = strlen (p2);
	  char *n = realloc (p1, l1 + l2 + 2);
	  if (!n)
	    break;
	  n[l1] = ' ';
	  memcpy (n + l1 + 1, p2, l2 + 1);
	  Kopf[Anzahl - 1] = n;
	}
      else if (*Zeile && Anzahl < KOPFMAXIMUM)
	{
	  // TODO: weniger malloc
	  char *neu = malloc (Laenge + 1);
	  if (!neu)
	    break;

	  memcpy (neu, Zeile, Laenge + 1);
	  Kopf[Anzahl] = neu;
	  ++Anzahl;
	}
    }
  while (*Zeile);

  Kopf[Anzahl] = NULL;

  return Kopf;
}


extern void
akfnetz_Kopffreigabe (char **Kopf)
{
  if (!Kopf)
    return;

  for (size_t i = 0; Kopf[i]; ++i)
    free (Kopf[i]);

  free (Kopf);
}


extern char *
akfnetz_Kopfeintrag (char **Kopf, const char *Name)
{
  if (!Kopf || !Name || !*Name)
    return NULL;

  size_t Laenge = strlen (Name);

  for (int i = 0; Kopf[i]; ++i)
    {
      if (!strncasecmp (Kopf[i], Name, Laenge) && Kopf[i][Laenge] == ':')
	{
	  char *z = Kopf[i] + Laenge + 1;
	  while (*z == ' ' || *z == '\t')
	    ++z;

	  return z;
	}
    }

  return NULL;
}


extern intmax_t
akfnetz_Kopfzahl (char **Kopf, const char *Name)
{
  char *p = akfnetz_Kopfeintrag (Kopf, Name);

  if (!p)
    {
      errno = EINVAL;
      return INTMAX_MIN;
    }

  return strtoimax (p, NULL, 10);
}
