/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2015-2017 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#define _POSIX_C_SOURCE 200112L

#include "akfnetz.h"
#include <unistd.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#if _POSIX_MAPPED_FILES+0 > 0 || defined(__dietlibc__)
#include <sys/mman.h>
#else
#undef PROT_READ
#undef PROT_WRITE
#undef MAP_SHARED
#undef MAP_FAILED
#undef mmap
#undef munmap
#define PROT_READ 0
#define PROT_WRITE 0
#define MAP_SHARED 0
#define MAP_FAILED NULL
#define mmap(a,b,c,d,e,f)  MAP_FAILED
#define munmap(a,b) (-1)
#endif

#ifndef MAP_FILE
#define MAP_FILE 0
#endif


struct akfnetz_Feld *akfnetz_Formularfelder;
static char *Puffer;
static size_t Maplaenge;


extern char *
akfnetz_Formularfeld (const char *Name)
{
  if (!akfnetz_Formularfelder || !Name || !*Name)
    return NULL;

  for (int i = 0; akfnetz_Formularfelder[i].Feld; ++i)
    if (!strcmp (Name, akfnetz_Formularfelder[i].Feld))
      return akfnetz_Formularfelder[i].Wert;

  return NULL;
}


extern void
akfnetz_Formularfreigabe (void)
{
  free (akfnetz_Formularfelder);
  akfnetz_Formularfelder = NULL;

  if (Maplaenge)
    (void) munmap (Puffer, Maplaenge);
  else
    free (Puffer);

  Puffer = NULL;
  Maplaenge = 0;
}


// Erstellt Liste akfnetz_Formularfelder
// Der Puffer wird hierbei in Einzelteile zerlegt und dekodiert
static void
Feldvorbereitung (void)
{
  if (!strchr (Puffer, '='))
    return;

  // Feldanzahl feststellen
  int Feldanzahl = 1;
  char *p;
  for (p = Puffer; *p; ++p)
    if (*p == '&' || *p == ';')
      ++Feldanzahl;

  // Das Array speichert Zeiger auf den Puffer.
  akfnetz_Formularfelder =
    malloc ((Feldanzahl + 1) * sizeof (*akfnetz_Formularfelder));

  if (!akfnetz_Formularfelder)
    {
      akfnetz_Formularfreigabe ();
      return;
    }

  char *s;
  int i = 0;
  for (char *Feld = strtok_r (Puffer, "&;", &s); Feld && i < Feldanzahl;
       Feld = strtok_r (NULL, "&;", &s))
    {
      char *Wert = strchr (Feld, '=');
      if (!Wert)
	continue;

      *Wert++ = '\0';
      akfnetz_Formularfelder[i].Feld = akfnetz_url_dekodieren (Feld);
      akfnetz_Formularfelder[i].Wert = akfnetz_url_dekodieren (Wert);
      ++i;
    }

  akfnetz_Formularfelder[i].Feld = akfnetz_Formularfelder[i].Wert = NULL;
}


extern void
akfnetz_Formulardaten (const char *Daten)
{
  akfnetz_Formularfreigabe ();

  if (!Daten || !*Daten)
    return;

  size_t Laenge = strlen (Daten) + 1;

  Puffer = malloc (Laenge);
  if (!Puffer)
    return;

  memcpy (Puffer, Daten, Laenge);
  Feldvorbereitung ();
}


static int
einblenden (size_t Laenge)
{
  void *e;

  e = mmap (NULL, Laenge + 1, PROT_READ | PROT_WRITE, MAP_SHARED | MAP_FILE,
	    STDIN_FILENO, 0);

  if (e == MAP_FAILED)
    return -1;

  Maplaenge = Laenge + 1;
  Puffer = e;
  Feldvorbereitung ();

  return 0;
}


extern void
akfnetz_Formulardaten_einlesen (size_t Laenge)
{
  akfnetz_Formularfreigabe ();

  if (Laenge <= 0)
    return;

  if (!einblenden (Laenge))
    return;

  Puffer = malloc (Laenge + 1);

  if (!Puffer)
    return;

  ssize_t r;
  size_t l = 0;
  do
    {
      r = read (STDIN_FILENO, Puffer + l, Laenge - l);
      if (r >= 0)
	l += r;
    }
  while ((r == -1 && errno == EINTR) || (r >= 0 && l < Laenge - 1));

  Puffer[Laenge] = '\0';
  Feldvorbereitung ();
}
